# -*- coding: utf-8 -*-
# HiveMind - Distributed mind map editor for Maemo 5 platform
# Copyright (C) 2010-2011 HiveMind developers
#
# HiveMind is the legal property of its developers, whose names are
# noticed in  or  annotations at the beginning of each
# module or class.
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02111-1301  USA

'''Command classes for mind map modification'''

from PyQt4.QtGui import QUndoCommand
from hivemind.attribute import readable


class AddNodeCommand(QUndoCommand):
    '''Undo stack command for node addition action'''

    readable('parentNode', 'childNode', 'childIndex')

    def __init__(self, parentNode, childNode, index = None):
        '''
        @param parentNode: parent node for new to be added node
        @type parentNode: Node

        @param childNode: node to add to parent
        @type childNode: Node

        @param index: order number of the child node
        @type index: int
        '''
        QUndoCommand.__init__(self)
        self.__parentNode = parentNode
        self.__childNode = childNode
        self.__childIndex = index

    def redo(self):
        ''' Add created node to it parent '''
        self.__parentNode.addChild(self.__childNode, self.__childIndex)

    def undo(self):
        ''' Remove created node '''
        self.__parentNode.removeChild(self.__childNode)


class RemoveNodeCommand(QUndoCommand):
    '''Undo stack command for node removal'''

    readable('currentNode')

    def __init__(self, currentNode):
        '''
        @param currentNode: node designated to be removed
        @type currentNode: Node
        '''
        QUndoCommand.__init__(self)
        self.__currentNode = currentNode
        self.__parentNode = self.__currentNode.parent
        self.__nodePosition = self.__parentNode.children.index(self.__currentNode)

    def redo(self):
        '''Remove designated node from the scene'''
        self.__parentNode.removeChild(self.__currentNode)

    def undo(self):
        '''Restore node and its tree back to scene'''
        self.__parentNode.addChild(self.__currentNode, self.__nodePosition)


class MoveNodeCommand(QUndoCommand):
    '''Undo stack command for move node from current position to the new ancestor'''

    readable('currentNode', 'newParentNode', 'newPosition', 'newSide')

    def __init__(self, currentNode, newParentNode, newPosition, newSide):
        '''
        @param currentNode: node designated to moved
        @type currentNode: Node
        @param newParentNode: new parent for selected node
        @type newParentNode: Node
        @param newPosition: location relative to parent
        @type newPosition: NodeLocation
        @param newSide: left or right side
        @type newSide: bool
        '''
        QUndoCommand.__init__(self)
        self.__currentNode = currentNode
        self.__currentParent = self.__currentNode.parent
        self.__currentPosition = self.__currentParent.getChildPosition(self.__currentNode)
        self.__currentSide = self.__currentNode.left
        self.__newParentNode = newParentNode
        self.__newPosition = newPosition
        self.__newSide = newSide

    def redo(self):
        '''Move node from current position to the new ancestor'''
        self.__currentNode.setAttributes(left = self.__newSide)
        self.__currentNode.moveNode(self.__newParentNode, self.__newPosition)

    def undo(self):
        '''Restore old ancestor at current node'''
        self.__currentNode.setAttributes(left = self.__currentSide)
        self.__currentNode.moveNode(self.__currentParent, self.__currentPosition)


class StateModificationCommand(QUndoCommand):
    '''Undo stack command for common node attribute modifications'''

    readable('node', 'stateAfter', 'stateBefore')

    def __init__(self, node, before, after):
        '''
        @param node: node, witch parameters are changing
        @type node: Node
        @param before: state of the node before modification
        @type before: Hash
        @param after: state of the node after modification
        @type after: Hash
        '''
        QUndoCommand.__init__(self)
        self.__node = node
        self.__stateBefore = before
        self.__stateAfter = after

    def isFoldCommand(self):
        '''Check if command change fold attribute'''
        return 'folded' in self.__stateAfter.keys()

    def undo(self):
        '''Change node attributes to it original state'''
        self.__node.setAttributes(**self.__stateBefore)

    def redo(self):
        '''Change node attributes to it new state'''
        self.__node.setAttributes(**self.__stateAfter)


class SimpleCommand(object):
    '''Base class for commands that are not stored in undo stack'''
    pass


class ActionCommand(SimpleCommand):
    '''Stores various commands'''

    readable('action')

    def __init__(self, action):
        '''
        @param action: Action to be executed
        @type action: str
        '''
        SimpleCommand.__init__(self)
        self.__action = action


class CurrentNodeCommand(SimpleCommand):
    '''Stores current node'''

    readable('node')

    def __init__(self, node):
        '''
        @param node: Current node
        @type node: Node
        '''
        SimpleCommand.__init__(self)
        self.__node = node
