# -*- coding: utf-8 -*-
# HiveMind - Distributed mind map editor for Maemo 5 platform
# Copyright (C) 2011 HiveMind developers
#
# HiveMind is the legal property of its developers, whose names are
# noticed in  or  annotations at the beginning of each
# module or class.
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02111-1301  USA
'''
This module contains classes that interconnect with the smart conference systemz
'''

try:
    from smart_m3.Node import TCPConnector, ParticipantNode
    M3_AVAILABLE = True
except ImportError:
    M3_AVAILABLE = False

import random
import socket
import copy
import traceback
from datetime import datetime
from hivemind.attribute import readable
from hivemind.commands import AddNodeCommand, StateModificationCommand
from PyQt4.QtCore import QThread, pyqtSignal, QString, Qt
from PyQt4.QtGui import QProgressDialog

class SmartSpace(object):
    '''
    Class manages connection to the smart space of the SCS

    @author: Andrew Vasilev
    '''

    def __init__(self, name, host, port):
        '''
        @param name: name of the smart space
        @type name: str
        @param host: ip address of the smart space
        @type host: str
        @param port: port to connect to
        @type port: int
        '''
        object.__init__(self)
        self.__node = ParticipantNode('HiveMind-KP')
        self.__smartSpaceLocation = (name, (TCPConnector, (host, port)))
        self.__node.join(self.__smartSpaceLocation)
        self.__subscriptions = {}

    def close(self):
        '''Disconnect from the smart space, close all transactions'''
        for transaction in self.__subscriptions.values():
            self.__node.CloseSubscribeTransaction(transaction)
        self.__node.leave(self.__smartSpaceLocation)
        self.__subscriptions.clear()

    def addData(self, subject, predicate, target, targetType = 'uri'):
        '''
        Insert information into the smart space
        @param subject: subject, cannot be None
        @type subject: str
        @param predicate: the relationship between subject and target, cannot be None
        @type predicate: str
        @param target: cannot be None
        @type target: str
        @param targetType: the type of the target: 'literal' or 'uri' which is set by default
        @type targetType: str
        '''
        transaction = self.__node.CreateInsertTransaction(self.__smartSpaceLocation)
        transaction.send([((subject, predicate, target), 'uri', targetType)])
        self.__node.CloseInsertTransaction(transaction)

    def deleteData(self, subject, predicate, target, targetType = 'uri'):
        '''
        Remove information from the smart space
        @param subject: cannot be none
        @type subject: str
        @param predicate: relationship between subject and target, cannot be None
        @type predicate: str
        @param target: cannot be None
        @type target: str
        @param targetType: type of the target: 'literal' or 'uri' which is set by default
        @type targetType: str
        '''
        transaction = self.__node.CreateRemoveTransaction(self.__smartSpaceLocation)
        transaction.remove([((subject, predicate, target), 'uri', targetType)])
        self.__node.CloseRemoveTransaction(transaction)

    def query(self, subject = None, predicate = None, target = None, targetType = 'uri'):
        '''
        Make a query for an information from the smart space
        @param subject: a subject of a query, may be None
        @type subject: str
        @param predicate: predicate, may be None
        @type predicate: str
        @param target: target, may be None
        @type target: str
        @param targetType: type of the target, may be 'uri' or 'literal'
        @type targetType: str
        @return: get all the triples
        @rtype: tuple
        '''
        transaction = self.__node.CreateQueryTransaction(self.__smartSpaceLocation)
        result = transaction.rdf_query(((subject, predicate, target), targetType))
        self.__node.CloseQueryTransaction(transaction)
        return result

    def queryConnections(self, subject):
        '''
        Query all triples of the specified subject and create map out of it
        @type subject: str
        @return: key-value pairs
        @rtype: dict
        '''
        triples = self.query(subject, None, None)
        connections = {}
        for triple, _ in triples:
            connections[triple[1]] = triple[2]
        return connections

    def subscribe(self, handler, subject = None, predicate = None, target = None,
            targetType = 'literal'):
        '''
        @param handler: subscription handler
        @type handler: object
        @param subject: a subject of a query, may be None
        @type subject: str
        @param predicate: predicate, may be None
        @type predicate: str
        @param target: target, may be None
        @param targetType: type of target to look: uri or literal
        @type targetType: str
        @type target: str
        @rtype: triple
        '''
        transaction = self.__node.CreateSubscribeTransaction(self.__smartSpaceLocation)
        self.__subscriptions[handler] = transaction
        return transaction.subscribe_rdf([((subject, predicate, target), 'literal')], handler)


class NodeHolder(object):
    '''Simple class that stores an node readable attribute'''

    readable('node')

    def __init__(self, node):
        '''
        @type node: Node
        '''
        self.__node = node


class NodeRepresented(object):
    '''Base class for all classes that have nodes representing them'''

    readable('name', 'node', 'value', 'label')

    def __init__(self, parent, name, value, label = None):
        '''
        @param parent: some object having node attribute to bound to
        @type parent: NodeRepresented or NodeHolder
        @type name: str
        @type value: str
        @type showLabel: bool
        '''
        self.__parent = parent
        self.__node = None
        self.__children = []
        self.__externalChildren = []
        self.__modificationCommands = []
        self.__name = name
        self.__value = value
        self.__label = label

    def addChild(self, child):
        '''
        Add child to the node represented object
        @type child: NodeRepresenting
        '''
        self.__externalChildren.append(child)

    def getChildByName(self, name):
        '''
        Get attribute object by name
        @type name: str
        @rtype: Attribute or None
        '''
        for attribute in self.__children:
            if attribute.name == name:
                return attribute
        return None

    def _nodeId(self):
        '''
        @return: id of the stored node
        @rtype: str
        '''
        return self.node.getAttributes('id')['id']

    def modifyNode(self, modificationCommand):
        '''
        Record state modification command, if needed
        @type modificationCommand: IconModification
        '''
        self.__modificationCommands.append(modificationCommand)

    def synchronizeStatus(self, mindMap):
        '''
        Synchronize element status with mind map status
        @param mindMap: current mind map
        @type mindMap: MindMap
        @return: a bunch of commands to update mind map
        @rtype: tuple of QUndoCommands
        '''
        for node in self.__externalChildren:
            self.__children.append(node.copy(self))
        self.__externalChildren = []
        commands = []
        if not self.__node:
            self.__node = self.__parent.node.createChildPrototype()
        if not mindMap.getNodeById(self._nodeId()):
            commands.append(self._nodeCreationCommand())
        for modification in self.__modificationCommands:
            commands.append(modification.command(self.__node))
        self.__modificationCommands = []
        for attribute in self.__children:
            commands.extend(attribute.synchronizeStatus(mindMap))
        return commands

    def _setInitalStatus(self):
        '''Set initial status of the node, assign corresponding fields'''
        self.node.setAttributes(text = self.__value, left = self.__parent.node.left)
        if not self.__parent.node.left:
            self.node.setAttributes(left = False)
        if self.__label:
            self.node.setAttributes(labelText = self.__label)

    def _nodeCreationCommand(self):
        '''
        Get command needed to create current object and it children
        @rtype: tuple of QUndoCommand
        '''
        self._setInitalStatus()
        return AddNodeCommand(self.__parent.node, self.node)

    def copy(self, newParent):
        '''
        Get a copy of a current NodeRepresented
        @param newParent: new parent to bound to
        '''
        clone = NodeRepresented(newParent, copy.copy(self.__name), copy.copy(self.__value),
                copy.copy(self.__label))
        for child in self.__children:
            clone.addChild(child)
        for child in self.__externalChildren:
            clone.addChild(child)
        for modification in self.__modificationCommands:
            clone.modifyNode(copy.deepcopy(modification))
        return clone


class ComplexAttribute(NodeRepresented):
    '''Complex attribute, that is shown with the several nodes'''

    def __init__(self, parent, name, value, separator = ','):
        '''
        @type parent: NodeHolder
        @type name: str
        @type value: str
        '''
        NodeRepresented.__init__(self, parent, name, name)
        for subValue in value.split(separator):
            if not subValue or subValue == '': continue
            self.addChild(NodeRepresented(self, subValue, subValue))


class SmartSpaceChecker(QThread):
    '''Special class that searches for smart spaces running smart conference'''

    found = pyqtSignal(bool)

    def __init__(self, parent = None):
        '''
        @param parent: parent object of the thread
        @type parent: QObject
        '''
        QThread.__init__(self, parent)
        self.__host = self.__port = self.__name = None

    def check(self, host, port, name):
        '''
        Check for an existence of smart space of Smart Conference in specified location
        @param host: ip address of smart space
        @type host: str
        @type port: int
        @type name: str
        '''
        self.__host = host
        self.__port = port
        self.__name = name
        self.start()

    def run(self):
        '''Run the check itself, start thread by calling check method'''
        node = ParticipantNode('HiveMind-Locator' + str(random.randint(100, 100000)))
        space = (self.__name, (TCPConnector, (self.__host, self.__port)))
        try:
            node.join(space)
        except socket.error:
            self.found.emit(False)
            return
        node.leave(space)
        self.found.emit(True)


class VisitorChecker(object):
    '''Handler for smart space visitors'''

    def __init__(self, robot, rootNode, parentWindow):
        '''
        @param robot: smart conference robot, holds the collection of talks and connection
        @type robot: SmartConferenceRobot
        @param rootNode: root node of current mind map
        @type rootNode: Node
        @param parentWindow: parent window to show progress dialog on
        @type parentWindow: QMainWindow
        '''
        self.__robot = robot
        self.__nodeHolder = NodeHolder(rootNode)
        self._collectInformation(parentWindow)
        self.__robot.connection.subscribe(self, None, 'Title', None)

    def _collectInformation(self, parent):
        '''
        Collect information about all titles available in the smart space
        @param parent: parent window to show progress dialog on
        @type parent: QMainWindow
        '''
        progress = QProgressDialog(parent.tr('Gathering information from smart space'),
                QString(), 5, 15, parent)
        progress.setWindowTitle(parent.tr('Gathering information'))
        progress.setWindowModality(Qt.WindowModal)
        talkTriples = self.__robot.connection.query(None, 'Title', None)
        progress.setMaximum(len(talkTriples))
        newTalkList = []
        index = 1
        for triple, _ in talkTriples:
            progress.setValue(index)
            index += 1
            found = False
            for talk in self.__robot.talks:
                if talk.value == triple[2]:
                    newTalkList.append(talk)
                    found = True
                    print 'found'
                    break
            if found: continue
            talk = self._getTalkById(self.__nodeHolder, triple[0], triple[2])
            newTalkList.append(talk)
        self.__robot.talks = newTalkList
        self.__robot.event.set()

    def _getTalkById(self, nodeHolder, talkId, talkName):
        '''
        Get information about speaker from SCS and create corresponding object
        @type nodeHolder: NodeHolder
        @param talkId: id of the talk
        @type talkId: str
        @param talkName: name of the talk
        @type talkName: str
        '''
        talk = NodeRepresented(nodeHolder, talkId, talkName)
        speakerTriple = self.__robot.connection.query(None, 'presents', talkId, 'literal')
        talk.addChild(self._getSpeakerById(speakerTriple[0][0][0], talk))
        talkProperties = self.__robot.connection.queryConnections(talkId)
        if talkProperties.has_key('url'):
            talk.addChild(NodeRepresented(talk, 'url', talkProperties['url'], 'video'))
        if talkProperties.has_key('keywords'):
            talk.addChild(ComplexAttribute(talk, 'keywords', talkProperties['Keywords']))
        if talkProperties.has_key('duration'):
            duration = NodeRepresented(talk, 'duration', 'duration')
            duration.addChild(NodeRepresented(duration, 'proposed',
                    talkProperties['duration'] + 'm', 'proposed'))
        talk.addChild(duration)
        return talk

    def _getSpeakerById(self, speakerId, talk):
        '''
        Get information about speaker from SCS and create corresponding object
        @param speakerId: id of the speaker in the SCS
        @type speakerId: str
        @param talk: speaker talk on the conference
        @type talk: Talk
        @return: created object for the speaker
        @rtype: Speaker
        '''
        speakerProperties = self.__robot.connection.queryConnections(speakerId)
        propertiesMap = {'language' : 'language', 'phone' : 'phone', 'email' : 'e-mail',
                'photoURI' : 'photo'}
        speaker = NodeRepresented(talk, speakerId, speakerProperties.pop('is'))
        speaker.addChild(ComplexAttribute(speaker, 'interests',
                speakerProperties.pop('interests')))
        for key, value in speakerProperties.iteritems():
            if propertiesMap.has_key(key):
                speaker.addChild(NodeRepresented(speaker, key, value, propertiesMap[key]))
        return speaker

    def handle(self, added, removed):
        '''
        Handle all data about visitors is the space
        @parse added: tuples representing added data
        @type added: tuple
        @param removed: tuples representing removed data
        @type removed: tuple
        '''
        self.__robot.lock.acquire()
        try:
            for triple, _ in removed:
                for talk in self.__robot.talks:
                    if talk.name == triple[0]:
                        self.__robot.talks.remove(talk)
            for triple, _ in added:
                found = False
                for talk in self.__robot.talks:
                    if triple[0] == talk.name:
                        found = True
                        break
                if found:
                    continue
                talk = self._getTalkById(self.__nodeHolder, triple[0], triple[2])
                self.__robot.addTalk(talk)
        except:
            traceback.print_stack()
        self.__robot.lock.release()
        self.__robot.event.set()

class EventConverter(QThread):
    '''Special hack to handle python blocking events'''

    received = pyqtSignal()

    def __init__(self, event, parent = None):
        '''
        @type event: Event
        @param parent: parent object for the thread
        @type parent: QObject
        '''
        QThread.__init__(self, parent)
        self.__event = event
        self.terminated.connect(self.deleteLater)

    def run(self):
        '''Start the thread execution'''
        while True:
            self.__event.wait()
            self.received.emit()
            self.__event.clear()


class TalkChangeHandler(object):
    '''Handles the change of current presentation'''

    def __init__(self, robot):
        '''
        @type robot: SmartSpaceRobot
        '''
        self.__robot = robot
        self.__robot.connection.subscribe(self, 'TS', 'is', None)
        self.__started = None
        self.__ended = None

    def handle(self, added, removed):
        '''
        Handle all data about visitors is the space
        @parse added: tuples representing added data
        @type added: tuple
        @param removed: tuples representing removed data
        @type removed: tuple
        '''
        self.__robot.lock.acquire()
        try:
            for triple, _ in added:
                if triple[2] == 'over':
                    self.__robot.currentTalk.modifyNode(IconModification([]))
                    self.__ended = datetime.now()
                    self._updateDuration()
                    self.__robot.currentTalk = None
                    self.__started = self.__ended = None
                elif triple[2] != 'None':
                    result = self.__robot.connection.query(triple[2], 'is_occupied_by', None)
                    presenter = result[0][0][2]
                    for talk in self.__robot.talks:
                        if talk.getChildByName(presenter):
                            self.__robot.currentTalk = talk
                            break
                    self.__robot.currentTalk.modifyNode(IconModification(['bookmark']))
                    self.__started = datetime.now()
                    self._updateDuration()
        except:
            traceback.print_stack()
        self.__robot.lock.release()
        self.__robot.event.set()

    def _updateDuration(self):
        '''
        Update duration node on the mind map
        '''
        if not self.__robot.currentTalk.getChildByName('duration'):
            self.__robot.currentTalk.addChild(NodeRepresented(self.__robot.currentTalk,
                    'duration', 'duration'))
        duration = self.__robot.currentTalk.getChildByName('duration')
        if self.__started and not duration.getChildByName('started'):
            duration.addChild(NodeRepresented(duration, 'started', str(self.__started.hour) + \
                    ':' + str(self.__started.minute), 'started'))
        if self.__ended and not duration.getChildByName('ended'):
            duration.addChild(NodeRepresented(duration, 'ended', str(self.__ended.hour) + \
                    ':' + str(self.__ended.minute), 'ended'))
            if self.__started:
                overall = (self.__ended - self.__started).seconds / 60
                duration.addChild(NodeRepresented(duration, 'overall', str(overall) + 'm',
                        'overall'))


class IconModification(object):
    '''Class representing icon modification action'''

    def __init__(self, newIcon):
        '''
        @param newIcon: new icon list for the node
        @type newIcon: list of str
        '''
        self.__newIcon = newIcon

    def command(self, node):
        '''
        Get command for current modification
        @param node: node to change
        @type node: Node
        '''
        before = node.getAttributes('icons')
        return StateModificationCommand(node, before, {'icons':self.__newIcon})


class SlideChangeHandler(object):
    '''Class handles the duty of slide change'''

    def __init__(self, robot):
        '''
        @type robot: SmartConferenceRobot
        '''
        object.__init__(self)
        self.__robot = robot
        self.__robot.connection.subscribe(self, 'current_slide', 'is')
        self.__currentSlide = None

    def handle(self, added, removed):
        '''
        Handle all add and remove tuples
        '''
        self.__robot.lock.acquire()
        try:
            prevNumber = 1000000000
            for triple, _ in removed:
                prevNumber = int(triple[2])
            for triple, _ in added:
                number = triple[2]
                if self.__currentSlide:
                    currentSlide = self.__robot.currentTalk.getChildByName(self.__currentSlide)
                    if currentSlide:
                        currentSlide.modifyNode(IconModification([]))
                        if not currentSlide.getChildByName('ended') and \
                                int(number) > prevNumber:
                            now = datetime.now()
                            currentSlide.addChild(NodeRepresented(currentSlide, 'ended',
                                    str(now.hour) + ':' + str(now.minute), 'ended'))
                self.__currentSlide = number
                currentSlide = self.__robot.currentTalk.getChildByName(self.__currentSlide)
                if not currentSlide:
                    currentSlide = NodeRepresented(self.__robot.currentTalk, number,
                            'slide #' + number)
                    self.__robot.currentTalk.addChild(currentSlide)
                    now = datetime.now()
                    currentSlide.addChild(NodeRepresented(currentSlide, 'started',
                            str(now.hour) + ':' + str(now.minute), 'started'))
                currentSlide.modifyNode(IconModification(['flag-orange']))
        except:
            traceback.print_stack()
        self.__robot.lock.release()
        self.__robot.event.set()
