# -*- coding: utf-8 -*-
# HiveMind - Distributed mind map editor for Maemo 5 platform
# Copyright (C) 2010-2011 HiveMind developers
#
# HiveMind is the legal property of its developers, whose names are
# noticed in  or  annotations at the beginning of each
# module or class.
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02111-1301  USA
#pylint: disable=W0603

'''
Application settings manager

@author: Ilya Paramonov
'''

from PyQt4.QtCore import QSettings, QObject, SIGNAL, QFileInfo
from PyQt4.QtGui import QFont, QColor
import os
import hivemind.sysconfig as sysconfig

__INT_PARAMETERS = ('defaultEdgeWidth', 'autoHideNodeMenuTimeout',
        'showNodeMenuAnimationDuration', 'hideNodeMenuAnimationDuration',
        'transportPingInterval', 'transportPingMaxCount', 'protocolPingInterval',
        'protocolPingMaxCount', 'autosaveInterval', 'maxRecentFiles',
        'defaultZoomLevel', 'maxNodeTextWidth', 'defaultXMPPPort', 'sidebarAnimationDuration')
'''
Integer parameters names
'''

__FONT_PARAMETERS = ('defaultFont', 'defaultLabelFont')
'''
Font parameters names
'''

__COLOR_PARAMETERS = ('defaultNodeTextColor', 'defaultNodeBackgroundColor',
                      'defaultEdgeColor', 'defaultLabelColor',
                      'defaultSelectedNodeBackgroundColor')
'''
Font parameters names
'''

__BOOLEAN_PARAMETERS = ('confirmDelete', 'autosaveEnabled', 'readOnly', 'firstClick',
                        'useExtendedMenu', 'presentationMode')
'''
Boolean parameters names
'''

__HIDDEN_PARAMETERS = ('password')
'''
Hidden parameters name
'''

__LIST_PARAMETERS = ('listIcons', 'listSvgIcons', 'recentFiles')
'''
List parameters name
'''

__CACHED_ATTRIBUTES = {}
'''
Cached attributes
'''

def translationDir():
    '''Return directory where application translations reside'''
    return "%s/translations" % sysconfig.BASEDIR


def get(parameter):
    '''
    Get parameter by name
    
    @return: parameter value
    @rtype: according to parameter type
    '''
    if parameter in __CACHED_ATTRIBUTES:
        return __CACHED_ATTRIBUTES[parameter]
    value = __settings.value(parameter)
    if value is None and (parameter not in __HIDDEN_PARAMETERS):
        raise Exception('Invalid parameter %s' % parameter)
    if parameter in __INT_PARAMETERS:
        __CACHED_ATTRIBUTES[parameter] = value.toInt()[0]
    elif parameter in __FONT_PARAMETERS:
        font = QFont()
        font.fromString(value.toString())
        __CACHED_ATTRIBUTES[parameter] = font
    elif parameter in __LIST_PARAMETERS:
        parameterList = unicode(value.toString()).split(',')
        if '' in parameterList:
            parameterList.remove('')
        __CACHED_ATTRIBUTES[parameter] = parameterList
    elif parameter in __COLOR_PARAMETERS:
        __CACHED_ATTRIBUTES[parameter] = QColor(value.toString())
    elif parameter in __BOOLEAN_PARAMETERS:
        return value.toBool()
#        if value is True or value is False:
#            return value
#        __CACHED_ATTRIBUTES[parameter] = {'true': True, 'false': False}.get(value.lower())
    else:
        __CACHED_ATTRIBUTES[parameter] = unicode(value.toString())
    return __CACHED_ATTRIBUTES[parameter]


def set(**parameters): #pylint: disable=W0622
    '''
    Set parameters
    '''
    for paramName, paramValue in parameters.iteritems():
        if paramName in __FONT_PARAMETERS:
            variant = paramValue.toString()
        elif paramName in __COLOR_PARAMETERS:
            variant = paramValue.name()
        elif paramName in __LIST_PARAMETERS:
            listStr = ','.join(paramValue)
            variant = listStr
        else:
            variant = paramValue
        __CACHED_ATTRIBUTES[paramName] = paramValue
        __settings.setValue(paramName, variant)
        __signalizer.emit(SIGNAL('settingsChanged'))

def addRecentFile(fileName):
    '''
    Add fileName to the list of recent files
    @param fileName: name of the recent edited file
    @type fileName: str
    '''
    #pylint: disable=E1103
    recentFiles = get('recentFiles')
    if fileName is not None:
        fileName = unicode(QFileInfo(fileName).absoluteFilePath())
        if fileName in recentFiles:
            recentFiles.remove(fileName)
        recentFiles.insert(0, fileName)
        set(recentFiles = recentFiles[:get('maxRecentFiles')])

def connect(receiver):
    '''
    Subscribe receiver to be called when settings changes
    
    @param receiver: callable object
    '''
    QObject.connect(__signalizer, SIGNAL('settingsChanged'), receiver)

def _reinitialize(path):
    '''
    Reinitialize __settings object
    
    We use '/' as a path in order to prevent using of user scope
    configuration file in unit test mode
    
    @param path: path to the user scope configuration file
    '''
    __CACHED_ATTRIBUTES.clear()
    QSettings.setPath(__settingsFormat, QSettings.UserScope, path)
    global __settings
    __settings = QSettings(__settingsFormat, QSettings.UserScope, 'hivemind', 'hivemind')


# Initialization
if os.name == "nt":
    __settingsFormat = QSettings.IniFormat
else:
    __settingsFormat = QSettings.NativeFormat
QSettings.setPath(__settingsFormat, QSettings.SystemScope, sysconfig.CONFIGDIR)
__settings = None
'''Qt settings manager'''
__signalizer = QObject()
'''Object used for emitting signals when settings changed'''
if 'UNITTEST' in os.environ:
    _reinitialize('/') # in order to prevert reading/writing
    def reinitialize(path = '/'):
        '''Reinitialize __settings object for unit testing case'''
        _reinitialize(path)
else:
    _reinitialize('%s/.config' % os.path.expanduser('~'))
