# -*- coding: utf-8 -*-
# HiveMind - Distributed mind map editor for Maemo 5 platform
# Copyright (C) 2010-2011 HiveMind developers
#
# HiveMind is the legal property of its developers, whose names are
# noticed in @author or @authors annotations at the beginning of each
# module or class.
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02111-1301  USA

'''
Helper widgets
'''

import hivemind.settings as settings
import hivemind.resource as resource
import math
from PyQt4.QtGui import QGraphicsView, QTextEdit, QGraphicsWidget, \
QListWidgetItem, QListWidget, QGraphicsScene, QAbstractButton, QWidget, \
QAction, QUndoCommand, QLineEdit, QSpinBox, QCheckBox, QComboBox, QLabel, \
QToolButton, QIcon, QApplication, QFormLayout, QFontComboBox, QFontDatabase, \
QHBoxLayout, QGraphicsBlurEffect, QFrame, QPalette, QPushButton, \
QColorDialog, QPainter, QKeyEvent, QPixmap, QListView, QKeySequence, QToolBar, \
QFont
from hivemind.attribute import readable
from PyQt4.QtCore import QAbstractAnimation, QObject, SIGNAL, QLocale, Qt, \
QRect, QEvent, QRectF, QSize, QParallelAnimationGroup, QPropertyAnimation, \
QPointF, QRegExp, QSequentialAnimationGroup, QEasingCurve, QTimer, pyqtSignal
from hivemind.gui_delegates import NodeFinder


class SearchingTrMixin(object):
    '''
    Mixin, defining the substitution for QObject.tr() method which searches translation
    along the inheritance hierarchy

    @author: Ilya Paramonov
    '''

    def tr(self, text):
        '''
        Search the class and its ancestors for the translation of the given text
        '''
        for clazz in self.__class__.mro():
            translatedText = QApplication.translate(clazz.__name__, text)
            if translatedText != text: return translatedText
            if clazz.__name__ == 'SearchingTrMixin': return text
        return text


class ScalingWidget(QGraphicsView):
    '''
    Special widget for widgets scaling

    @author: Alexander Kulikov
    '''
    def __init__(self, widget, scale, parent = None):
        '''
        Initialize widget

        @type widget: QWidget
        @type scale: int
        @type parent: QWidget
        '''
        QGraphicsView.__init__(self, parent)
        self.__widget = widget
        self.__scene = QGraphicsScene()
        self.setScene(self.__scene)
        self.__scene.addWidget(widget)
        self.__scale = scale
        self.scale(self.__scale, self.__scale)
        self.setRenderHints(QPainter.Antialiasing)

    def resizeEvent(self, event):
        '''
        Resize event handler

        @type event: QResizeEvent
        '''
        width = event.size().width() / self.__scale
        height = event.size().height() / self.__scale
        self.__scene.setSceneRect(0, 0, width, height)
        self.__widget.resize(width, height)
        QGraphicsView.resizeEvent(self, event)


class SpecialTextEdit(QTextEdit):
    '''
    Special text edit widget which handles Enter as newline character insertion
    and emits accept event on Ctrl-Enter

    @author: Alexander Kulikov
    '''

    accept = pyqtSignal()

    def __init__(self, parent = None):
        '''
        Initialize widget
        @type parent: QWidget
        '''
        QTextEdit.__init__(self, parent)

    def keyPressEvent(self, event):
        '''
        Key press event handler
        @type event: QKeyEvent
        '''
        if event.key() == Qt.Key_Return or event.key() == Qt.Key_Enter:
            modifier = event.modifiers() & ~Qt.KeypadModifier
            if modifier == Qt.NoModifier:
                self.accept.emit()
                return
            if modifier == Qt.ControlModifier:
                event = QKeyEvent(QEvent.KeyPress, event.key(), Qt.NoModifier)
            QTextEdit.keyPressEvent(self, event)
        else:
            QTextEdit.keyPressEvent(self, event)


class IconItem(QGraphicsWidget):
    '''
    Graphics widget provides rendering icons

    @author: Alexander Kulikov
    '''

    readable('icon')

    def __init__(self, icon, size, iconSize, parent = None):
        '''
        Initialize widget with specified icon name
        @type icon: string
        @type size: int
        @type iconSize: int
        @type parent: QGraphicsItem
        '''
        QGraphicsWidget.__init__(self)
        self.__icon = icon
        self.__size = size
        self.__drawFrame = True
        self.__iconSize = iconSize
        self.resize(size, size)

    def setSize(self, size):
        '''Set item size'''
        self.__size = size

    def setIconSize(self, size):
        '''Set icon size'''
        self.__iconSize = size

    def paint(self, paint, options, widget):
        '''Paint icon item'''
        delta = (self.__size - self.__iconSize) / 2
        if isinstance(paint, QPainter):
            paint.setRenderHint(QPainter.Antialiasing)
            paint.setBrush(self.palette().brush(widget.backgroundRole()))
            paint.drawRoundedRect(0, 0, self.__size, self.__size, 6, 6)
            svgIcon = resource.getSvgIconByName(self.__icon)
            iconRect = QRectF(delta, delta, self.__iconSize, self.__iconSize)
            if svgIcon:
                svgIcon.render(paint, iconRect)
            else:
                paint.drawImage(iconRect, resource.getIconByName(self.__icon))


class ListWidgetItem(QListWidgetItem):
    '''
    List widget item for drawing icons in icon table
    @author: Alexander Kulikov
    '''

    readable('icon')

    def __init__(self, icon, parent = None):
        QListWidgetItem.__init__(self, parent)
        self.__icon = icon
        if resource.getSvgIconByName(icon):
            pixmap = QPixmap(32, 32)
            pixmap.fill(Qt.transparent)
            painter = QPainter()
            painter.begin(pixmap)
            painter.setRenderHint(QPainter.Antialiasing)
            resource.getSvgIconByName(icon).render(painter, QRectF(0, 0, 32, 32))
            painter.end()
        else:
            pixmap = QPixmap.fromImage(resource.getIconByName(icon).scaled(QSize(32, 32)))
        self.setIcon(QIcon(pixmap))

class IconTable(QListWidget):
    '''
    Icon table provides icon location in a grid
    '''
    def __init__(self, parent = None):
        '''
        Initialize icon table
        @type parent: QWidget
        '''
        QListWidget.__init__(self, parent)
        self.setViewMode(QListView.IconMode)
        self.setResizeMode(QListView.Adjust)
        self.setMovement(QListView.Static)
        self.setSelectionMode(QListView.NoSelection)
        for icon in settings.get("listIcons"):
            item = ListWidgetItem(icon)
            self.addItem(item)
        self.itemClicked[QListWidgetItem].connect(self._onClick)

    def _onClick(self, item):
        '''Emit iconClicked signal on item click'''
        self.emit(SIGNAL('iconClicked'), item.icon)


class IconList(QGraphicsScene):
    '''
    Icon list provides icon location in an animated list

    @author: Alexander Kulikov
    '''

    DEFAULT_HORIZONTAL_SPACING = 10
    '''Default horizontal space between icon items'''

    DEFAULT_ICON_SIZE = 32
    '''Default icon size into item'''

    DEFAULT_ITEM_SIZE = 64
    '''Default item size'''

    __MIN_DRAG_PATH = 10
    '''Minimum drag path length'''

    MAX_ICON_COUNT = 10
    '''Maximum icon count'''

    def __init__(self, icons, parent = None):
        '''
        Initialize with specified icon list
        @type icons: list
        @type parent: QObject
        '''
        QGraphicsScene.__init__(self, parent)
        self.__horizontalSpacing = self.DEFAULT_HORIZONTAL_SPACING
        self.__itemSize = self.DEFAULT_ITEM_SIZE
        self.__iconSize = self.DEFAULT_ICON_SIZE
        self._updateSize()
        self.__items = []
        self.__startPos = self.__draggedPath = 0
        self.__item = None
        for icon in icons:
            self.addIcon(icon)

    def _updateSize(self):
        '''Update scene size'''
        self.setSceneRect(0, 0, (self.__itemSize + self.__horizontalSpacing) * \
                           self.MAX_ICON_COUNT - self.__horizontalSpacing, \
                           self.__itemSize)

    def _updatePosition(self):
        '''Calculate new position for icon items'''
        for i in xrange(0, len(self.__items)):
            item = self.__items[i]
            if isinstance(item, IconItem):
                x = i * (self.__itemSize + self.__horizontalSpacing)
                item.setPos(x, 0)

    def setItemSize(self, itemSize):
        '''
        Set item size of current items
        @type itemSize: int
        '''
        self.__itemSize = itemSize
        self._updateSize()
        for item in self.items():
            if isinstance(item, IconItem):
                item.setSize(itemSize)
        self._updatePosition()

    def setHorizontalSpacing(self, spacing):
        '''
        Set horizontal spacing beetwing icon items
        @type spacing: int
        '''
        self.__horizontalSpacing = spacing
        self._updateSize()
        self._updatePosition()

    def setIconSize(self, iconSize):
        '''
        Set icon size of current items
        @type iconSize: int
        '''
        self.__iconSize = iconSize
        self._updateSize()
        for item in self.items():
            if isinstance(item, IconItem):
                item.setIconSize(iconSize)

    def addIcon(self, icon):
        '''
        Add new icon in the icon list
        @type icon: string
        '''
        index = len(self.__items)
        if (index < self.MAX_ICON_COUNT):
            item = IconItem(icon, self.__itemSize, self.__iconSize, self)
            item.setZValue(len(self.__items))
            x = index * (item.rect().width() + self.__horizontalSpacing)
            item.setPos(x, 0)
            self.__items.append(item)
            self.addItem(item)

    def removeIcon(self, item):
        '''
        Remove icon widget from the icon list
        @type item: IconWidget
        '''
        self.removeItem(item)
        self.__items.remove(item)
        item.setParentItem(None)

    def getIcons(self):
        '''
        Get icon name list
        @rtype: list
        '''
        return [item.icon for item in self.__items]

    def mousePressEvent(self, event):
        '''
        Mouse press event handler
        @type event: QMouseEvent
        '''
        QGraphicsScene.mousePressEvent(self, event)
        self.__startPos = event.screenPos()
        self.__item = self.itemAt(event.scenePos())
        if self.__item is not None:
            self.__item.setZValue(len(self.__items))
        self.__draggedPath = 0

    def mouseMoveEvent(self, event):
        '''
        Mouse move event handler
        @type event: QMouseEvent
        '''
        delta = event.screenPos() - self.__startPos
        self.__draggedPath = max(self.__draggedPath, delta.manhattanLength())
        if self.__item is not None:
            self._moveItem(self.__item, event.scenePos() - event.lastScenePos())

    def mouseReleaseEvent(self, event):
        '''
        Mouse release event handler
        @type event: QMouseEvent
        '''
        QGraphicsScene.mouseReleaseEvent(self, event)
        if self.__item is not None:
            self.__item.setZValue(self.__items.index(self.__item))
        if self.__draggedPath < self.__MIN_DRAG_PATH:
            item = self.itemAt(event.scenePos())
            if item is not None:
                self.removeIcon(item)
        self.__item = None
        self._generateAnimation()

    def _moveItem(self, item, delta):
        '''
        Move icon widget in the icon list and on the scene
        @type item: IconWidget
        @type delta: QPoint
        '''
        delta.setY(0)
        item.setPos(item.pos() + delta)
        index = self.__items.index(item)
        maxIndex = len(self.__items) - 1
        left = (index - 1) * (self.__itemSize + self.__horizontalSpacing)
        right = (index + 1) * (self.__itemSize + self.__horizontalSpacing)
        if index > 0 and item.pos().x() < left + self.__itemSize / 2:
            self.__items[index - 1], self.__items[index] = \
            self.__items[index], self.__items[index - 1]
            self._generateAnimation(item)
        elif index < maxIndex and item.pos().x() > right - self.__itemSize / 2:
            self.__items[index + 1], self.__items[index] = \
            self.__items[index], self.__items[index + 1]
            self._generateAnimation(item)

    def _generateAnimation(self, exception = None):
        '''
        Create and start move animation for icon widgets
        @param exception: Widget which does not require animation
        @type exception: QWidget
        '''
        group = QParallelAnimationGroup(self)
        for i in xrange(0, len(self.__items)):
            item = self.__items[i]
            if item == exception: continue
            left = i * (self.__itemSize + self.__horizontalSpacing)
            deltaX = left - item.pos().x()
            if deltaX != 0:
                animation = QPropertyAnimation(item, 'pos', self)
                animation.setDuration(abs(deltaX) * 3)
                animation.setStartValue(item.pos())
                animation.setEndValue(QPointF(left, 0))
                group.addAnimation(animation)
        group.start()


class ActionImageButton(QAbstractButton):
    '''
    Command widget that only shows the icon of the action and tightly assosiated with it.
    
    @author: Andrew Vasilev
    '''

    __MAX_ICON_HEIGHT = 64
    '''Maximum height and width of the icon'''

    def __init__(self, action, parent = None, sizeHint = None):
        '''
        @param action: action to bound state of the button
        @type action: QAction
        @param parent: parent widget to paint button on
        @type parent: QWidget
        @param sizeHint: maximum height\width of the icon to draw
        @type sizeHint: int
        '''
        QAbstractButton.__init__(self, parent)
        self.__action = action
        self._syncProperties()
        self._connectToAction()
        self._setupSizeHint(sizeHint)

    def _syncProperties(self):
        '''Synchronize action properties with the button properties and update the view'''
        self.setAutoRepeat(self.__action.autoRepeat())
        self.setCheckable(self.__action.isCheckable())
        self.setEnabled(self.__action.isEnabled())
        self.update()

    def _connectToAction(self):
        '''Create connections to the action state and property changes and enable acction
        trigger on button click'''
        self.__action.changed.connect(self._syncProperties)
        self.__action.toggled.connect(self.setChecked)
        self.clicked.connect(self.__action.trigger)

    def _setupSizeHint(self, sizeHint):
        '''
        Setup size hint attribute for the button. It cannot be greater than 
        ActionImageButton.__MAX_ICON_HEIGHT and the size of the icon.
        @param sizeHint: proposed height/width of the button
        @type sizeHint: int
        '''
        #pylint: disable=W0201
        defaultSize = ActionImageButton.__MAX_ICON_HEIGHT if sizeHint is None else sizeHint
        self.__sizeHint = QSize(defaultSize, defaultSize)
        newSize = self.__action.icon().actualSize(self.__sizeHint)
        if newSize.height() < self.__sizeHint.height():
            self.__sizeHint = newSize

    def sizeHint(self):
        '''
        @return: recomended size for the widget
        @rtype: QSize
        '''
        return self.__sizeHint

    def paintEvent(self, paintEvent):
        '''
        Paint action icon in the corresponding state
        @param paintEvent: paint event representation
        @type paintEvent: QPaintEvent 
        '''
        painter = QPainter(self)
        if not self.__action.isEnabled(): mode = QIcon.Disabled
        elif self.isDown() or self.__action.isChecked(): mode = QIcon.Selected
        else: mode = QIcon.Normal
        self.__action.icon().paint(painter, self.rect(), Qt.AlignCenter, mode)


class StatusAction(QAction):
    '''
    Action with support for statuses

    @author: Oleg Kandaurov
    '''
    readable('image')

    def __init__(self, text, imageBaseName, status, parent = None):
        '''
        Constructs action
        @param text: Text of the action
        @imageBaseName text: str
        @param imageBaseName: Type of the action
        @imageBaseName imageBaseName: str
        @param status: Initial status of the action
        @imageBaseName status: str
        @param parent: Parent
        @imageBaseName parent: QObject
        '''
        QAction.__init__(self, text, parent)
        SIGNAL('imageChanged(QImage)')
        SIGNAL('statusChanged(str)')
        self.__imageBaseName = imageBaseName
        self._setStatus(status)

    def _setStatus(self, status):
        '''
        Set status of the action and change icon according to status
        '''
        #pylint: disable=W0201
        self.__status = status
        self.__image = resource.getImage('%s_%s' % (self.__imageBaseName, self.__status))
        icon = QIcon()
        icon.addPixmap(self.__image, QIcon.Disabled)
        self.setIcon(icon)
        self.emit(SIGNAL('statusChanged'), self.__status)

    def _getStatus(self):
        '''
        Get status of the action
        '''
        return self.__status

    status = property(_getStatus, _setStatus, None, 'Status of the Action')


class WidgetVisibilityAnimation(QAbstractAnimation):
    '''
    Widget visibility changer for the animation framework
    
    @author: Andrew Vasilev
    '''

    def __init__(self, parent, visibility):
        '''
        Initialize animation
        
        @param parent: target widget for visibility change
        @type parent: QWidget
        
        @param visibility: the visibility value
        @param visibility: bool
        '''
        QAbstractAnimation.__init__(self, parent)
        self.__widget = parent
        self.__visibility = visibility

    def updateCurrentTime(self, currentTime):
        '''Update widget property at passed time'''
        if self.__visibility:
            self.__widget.show()
        else:
            self.__widget.hide()

    def duration(self):
        '''Animation is instant'''
        #pylint: disable = R0201
        return 0


class ActionModificationCommand(QUndoCommand):
    '''
    Undo stack common comand. Stores do and undo functions
    '''
    def __init__(self, do, undo):
        '''
        @param do: function that makes changes
        @param undo: function that withdarws changes
        '''
        QUndoCommand.__init__(self)
        self.__do = do
        self.__undo = undo

    def undo(self):
        '''Restore previous state by calling undo function'''
        self.__undo()

    def redo(self):
        '''Move to next state by calling do function'''
        self.__do()

class ColorLabel(QLabel):
    '''Label component that displays selected color'''

    def __init__(self, parent = None):
        QLabel.__init__(self, parent)
        self.__palette = self.palette()
        self.__effect = QGraphicsBlurEffect()
        self.__effect.setBlurRadius(2)
        self.setGraphicsEffect(self.__effect)
        self.setAlignment(Qt.AlignCenter)
        self.setAutoFillBackground(True)
        self.setFrameStyle(QFrame.StyledPanel | QFrame.Raised)

    def setColor(self, color):
        '''
        Set color to dispay on the label
        @type color: QColor
        '''
        self.__palette.setColor(QPalette.Window, color)
        self.setPalette(self.__palette)


class SettingsCreator(QObject):
    '''
    Create and layout application settings
    @author: Oleg Kandaurov
    '''
    def __init__(self, settings, parent = None):
        '''
        @param settings: Settings that are need to create
        @type settings: list
        @type parent: QWidget
        '''
        #pylint: disable=W0621
        QObject.__init__(self, parent)
        self.__settings = settings
        self.__mainLayout = QFormLayout()
        self.__component = dict()
        self._createDecorations()

    def getLayout(self):
        '''Return resulting layout'''
        return self.__mainLayout

    def _createDecorations(self):
        '''
        Create components
        '''
        for setting in self.__settings:
            getattr(self, '_create%s%s' % (setting[0].upper(), setting[1:]))()

    def retrieve(self):
        '''
        Retrieve settings
        '''
        newSettings = list()
        for setting in self.__settings:
            componentSetting = self.__component[setting].retrieve()
            for component, value in componentSetting.iteritems():
            # Append only changed settings due to performance issues
                if settings.get(component) != value:
                    newSettings.append({component:value})
        return newSettings

    def _createLocale(self):
        '''
        Create locale components
        '''
        languages = [{'English': QLocale(QLocale.English).name()},
                {'Russian': QLocale(QLocale.Russian).name()},
                {'Deutsch': QLocale(QLocale.German).name()}]
        binder = ComboBoxBinder('locale', settings.get('locale'), languages)
        self.__mainLayout.addRow(self.tr('Language'), binder)
        self.__component['locale'] = binder

    def _createDefaultZoomLevel(self):
        '''
        Create defaultZoomLevel components
        '''
        binder = SpinBoxBinder('defaultZoomLevel',
                settings.get('defaultZoomLevel'), -5, 5)
        self.__mainLayout.addRow(self.tr('Default zoom level'), binder)
        self.__component['defaultZoomLevel'] = binder

    def _createConfirmDelete(self):
        '''
        Create confirmDelete components
        '''
        binder = CheckBoxBinder('confirmDelete', settings.get('confirmDelete'))
        binder.setText(self.tr('Confirm node deletion'))
        self.__mainLayout.addRow(binder)
        self.__component['confirmDelete'] = binder

    def _createAutoHideNodeMenuTimeout(self):
        '''
        Create autoHideNodeMenuTimeout components
        '''
        binder = SpinBoxBinder('autoHideNodeMenuTimeout',
                settings.get('autoHideNodeMenuTimeout'), 0, 10000)
        self.__mainLayout.addRow(self.tr('Menu hide timeout'), binder)
        self.__component['autoHideNodeMenuTimeout'] = binder

    def _createShowNodeMenuAnimationDuration(self):
        '''
        Create showNodeMenuAnimationDuration components
        '''
        binder = SpinBoxBinder('showNodeMenuAnimationDuration',
                settings.get('showNodeMenuAnimationDuration'), 0, 10000)
        self.__mainLayout.addRow(self.tr('Menu appearance duration'), binder)
        self.__component['showNodeMenuAnimationDuration'] = binder

    def _createHideNodeMenuAnimationDuration(self):
        '''
        Create hideNodeMenuAnimationDuration components
        '''
        binder = SpinBoxBinder('hideNodeMenuAnimationDuration',
                settings.get('hideNodeMenuAnimationDuration'), 0, 10000)
        self.__mainLayout.addRow(self.tr('Menu disappearance duration'), binder)
        self.__component['hideNodeMenuAnimationDuration'] = binder

    def _createDefaultFont(self):
        '''
        Create defaultFont components
        '''
        binder = FontBinder('defaultFont', settings.get('defaultFont'))
        layout = QHBoxLayout()
        layout.addWidget(binder.fontCombo)
        layout.addWidget(binder.sizeCombo)
        self.__mainLayout.addRow(self.tr('Text font'), layout)
        self.__component['defaultFont'] = binder

    def _createDefaultLabelFont(self):
        '''
        Create defaultLabelFont components
        '''
        binder = FontBinder('defaultLabelFont',
                settings.get('defaultLabelFont'))
        layout = QHBoxLayout()
        layout.addWidget(binder.fontCombo)
        layout.addWidget(binder.sizeCombo)
        self.__mainLayout.addRow(self.tr('Label font'), layout)
        self.__component['defaultLabelFont'] = binder

    def _createDefaultNodeTextColor(self):
        '''
        Create defaultNodeTextColor components
        '''
        binder = ColorLabelBinder('defaultNodeTextColor',
                settings.get('defaultNodeTextColor'))
        binderButton = QPushButton(self.tr('Choose color'))
        layout = QHBoxLayout()
        layout.addWidget(binder, Qt.AlignLeft)
        layout.addWidget(binderButton, Qt.AlignLeft)
        self.__mainLayout.addRow(self.tr('Text color'), layout)
        binderButton.clicked.connect(binder.open)
        self.__component['defaultNodeTextColor'] = binder

    def _createDefaultNodeBackgroundColor(self):
        '''
        Create defaultNodeBackgroundColor components
        '''
        binder = ColorLabelBinder('defaultNodeBackgroundColor',
                settings.get('defaultNodeBackgroundColor'))
        binderButton = QPushButton(self.tr('Choose color'))
        layout = QHBoxLayout()
        layout.addWidget(binder, Qt.AlignLeft)
        layout.addWidget(binderButton, Qt.AlignLeft)
        self.__mainLayout.addRow(self.tr('Background color'), layout)
        binderButton.clicked.connect(binder.open)
        self.__component['defaultNodeBackgroundColor'] = binder

    def _createDefaultEdgeColor(self):
        '''
        Create defaultEdgeColor components
        '''
        binder = ColorLabelBinder('defaultEdgeColor',
                settings.get('defaultEdgeColor'))
        binderButton = QPushButton(self.tr('Choose color'))
        layout = QHBoxLayout()
        layout.addWidget(binder, Qt.AlignLeft)
        layout.addWidget(binderButton, Qt.AlignLeft)
        self.__mainLayout.addRow(self.tr('Edge color'), layout)
        binderButton.clicked.connect(binder.open)
        self.__component['defaultEdgeColor'] = binder

    def _createDefaultLabelColor(self):
        '''
        Create defaultLabelColor components
        '''
        binder = ColorLabelBinder('defaultLabelColor',
                settings.get('defaultLabelColor'))
        binderButton = QPushButton(self.tr('Choose color'))
        layout = QHBoxLayout()
        layout.addWidget(binder, Qt.AlignLeft)
        layout.addWidget(binderButton, Qt.AlignLeft)
        self.__mainLayout.addRow(self.tr('Label color'), layout)
        binderButton.clicked.connect(binder.open)
        self.__component['defaultLabelColor'] = binder

    def _createDefaultEdgeWidth(self):
        '''
        Create defaultEdgeWidth components
        '''
        width = [{'1': 1}, {'2': 2}, {'4': 4}, {'8': 8}, {self.tr('Thin'): 11}]
        binder = ComboBoxBinder('defaultEdgeWidth',
                settings.get('defaultEdgeWidth'), width)
        self.__mainLayout.addRow(self.tr('Edge width'), binder)
        self.__component['defaultEdgeWidth'] = binder

    def _createDefaultSelectedNodeBackgroundColor(self):
        '''
        Create defaultSelectedNodeBackgroundColor component
        '''
        binder = ColorLabelBinder('defaultSelectedNodeBackgroundColor',
                settings.get('defaultSelectedNodeBackgroundColor'))
        binderButton = QPushButton(self.tr('Choose color'))
        layout = QHBoxLayout()
        layout.addWidget(binder, Qt.AlignLeft)
        layout.addWidget(binderButton, Qt.AlignLeft)
        self.__mainLayout.addRow(self.tr('Selected node background'), layout)
        binderButton.clicked.connect(binder.open)
        self.__component['defaultSelectedNodeBackgroundColor'] = binder

    def _createAutosaveEnabled(self):
        '''
        Create autosaveEnabled component
        '''
        binder = CheckBoxBinder('autosaveEnabled', settings.get('autosaveEnabled'))
        binder.setText(self.tr('Autosave of mind map'))
        self.__mainLayout.addRow(binder)
        self.__component['autosaveEnabled'] = binder

    def _createAutosaveInterval(self):
        '''
        Create autosaveInterval component
        '''
        binder = SpinBoxBinder('autosaveInterval',
                settings.get('autosaveInterval'), 1, 600)
        binder.setSuffix(self.tr(' sec'))
        self.__mainLayout.addRow(self.tr('Autosave interval'), binder)
        self.__component['autosaveInterval'] = binder

    def _createUseExtendedMenu(self):
        '''
        Create useExtendedMenu component
        '''
        binder = CheckBoxBinder('useExtendedMenu', settings.get('useExtendedMenu'))
        binder.setText(self.tr('Extended menu'))
        self.__mainLayout.addRow(binder)
        self.__component['useExtendedMenu'] = binder

    def _createMaxNodeTextWidth(self):
        '''
        Create maxNodeTextWidth component
        '''
        binder = SpinBoxBinder('maxNodeTextWidth',
                settings.get('maxNodeTextWidth'), 50, 1000)
        binder.setSuffix(self.tr(' px'))
        self.__mainLayout.addRow(self.tr('Maximum width of node text'), binder)
        self.__component['maxNodeTextWidth'] = binder

    def _createAccessModel(self):
        '''Create accessModel component'''
        accessModels = [{self.tr('Open'): 'open'},
                {self.tr('Interactive'): 'authorize'},
                {self.tr('Contact list'): 'roster'},
                {self.tr('White list'): 'whitelist'}]
        binder = ComboBoxBinder('accessModel', settings.get('accessModel'), accessModels)
        self.__mainLayout.addRow(self.tr('Trust level'), binder)
        self.__component['accessModel'] = binder

    def _createDefaultAffiliation(self):
        '''Create defaultAffiliation component'''
        affiliations = [{self.tr('None'): 'outcast'},
                {self.tr('Restricted'): 'member'},
                {self.tr('Full'): 'publisher'},
                {self.tr('Owner'): 'owner'}]
        binder = ComboBoxBinder('defaultAffiliation', settings.get('defaultAffiliation'),
                affiliations)
        self.__mainLayout.addRow(self.tr('Default permission'), binder)
        self.__component['defaultAffiliation'] = binder

    def _createPublishModel(self):
        '''Create publishModel component'''
        publishModels = [{self.tr('Open'): 'subscribers'},
                {self.tr('Restricted'): 'publishers'}]
        binder = ComboBoxBinder('publishModel', settings.get('publishModel'), publishModels)
        self.__mainLayout.addRow(self.tr('Permission policy'), binder)
        self.__component['publishModel'] = binder

    def _createPresentationMode(self):
        '''Create presentationMode component'''
        binder = CheckBoxBinder('presentationMode', settings.get('presentationMode'))
        binder.setText(self.tr('Presentation mode'))
        self.__mainLayout.addRow(binder)
        self.__component['presentationMode'] = binder

    def _createShortcuts(self):
        '''
        Create shortcuts table component
        '''
        shortcuts = [('addNodeShortcut', self.tr('Add child node')),
                ('addSiblingNodeBelowShortcut', self.tr('Add sibling node below')),
                ('addSiblingNodeAboveShortcut', self.tr('Add sibling node above')),
                ('editNodeShortcut', self.tr('Edit node')),
                ('removeNodeShortcut', self.tr('Remove node')),
                ('editEdgeShortcut', self.tr('Edit edge')),
                ('editLabelShortcut', self.tr('Edit label')),
                ('editNodeIconsShortcut', self.tr('Edit node icons')),
                ('foldNodeShortcut', self.tr('Fold node')),
                ('editHyperlinkShortcut', self.tr('Edit nyperlink')),
                ('openHyperlinkShortcut', self.tr('Open nyperlink')),
                ('selectLeftNodeShortcut', self.tr('Select left node')),
                ('selectRightNodeShortcut', self.tr('Select right node')),
                ('selectUpNodeShortcut', self.tr('Select up node')),
                ('selectDownNodeShortcut', self.tr('Select down node')),
                ('selectRootNodeShortcut', self.tr('Select root node')),
                ('enterTransferModeShortcut', self.tr('Enter transfer mode')),
                ('cancelTransferModeShortcut', self.tr('Cancel transfer mode')),
                ('putNodeShortcut', self.tr('Put node')),
                ('putNodeBelowShortcut', self.tr('Put node below')),
                ('putNodeAboveShortcut', self.tr('Put node above'))]
        binder = ComponentBinderGroup()
        for shortcut in shortcuts:
            editor = ShortcutEditBinder(shortcut[0], settings.get(shortcut[0]))
            binder.addComponent(editor)
            self.__mainLayout.addRow(shortcut[1], editor)
        self.__component['shortcuts'] = binder


class ComponentBinder(object):
    '''
    Abstract component binder
    @author: Oleg Kandaurov
    '''
    def __init__(self, paramName, parent = None):
        '''
        Initialize binder
        @param paramName: Parameter that is binded to component
        @type paramName: str
        @type parent: QWidget
        '''
        self._value = None
        self._paramName = paramName

    def isValid(self, value):
        '''
        @return: whether value correct or not
        @rtype: bool
        '''
        #pylint: disable=R0201
        return True

    def retrieve(self):
        '''
        @return: pair parameter and value
        @rtype: dict
        '''
        if self.isValid(self._value):
            return {self._paramName: self._value}
        raise ValueError('Stored value is incorrect')

    def _getName(self):
        '''Retrieve name of parameter'''
        return self._paramName

    def _getValue(self):
        '''Retrieve current value'''
        return self._value

    def _setValue(self, value):
        '''
        Sets new value
        @param value: new value
        '''
        if self.isValid(value):
            self._value = value

    name = property(_getName, None, None, 'Parameter name')
    value = property(_getValue, None, None, 'Binder value')

class ComponentBinderGroup():
    '''
    Component binder collection
    '''
    def __init__(self, components = None):
        '''
        Initialize group
        @param components: list
        '''
        self.__collection = []

    def addComponent(self, component):
        '''
        Add component to collection
        @param component: ComponentBinder
        '''
        if not component in self.__collection:
            self.__collection.append(component)

    def removeComponent(self, component):
        '''
        Remove component from collection
        @param component: ComponentBinder
        '''
        self.__collection.remove(component)

    def retrieve(self):
        '''
        @return: pair parameter and value
        @rtype: dict
        '''
        params = {}
        for component in self.__collection:
            params.update(component.retrieve())
        return params


class LineEditBinder(QLineEdit, ComponentBinder):
    '''
    QLineEdit binder
    @author: Oleg Kandaurov
    '''
    def __init__(self, paramName, text, regexp = None, parent = None):
        '''
        Initialize binder
        @param paramName: Parameter that is binded to component
        @type paramName: str
        @param text: Initial text
        @type text: str
        @param regexp: Regular expression needed to inspect line edit values
        @type regexp: QRegExp
        @type parent: QWidget
        '''
        QLineEdit.__init__(self, parent)
        ComponentBinder.__init__(self, paramName, parent)
        self._value = text
        self.__regexp = QRegExp('.*') if regexp is None else regexp
        self.setText(text)
        self.textChanged.connect(self._onTextChanged)

    def _onTextChanged(self, value):
        '''
        Handler for modification of text in line edit.
        This handler converts QString to Python string in the field _value
        @type value: QString
        '''
        self._setValue(unicode(value))

    def isValid(self, value = None):
        '''
        @return: whether value correct or not
        @rtype: bool
        '''
        if value is None: value = self.text()
        if self.__regexp.exactMatch(value):
            return True
        return False


class ShortcutEditBinder(QLineEdit, ComponentBinder):
    '''
    QLineEdit binder for shortcut setup
    @author: Alexander Kulikov
    '''
    __NORMAL_STATE = 'normal'
    '''Show shortcut state'''

    __EDITING_STATE = 'editing'
    '''Edit shortcut state'''

    def __init__(self, paramName, text, parent = None):
        '''
        Initialize binder
        @param paramName: Parameter that is binded to component
        @type paramName: str
        @param text: Initial text
        @type text: str
        @type parent: QWidget
        '''
        QLineEdit.__init__(self, parent)
        ComponentBinder.__init__(self, paramName, parent)
        self._value = text
        self.setText(text)
        self.setReadOnly(True)
        self.__state = self.__NORMAL_STATE
        self.__oldShortcut = ''
        self.textChanged.connect(self._setValue)

    def mousePressEvent(self, event):
        '''
        Mouse press event handler
        @type event: QMouseEvent
        '''
        if event.buttons() & Qt.LeftButton:
            if self.__state == self.__NORMAL_STATE:
                self._enterEditState()
            else:
                self._setShortcut(self.__oldShortcut)

    def _enterEditState(self):
        '''Move shortcut binder into edit state'''
        if self.__state == self.__NORMAL_STATE:
            self.__state = self.__EDITING_STATE
            palette = QPalette(self.palette())
            palette.setColor(QPalette.Base, Qt.red)
            self.setPalette(palette)
            self.__oldShortcut = self.text()
            self.setText(self.tr('Enter new shortcut'))

    def _setShortcut(self, key):
        '''
        Set shortcut to input widget
        @type key: int or string
        '''
        self.setText(QKeySequence(key).toString())
        self.__state = self.__NORMAL_STATE
        self.setPalette(QPalette())
        self.selectAll()

    def keyPressEvent(self, event):
        '''
        Key press event handler
        @type event: QKeyEvent
        '''
        if self.__state == self.__NORMAL_STATE:
            return
        if event.key() == Qt.Key_Escape:
            self._setShortcut(self.__oldShortcut)
            return
        modifiers = int(event.modifiers())
        key = modifiers
        if event.key() not in [-1, Qt.Key_Control, Qt.Key_Alt, Qt.Key_AltGr,
                Qt.Key_Shift, Qt.Key_Meta]:
            key += int(event.key())
        if key > 0 and (key & (~modifiers) > 0):
            self._setShortcut(key)

    def focusOutEvent(self, event):
        '''
        Focus out event handler
        @type event: QFocusEvent
        '''
        QLineEdit.focusOutEvent(self, event)
        if self.__state == self.__EDITING_STATE:
            self._setShortcut(self.__oldShortcut)
        self.deselect()

    def isValid(self, value = None):
        '''
        @return: whether value correct or not (always True)
        @rtype: bool
        '''
        if value is None: value = self.text()
        return True


class SpinBoxBinder(QSpinBox, ComponentBinder):
    '''
    QSpinBox binder
    @author: Oleg Kandaurov
    '''

    def __init__(self, paramName, value, minValue, maxValue, parent = None):
        '''
        Initialize binder
        @param paramName: Parameter that is binded to component
        @type paramName: str
        @param value: Initial value
        @type value: int
        @param minValue: Minimum value
        @type minValue: int
        @param maxValue: Maximum value
        @type maxValue: int
        @type parent: Qwidget
        '''
        #pylint: disable=R0913
        QSpinBox.__init__(self, parent)
        ComponentBinder.__init__(self, paramName, parent)
        self.setMinimum(minValue)
        self.setMaximum(maxValue)
        self.setValue(value)
        self._value = value
        self.valueChanged.connect(self._setValue)


class CheckBoxBinder(QCheckBox, ComponentBinder):
    '''
    QCheckBox binder
    @author: Oleg Kandaurov
    '''
    def __init__(self, paramName, value, parent = None):
        '''
        Initialize binder
        @param paramName: Parameter that is binded to component
        @type paramName: str
        @param value: Initial state
        @type value: bool
        @type parent: Qwidget
        '''
        QCheckBox.__init__(self, parent)
        ComponentBinder.__init__(self, paramName, parent)
        self._value = value
        self.toggled.connect(self._setValue)
        self.setChecked(value)


class ComboBoxBinder(QComboBox, ComponentBinder):
    '''
    QComboBox binder
    @author: Oleg Kandaurov
    '''
    def __init__(self, paramName, value, data, parent = None):
        '''
        Initialize binder
        @param paramName: Parameter that is binded to component
        @type paramName: str
        @param value: Initial string
        @type value: str
        @param data: Possible values in QComboBox
        @type data: list of dicts
        @type parent: QWidget
        '''
        QComboBox.__init__(self, parent)
        ComponentBinder.__init__(self, paramName, parent)
        self.__data = data
        self._value = value
        for values in data:
            self.addItem(values.keys().pop(), values.values().pop())
        self.setCurrentIndex(self.findData(value))
        self.activated.connect(self._setValue)

    def _setValue(self, index):
        '''
        Sets new value
        @param value: index in combo box
        @type value: int
        '''
        self._value = self.__data[index].values().pop()


class ColorLabelBinder(ColorLabel, ComponentBinder):
    '''
    QColorDialog binder
    @author: Oleg Kandaurov
    '''
    def __init__(self, paramName, color, parent = None):
        '''
        Initialize binder
        @param paramName: Parameter that is binded to component
        @type paramName: str
        @param color: Initial color
        @type color: QColor
        @type parent: QWidget
        '''
        ColorLabel.__init__(self, parent)
        ComponentBinder.__init__(self, paramName, parent)
        self._setValue(color)
        self.setColor(color)

    def open(self):
        '''
        Open color dialog and sets color
        '''
        color = QColorDialog.getColor(self._value, self)
        self._setValue(color)

    def isValid(self, color):
        '''
        @return: whether color valid or not
        @rtype: bool
        '''
        return color.isValid()

    def _setValue(self, value):
        '''
        Sets new value
        @param value: new value
        '''
        if self.isValid(value):
            self.setColor(value)
            self._value = value


class FontBinder(QWidget, ComponentBinder):
    '''
    Font binder
    @author: Oleg Kandaurov
    '''
    readable('fontCombo', 'sizeCombo')
    def __init__(self, paramName, font, parent = None):
        '''
        Initialize binder
        @param paramName: Parameter that is binded to component
        @type paramName: str
        @param font: initial font
        @type font: QFont
        @type parent: QWidget
        '''
        QWidget.__init__(self, parent)
        ComponentBinder.__init__(self, paramName, parent)
        font = QFont(font)
        self._value = font
        self.__font = font
        self.__size = font.pointSize()
        self.__sizeCombo = QComboBox()
        self.__sizeCombo.addItem(str(self.__size))
        self.__sizeCombo.setEditable(True)
        self.__fontCombo = QFontComboBox()
        self.__fontCombo.setCurrentFont(font)
        self._setFont(font)
        self.__fontCombo.currentFontChanged.connect(self._setFont)
        self.__sizeCombo.currentIndexChanged.connect(self._setSize)

    def _setFont(self, font):
        '''
        Set font and update available sizes
        @type font: QFont
        '''
        self.__font = font
        self._fontSizes(font)
        self._setSizeIndex(self.__size)
        self.__font.setPointSize(self.__size)
        self._setValue(self.__font)

    def _setSize(self, index):
        '''
        Set size of font
        @type index: int
        '''
        if index == -1:
            self.__sizeCombo.addItem(str(self.__size))
            return
        self.__size = self.__sizeCombo.itemText(index).toInt()[0]
        self.__font.setPointSize(self.__size)
        self._setValue(self.__font)

    def _setSizeIndex(self, size):
        '''
        Set font size in QComboBox
        @type size: int
        '''
        index = self.__sizeCombo.findText(str(size), Qt.MatchExactly)
        if index == -1:
            self.__sizeCombo.addItem(str(size))
        else:
            self.__sizeCombo.setCurrentIndex(index)

    def _fontSizes(self, font):
        '''
        Add to QComboBox sizes supported by current font
        @type font: QFont
        '''
        self.__sizeCombo.clear()
        sizes = [str(value) for value in QFontDatabase().pointSizes(font.family())]
        if str(self.__size) not in sizes:
            sizes.append(str(self.__size))
        self.__sizeCombo.addItems(sizes)


class FullScreenExitButton(QToolButton):
    '''
    Full screen exit button for Maemo platform

    @author: Oleg Kandaurov
    '''
    def __init__(self, parent = None):
        '''
        @type parent: QWidget
        '''
        QToolButton.__init__(self, parent)
        self.setIcon(QIcon.fromTheme('general_fullsize'))
        self.setFixedSize(self.sizeHint())
        pal = self.palette()
        backgroundColor = pal.color(self.backgroundRole())
        backgroundColor.setAlpha(128)
        pal.setColor(self.backgroundRole(), backgroundColor)
        self.setPalette(pal)
        self.setAutoFillBackground(True)
        self.setVisible(False)
        self.clicked.connect(parent.showNormal)
        parent.installEventFilter(self)

    def eventFilter(self, obj, event):
        '''
        Catch Windows state changed and Resize events from parent object
        @type obj: QEvent
        @type event: QEvent
        '''
        parent = self.parentWidget()
        if obj != self.parent():
            return QToolButton.eventFilter(self, obj, event)
        isFullScreen = parent.windowState() & Qt.WindowFullScreen
        if event.type() == QEvent.WindowStateChange:
            if isFullScreen:
                self.setVisible(True)
                self.raise_()
            else:
                self.setVisible(False)
        elif event.type() == QEvent.Resize:
            if self.isVisible():
                self.move(parent.width() - self.width(), parent.height() - self.height())
        return QToolButton.eventFilter(self, obj, event)

class MaemoNodeMenu(QObject):
    '''
    Context menu for Maemo Platform

    @author: Andrew Vasilev, Alexander Kulikov
    '''
    readable('item')

    isShowed = pyqtSignal()

    def __init__(self, parent):
        '''
        @param parent: the main scene view to place menu on top of
        @type parent: MainView
        '''
        QObject.__init__(self, parent)
        self._mapView = parent
        self._menuButtons = {}
        self.__basePoint = None
        self.__showMenuAnimation = self.__hideMenuAnimation = None
        self.__showAnimationTransformation = None
        self.__hideAnimationTransformation = None
        self.__radiusOne = self.__radiusTwo = None
        self.__item = None
        parent.mapClicked.connect(self.hideMenu)
        self.__hideMenuTimer = QTimer(self)
        self.__hideMenuTimer.setInterval(settings.get('autoHideNodeMenuTimeout'))
        self.__hideMenuTimer.setSingleShot(True)
        self.__hideMenuTimer.timeout.connect(self.hideMenu)

    __INITIAL_ANGLE = math.pi * 1.2
    '''Start angle for button positioning'''

    def addAction(self, action, iconName):
        '''
        Add action to maemo menu
        @type action: QAction
        @type iconName: str
        '''
        action.setIcon(resource.getIcon(iconName))
        button = ActionImageButton(action, self._mapView)
        self._connectButtonToAction(button, action)

    def _connectButtonToAction(self, button, action):
        '''
        Link button with action
        @type button: QAbstractButton
        @type action: QAction
        '''
        self._menuButtons[action] = button
        button.hide()
        button.clicked.connect(self.hideMenu)

    def _restartHideMenuTimer(self):
        '''Restart hide menu timer'''
        self.__hideMenuTimer.stop()
        self.__hideMenuTimer.start()

    def _calculatePosition(self, angle):
        '''
        @param angle: angle of the point in radians
        @type angle: float
        
        @return: the point on ellipse, identified by angle
        @rtype: float tuple 
        '''
        x = self.__basePoint.x() + self.__radiusOne * \
                math.cos(angle + MaemoNodeMenu.__INITIAL_ANGLE)
        y = self.__basePoint.y() + self.__radiusTwo * \
                math.sin(angle + MaemoNodeMenu.__INITIAL_ANGLE)
        return x, y

    def _positionMenu(self):
        '''
        Position buttons on the view and start animation
        '''
        self._generateAnimations()
        step = 0.2
        angle = 0
        while angle < 2 * math.pi:
            if self._isButtonInside(*self._calculatePosition(angle)):
                angle = self._positionMenuButtons(angle)
            angle += step
        self.__showMenuAnimation.start()

    __ICON_SIZE = 80
    '''Icon size'''

    def _isButtonInside(self, x, y):
        '''
        Check whether button fits the view if placed on (x, y) coordinates
          
        @return: True if button fits on the view
        @rtype: bool
        '''
        buttonRect = QRect(x - self.__ICON_SIZE / 2, y - self.__ICON_SIZE / 2,
                self.__ICON_SIZE, self.__ICON_SIZE)
        viewRectangle = self._mapView.rect()
        return viewRectangle.contains(buttonRect)

    @classmethod
    def _distance(cls, pointOne, pointTwo):
        '''
        @param pointOne: first point coordinates
        @type pointOne: tuple
        
        @param pointTwo: second point coordinates
        @type pointTwo: tuple
        
        @return: distance between passed points
        @rtype: float 
        '''
        return math.sqrt((pointOne[0] - pointTwo[0]) ** 2
                         + (pointOne[1] - pointTwo[1]) ** 2)

    __MIN_DISTANCE = 84
    '''Minimal distance between buttons'''

    def _positionMenuButtons(self, startAngle):
        '''
        Position menu buttons starting from the passed angle
        
        @param startAngle: base angle for positioning in radian measurement 
        @type startAngle: float
        
        @return: 
        @rtype: float
        '''
        angle = startAngle
        angleStep = math.pi / 60
        basePoint = self.__basePoint
        centerRect = QRect(basePoint.x(), basePoint.y(), 0, 0)
        for action, button in self._menuButtons.iteritems():
            if action.isEnabled():
                curPos = self._calculatePosition(angle)
                endRect = QRect(curPos[0] - self.__ICON_SIZE / 2, curPos[1] - self.__ICON_SIZE\
                         / 2, self.__ICON_SIZE, self.__ICON_SIZE)
                showAnimation = self.__showAnimationTransformation[button]
                if button.isVisible():
                    showAnimation.setStartValue(button.geometry())
                else:
                    showAnimation.setStartValue(centerRect)
                    showAnimation.setKeyValueAt(0.65,
                            self._calculateTransitionalPosition(centerRect, endRect, 0.4))
                showAnimation.setEndValue(endRect)
                hideAnimation = self.__hideAnimationTransformation[button]
                hideAnimation.setStartValue(endRect)
                hideAnimation.setEndValue(centerRect)
                # calculate position for next button
                while True:
                    angle += angleStep
                    nextPos = self._calculatePosition(angle)
                    if self._distance(curPos, nextPos) >= MaemoNodeMenu.__MIN_DISTANCE:
                        if not self._isButtonInside(*nextPos):
                            return angle
                        break
        return math.pi * 2

    @classmethod
    def _calculateTransitionalPosition(cls, startPoint, endPoint, coef):
        '''
        Calculate transitional position for animation between
        startPoint and endPoint.
        
        @param startPoint: start point of animation
        @type startPoint: QRect
        
        @param endPoint: end point of animation
        @type endPoint: QRect
        
        @return: transition point to emulate twisting animation
        @rtype: QRect
        '''
        vectorX = endPoint.x() - startPoint.x()
        vectorY = endPoint.y() - startPoint.y()
        midPointX = vectorX * 0.65 + vectorY * coef + startPoint.x()
        midPointY = vectorY * 0.65 - vectorX * coef + startPoint.y()
        midPointWidth = math.fabs((endPoint.width() - startPoint.width()) / 2.0)
        midPointHeight = math.fabs((endPoint.height() - startPoint.height()) / 2.0)
        return QRect(midPointX, midPointY, midPointWidth, midPointHeight)

    def _generateAnimations(self):
        '''
        Create and set up show and hide animations for menu buttons 
        '''
        self.__showAnimationTransformation = dict()
        self.__hideAnimationTransformation = dict()
        self.__showMenuAnimation = QParallelAnimationGroup()
        self.__hideMenuAnimation = QParallelAnimationGroup()

        for action, button in self._menuButtons.iteritems():
            if button.isVisible() and not action.isEnabled():
                centrRect = QRect(self.__basePoint.x(), self.__basePoint.y(), 0, 0)
                hideAnimation = QSequentialAnimationGroup()

                hideTransformationAnimation = QPropertyAnimation(button, 'geometry')
                hideTransformationAnimation.setDuration(
                        settings.get('hideNodeMenuAnimationDuration'))
                hideTransformationAnimation.setEasingCurve(QEasingCurve.OutExpo)
                hideTransformationAnimation.setStartValue(button.geometry())
                hideTransformationAnimation.setEndValue(centrRect)

                hideVisibilityAnimation = WidgetVisibilityAnimation(button, False)

                hideAnimation.addAnimation(hideTransformationAnimation)
                hideAnimation.addAnimation(hideVisibilityAnimation)
                self.__showMenuAnimation.addAnimation(hideAnimation)

        for action, button in self._menuButtons.iteritems():
            if action.isEnabled():
                showAnimation = QSequentialAnimationGroup()

                showTransformationAnimation = QPropertyAnimation(button, 'geometry')
                showTransformationAnimation.setDuration(
                        settings.get('showNodeMenuAnimationDuration'))
                showTransformationAnimation.setEasingCurve(QEasingCurve.InSine)
                self.__showAnimationTransformation[button] = showTransformationAnimation

                showVisibilityAnimation = WidgetVisibilityAnimation(button, True)

                showAnimation.addAnimation(showVisibilityAnimation)
                showAnimation.addAnimation(showTransformationAnimation)
                self.__showMenuAnimation.addAnimation(showAnimation)

                hideAnimation = QSequentialAnimationGroup()

                hideTransformationAnimation = QPropertyAnimation(button, 'geometry')
                hideTransformationAnimation.setDuration(
                        settings.get('hideNodeMenuAnimationDuration'))
                hideTransformationAnimation.setEasingCurve(QEasingCurve.OutExpo)
                self.__hideAnimationTransformation[button] = hideTransformationAnimation

                hideVisibilityAnimation = WidgetVisibilityAnimation(button, False)

                hideAnimation.addAnimation(hideTransformationAnimation)
                hideAnimation.addAnimation(hideVisibilityAnimation)

                self.__hideMenuAnimation.addAnimation(hideAnimation)

    def _evaluateMenuBoundaries(self, item):
        '''
        @param item: selected item on the scene
        @type item: NodeDelegate
        '''
        rect = item.textRectOnScene()
        self._mapView.ensureVisible(rect, 120, 110)
        self.__basePoint = self._mapView.mapFromScene(rect.center())
        if rect.width() > 100:
            self.__radiusOne = rect.width() / 2.0 + 120
        else:
            self.__radiusOne = 105
        if rect.height() > 60:
            self.__radiusTwo = rect.height() / 2.0 + 100
        else:
            self.__radiusTwo = 80

    def show(self, item):
        '''
        Show main menu on top of selected item
        @param item: selected item on the scene
        @type item: NodeDelegate
        '''
        self.__item = item
        self.isShowed.emit()
        self._evaluateMenuBoundaries(item)
        self._positionMenu()
        self._restartHideMenuTimer()

    def hideMenu(self):
        '''Hide menu'''
        for button in self._menuButtons.values():
            if not button.isHidden():
                self.__hideMenuAnimation.start()
                return

    def _isActive(self):
        '''Whether menu active or not'''
        if self.__hideMenuTimer.isActive():
            return True
        return False

    active = property(_isActive, None, None, 'State of the menu')


class MaemoSidePanel(QObject):
    '''
    Side panel

    @author: Oleg Kandaurov
    '''

    '''Distance between buttons'''
    __DISTANCE = 24

    '''Size of all buttons in sidebar'''
    __SIZE = 64

    '''
    MainView always returns height for fullscreen mode. Therefore we can not retrieve
    screen bottom for normal mode/ I added hard coded vertical position of sidebar button
    '''
    __SIDEBAR_BUTTON_POSITION = 360

    def __init__(self, sidebarAction, parent):
        '''
        @param sidebarAction: Action for sidebar activation
        @type sidebarAction: QAction
        @type parent: MainView
        '''
        QObject.__init__(self, parent)
        self.__sidebarAction = sidebarAction
        self.__mapView = parent
        self.__buttons = []
        self.__position = {}
        self.__showAnimation = QParallelAnimationGroup()
        self.__hideAnimation = QParallelAnimationGroup()
        self.__sidebarButton = ActionImageButton(sidebarAction, self.__mapView)
        self.__sidebarButton.move(0, self.__SIDEBAR_BUTTON_POSITION)
        sidebarAction.triggered.connect(self._startAnimation)

    def _startAnimation(self):
        '''Start various animations according to sidebar status'''
        if self.__sidebarAction.status == 'closed':
            self._generateShowAnimation()
            self.__showAnimation.start()
        elif self.__sidebarAction.status == 'opened':
            self._generateHideAnimation()
            self.__hideAnimation.start()

    def _toggleSidebarStatus(self):
        '''Toggle sidebar button'''
        if self.__sidebarAction.status == 'closed':
            self.__sidebarAction.status = 'opened'
        elif self.__sidebarAction.status == 'opened':
            self.__sidebarAction.status = 'closed'

    def addAction(self, action):
        '''
        @type action: QAction
        '''
        button = ActionImageButton(action, self.__mapView)
        button.hide()
        self.__buttons.append(button)
        for pos, button in enumerate(self.__buttons):
            offset = (self.__SIZE + self.__DISTANCE) * pos
            self.__position[button] = {
                    'start': QRect(-self.__SIZE, offset, self.__SIZE, self.__SIZE),
                    'end': QRect(0, offset, self.__SIZE, self.__SIZE)}

    def _generateShowAnimation(self):
        '''Generate show animation'''
        self.__showAnimation = QParallelAnimationGroup()
        for button in self.__buttons:
            showMovementAnimation = QPropertyAnimation(button, 'geometry')
            showMovementAnimation.setDuration(settings.get('sidebarAnimationDuration'))
            showMovementAnimation.setEasingCurve(QEasingCurve.Linear)
            showMovementAnimation.setStartValue(self.__position[button]['start'])
            showMovementAnimation.setEndValue(self.__position[button]['end'])
            showAnimation = QSequentialAnimationGroup()
            showAnimation.addAnimation(WidgetVisibilityAnimation(button, True))
            showAnimation.addAnimation(showMovementAnimation)
            self.__showAnimation.addAnimation(showAnimation)
        showMovementAnimation = QPropertyAnimation(self.__sidebarButton, 'geometry')
        showMovementAnimation.setDuration(settings.get('sidebarAnimationDuration'))
        showMovementAnimation.setEasingCurve(QEasingCurve.Linear)
        showMovementAnimation.setStartValue(QRect(0, self.__SIDEBAR_BUTTON_POSITION,
                self.__SIZE, self.__SIZE))
        showMovementAnimation.setEndValue(QRect(self.__SIZE + self.__DISTANCE, 360,
                self.__SIZE, self.__SIZE))
        self.__showAnimation.addAnimation(showMovementAnimation)
        self.__showAnimation.finished.connect(self._toggleSidebarStatus)

    def _generateHideAnimation(self):
        '''Generate hide animation'''
        self.__hideAnimation = QParallelAnimationGroup()
        for button in self.__buttons:
            hideMovementAnimation = QPropertyAnimation(button, 'geometry')
            hideMovementAnimation.setDuration(settings.get('sidebarAnimationDuration'))
            hideMovementAnimation.setEasingCurve(QEasingCurve.Linear)
            hideMovementAnimation.setStartValue(self.__position[button]['end'])
            hideMovementAnimation.setEndValue(self.__position[button]['start'])
            hideAnimation = QSequentialAnimationGroup()
            hideAnimation.addAnimation(hideMovementAnimation)
            hideAnimation.addAnimation(WidgetVisibilityAnimation(button, False))
            self.__hideAnimation.addAnimation(hideAnimation)
        hideMovementAnimation = QPropertyAnimation(self.__sidebarButton, 'geometry')
        hideMovementAnimation.setDuration(settings.get('sidebarAnimationDuration'))
        hideMovementAnimation.setEasingCurve(QEasingCurve.Linear)
        hideMovementAnimation.setStartValue(QRect(self.__SIZE + self.__DISTANCE,
                self.__SIDEBAR_BUTTON_POSITION, self.__SIZE, self.__SIZE))
        hideMovementAnimation.setEndValue(QRect(0, 360, self.__SIZE, self.__SIZE))
        self.__hideAnimation.addAnimation(hideMovementAnimation)
        self.__hideAnimation.finished.connect(self._toggleSidebarStatus)


class FindToolBar(SearchingTrMixin, QToolBar):
    '''
    Abstract class for find toolBar
    '''
    def __init__(self, controller, parent = None):
        '''
        @type controller: MainWindowController
        @type parent: QWidget
        '''
        QToolBar.__init__(self, parent)
        SearchingTrMixin.__init__(self)
        self.setWindowTitle(self.tr('Find'))
        self._finder = NodeFinder(controller)

    def setDefaultFocus(self):
        '''
        Set focus on default item
        '''
        self._findLineEdit.setFocus()

    def _createComponents(self):
        '''
        Create components
        '''
        #pylint: disable = W0201
        self._findWidget = QWidget()
        self._findLabel = QLabel(self.tr('Find:'))
        self._findLineEdit = QLineEdit()
        self._highlightCheckBox = QCheckBox(self.tr("&Highlight all"))
        self._caseSensitiveCheckBox = QCheckBox(self.tr("Case &sensitive"))
        self._statusLabel = QLabel()
        self._highlightCheckBox.clicked.connect(self._finder.highlightNodes)
        self._caseSensitiveCheckBox.clicked.connect(self._updateFinder)
        self._findLineEdit.textEdited.connect(self._updateFinder)
        QObject.connect(self, SIGNAL('visibilityChanged(bool)'), self._updateFinder)
        self._finder.statusChanged.connect(self._updateStatusLabel)

    def _updateFinder(self):
        '''
        Update finder
        '''
        if self.isVisible():
            self._finder.findNodes(self._findLineEdit.text(),
                    self._caseSensitiveCheckBox.isChecked(),
                    self._highlightCheckBox.isChecked())
        else:
            self._finder.findNodes('')

    def _updateStatusLabel(self, status):
        '''
        Update status label
        @type status: str
        '''
        if status == 'none': self._statusLabel.setText('')
        if status == 'not_found': self._statusLabel.setText(self.tr('No found nodes'))
        if status == 'first': self._statusLabel.setText(self.tr('First found node'))
        if status == 'last': self._statusLabel.setText(self.tr('Last found node'))

