# -*- coding: utf-8 -*-
# HiveMind - Distributed mind map editor for Maemo 5 platform
# Copyright (C) 2011 HiveMind developers
#
# HiveMind is the legal property of its developers, whose names are
# noticed in  or  annotations at the beginning of each
# module or class.
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02111-1301  USA
'''
This module contains classes that interconnect with the smart conference systemz
'''

try:
    from smart_m3.Node import TCPConnector, ParticipantNode
    M3_AVAILABLE = True
except ImportError:
    M3_AVAILABLE = False

from hivemind.attribute import readable
from hivemind.commands import AddNodeCommand, StateModificationCommand

class SmartSpace(object):
    '''
    Class manages connection to the smart space of the SCS

    @author: Andrew Vasilev
    '''

    def __init__(self, name, host, port):
        '''
        @param name: name of the smart space
        @type name: str
        @param host: ip address of the smart space
        @type host: str
        @param port: port to connect to
        @type port: int
        '''
        self.__node = ParticipantNode('HiveMind-KP')
        self.__smartSpaceLocation = (name, (TCPConnector, (host, port)))
        self.__node.join(self.__smartSpaceLocation)

    def close(self):
        '''Disconnect from the smart space'''
        self.__node.leave(self.__smartSpaceLocation)

    def addData(self, subject, predicate, target, objectType = 'uri'):
        '''
        Insert information into the smart space
        @param subject: subject, cannot be None
        @type subject: str
        @param predicate: the relationship between subject and target, cannot be None
        @type predicate: str
        @param target: cannot be None
        @type target: str
        @param objectType: the type of the target: 'literal' or 'uri' which is set by default
        @type objectType: str
        '''
        transaction = self.__node.CreateInsertTransaction(self.__smartSpaceLocation)
        transaction.send([((subject, predicate, target), 'uri', objectType)])
        transaction.close()

    def deleteData(self, subject, predicate, target, objectType = 'uri'):
        '''
        Remove information from the smart space
        @param subject: cannot be none
        @type subject: str
        @param predicate: relationship between subject and target, cannot be None
        @type predicate: str
        @param target: cannot be None
        @type target: str
        @param objectType: type of the target: 'literal' or 'uri' which is set by default
        @type objectType: str
        '''
        transaction = self.__node.CreateRemoveTransaction(self.__smartSpaceLocation)
        transaction.remove([((subject, predicate, target), 'uri', objectType)])
        transaction.close()

    def query(self, subject, predicate, target, objectType = 'uri'):
        '''
        Make a query for an information from the smart space
        @param subject: a subject of a query, may be None
        @type subject: str
        @param predicate: predicate, may be None
        @type predicate: str
        @param target: target, may be None
        @type target: str
        @param objectType: type of the target, may be 'uri' or 'literal'
        @type objectType: str
        '''
        transaction = self.__node.CreateQueryTransaction(self.__smartSpaceLocation)
        result = transaction.rdf_query(((subject, predicate, target), objectType))
        transaction.close()
        return result


class NodeHolder(object):
    '''Simple class that stores an node readable attribute'''

    readable('node')

    def __init__(self, node):
        '''
        Add ne
        @type node: Node
        '''
        self.__node = node


class NodeRepresented(NodeHolder):
    '''Base class for all classes that have nodes representing them'''

    def __init__(self, parent):
        '''
        @param parent: some object having node attribute to bound to
        @type parent: NodeRepresented
        '''
        self.__parent = parent
        NodeHolder.__init__(self, self.__parent.node.createChildPrototype())
        self.__attributes = []

    def addAttribute(self, attribute):
        '''
        Add attribute to the attribute list
        @type attribute: Attribute
        '''
        self.__attributes.append(attribute)

    def getAttributeByName(self, name):
        '''
        Get attribute object by name
        @type name: str
        @rtype: Attribute or None
        '''
        for attribute in self.__attributes:
            if attribute.name == name:
                return attribute
        return None

    def nodeCreationCommand(self):
        '''
        Get command needed to create current object and it children
        @rtype: tuple of QUndoCommand
        '''
        commands = [AddNodeCommand(self.__parent.node, self.node)]
        for attribute in self.__attributes:
            commands.extend(attribute.nodeCreationCommand())
        return commands


class Attribute(NodeRepresented):
    '''Class for representing single attribute of the element'''

    readable('name')

    def __init__(self, parent, name, value, showLabel = True):
        '''
        @param parent: parent NodeRepresented object
        @type parent: NodeRepresented
        @type name: str
        @type value: str
        @type label: str
        @type showLabel: bool
        '''
        NodeRepresented.__init__(self, parent)
        self.__name = name
        self.__value = value
        self.__showLabel = showLabel
        self.node.setAttributes(text = self.__value)
        if self.__showLabel:
            self.node.setAttributes(labelText = self.__name)

    def updateNodeCommand(self):
        '''
        Get command to update node to current status
        @rtype: StateModificationCommand
        '''
        previousAttributes = self.node.getAttributes(('text', 'labelText'))
        nextAttributes = {'text' : self.__value, 'labelText' : self.__label}
        return StateModificationCommand(self.node, previousAttributes, nextAttributes)


class ComplexAttribute(Attribute):
    '''Complex attribute, that is shown with the several nodes'''

    def __init__(self, parent, name, value, separator = ','):
        '''
        @type parent: NodeHolder
        @type name: str
        @type value: str
        '''
        Attribute.__init__(self, parent, value, name, False)
        for subValue in value.split(separator):
            if not subValue or subValue == '': continue
            self.addAttribute(Attribute(self, None, subValue, False))


class Talk(NodeRepresented):
    '''Class, representing talk on the conference'''

    def __init__(self, parent, handle, title):
        '''
        @type handle: unique handle of the talk in the smart conference
        @param handle: str
        @type title: str
        @param parent: parent object to bound to
        @type parent: NodeRepresented
        '''
        NodeRepresented.__init__(self, parent)
        self.__handle = handle
        self.__title = title
        self.__speaker = None
        self.node.setAttributes(text = self.__title, left = False)

    def nodeCreationCommand(self):
        '''
        Get the command needed to create the corresponding node on the mind map
        @rtype: list of AddNodeCommand
        '''
        commands = NodeRepresented.nodeCreationCommand(self)
        commands.extend(self.__speaker.nodeCreationCommand())
        return commands

    def _getSpeaker(self):
        '''Get speaker for the talk'''
        return self.__speaker

    def _setSpeaker(self, speaker):
        '''Set new speaker for the conference'''
        self.__speaker = speaker

    speaker = property(_getSpeaker, _setSpeaker, None, None)

    def __str__(self):
        '''String representation of the talk'''
        return self.__handle + ': ' + self.__title


class Speaker(NodeRepresented):
    '''Speaker on the conference'''

    def __init__(self, handle, name, talk):
        '''
        Initialize speaker class
        @param handle: handle of the speaker in the smart conference system
        @type handle: str
        @param name: name of the speaker
        @type name: str
        @param talk: corresponding talk on the conference
        @type talk: Talk
        '''
        NodeRepresented.__init__(self, talk)
        self.__handle = handle
        self.__name = name
        self.node.setAttributes(text = self.__name, labelText = 'speaker')

    def __str__(self):
        '''String representation of the speaker'''
        return self.__name

