# -*- coding: utf-8 -*-
# HiveMind - Distributed mind map editor for Maemo 5 platform
# Copyright (C) 2011 HiveMind developers
#
# HiveMind is the legal property of its developers, whose names are
# noticed in  or  annotations at the beginning of each
# module or class.
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02111-1301  USA
'''
Module contains special robots for filling mind map with external data in
'''

from hivemind.smartconference.kp import M3_AVAILABLE, SmartSpace, Talk, Speaker, Attribute, \
    NodeHolder, ComplexAttribute

class SmartConferenceRobot(object):
    '''
    Creates specialized data from the smart conference system. This robot may work for a long
    time.

    @author: Andrew Vasilev
    '''
    def __init__(self, mindMapDelegate, networkController):
        '''
        Initialize the smart conference robot
        @param mindMapDelegate: mind map delegate, holds current mind map
        @type mindMapDelegate: MindMapDelegate
        @param networkController: controller to publish commands to
        @type networkController: NetworkController
        '''
        self.__connection = None
        self.__mindMapDelegate = mindMapDelegate
        self.__networkController = networkController
        self.__talks = None

    def start(self):
        '''Begin data collection from the smart space'''
        if not M3_AVAILABLE:
            raise Exception('Smart M3 python support is not installed')
        try:
            self.__connection = SmartSpace('X', '192.168.0.217', 10010)
        except:
            raise Exception('Unable to connect to the smart space')
        self._collectInformation()

    def _collectInformation(self):
        '''Collect information about all titles available in the smart space'''
        self.__talks = []
        nodeHolder = NodeHolder(self.__mindMapDelegate.mindMap.root)
        talkTriples = self.__connection.query(None, 'Title', None)
        for triple, _ in talkTriples:
            speakerTriple = self.__connection.query(None, 'presents', triple[0], 'literal')
            talk = Talk(nodeHolder, triple[0], triple[2])
            talk.speaker = self._getSpeakerById(speakerTriple[0][0][0], talk)
            talkProperties = self.__connection.query(triple[0], None, None)
            propertiesMap = {'url':'url', 'Keywords' : 'keywords', 'duration' : 'duration'}
            attributes = {}
            for triple, _ in talkProperties:
                if triple[1] in propertiesMap:
                    attributes[propertiesMap[triple[1]]] = triple[2]
            duration = Attribute(talk, 'duration', 'duration', False)
            duration.addAttribute(Attribute(duration, 'proposed', attributes.pop('duration') +
                    'm'))
            talk.addAttribute(duration)
            talk.addAttribute(ComplexAttribute(talk, 'keywords', attributes.pop('keywords')))
            for key in attributes:
                talk.addAttribute(Attribute(talk, key, attributes[key]))
            self.__talks.append(talk)
            for command in talk.nodeCreationCommand():
                self.__networkController.publishCommand(command)

    def _getSpeakerById(self, speakerId, talk):
        '''
        Get information about speaker from SCS and create corresponding object
        @param speakerId: id of the speaker in the SCS
        @type speakerId: str
        @param talk: speaker talk on the conference
        @type talk: Talk
        @return: created object for the speaker
        @rtype: Speaker
        '''
        speakerTriples = self.__connection.query(speakerId, None, None)
        propertiesMap = {'is' : 'name', 'interests' : 'interests', 'language' : 'language',
                'phone' : 'phone', 'email' : 'e-mail', 'photoURI' : 'photo'}
        attributes = {}
        for triple, _ in speakerTriples:
            if triple[1] in propertiesMap:
                attributes[propertiesMap[triple[1]]] = triple[2]
        speaker = Speaker(speakerId, attributes['name'], talk)
        attributes.pop('name')
        speaker.addAttribute(ComplexAttribute(speaker, 'interests',
                attributes.pop('interests')))
        for key in attributes:
            speaker.addAttribute(Attribute(speaker, key, attributes[key]))
        return speaker

    def stop(self):
        '''Stop data collection from the smart space'''
        if self.__connection:
            self.__connection.close()
        self.__connection = None
