# -*- coding: utf-8 -*-
# HiveMind - Distributed mind map editor for Maemo 5 platform
# Copyright (C) 2010-2011 HiveMind developers
#
# HiveMind is the legal property of its developers, whose names are
# noticed in @author or @authors annotations at the beginning of each
# module or class.
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02111-1301  USA

'''
Main window class for Mobile platform
'''

from PyQt4.QtCore import SIGNAL, Qt, pyqtSignal
from PyQt4.QtGui import QAction, QIcon, QMainWindow, QApplication
from hivemind.main_window import MainView, AbstractMainWindow
from hivemind.gui_widgets import MaemoSidePanel, FullScreenExitButton, MaemoNodeMenu
from hivemind.attribute import readable
import hivemind.settings as settings
from hivemind.maemo.dialogs import MaemoExtendedMenuDialog, MaemoNetworkMainDialog


class MaemoMainView(MainView):
    '''
    Mouse input filter for the Maemo platform
    @author: Andrew Vasilev
    '''

    mapClicked = pyqtSignal()

    __MIN_DRAG_PATH = 50
    '''Minimum drag path length for the Maemo platform'''

    def __init__(self, scene, actionBag):
        '''
        @type scene: MindMapDelegate
        @type actionBag: ActionBag
        '''
        MainView.__init__(self, scene, actionBag)
        self.viewport().setCursor((Qt.BlankCursor))

    def mouseReleaseEvent(self, event):
        '''
        Mouse release event handler.
        Emits itemClicked, mapClicked signals.

        @type event: QMouseEvent
        '''
        MainView.mouseReleaseEvent(self, event)
        if self.draggedPath < self.__MIN_DRAG_PATH:
            item = self.scene().itemAt(self.mapToScene(event.pos()))
            parent, childLocation = self.scene().parentAt(self.mapToScene(event.pos()))
            if parent is not None:
                self.emit(SIGNAL('itemClicked'), False, item, parent, childLocation)
            if item is None:
                self.mapClicked.emit()


class MaemoMainWindow(AbstractMainWindow):
    '''
    Main window of HiveMind program for Maemo platform

    @author: Alexander Kulikov
    '''

    readable('mindMapView', 'fullScreenExitButton')

    def __init__(self, actionBag, mindMapScene):
        '''Create new window'''
        AbstractMainWindow.__init__(self, actionBag, mindMapScene)
        QApplication.setOverrideCursor(Qt.BlankCursor)
        self.__actionBag = actionBag
        self.__mindMapView = MaemoMainView(mindMapScene, actionBag)
        self.setCentralWidget(self.__mindMapView)
        self.__fullScreenAction = QAction(QIcon.fromTheme('view-fullscreen'),
                self.tr('&Full screen'), self)
        self.__fullScreenExitButton = FullScreenExitButton(self)
        self.__fullScreenAction.triggered.connect(self.showFullScreen)
        networkDialog = MaemoNetworkMainDialog(actionBag, self)
        actionBag.connectionStatusAction.triggered.connect(networkDialog.exec_)
        self.__contextMenu = self._createContextMenu()
        self.__additionalContextMenu = self._createAdditionalContextMenu()
        self._configureContextMenuSubstitution()
        self._createMainMenu()
        self._createSidePanel()
        self.setAttribute(Qt.WA_Maemo5StackedWindow)
        self.__mindMapView.setVerticalScrollBarPolicy(Qt.ScrollBarAlwaysOff)
        self.__mindMapView.setHorizontalScrollBarPolicy(Qt.ScrollBarAlwaysOff)

    def _createSidePanel(self):
        '''Create side panel'''
        sidePanel = MaemoSidePanel(self.__actionBag.sidebarAction, self.__mindMapView)
        sidePanel.addAction(self.__actionBag.connectionStatusAction)
        sidePanel.addAction(self.__actionBag.zoomInAction)
        sidePanel.addAction(self.__actionBag.zoomOutAction)
        sidePanel.addAction(self.__actionBag.undoAction)
        sidePanel.addAction(self.__actionBag.redoAction)

    def setWindowTitle(self, fileName):
        '''
        Set new title for the window
        @param fileName: filename of opened mind map, can be None
        @type fileName: str
        '''
        if fileName:
            QMainWindow.setWindowTitle(self, fileName)
        else:
            QMainWindow.setWindowTitle(self, 'HiveMind')

    def _createContextMenu(self):
        '''
        Create general context menu
        @rtype: MaemoNodeMenu
        '''
        contextMenu = MaemoNodeMenu(self.__mindMapView)
        contextMenu.addAction(self._actionBag.addNodeAction, 'add_node')
        contextMenu.addAction(self._actionBag.removeNodeAction, 'remove_node')
        contextMenu.addAction(self._actionBag.editNodeAction, 'edit_node')
        contextMenu.addAction(self._actionBag.foldNodeAction, 'hide_subtree')
        self._actionBag.foldNodeAction.setCheckable(True)
        return contextMenu

    def _createAdditionalContextMenu(self):
        '''
        Create additional context menu
        @rtype: MaemoNodeMenu
        '''
        additionalContextMenu = MaemoNodeMenu(self.__mindMapView)
        additionalContextMenu.addAction(self._actionBag.editNodeIconsAction, 'edit_icons')
        additionalContextMenu.addAction(self._actionBag.editEdgeAction, 'edit_edge')
        additionalContextMenu.addAction(self._actionBag.enterTransferModeAction, 'move_node')
        additionalContextMenu.addAction(self._actionBag.editLabelAction, 'label_edge')
        additionalContextMenu.addAction(self._actionBag.openHyperlinkAction, 'open_link')
        return additionalContextMenu

    def _configureContextMenuSubstitution(self):
        '''
        Set event handlers to hide any context menu when it is shown
        and the other menu is requested to show
        '''
        self.__contextMenu.isShowed.connect(self.__additionalContextMenu.hideMenu)
        self.__additionalContextMenu.isShowed.connect(self.__contextMenu.hideMenu)

    def _createMainMenu(self):
        '''Create main menu'''
        menu = self.menuBar().addMenu('')
        menu.addAction(self._actionBag.fileNewAction)
        menu.addAction(self._actionBag.fileOpenAction)
        menu.addAction(self._actionBag.fileSaveAction)
        menu.addAction(self._actionBag.fileSaveAsAction)
        menu.addAction(self._actionBag.exportToImageAction)
        menu.addAction(self.__fullScreenAction)
        menu.addAction(self._actionBag.findAction)
        menu.addAction(self._actionBag.settingsAction)
        menu.addAction(self._actionBag.aboutAction)

    def showMenu(self, rightButton, item):
        '''
        Show various menus
        @param rightButton: not used on maemo
        @type rightButton: bool
        @param item: clicked node
        @type item: NodeDelegate
        '''
        if settings.get('firstClick') is True:
            extendedMenu = MaemoExtendedMenuDialog(self.__actionBag, self, 5000, True)
            settings.set(firstClick = False)
            extendedMenu.exec_()
        elif settings.get('useExtendedMenu') is True:
            extendedMenu = MaemoExtendedMenuDialog(self.__actionBag, self)
            extendedMenu.exec_()
        else:
            if self.__additionalContextMenu.active:
                self.__contextMenu.show(item)
            elif self.__contextMenu.active:
                if self.__contextMenu.item == item:
                    self.__additionalContextMenu.show(item)
                else:
                    self.__contextMenu.show(item)
            else:
                self.__contextMenu.show(item)
