# -*- coding: utf-8 -*-
# HiveMind - Distributed mind map editor for Maemo 5 platform
# Copyright (C) 2010-2011 HiveMind developers
#
# HiveMind is the legal property of its developers, whose names are
# noticed in @author or @authors annotations at the beginning of each
# module or class.
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02111-1301  USA

'''
Dialogs for Maemo platform
'''

import hivemind.resource as resource
from PyQt4.QtGui import QFrame, QWidget, QGridLayout, QStackedWidget, QVBoxLayout, \
QRadioButton, QHBoxLayout, QToolBar, QIcon, QPalette, QPushButton, QLabel, \
QTextBrowser, QDialog, QFont, QDesktopServices, \
QAction, QPainter, QBrush, QPen, QColor
from PyQt4.QtCore import Qt, QSize, QTimer, QUrl, QEvent
from hivemind.dialogs import EditNodeDialog, EditLabelDialog, IconDialog, \
AboutDialog, EditEdgeDialog, SettingsDialog, AffiliationDialog, NetworkSettingsDialog, \
NetworkClientDialog, NetworkConnectionDialog
from hivemind.gui_widgets import ScalingWidget, ActionImageButton, FindToolBar
from hivemind.attribute import adaptCallable
import hivemind.__version__ as __version__


class MaemoDialogMixin:
    '''
    Mixin for Maemo dialogs

    @author: Alexander Kulikov
    '''

    def __init__(self):
        '''Initialize mixin'''
        self.setWindowFlags(Qt.Window)
        self.setAttribute(Qt.WA_Maemo5StackedWindow) #pylint: disable = E1103

    def exec_(self):
        '''This method is overriden to call show() as it is necessary on Maemo'''
        self.show()


class MaemoEditNodeDialog(MaemoDialogMixin, EditNodeDialog):
    '''
    Edit node dialog for Maemo platform

    @author: Alexander Kulikov
    '''

    def __init__(self, node, parent = None):
        '''
        Initialize dialog with specific node
        @type node: Node
        @type parent: QWidget
        '''
        EditNodeDialog.__init__(self, node, parent)
        MaemoDialogMixin.__init__(self)
        self._initComponents()
        self._showTextSourceButtons(self._isHtml)

    def _initComponents(self):
        '''Form initialization'''
        #pylint: disable = W0201
        toolbar = self._createToolbar()
        #create edit widget
        scalingWidget = ScalingWidget(self._editor, 2.0)
        self._editor.setFrameShape(QFrame.Box)
        editWidget = QWidget()
        editLayout = QGridLayout()
        editWidget.setLayout(editLayout)
        editLayout.addWidget(toolbar, 0, 0, 1, 6)
        editLayout.addWidget(scalingWidget, 1, 0, 1, 6)
        #create html widget
        scalingWidget = ScalingWidget(self._htmlEditor, 1.5)
        self._htmlEditor.setFrameShape(QFrame.Box)
        htmlWidget = QWidget()
        htmlLayout = QGridLayout()
        htmlWidget.setLayout(htmlLayout)
        htmlLayout.addWidget(scalingWidget, 0, 0)
        #create stacked widget
        self.__stackedWidget = QStackedWidget()
        self.__textIndex = self.__stackedWidget.addWidget(editWidget)
        self.__htmlIndex = self.__stackedWidget.addWidget(htmlWidget)
        mainLayout = QVBoxLayout()
        self.__showText = QRadioButton(self.tr('Text'))
        self.__showHtml = QRadioButton(self.tr('Source'))
        self.__showText.setChecked(True)
        mainLayout.addWidget(self.__stackedWidget)
        bottomLayout = QHBoxLayout()
        bottomLayout.addWidget(self.__showText)
        bottomLayout.addWidget(self.__showHtml)
        bottomLayout.addWidget(self._propagateFormatting)
        bottomLayout.addStretch()
        bottomLayout.addWidget(self._buttonBox)
        mainLayout.addLayout(bottomLayout)
        self._editor.setFocus()
        self.setLayout(mainLayout)
        self.__showHtml.toggled.connect(adaptCallable(self._changedMode))
        self._textHtmlAction.toggled.connect(adaptCallable(self._showTextSourceButtons))

    def _createToolbar(self):
        '''Create editor toolbar'''
        toolbar = QToolBar(self)
        self._boldAction.setIcon(QIcon(resource.getImage('node_editor_bold')))
        toolbar.addAction(self._boldAction)
        self._italicAction.setIcon(QIcon(resource.getImage('node_editor_italic')))
        toolbar.addAction(self._italicAction)
        self._underlineAction.setIcon(QIcon(resource.getImage('node_editor_underline')))
        toolbar.addAction(self._underlineAction)
        self._textFontAction.setIcon(QIcon(resource.getImage('node_editor_font')))
        toolbar.addAction(self._textFontAction)
        self._textHtmlAction.setIcon(QIcon(resource.getImage('node_editor_html')))
        toolbar.addAction(self._textHtmlAction)
        self._colorAction.setIcon(QIcon(resource.getImage('node_editor_font_color')))
        toolbar.addAction(self._colorAction)
        self._backgroundAction.setIcon(QIcon(
                        resource.getImage('node_editor_background_color')))
        toolbar.addAction(self._backgroundAction)
        self._clearFormatAction.setIcon(QIcon(resource.getImage('node_editor_clear')))
        toolbar.addAction(self._clearFormatAction)
        self._editHyperlinkAction.setIcon(QIcon(resource.getImage('node_editor_link')))
        toolbar.addAction(self._editHyperlinkAction)
        return toolbar

    def _changedMode(self, isSourceMode):
        '''
        Modificate dialog when html source was selected
        @type isSourceMode: bool
        '''
        self._isHtmlSourceMode = (isSourceMode == self.__htmlIndex)
        if isSourceMode == self.__htmlIndex:
            self._htmlEditor.setPlainText(self._editor.toHtml())
            self.__stackedWidget.setCurrentIndex(self.__htmlIndex)
        else:
            self._editor.setHtml(self._htmlEditor.toPlainText())
            self.__stackedWidget.setCurrentIndex(self.__textIndex)

    def _showTextSourceButtons(self, isHtml):
        '''
        Show switch buttons of text/source when html mode was selected
        @type isHtml: bool
        '''
        self.__showText.setVisible(isHtml)
        self.__showHtml.setVisible(isHtml)
        self._propagateFormatting.setVisible(not isHtml)

class MaemoEditLabelDialog(MaemoDialogMixin, EditLabelDialog):
    '''
    Edit label dialog for Maemo platform

    @author: Oleg Kandaurov
    '''
    def __init__(self, node, parent = None):
        '''
        Initialize dialog with specific node
        @type node: Node
        @type parent: QWidget
        '''
        EditLabelDialog.__init__(self, node, parent)
        MaemoDialogMixin.__init__(self)
        self._initComponents()

    def _initComponents(self):
        '''Form initialization'''
        #create toolbar
        self._boldAction.setIcon(QIcon(resource.getImage('node_editor_bold')))
        self._italicAction.setIcon(QIcon(resource.getImage('node_editor_italic')))
        self._underlineAction.setIcon(QIcon(resource.getImage('node_editor_underline')))
        centralWidget = ScalingWidget(self._editor, 2.0)
        self._editor.setFrameShape(QFrame.Box)
        self._layoutComponents(centralWidget)


class MaemoIconDialog(MaemoDialogMixin, IconDialog):
    '''
    Icon selection dialog for Maemo platform

    @author: Alexander Kulikov
    '''

    def __init__(self, node, parent = None):
        '''
        Initialize with specific node
        @type node: Node
        @type parent: QWidget
        '''
        IconDialog.__init__(self, node, parent)
        MaemoDialogMixin.__init__(self)
        self._initComponents()

    def _initComponents(self):
        '''Form initialization'''
        layout = QGridLayout()
        self._iconList.setBackgroundRole(QPalette.Window)
        self._iconList.setStyleSheet('''QGraphicsView {border: 1px solid gray;
                            border-radius: 5px; padding: 1 1px;}''')
        self._iconTable.setStyleSheet('''QListWidget {border: 1px solid gray;
                            border-radius: 5px; padding: 1 1px;}''')
        self._iconTable.setGridSize(QSize(68, 68))
        self._iconList.setMinimumHeight(self._listScene.height() + 8)
        self._iconList.setMaximumHeight(self._listScene.height() + 8)
        layout.addWidget(self._iconTable, 0, 0, 2, 4)
        layout.addWidget(self._iconList, 2, 0, 1, 4)
        layout.addWidget(self._buttonBox, 3, 0, 1, 4)
        self.setLayout(layout)


class MaemoAboutDialog(AboutDialog):
    '''
    About dialog for Maemo platform

    @author: Oleg Kandaurov, Alexander Kulikov
    '''
    def __init__(self, parent = None):
        AboutDialog.__init__(self, parent)

    def _createComponents(self):
        '''Create components'''
        #pylint: disable = W0201
        self._logo.setPixmap(self._logoImage)
        self._visitWebsite = QPushButton(self.tr('Visit website'))
        self._reportBug = QPushButton(self.tr('Report bug'))
        self._name = QLabel('<h2>%s %s</h2>' % (self._appName, __version__.VERSION))
        self._about = QTextBrowser()
        self._about.setHtml(('<center>%s</center>' +
                '<ul style="list-style-type: square;margin: 0px 0px 0px 40px">' +
                '<li>%s<li>%s<li>%s<li>%s<li>%s<li>%s</ul><center>%s<br><br>' +
                '%s<br>%s<br><br>%s<br>%s</center>') % (
                self._copyright,
                self._generateAuthorInfo('AndreyVasilev'),
                self._generateAuthorInfo('IlyaParamonov'),
                self._generateAuthorInfo('OlegKandaurov'),
                self._generateAuthorInfo('AlexanderKulikov'),
                self._generateAuthorInfo('EldarMamedov'),
                self._generateAuthorInfo('AndreyGolovchenko'),
                self._license,
                self._development, (self._linkTemplate % (self._fructLink, self._fruct)),
                self._generatePartcipantInfo('NicuBuculei'),
                self._generatePartcipantInfo('Mentalrey')))
        self._about.setOpenExternalLinks(True)
        self._visitWebsite.clicked.connect(self._openWebsite)
        self._reportBug.clicked.connect(self._openBugReport)

    def _generatePartcipantInfo(self, name):
        '''
        Generate formatted information about participant person
        @type name: str
        '''
        infoTemplate = ('%s: %s')
        return infoTemplate % (self._participantInfo[name]['role'], self._linkTemplate %
            (self._participantInfo[name]['site'], self._participantInfo[name]['name']))

    def _generateAuthorInfo(self, name):
        '''
        Generate formatted information about author person
        @type name: str
        '''
        infoTemplate = ('<a href="%s">%s</a>, %s')
        return infoTemplate % (self._authorInfo[name]['site'], self._authorInfo[name]['name'],
                self._authorInfo[name]['role'])

    def _initComponents(self):
        '''Form initialization'''
        mainLayout = QGridLayout()
        mainLayout.addWidget(self._logo, 0, 14, 16, 6)
        mainLayout.addWidget(self._name, 0, 0, 3, 14, Qt.AlignCenter)
        mainLayout.addWidget(self._info, 2, 0, 3, 14, Qt.AlignCenter)
        mainLayout.addWidget(self._about, 5, 0, 15, 14)
        buttonsLayout = QVBoxLayout()
        buttonsLayout.addWidget(self._visitWebsite)
        buttonsLayout.addWidget(self._reportBug)
        mainLayout.addLayout(buttonsLayout, 16, 14, 4, 6)
        self.setLayout(mainLayout)

    def _openWebsite(self):
        '''Open project web page'''
        QDesktopServices.openUrl(QUrl(self._siteLink))

    def _openBugReport(self):
        '''Open bug report page'''
        QDesktopServices.openUrl(QUrl(self._bugLink))


class MaemoEditEdgeDialog(MaemoDialogMixin, EditEdgeDialog):
    '''
    Dialog for editing decoration and style of a joint link for Maemo platform
    @author: Oleg Kandaurov
    '''
    def __init__(self, node = None, parent = None):
        '''
        Initialize dialog with specific node
        @type node: Node
        '''
        EditEdgeDialog.__init__(self, node, parent)
        MaemoDialogMixin.__init__(self)
        self._initComponents()


class MaemoSettingsDialog(MaemoDialogMixin, SettingsDialog):
    '''
    Settings dialog for Maemo platform
    @author: Oleg Kandaurov
    '''
    def __init__(self, parent = None):
        '''
        @type parent: QWidget
        '''
        SettingsDialog.__init__(self, parent)
        MaemoDialogMixin.__init__(self)
        self._createTab(self.tr('General'), ['locale', 'confirmDelete', 'useExtendedMenu',
                'autosaveEnabled', 'autosaveInterval'], True)
        self._createTab(self.tr('Node'), ['defaultFont', 'defaultNodeTextColor',
                'defaultNodeBackgroundColor', 'defaultSelectedNodeBackgroundColor'])
        self._createTab(self.tr('Edge && Label'), ['defaultEdgeWidth', 'defaultEdgeColor',
                'defaultLabelFont', 'defaultLabelColor'])
        self._createTab(self.tr('View'), ['autoHideNodeMenuTimeout',
                'showNodeMenuAnimationDuration', 'hideNodeMenuAnimationDuration',
                'defaultZoomLevel', 'maxNodeTextWidth'], True)
        self._createTab(self.tr('Shortcuts'), ['shortcuts'], True)


class MaemoAffiliationDialog(MaemoDialogMixin, AffiliationDialog):
    '''
    Affiliation dialog for Maemo platform
    @author: Oleg Kandaurov
    '''
    def __init__(self, model, parent = None):
        AffiliationDialog.__init__(self, model, parent)
        MaemoDialogMixin.__init__(self)


class MaemoExtendedMenuDialog(QDialog):
    '''
    Extended menu dialog for Maemo platform
    @author: Alexander Kulikov
    '''
    def __init__(self, actionBag, parent = None, freezeTime = 0, showHints = False):
        '''
        @type actionBag: ActionBag
        @param freezeTime: Time interval during which dialog can not be rejected
        @type freezeTime: int
        @param showHints: Show hints or not
        @type showHints: bool
        @type parent: QWidget
        '''
        QDialog.__init__(self, parent)
        if showHints:
            self.setWindowTitle(self.tr('Context menu usage'))
        else:
            self.setWindowTitle(self.tr('Extended menu'))
        self.__freezed = freezeTime != 0
        self.__showHints = showHints
        QTimer.singleShot(freezeTime, self._unfreeze)
        self._createComponents(actionBag)

    def _createComponents(self, actionBag):
        '''
        Create dialog widgets
        @type actionBag: ActionBag
        '''
        def createLabel(text):
            '''Creates label with special properties'''
            label = QLabel(text)
            label.setAlignment(Qt.AlignCenter)
            return label

        layout = QGridLayout()
        font = QFont()
        font.setPixelSize(24)
        if self.__showHints:
            singleClickLabel = QLabel(self.tr('Single-click on the node to show ' +
                    'the main menu:'))
            singleClickLabel.setFont(font)
            doubleClickLabel = QLabel(self.tr('Double-click on the node to show the ' +
                    'additional menu:'))
            doubleClickLabel.setFont(font)
            layout.addWidget(singleClickLabel, 0, 0, 1, 12, Qt.AlignCenter)
            layout.addWidget(doubleClickLabel, 3, 0, 1, 12, Qt.AlignCenter)
        layout.addWidget(self._createActionWidget(actionBag.addNodeAction), 1, 0, 1, 3,
                Qt.AlignCenter)
        layout.addWidget(createLabel(self.tr('Add node')), 2, 0, 1, 3,
                Qt.AlignHCenter | Qt.AlignTop)
        layout.addWidget(self._createActionWidget(actionBag.editNodeAction), 1, 3, 1, 3,
                Qt.AlignCenter)
        layout.addWidget(createLabel(self.tr('Edit node')), 2, 3, 1, 3,
                Qt.AlignHCenter | Qt.AlignTop)
        layout.addWidget(self._createActionWidget(actionBag.removeNodeAction), 1, 6, 1, 3,
                Qt.AlignCenter)
        layout.addWidget(createLabel(self.tr('Remove node')), 2, 6, 1, 3,
                Qt.AlignHCenter | Qt.AlignTop)
        layout.addWidget(self._createActionWidget(actionBag.foldNodeAction), 1, 9, 1, 3,
                Qt.AlignCenter)
        if actionBag.foldNodeAction.isChecked():
            foldLabel = createLabel(self.tr('Unfold node subtree'))
        else:
            foldLabel = createLabel(self.tr('Fold node subtree'))
        foldLabel.setWordWrap(True)
        layout.addWidget(foldLabel, 2, 9, 1, 3, Qt.AlignHCenter | Qt.AlignTop)

        layout.addWidget(self._createActionWidget(actionBag.editNodeIconsAction),
                6, 0, 1, 4, Qt.AlignCenter)
        layout.addWidget(createLabel(self.tr('Select icons')), 7, 0, 1, 4,
                Qt.AlignHCenter | Qt.AlignTop)
        layout.addWidget(self._createActionWidget(actionBag.editEdgeAction), 6, 4, 1, 4,
                Qt.AlignCenter)
        layout.addWidget(createLabel(self.tr('Edit edge')), 7, 4, 1, 4, Qt.AlignHCenter |
                Qt.AlignTop)
        layout.addWidget(self._createActionWidget(actionBag.enterTransferModeAction),
                6, 8, 1, 4, Qt.AlignCenter)
        layout.addWidget(createLabel(self.tr('Move node')), 7, 8, 1, 4, Qt.AlignHCenter |
                Qt.AlignTop)

        layout.addWidget(self._createActionWidget(actionBag.editLabelAction), 4, 2, 1, 4,
                Qt.AlignCenter)
        layout.addWidget(createLabel(self.tr('Edit edge label')), 5, 2, 1, 4,
                Qt.AlignHCenter | Qt.AlignTop)
        layout.addWidget(self._createActionWidget(actionBag.openHyperlinkAction),
                4, 6, 1, 4, Qt.AlignCenter)
        layout.addWidget(createLabel(self.tr('Go to hyperlink')), 5, 6, 1, 4,
                Qt.AlignHCenter | Qt.AlignTop)
        self.setLayout(layout)

    def _unfreeze(self):
        '''Unfreeze reject functionality'''
        self.__freezed = False

    def reject(self):
        '''Close dialog with reject state returned'''
        if self.__freezed == False:
            QDialog.reject(self)

    def _createActionWidget(self, action):
        '''
        Create action widget
        @type action: QAction
        '''
        button = ActionImageButton(action)
        button.clicked.connect(self.accept)
        return button


class MaemoNetworkSettingsDialog(MaemoDialogMixin, NetworkSettingsDialog):
    '''
    Network settings dialog for Maemo platfrom
    @author: Oleg Kandaurov
    '''

    def __init__(self, parent = None):
        NetworkSettingsDialog.__init__(self, parent)
        MaemoDialogMixin.__init__(self)


class MaemoNetworkClientDialog(MaemoDialogMixin, NetworkClientDialog):
    '''
    Client dialog for Maemo platfrom
    @author: Oleg Kandaurov
    '''

    def __init__(self, rosterModel, parent = None):
        NetworkClientDialog.__init__(self, rosterModel, parent)
        MaemoDialogMixin.__init__(self)


class MaemoNetworkConnectionDialog(MaemoDialogMixin, NetworkConnectionDialog):
    '''
    Connection dialog for Maemo platfrom
    @author: Oleg Kandaurov
    '''

    def __init__(self, parent = None):
        NetworkConnectionDialog.__init__(self, parent)
        MaemoDialogMixin.__init__(self)


class MaemoNetworkMainDialog(MaemoDialogMixin, QDialog):
    '''
    Main network dialog for Maemo platform
    @author: Oleg Kandaurov
    '''
    def __init__(self, actionBag, parent = None):
        '''Constructor'''
        QDialog.__init__(self, parent)
        MaemoDialogMixin.__init__(self)
        self.__bag = actionBag
        self._createComponents()
        self._initComponents()
        self._connectActions()

    def _createComponents(self):
        '''Create components'''
        #pylint: disable = W0201
        self.__editAffiliationButton = QPushButton(self.__bag.editAffiliationAction.text())
        self.__disconnectXMPPButton = QPushButton(self.__bag.disconnectXMPPAction.text())
        self.__stopProtocolButton = QPushButton(self.__bag.stopProtocolAction.text())
        self.__startServiceButton = QPushButton(self.__bag.startServiceAction.text())
        self.__startClientButton = QPushButton(self.__bag.startClientAction.text())
        self.__networkSettingsButton = QPushButton(self.__bag.networkSettingsAction.text())
        self.__networkHelpButton = QPushButton(self.__bag.networkHelpAction.text())

    def _initComponents(self):
        '''Init components'''
        mainLayout = QVBoxLayout()
        mainLayout.addWidget(self.__startServiceButton)
        mainLayout.addWidget(self.__startClientButton)
        mainLayout.addWidget(self.__stopProtocolButton)
        mainLayout.addWidget(self.__disconnectXMPPButton)
        mainLayout.addWidget(self.__editAffiliationButton)
        layout = QHBoxLayout()
        layout.addWidget(self.__networkHelpButton)
        layout.addWidget(self.__networkSettingsButton)
        mainLayout.addLayout(layout)
        self.setLayout(mainLayout)

    def _connectActions(self):
        '''Connect actions'''
        self.__editAffiliationButton.clicked.connect(self.__bag.editAffiliationAction.trigger)
        self.__disconnectXMPPButton.clicked.connect(self.__bag.disconnectXMPPAction.trigger)
        self.__stopProtocolButton.clicked.connect(self.__bag.stopProtocolAction.trigger)
        self.__startServiceButton.clicked.connect(self.__bag.startServiceAction.trigger)
        self.__startClientButton.clicked.connect(self.__bag.startClientAction.trigger)
        self.__networkHelpButton.clicked.connect(self.__bag.networkHelpAction.trigger)
        self.__networkSettingsButton.clicked.connect(self.__bag.networkSettingsAction.trigger)
        self.__bag.editAffiliationAction.changed.connect(self._updateState)
        self.__bag.disconnectXMPPAction.changed.connect(self._updateState)
        self.__bag.stopProtocolAction.changed.connect(self._updateState)
        self.__bag.startServiceAction.changed.connect(self._updateState)
        self.__bag.startClientAction.changed.connect(self._updateState)
        self.__bag.networkHelpAction.changed.connect(self._updateState)
        self.__bag.networkSettingsAction.changed.connect(self._updateState)

    def _updateState(self):
        '''Update button states acccording to action state'''
        self.__editAffiliationButton.setEnabled(self.__bag.editAffiliationAction.isEnabled())
        self.__disconnectXMPPButton.setEnabled(self.__bag.disconnectXMPPAction.isEnabled())
        self.__stopProtocolButton.setEnabled(self.__bag.stopProtocolAction.isEnabled())
        self.__startServiceButton.setEnabled(self.__bag.startServiceAction.isEnabled())
        self.__startClientButton.setEnabled(self.__bag.startClientAction.isEnabled())
        self.__networkHelpButton.setEnabled(self.__bag.networkHelpAction.isEnabled())
        self.__networkSettingsButton.setEnabled(self.__bag.networkSettingsAction.isEnabled())


class MaemoFindToolBar(FindToolBar):
    '''
    Find tool bar for Desktop platform
    '''

    __HEIGHT = 82
    '''Height of the tool bar'''

    __WIDTH = 630
    '''Width of the tool bar'''

    __RADIUS = 20
    '''Corner radius'''

    def __init__(self, controller, parent = None):
        '''
        @type controller: MainWindowController
        @type parent: QWidget
        '''
        FindToolBar.__init__(self, controller, parent)
        self._findWidget = QWidget()
        self.resize(self.__WIDTH, self.__HEIGHT)
        self.setAllowedAreas(Qt.NoToolBarArea)
        self.setMovable(False)
        self.setFloatable(False)
        self._createComponents()
        self._initComponents()
        parent.installEventFilter(self)

    def _createComponents(self):
        '''
        Create components
        '''
        #pylint: disable = W0201
        FindToolBar._createComponents(self)
        self.__prevAction = QAction(self)
        self.__nextAction = QAction(self)
        self.__closeAction = QAction(self)
        self.__prevAction.setIcon(QIcon(resource.getImage('find_previous')))
        self.__nextAction.setIcon(QIcon(resource.getImage('find_next')))
        self.__closeAction.setIcon(QIcon(resource.getImage('find_close')))
        self.__nextAction.triggered.connect(self._finder.nextFoundNode)
        self.__prevAction.triggered.connect(self._finder.prevFoundNode)
        self.__closeAction.triggered.connect(self.close)
        self.__prevButton = ActionImageButton(self.__prevAction)
        self.__nextButton = ActionImageButton(self.__nextAction)
        self.__closeButton = ActionImageButton(self.__closeAction)

    def _initComponents(self):
        '''
        Init components
        '''
        buttonLayout = QHBoxLayout()
        buttonLayout.setSpacing(30)
        buttonLayout.addWidget(self.__prevButton)
        buttonLayout.addWidget(self.__nextButton)
        mainLayout = QHBoxLayout()
        mainLayout.addWidget(self._findLabel)
        mainLayout.addWidget(self._findLineEdit)
        mainLayout.addLayout(buttonLayout)
        mainLayout.addStretch()
        mainLayout.addWidget(self.__closeButton)
        self._findWidget.setLayout(mainLayout)
        self.addWidget(self._findWidget)
        self._findLineEdit.setStyleSheet("QLineEdit { border-width: 8px;"
                 + " border-color: white; border-style: solid;"
                 + " border-radius: 12px; background: white; }")

    def paintEvent(self, paintEvent):
        '''
        Paint black background
        @param paintEvent: paint event representation
        @type paintEvent: QPaintEvent
        '''
        painter = QPainter(self)
        background = QColor(self.backgroundRole())
        background.setAlpha(128)
        brush = QBrush(background)
        pen = QPen(background)
        painter.fillRect(0, self.__RADIUS, self.__WIDTH, self.__HEIGHT - self.__RADIUS, brush)
        painter.fillRect(self.__RADIUS, 0, self.__WIDTH - self.__RADIUS, self.__RADIUS, brush)
        painter.setPen(pen)
        painter.setBrush(brush)
        painter.drawPie(0, 0, (self.__RADIUS - 1) * 2, (self.__RADIUS - 1) * 2, 180 * 8,
                180 * 8)

    def eventFilter(self, obj, event):
        '''
        Catch Windows state changed and Resize events from parent object
        @type obj: QEvent
        @type event: QEvent
        '''
        parent = self.parentWidget()
        if obj != self.parent():
            return QToolBar.eventFilter(self, obj, event)
        isFullScreen = parent.windowState() & Qt.WindowFullScreen
        if event.type() == QEvent.Resize:
            if isFullScreen:
                self.resize(self.width() - parent.fullScreenExitButton.width(), self.__HEIGHT)
                self.move(parent.width() - self.width() - parent.fullScreenExitButton.width(),
                         parent.height() - parent.fullScreenExitButton.height())
            else:
                self.resize(self.__WIDTH, self.__HEIGHT)
                self.move(parent.width() - self.width() + 10, parent.height() - self.height()
                        + 10)
        return QToolBar.eventFilter(self, obj, event)

