# -*- coding: utf-8 -*-
# HiveMind - Distributed mind map editor for Maemo 5 platform
# Copyright (C) 2010-2011 HiveMind developers
#
# HiveMind is the legal property of its developers, whose names are
# noticed in @author or @authors annotations at the beginning of each
# module or class.
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02111-1301  USA

'''
GUI factory

@author: Alexander Kulikov, Ilya Paramonov
'''

from PyQt4.QtGui import QFileDialog
import hivemind.sysconfig as sysconfig
from hivemind.dialogs import SideSelectionDialog, NetworkAuthorizationDialog, \
        NetworkErrorDialog, EditHyperlinkDialog

if sysconfig.PLATFORM == 'maemo':
    from hivemind.maemo.main_window import MaemoMainWindow
    from hivemind.maemo.dialogs import MaemoEditNodeDialog, MaemoEditLabelDialog
    from hivemind.maemo.dialogs import MaemoIconDialog, MaemoAboutDialog, MaemoEditEdgeDialog
    from hivemind.maemo.dialogs import MaemoSettingsDialog, MaemoNetworkConnectionDialog
    from hivemind.maemo.dialogs import MaemoAffiliationDialog, MaemoFindToolBar
    from hivemind.maemo.dialogs import MaemoNetworkSettingsDialog, MaemoNetworkClientDialog
else:
    from hivemind.desktop.main_window import DesktopMainWindow
    from hivemind.desktop.dialogs import DesktopEditNodeDialog, DesktopEditLabelDialog
    from hivemind.desktop.dialogs import DesktopIconDialog, DesktopAboutDialog, \
        DesktopFindToolBar
    from hivemind.desktop.dialogs import DesktopEditEdgeDialog, DesktopNetworkSettingsDialog
    from hivemind.desktop.dialogs import DesktopSettingsDialog, DesktopNetworkConnectionDialog
    from hivemind.desktop.dialogs import DesktopAffiliationDialog, DesktopNetworkClientDialog

'''Default parent for all application dialogs (actually, main window)'''
__defaultParent = None

def defaultParent():
    '''
    @return: default parent for all dialogs
    @rtype: QWidget
    '''
    return __defaultParent

def createMainWindow(actionBag, mindMapScene):
    '''
    Factory for creating families of related main windows
    @type actionBag: ActionBag
    @type mindMapScene: MindMapDelegate
    '''
    global __defaultParent #pylint: disable=W0603
    if sysconfig.PLATFORM == 'maemo':
        __defaultParent = MaemoMainWindow(actionBag, mindMapScene)
    else:
        __defaultParent = DesktopMainWindow(actionBag, mindMapScene)
    return __defaultParent

def createEditNodeDialog(node):
    '''
    Factory for creating families of related node editing dialogs
    @type node: Node
    '''
    if sysconfig.PLATFORM == 'maemo':
        return MaemoEditNodeDialog(node, __defaultParent)
    return DesktopEditNodeDialog(node, __defaultParent)

def createEditEdgeDialog(node):
    '''
    Factory for creating families of related edge editing dialogs
    @type node: Node
    '''
    if sysconfig.PLATFORM == 'maemo':
        return MaemoEditEdgeDialog(node, __defaultParent)
    return DesktopEditEdgeDialog(node, __defaultParent)

def createIconDialog(node):
    '''
    Factory for creating families of related icon dialogs
    @type node: Node
    '''
    if sysconfig.PLATFORM == 'maemo':
        return MaemoIconDialog(node, __defaultParent)
    return DesktopIconDialog(node, __defaultParent)

def createAboutDialog():
    '''
    Factory for creating families of related about dialogs
    '''
    if sysconfig.PLATFORM == 'maemo':
        return MaemoAboutDialog(__defaultParent)
    return DesktopAboutDialog(__defaultParent)

def createFindToolBar(controller):
    '''
    Factory for creating families of related find dialogs
    @type controller: MainWindowController
    '''
    if sysconfig.PLATFORM == 'maemo':
        return MaemoFindToolBar(controller, __defaultParent)
    return DesktopFindToolBar(controller, __defaultParent)

def createSettingsDialog():
    '''
    Factory for creating families of related settings dialogs
    @type parent: QWidget
    '''
    if sysconfig.PLATFORM == 'maemo':
        return MaemoSettingsDialog(__defaultParent)
    return DesktopSettingsDialog(__defaultParent)

def createEditLabelDialog(node):
    '''
    Factory for creating families of related edge label editing dialogs
    @type node: Node
    '''
    if sysconfig.PLATFORM == 'maemo':
        return MaemoEditLabelDialog(node, __defaultParent)
    return DesktopEditLabelDialog(node, __defaultParent)

def createSideSelectionDialog():
    '''
    Factory for creating families of related side selection dialogs
    '''
    return SideSelectionDialog(__defaultParent)

def createAffiliationDialog(model):
    '''
    Factory for creating families of related side affiliation dialogs
    '''
    if sysconfig.PLATFORM == 'maemo':
        return MaemoAffiliationDialog(model, __defaultParent)
    return DesktopAffiliationDialog(model, __defaultParent)

def createNetworkAuthorizationDialog(requestor):
    '''
    Factory for creating families of network authorization dialogs
    @type requestor: str
    '''
    return NetworkAuthorizationDialog(requestor, __defaultParent)

def createNetworkSettingsDialog():
    '''
    Factory for creating families of related network settings dialogs
    '''
    if sysconfig.PLATFORM == 'maemo':
        return MaemoNetworkSettingsDialog(__defaultParent)
    return DesktopNetworkSettingsDialog(__defaultParent)

def createNetworkClientDialog(rosterModel):
    '''
    Factory for creating families of related client dialogs
    '''
    if sysconfig.PLATFORM == 'maemo':
        return MaemoNetworkClientDialog(rosterModel, __defaultParent)
    return DesktopNetworkClientDialog(rosterModel, __defaultParent)

def createNetworkConnectionDialog():
    '''
    Factory for creating families of related network connection dialogs
    '''
    if sysconfig.PLATFORM == 'maemo':
        return MaemoNetworkConnectionDialog(__defaultParent)
    return DesktopNetworkConnectionDialog(__defaultParent)

def createNetworkErrorDialog():
    '''
    Factory for creating families of network error dialogs
    '''
    return NetworkErrorDialog(__defaultParent)

def createEditHyperlinkDialog(node, curdir):
    '''
    Factory for creating families of hyperlink edit dialogs
    @type node: Node
    @type curdir: string
    '''
    return EditHyperlinkDialog(node, curdir, __defaultParent)

def invokeSaveFileDialog(title, baseDirectory, fileFilter, selectedFilter):
    '''
    Factory for creating save file dialog
    @param title: dialog title
    @type title: str
    @param baseDirectory: base save file directory
    @type baseDirectory: str
    @param fileFilter: file type fileFilter
    @type fileFilter: str
    @param selectedFilter: default fileFilter selected, may be None
    @type selectedFilter: str
    @return: selected file name and active fileFilter
    @rtype: [QString, QString]
    '''
    if sysconfig.PLATFORM == 'maemo':
        baseDirectory = '/'
    return QFileDialog.getSaveFileNameAndFilter(defaultParent(), title, baseDirectory,
            fileFilter, selectedFilter)
