# -*- coding: utf-8 -*-
# HiveMind - Distributed mind map editor for Maemo 5 platform
# Copyright (C) 2011 HiveMind developers
#
# HiveMind is the legal property of its developers, whose names are
# noticed in  or  annotations at the beginning of each
# module or class.
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02111-1301  USA

'''
This module contains classes that interconnect with the smart conference systemz
'''

from smart_m3.Node import TCPConnector, ParticipantNode

class SmartSpace(object):
    '''
    Class manages connection to the smart space of the SCS

    @author: Andrew Vasilev
    '''

    def __init__(self, name, host, port):
        '''
        @param name: name of the smart space
        @type name: str
        @param host: ip address of the smart space
        @type host: str
        @param port: port to connect to
        @type port: int
        '''
        self.__node = ParticipantNode('HiveMind-KP')
        self.__smartSpaceLocation = (name, (TCPConnector, (host, port)))
        self.__node.join(self.__smartSpaceLocation)

    def close(self):
        '''Disconnect from the smart space'''
        self.__node.leave(self.__smartSpaceLocation)

    def addData(self, subject, predicate, object, objectType = 'uri'):
        '''
        Insert information into the smart space
        @param subject: subject, cannot be None
        @type subject: str
        @param predicate: the relationship between subject and object, cannot be None
        @type predicate: str
        @param object: cannot be None
        @type object: str
        @param objectType: the type of the object: 'literal' or 'uri' which is set by default
        @type objectType: str
        '''
        transaction = self.__node.CreateInsertTransaction(self.__smartSpaceLocation)
        transaction.send([((subject, predicate, object), 'uri', objectType)])
        transaction.close()

    def deleteData(self, subject, predicate, object, objectType = 'uri'):
        '''
        Remove information from the smart space
        @param subject: cannot be none
        @type subject: str
        @param predicate: relationship between subject and object, cannot be None
        @type predicate: str
        @param object: cannot be None
        @type object: str
        @param objectType: type of the object: 'literal' or 'uri' which is set by default
        @type objectType: str
        '''
        transaction = self.__node.CreateRemoveTransaction(self.__smartSpaceLocation)
        transaction.remove([((subject, predicate, object), 'uri', objectType)])
        transaction.close()

    def query(self, subject, predicate, object, objectType = 'uri'):
        '''
        Make a query for an information from the smart space
        @param subject: a subject of a query, may be None
        @type subject: str
        @param predicate: predicate, may be None
        @type predicate: str
        @param object: object, may be None
        @type object: str
        @param objectType: type of the object, may be 'uri' or 'literal
        @type objectType: str
        '''
        transaction = self.__node.CreateQueryTransaction(self.__smartSpaceLocation)
        result = transaction.rdf_query(((subject, predicate, object), objectType))
        transaction.close()
        return result
