# -*- coding: utf-8 -*-
# HiveMind - Distributed mind map editor for Maemo 5 platform
# Copyright (C) 2010-2011 HiveMind developers
#
# HiveMind is the legal property of its developers, whose names are
# noticed in  or  annotations at the beginning of each
# module or class.
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02111-1301  USA

'''
Network related models and delegates
'''
from PyQt4.QtGui import QStyledItemDelegate, QComboBox
from PyQt4.QtCore import Qt, QAbstractListModel, QModelIndex, QAbstractTableModel
from hivemind.attribute import readable

class RosterModel(QAbstractListModel):
    '''
    Model for roster
    @author: Oleg Kandaurov
    '''

    def __init__(self, parent = None):
        QAbstractListModel.__init__(self, parent)
        self.__roster = list()

    def data(self, index, role = Qt.DisplayRole):
        '''
        Returns the data stored under the given role for the item referred to by the index.
        @type index: QModelIndex
        @type role: int
        '''
        if not index.isValid() or index.row() not in range(0, len(self.__roster)):
            return None
        item = self.__roster[index.row()]
        if role == Qt.DisplayRole:
            return item.name if item.name is not None else item.jid.userhost()
        elif role == Qt.EditRole:
            return item.jid.userhost()
        return None

    def headerData(self, section, orientation, role = Qt.DisplayRole):
        '''
        Returns the data for the given role and section
        in the header with the specified orientation.
        @type section: int
        @type orientation: Qt.Orientation
        @type role: int
        '''
        if role != Qt.DisplayRole or orientation == Qt.Vertical:
            return None
        return self.tr('Name')

    def rowCount(self, parent = QModelIndex()):
        '''
        Returns the number of rows under the given parent.
        @type parent: QModelIndex
        '''
        return len(self.__roster)

    def setContact(self, newItem):
        '''
        Add or update contact in the model
        @type newItem: RosterItem
        '''
        for index, item in enumerate(self.__roster):
            if item.jid == newItem.jid:
                self.__roster[index] = newItem
                self.reset()
                return
        self.__roster.append(newItem)
        self.reset()

    def removeContact(self, entity):
        '''
        Remove contact from the model
        @type entity: JID
        '''
        for item in self.__roster:
            if item.jid == entity:
                self.__roster.remove(item)
        self.reset()

    def setRoster(self, roster):
        '''
        Setter for roster
        @type roster: Roster as a mapping from L{JID} to L{RosterItem}.
        '''
        #pylint: disable=W0612
        self.__roster = [item for jid, item in roster.iteritems()]
        self.reset()

    def getRoster(self):
        '''
        Getter for roster
        @rtype: list of JID
        '''
        return [item.jid for item in self.__roster]

    roster = property(getRoster, setRoster, None, 'Roster')


class AffiliationModel(QAbstractTableModel):
    '''
    Model for affiliation dialog
    @author: Oleg Kandaurov
    '''

    readable('mapAffiliations')
    def __init__(self, parent = None):
        QAbstractTableModel.__init__(self, parent)
        self.__mapStates = {'subscribed': self.tr('Connected'),
                'unsubscribed': self.tr('Disconnected')}
        self.__mapAffiliations = {'publisher': self.tr('Full'),
                'member' : self.tr('Restricted'),
                'outcast': self.tr('None'),
                'owner': self.tr('Owner')}
        self.__column = {'jid': 0, 'affiliation': 1, 'state': 2}
        self.__mapHeaderLabels = {self.__column['jid']: self.tr('JID'),
                self.__column['affiliation']: self.tr('Permission'),
                self.__column['state']: self.tr('State')}
        self.__participants = list()

    def data(self, index, role = Qt.DisplayRole):
        '''
        Returns the data stored under the given role for the item referred to by the index.
        @type index: QModelIndex
        @type role: int
        '''
        if not index.isValid() or index.row() not in range(0, self.rowCount()) \
                or index.column() not in range(0, self.columnCount()) \
                or role == Qt.CheckStateRole:
            return None
        if role == Qt.TextAlignmentRole:
            return Qt.AlignVCenter | Qt.AlignLeft
        item = self.__participants[index.row()]
        mapData = {self.__column['jid']: item['entity'].userhost(),
                self.__column['affiliation']: self.__mapAffiliations[item['affiliation']],
                self.__column['state']: self.__mapStates[item['state']]}
        return mapData[index.column()]

    def flags(self, index):
        '''
        Returns the item flags for the given index
        @type index: QModelIndex
        @rtype: ItemFlags
        '''
        if not index.isValid() or index.column() == self.__column['jid'] \
                or index.column() == self.__column['state']:
            return Qt.ItemIsEnabled
        return Qt.ItemIsEnabled | Qt.ItemIsEditable | Qt.ItemIsSelectable

    def setData(self, index, value, role = Qt.EditRole):
        '''
        Sets the role data for the item at index to value.
        Returns true if successful; otherwise returns false.
        @type index: QModelIndex
        @type role: int
        '''
        if not index.isValid() or index.row() not in range(0, self.rowCount()) \
                or index.column() not in range(0, self.columnCount()) or \
                index.column() != self.__column['affiliation'] or role != Qt.EditRole:
            return False
        self.__participants[index.row()]['affiliation'] = value
        self.reset()
        return True

    def headerData(self, section, orientation, role = Qt.DisplayRole):
        '''
        Returns the data for the given role and section
        in the header with the specified orientation.
        @type section: int
        @type orientation: Qt.Orientation
        @type role: int
        '''
        if role != Qt.DisplayRole or orientation == Qt.Vertical:
            return None
        return self.__mapHeaderLabels[section]

    def rowCount(self, parent = QModelIndex()):
        '''
        Returns the number of rows under the given parent.
        @type parent: QModelIndex
        '''
        return len(self.__participants)

    def columnCount(self, parent = QModelIndex()):
        '''
        Returns the number of columns for the children of the given parent.
        @type parent: QModelIndex
        '''
        return len(self.__column)

    def updateState(self, entity, state):
        '''
        Update status of widgets according to received event
        @param entity: Subscriber that generates subscription event
        @type entity: JID
        @param status: Status of subscription
        @type status: str
        '''
        entity = entity.userhostJID()
        for item in self.__participants:
            if item['entity'] == entity:
                item['state'] = state
        self.reset()

    def setAffiliation(self, entity, affiliation):
        '''
        Set new affiliation for entity
        @type entity: JID
        @type affiliation: str
        '''
        entity = entity.userhostJID()
        for index, item in enumerate(self.__participants):
            if item['entity'] == entity:
                self.__participants[index]['affiliation'] = affiliation
                self.reset()
                return
        self.__participants.append({'entity': entity, 'affiliation': affiliation,
                'state':'unsubscribed'})
        self.reset()

    def getAffiliations(self):
        '''
        Extract affiliations from model
        @return: Dictionary of jid and affiliation
        @rtype: dict
        '''
        return dict((item['entity'].userhost(), item['affiliation'])
                for item in self.__participants)

    affiliations = property(getAffiliations, None, None, 'Affiliations')


class AffiliationDelegate(QStyledItemDelegate):
    '''
    Delegate for affiliation combo box
    @author: Oleg Kandaurov
    '''

    def __init__(self, parent = None):
        '''Constructor'''
        QStyledItemDelegate.__init__(self, parent)

    def createEditor(self, parent, option, index):
        '''
        Returns the widget used to edit the item specified by index for editing.
        The parent widget and style option are used to control how the editor widget appears.
        '''
        editor = QComboBox(parent)
        for data, name in index.model().mapAffiliations.iteritems():
            editor.addItem(name, data)
        return editor

    def setEditorData(self, editor, index):
        '''
        Sets the data to be displayed and edited by the editor
        from the data model item specified by the model index.
        '''
        value = index.model().data(index, Qt.DisplayRole)
        if value: editor.setCurrentIndex(editor.findText(value))

    def setModelData(self, editor, model, index):
        '''
        Gets data from the editor widget and stores
        it in the specified model at the item index.
        '''
        value = editor.itemData(editor.currentIndex())
        if value: model.setData(index, value, Qt.EditRole)

    def updateEditorGeometry(self, editor, option, index):
        '''
        Updates the editor for the item specified
        by index according to the style option given.
        '''
        editor.setGeometry(option.rect)
