# -*- coding: utf-8 -*-
# HiveMind - Distributed mind map editor for Maemo 5 platform
# Copyright (C) 2010-2011 HiveMind developers
#
# HiveMind is the legal property of its developers, whose names are
# noticed in @author or @authors annotations at the beginning of each
# module or class.
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02111-1301  USA


'''
Main window classes
'''

'''Install and start QtReactor'''
import sys
from PyQt4.QtGui import QApplication
app = QApplication(sys.argv)
from qt4reactor import qt4reactor
qt4reactor.install()
from hivemind.network import network_core
network_core.startReactor()

import getopt
import os
import re
from twisted.python import log
from PyQt4.QtGui import QActionGroup, QGraphicsBlurEffect, QMessageBox, QUndoStack, \
QFileDialog, QPixmap, QPainter, QAction, QDesktopServices, QPrintDialog, QPrinter
from PyQt4.QtCore import QObject, SIGNAL, QTimer, QLocale, QTranslator, QLibraryInfo, \
Qt, QRectF, QUrl
import hivemind.gui_factory as gui_factory
import hivemind.parser as parser
import hivemind.settings as settings
import hivemind.sysconfig as sysconfig
from hivemind.action_bag import ActionBag
from hivemind.attribute import writable, adaptCallable
from hivemind.commands import AddNodeCommand, StateModificationCommand, RemoveNodeCommand, \
MoveNodeCommand, ActionCommand, CurrentNodeCommand, HivemindCommand
from hivemind.core import MindMap
from hivemind.gui_delegates import MindMapDelegate, NodeLocation
from hivemind.gui_widgets import readable
from hivemind.robots import SmartConferenceRobot


class MainWindowController(QObject):
    '''
    Base controller class for main window widgets
    @author: Andrew Vasilev, Alexander Kulikov, Ilya Paramonov
    '''

    writable('mode')
    readable('mindMapScene')

    MODE_NORMAL = 'normal'
    '''All actions are parsed as normal'''

    MODE_TRANSFER = 'transfer'
    '''Map is in transfer mode'''

    def __init__(self):
        QObject.__init__(self)
        '''Init widgets and configure application'''
        self.__findWidget = None
        self.__actionBag = ActionBag(self)
        self.__mode = self.MODE_NORMAL
        self._createActions()
        self._createStateModificationActions()
        self.__mindMapScene = MindMapDelegate(self.__actionBag)
        QObject.connect(self.__mindMapScene, SIGNAL('currentNodeChanged'),
                     adaptCallable(self._setActionsState))
        self.__mainWindow = gui_factory.createMainWindow(self.__actionBag, self.__mindMapScene)
        self.__mindMapController = MindMapController(self.__mindMapScene, self.__actionBag)
        self.__networkController = network_core.NetworkController(
                self.__mindMapController, self.__actionBag)
        self.__smartConferenceRobot = SmartConferenceRobot(self.__mindMapScene)
        QObject.connect(self.__actionBag.startSmartConferenceRobot, SIGNAL('triggered()'), self.__smartConferenceRobot.start)
        QObject.connect(self.__actionBag.stopSmartConferenceRobot, SIGNAL('triggered()'), self.__smartConferenceRobot.stop)
        QObject.connect(self.__mainWindow.mindMapView, SIGNAL('itemClicked'), adaptCallable(self.onItemClicked))
        QObject.connect(self.__mainWindow, SIGNAL('closeEvent'), self._closeEvent)
        _configureApplication(self.__mindMapController)
        self.__mainWindow.showMaximized()

    def _setActionsState(self, currentNode):
        '''
        Set properties of the actions according to the current node selection
        @type currentNode: Node
        '''
        self.__widthActions[currentNode.edgeWidth].setChecked(True)
        self.__styleActions[currentNode.edgeStyle].setChecked(True)
        self.__actionBag.foldNodeAction.setEnabled(len(currentNode.children) > 0)
        self.__actionBag.foldNodeAction.setText(
               self.tr('Unfold node') if currentNode.folded else self.tr('Fold node'))
        self.__actionBag.foldNodeAction.setChecked(currentNode.folded is not None)
        self.__actionBag.removeNodeAction.setEnabled(
                currentNode.parent is not None and self.__mode == self.MODE_NORMAL)
        self.__actionBag.editLabelAction.setEnabled(
                currentNode.parent is not None and self.__mode == self.MODE_NORMAL)
        self.__actionBag.enterTransferModeAction.setEnabled(
                currentNode.parent is not None and self.__mode == self.MODE_NORMAL)
        self.__actionBag.addSiblingNodeBelowAction.setEnabled(
                currentNode.parent is not None and self.__mode == self.MODE_NORMAL)
        self.__actionBag.addSiblingNodeAboveAction.setEnabled(
                currentNode.parent is not None and self.__mode == self.MODE_NORMAL)
        self.__actionBag.editEdgeAction.setEnabled(self.__mode == self.MODE_NORMAL)
        self.__actionBag.addNodeAction.setEnabled(self.__mode == self.MODE_NORMAL)
        self.__actionBag.editNodeIconsAction.setEnabled(self.__mode == self.MODE_NORMAL)
        self.__actionBag.editNodeAction.setEnabled(self.__mode == self.MODE_NORMAL)
        self.__actionBag.selectRootNodeAction.setEnabled(self.__mode == self.MODE_NORMAL)
        self.__actionBag.cancelTransferModeAction.setEnabled(self.__mode == self.MODE_TRANSFER)
        self.__actionBag.putNodeAction.setEnabled(self.__mode == self.MODE_TRANSFER)
        self.__actionBag.putNodeBelowAction.setEnabled(self.__mode == self.MODE_TRANSFER)
        self.__actionBag.putNodeAboveAction.setEnabled(self.__mode == self.MODE_TRANSFER)
        self.__actionBag.showContextMenuAction.setEnabled(self.__mode == self.MODE_NORMAL)
        self.__actionBag.openHyperlinkAction.setEnabled(
                self.__mode == self.MODE_NORMAL and currentNode.link is not None)

    def _connectStateModificationAction(self, action, alteredAttributes, checkable = True):
        '''
        Connect action to a handler which alters given attributes
        @type action: QAction
        @param alteredAttributes: attributes to be altered by the handler
        @type alteredAttributes: dict
        @type checkable: bool
        '''
        def handler():
            self.createStateModificationCommand(
                    self.__mindMapScene.currentNode, alteredAttributes)
        QObject.connect(action, SIGNAL('triggered()'), handler)
        action.setCheckable(checkable)

    def _createStateModificationActions(self):
        '''Create state modification actions'''
        self._connectStateModificationAction(self.__actionBag.parentWidthAction,
            {'edgeWidth': None})
        self._connectStateModificationAction(self.__actionBag.thinWidthAction,
            {'edgeWidth': 11})
        self._connectStateModificationAction(self.__actionBag.oneWidthAction,
            {'edgeWidth': 1})
        self._connectStateModificationAction(self.__actionBag.twoWidthAction,
            {'edgeWidth': 2})
        self._connectStateModificationAction(self.__actionBag.fourWidthAction,
            {'edgeWidth': 4})
        self._connectStateModificationAction(self.__actionBag.eightWidthAction,
            {'edgeWidth': 8})
        self._connectStateModificationAction(self.__actionBag.parentStyleAction,
            {'edgeStyle': None})
        self._connectStateModificationAction(self.__actionBag.linearStyleAction,
            {'edgeStyle': 'linear'})
        self._connectStateModificationAction(self.__actionBag.bezierStyleAction,
            {'edgeStyle': 'bezier'})
        self._connectStateModificationAction(self.__actionBag.sharpLinearStyleAction,
            {'edgeStyle': 'sharp_linear'})
        self._connectStateModificationAction(self.__actionBag.sharpBezierStyleAction,
            {'edgeStyle': 'sharp_bezier'})
        self.__widthActions = {None : self.__actionBag.parentWidthAction,
                               11 : self.__actionBag.thinWidthAction,
                               1 : self.__actionBag.oneWidthAction,
                               2 : self.__actionBag.twoWidthAction,
                               4 : self.__actionBag.fourWidthAction,
                               8 : self.__actionBag.eightWidthAction}
        self.__styleActions = {'linear' : self.__actionBag.linearStyleAction,
                               'bezier' : self.__actionBag.bezierStyleAction,
                               'sharp_linear' : self.__actionBag.sharpLinearStyleAction,
                               'sharp_bezier' : self.__actionBag.sharpBezierStyleAction,
                               None : self.__actionBag.parentStyleAction}
        group = QActionGroup(self)
        for action in self.__widthActions.values():
            group.addAction(action)
        group = QActionGroup(self)
        for action in self.__styleActions.values():
            group.addAction(action)

    def _createActions(self):
        '''Create actions'''
        QObject.connect(self.__actionBag.editEdgeAction, SIGNAL('triggered()'), self.editEdge)
        QObject.connect(self.__actionBag.editLabelAction, SIGNAL('triggered()'), self.editLabel)
        QObject.connect(self.__actionBag.editNodeAction, SIGNAL('triggered()'), self.editNode)
        QObject.connect(self.__actionBag.editNodeIconsAction, SIGNAL('triggered()'), self.editNodeIcons)
        QObject.connect(self.__actionBag.foldNodeAction, SIGNAL('triggered()'), self.foldNode)
        QObject.connect(self.__actionBag.addNodeAction, SIGNAL('triggered()'), self.addNode)
        QObject.connect(self.__actionBag.addSiblingNodeBelowAction, SIGNAL('triggered()'), self._addSiblingNodeBelowAction)
        QObject.connect(self.__actionBag.addSiblingNodeAboveAction, SIGNAL('triggered()'), self._addSiblingNodeAboveAction)
        QObject.connect(self.__actionBag.removeNodeAction, SIGNAL('triggered()'), self.removeNode)
        QObject.connect(self.__actionBag.enterTransferModeAction, SIGNAL('triggered()'), self._startNodeMoving)
        QObject.connect(self.__actionBag.cancelTransferModeAction, SIGNAL('triggered()'), self._cancelNodeMoving)
        QObject.connect(self.__actionBag.putNodeAction, SIGNAL('triggered()'), self._putNode)
        QObject.connect(self.__actionBag.putNodeBelowAction, SIGNAL('triggered()'), self._putNodeBelow)
        QObject.connect(self.__actionBag.putNodeAboveAction, SIGNAL('triggered()'), self._putNodeAbove)
        QObject.connect(self.__actionBag.undoAction, SIGNAL('triggered()'), self._doUndo)
        QObject.connect(self.__actionBag.redoAction, SIGNAL('triggered()'), self._doRedo)
        QObject.connect(self.__actionBag.aboutAction, SIGNAL('triggered()'), self.onAbout)
        QObject.connect(self.__actionBag.exitAction, SIGNAL('triggered()'), self._closeMainWindow)
        QObject.connect(self.__actionBag.settingsAction, SIGNAL('triggered()'), self.onSettings)
        QObject.connect(self.__actionBag.findAction, SIGNAL('triggered()'), self.onFind)
        QObject.connect(self.__actionBag.showContextMenuAction, SIGNAL('triggered()'), self._virtualClick)
        QObject.connect(self.__actionBag.editHyperlinkAction, SIGNAL('triggered()'), self.editHyperlink)
        QObject.connect(self.__actionBag.openHyperlinkAction, SIGNAL('triggered()'), self.openHyperlink)

    def _doUndo(self):
        '''
        Record undo action
        '''
        self._recordCommand(ActionCommand('undo'))

    def _doRedo(self):
        '''
        Record redo action
        '''
        self._recordCommand(ActionCommand('redo'))

    def createStateModificationCommand(self, selectedNode, alteredAttributes):
        '''
        Create modification command using modified attributes on selected node

        @param selectedNode: the object of modification
        @type selectedNode: Node

        @param alteredAttributes: modified attributes
        @type alteredAttributes: Hash
        '''
        initialAttributeValues = selectedNode.getAttributes(getAll = True)
        unmodifiedAttributes = {}
        for key in alteredAttributes.keys():
            unmodifiedAttributes[key] = initialAttributeValues[key]
        self._recordCommand(StateModificationCommand(selectedNode, unmodifiedAttributes,
                                                     alteredAttributes))

    def _recordStateModificationCommand(self, node, dialog):
        '''
        Store node modifications made by dialog in the undo stack

        @param node: the object of modification
        @type node: Node

        @param dialog: dialog that performs change, must emit accepted() signal
                       after node modification
        @type dialog: QDialog
        '''
        initialAttributeValues = node.getAttributes(getAll = True)
        def _accept():
            alteredAttributes = dialog.attributes()
            # need to fix it, must use initialAttributeValues
            unmodifiedAttributes = {}
            for key in alteredAttributes.keys():
                if initialAttributeValues.has_key(key):
                    unmodifiedAttributes[key] = initialAttributeValues[key]
            self._recordCommand(StateModificationCommand(node, unmodifiedAttributes,
                    alteredAttributes))
        QObject.connect(dialog, SIGNAL('accepted()'), _accept)
        dialog.exec_()

    def _recordCommand(self, command):
        '''
        Process created command
        
        @param command: undo stack command to 
        '''
        self.__networkController.publishCommand(command)

    def _virtualClick(self):
        '''Emulate mouse click event'''
        self.__mainWindow.showMenu(True, self.__mindMapScene.currentNode)

    def onItemClicked(self, rightButton, item, parent, childLocation):
        if item: self._recordCommand(CurrentNodeCommand(item.node))
        if self.__mode == MainWindowController.MODE_NORMAL:
            if item is not None:
                self.__mainWindow.showMenu(rightButton, item)
        else:
            if parent.node.folded is None:
                self._finishNodeMoving(parent, childLocation)
            else:
                self.__mindMapScene.currentNode = item.node
                self.foldNode()

    def _addSiblingNodeBelowAction(self):
        '''Create and edit a sibling node, add it under selected node'''
        parentNode = self.__mindMapScene.currentNode.parent
        index = parentNode.children.index(self.__mindMapScene.currentNode)
        if parentNode is None:
            return
        def _accept():
            '''Process accept action'''
            newNode.setAttributes(**editor.attributes())
            self._recordCommand(AddNodeCommand(parentNode, newNode, index + 1))
        newNode = parentNode.createChildPrototype()
        newNode.setAttributes(left = self.__mindMapScene.currentNode.left)
        editor = gui_factory.createEditNodeDialog(newNode)
        QObject.connect(editor, SIGNAL('accepted()'), _accept)
        editor.exec_()

    def _addSiblingNodeAboveAction(self):
        '''Create and edit a sibling node, add it over selected node'''
        parentNode = self.__mindMapScene.currentNode.parent
        index = parentNode.children.index(self.__mindMapScene.currentNode)
        if parentNode is None:
            return
        def _accept():
            '''Process accept action'''
            newNode.setAttributes(**editor.attributes())
            self._recordCommand(AddNodeCommand(parentNode, newNode, index))
        newNode = parentNode.createChildPrototype()
        newNode.setAttributes(left = self.__mindMapScene.currentNode.left)
        editor = gui_factory.createEditNodeDialog(newNode)
        QObject.connect(editor, SIGNAL('accepted()'), _accept)
        editor.exec_()

    def _startNodeMoving(self):
        '''
        Switch controller and view to transfer mode
        '''
        self.__mode = self.MODE_TRANSFER
        self.__inactiveEffect = QGraphicsBlurEffect()
        self.__inactiveEffect.setBlurRadius(3)
        self.__movingNode = self.__mindMapScene.currentNode
        self.__mindMapScene.getDelegate(self.__movingNode) \
                .setGraphicsEffect(self.__inactiveEffect)
        self._setActionsState(self.__mindMapScene.currentNode)

    def _cancelNodeMoving(self):
        '''Cancel transfer mode. Switch controller and view to normal mode'''
        self.__mode = self.MODE_NORMAL
        self.__mindMapScene.getDelegate(self.__movingNode).setGraphicsEffect(None)
        self._setActionsState(self.__mindMapScene.currentNode)

    def _putNode(self):
        '''Move relocable node from old position to new parent'''
        if self.__mindMapScene.currentNode.folded:
            self.foldNode()
        else:
            if self.__mindMapScene.currentNode.parent is None:
                dialog = gui_factory.createSideSelectionDialog()
                dialog.exec_()
                if dialog.leftSide is None:
                    self._cancelNodeMoving()
                    return
                location = NodeLocation(not dialog.leftSide)
            else:
                location = NodeLocation()
            self._finishNodeMoving(self.__mindMapScene.getDelegate(
                    self.__mindMapScene.currentNode), location)

    def _putNodeBelow(self):
        '''Move relocable node as sibling below relative to current node'''
        parent = self.__mindMapScene.currentNode.parent
        if parent is not None:
            location = NodeLocation(not self.__mindMapScene.currentNode.left, None,
                    parent.getChildPosition(self.__mindMapScene.currentNode) + 1)
            self._finishNodeMoving(self.__mindMapScene.getDelegate(parent),
                    location)

    def _putNodeAbove(self):
        '''Move relocable node as sibling above relative to current node'''
        parent = self.__mindMapScene.currentNode.parent
        if parent is not None:
            location = NodeLocation(not self.__mindMapScene.currentNode.left, None,
                    parent.getChildPosition(self.__mindMapScene.currentNode))
            self._finishNodeMoving(self.__mindMapScene.getDelegate(parent),
                    location)

    def _finishNodeMoving(self, parent, location):
        '''
        Switch controller to normal mode
        @type parent: NodeDelegate
        @type location: NodeLocation
        '''
        self.__mode = self.MODE_NORMAL
        self.__mindMapScene.getDelegate(self.__movingNode).setGraphicsEffect(None)
        if parent is None: return
        if self.__movingNode == parent.node: return
        self.moveNode(self.__movingNode, parent.node, location)
        self._setActionsState(self.__mindMapScene.currentNode)

    @classmethod
    def onAbout(cls):
        '''
        Process about action
        '''
        gui_factory.createAboutDialog().exec_()

    @classmethod
    def onSettings(cls):
        '''
        Process settings action
        '''
        gui_factory.createSettingsDialog().exec_()

    def onFind(self):
        '''
        Process find action
        '''
        if self.__findWidget is None:
            self.__findWidget = gui_factory.createFindToolBar(self)
            if sysconfig.PLATFORM == 'maemo':
                self.__findWidget.move(self.__mainWindow.width() - self.__findWidget.width() + 10,
                        self.__mainWindow.height() - self.__findWidget.height() + 10)
            else:
                self.__mainWindow.addToolBar(Qt.BottomToolBarArea, self.__findWidget)
        if self.__findWidget.isVisible():
            self.__findWidget.setVisible(False)
        else:
            self.__findWidget.setVisible(True)
            self.__findWidget._findLineEdit.setFocus()

    def editEdge(self):
        '''Edit node's edge decorations and style'''
        dialog = gui_factory.createEditEdgeDialog(self.__mindMapScene.currentNode)
        self._recordStateModificationCommand(self.__mindMapScene.currentNode, dialog)

    def editNode(self):
        '''Edit selected node'''
        dialog = gui_factory.createEditNodeDialog(self.__mindMapScene.currentNode)
        self._recordStateModificationCommand(self.__mindMapScene.currentNode, dialog)

    def editNodeIcons(self):
        '''Edit set of icons for selected node'''
        dialog = gui_factory.createIconDialog(self.__mindMapScene.currentNode)
        self._recordStateModificationCommand(self.__mindMapScene.currentNode, dialog)

    def showNode(self, node):
        '''
        Show node if it was invisible
        @param node: node to show
        @type node: Node
        '''
        nodeParent = node.parent
        while nodeParent:
            if nodeParent.folded: self.foldNode(nodeParent)
            nodeParent = nodeParent.parent

    def foldNode(self, node = None):
        '''
        Fold node subtree
        @param node: node to fold
        @type node: Node
        '''
        if node is None:
            node = self.__mindMapScene.currentNode
        unmodifiedAttributes = {}
        alteredAttributes = {}
        unmodifiedAttributes['folded'] = node.folded
        if node.folded is None:
            alteredAttributes['folded'] = True
        else:
            alteredAttributes['folded'] = None
        self._recordCommand(StateModificationCommand(node, unmodifiedAttributes,
                alteredAttributes))

    def addNode(self):
        '''
        Create and edit a child node, add it to selected node
        '''
        parentNode = self.__mindMapScene.currentNode
        def _accept():
            '''Process accept action'''
            if self.__mindMapScene.currentNode.parent is None:
                dialog = gui_factory.createSideSelectionDialog()
                dialog.exec_()
                if dialog.leftSide is None:
                    # User doesn't accept dialog
                    return
                childNode.setAttributes(left = dialog.leftSide)
            if self.__mindMapScene.currentNode.folded:
                self.foldNode()
            childNode.setAttributes(**editor.attributes())
            self._recordCommand(AddNodeCommand(parentNode, childNode))
        childNode = parentNode.createChildPrototype()
        editor = gui_factory.createEditNodeDialog(childNode)
        QObject.connect(editor, SIGNAL('accepted()'), _accept)
        editor.exec_()

    def removeNode(self):
        '''
        Remove selected node
        '''
        def _recordCommand():
            self._recordCommand(RemoveNodeCommand(self.__mindMapScene.currentNode))
        if settings.get('confirmDelete'):
            ret = QMessageBox.question(self.__mainWindow, self.tr('Confirm delete'),
                  self.tr('Do you really want to delete the node?'),
                  QMessageBox.Yes, QMessageBox.No)
            if ret == QMessageBox.Yes:
                _recordCommand()
        else:
            _recordCommand()

    def moveNode(self, roamingNode, parentNode, location):
        '''
        Move node from current position to the new ancestor
        @param roamingNode: node to move
        @type roamingNode: Node
        @param parentNode: new parent for selected node
        @type parentNode: Node
        @param location: location relative to parent
        @type location: NodeLocation
        '''
        newPosition = location.position
        if newPosition is None:
            if location.top:
                newPosition = 0
            else:
                newPosition = len(parentNode.children)
        newSide = None
        if parentNode.parent is None:
            newSide = not location.right
        self._recordCommand(MoveNodeCommand(roamingNode, parentNode, newPosition, newSide))

    def editLabel(self):
        '''Edit edge label'''
        dialog = gui_factory.createEditLabelDialog(self.__mindMapScene.currentNode)
        self._recordStateModificationCommand(self.__mindMapScene.currentNode, dialog)

    def editHyperlink(self):
        '''Edit hyperlink'''
        dialog = gui_factory.createEditHyperlinkDialog(self.__mindMapScene.currentNode,
                self.__mindMapController.fileDirectory())
        self._recordStateModificationCommand(self.__mindMapScene.currentNode, dialog)

    def openHyperlink(self):
        '''Open node's hyperlink'''
        if self.__mindMapScene.currentNode.link:
            QDesktopServices.openUrl(QUrl(self.__mindMapScene.currentNode.link))

    def _closeMainWindow(self):
        '''Close main window of the application'''
        self.__mainWindow.close()

    def _closeEvent(self):
        '''Close network subsystem'''
        self.__smartConferenceRobot.stop()
        self.__networkController.terminateNetworkConnection()


class MindMapController(QObject):
    '''
    Mind map controller class for save and load mind maps
    @author: Andrew Vasilev, Alexander Kulikov
    '''

    readable('initialMindMap', 'undoStack')

    def __init__(self, mindMapScene, actionBag):
        '''
        @type mindMapScene: MindMapDelegate
        @type actionBag: ActionBag
        '''
        QObject.__init__(self)
        self.__mindMapScene = mindMapScene
        self.__actionBag = actionBag
        self._loadUndoStack(QUndoStack())
        self.__fileName = None
        self.__autosaved = False
        self.__autosaveTimer = QTimer(self)
        self.__autosaveTimer.setSingleShot(False)
        QObject.connect(self.__autosaveTimer, SIGNAL('timeout()'), self._autosaveMindMap)
        settings.connect(self._updateAutosave)
        self._updateAutosave()
        QObject.connect(self.__actionBag.fileNewAction, SIGNAL('triggered()'), self.onFileNew)
        QObject.connect(self.__actionBag.fileOpenAction, SIGNAL('triggered()'), self.onFileOpen)
        QObject.connect(self.__actionBag.fileSaveAction, SIGNAL('triggered()'), self.onFileSave)
        QObject.connect(self.__actionBag.fileSaveAsAction, SIGNAL('triggered()'), self.onFileSaveAs)
        QObject.connect(self.__actionBag.exportToImageAction, SIGNAL('triggered()'), self.onExportToImage)
        QObject.connect(self.__actionBag.printAction, SIGNAL('triggered()'), self.onPrint)
        for action in self.__actionBag.recentFileActions:
            QObject.connect(action, SIGNAL('triggered()'), self.openRecentFile)

    def _loadUndoStack(self, undoStack):
        '''
        Load new undo stack
        @param undoStack: UndoStack to be loaded
        @type undoStack: QUndoStack
        '''
        self.__undoStack = undoStack
        self.__actionBag.undoAction.setEnabled(undoStack.canRedo())
        self.__actionBag.redoAction.setEnabled(undoStack.canUndo())
        QObject.connect(self.__undoStack, SIGNAL('canUndoChanged(bool)'), adaptCallable(self.__actionBag.undoAction.setEnabled))
        QObject.connect(self.__undoStack, SIGNAL('canRedoChanged(bool)'), adaptCallable(self.__actionBag.redoAction.setEnabled))
        QObject.connect(self.__undoStack, SIGNAL('cleanChanged(bool)'), adaptCallable(self._undoStackStateChanged))

    def loadUndoStackFromNetwork(self, stack):
        '''
        Deserialize stack and load it
        @param stack: serialized stack
        @type stack: str
        '''
        self._loadUndoStack(parser.deserializeCommandStack(stack,
                self.__mindMapScene.mindMap))

    def _autosaveMindMap(self):
        '''Autosave current map'''
        if self.__undoStack.canUndo():
            if self.__fileName:
                path = self._generateAutosaveFilename(self.__fileName)
                parser.storeMindMap(path, self.__mindMapScene.mindMap)

    def _removeAutosave(self):
        '''Remove autosave of current map'''
        autosave = self._generateAutosaveFilename(self.__fileName)
        if autosave is not None:
            if os.path.isfile(autosave):
                os.remove(autosave)

    def _updateAutosave(self):
        '''Update autosave timer'''
        if settings.get('autosaveEnabled') == True:
            if settings.get('autosaveInterval') * 1000 != self.__autosaveTimer.interval():
                self.__autosaveTimer.start(settings.get('autosaveInterval') * 1000)
        else:
            self.__autosaveTimer.stop()

    def _getSaveFileName(self, title, fileTypes):
        '''
        This method call save dialog and return file name for saving
        @param title: title of the save dialog
        @type title: QString
        @param fileTypes: file types for save dialog filter
        @type fileTypes: hash
        @return: name of selected file
        @rtype: str
        '''
        filter = ''
        for type in fileTypes:
            filter = filter + type[0] + ';;'
        selectedFilter = fileTypes[0][0]
        fileName, currentFilter = gui_factory.invokeSaveFileDialog(title, self.fileDirectory(),
                filter, selectedFilter)
        if len(currentFilter) == 0:
            currentFilter = selectedFilter
        if len(fileName) == 0:
            return None
        for type in fileTypes:
            if type[0] == currentFilter:
                fileType = type
                if not re.match(fileType[1], fileName): fileName += fileType[2]
                break
        return unicode(fileName)

    def onExportToImage(self):
        '''Export mind map to image'''
        fileTypes = [[self.tr('PNG image (*.png)'), '.*\.png', '.png'],
                [self.tr('JPEG image (*.jpg *.jpeg)'), '.*\.(jpg|jpeg)', '.jpg'],
                [self.tr('Windows BMP image (*.bmp)'), '.*\.bmp', '.bmp']]
        fileName = self._getSaveFileName(self.tr('Save as image'), fileTypes)
        if not fileName:
            return
        size = self.__mindMapScene.itemsBoundingRect().size()
        pixmap = QPixmap(size.width(), size.height())
        pixmap.fill(Qt.white)
        painter = QPainter(pixmap)
        painter.setRenderHint(QPainter.Antialiasing)
        self.__mindMapScene.render(painter, QRectF(),
                self.__mindMapScene.itemsBoundingRect())
        painter.end()
        pixmap.save(fileName)

    def onPrint(self):
        '''Print mind map'''
        printer = QPrinter(QPrinter.HighResolution)
        printDialog = QPrintDialog(printer, gui_factory.defaultParent())
        if printDialog.exec_():
            painter = QPainter(printer)
            self.__mindMapScene.render(painter, QRectF(),
                self.__mindMapScene.itemsBoundingRect())
            painter.end()

    def onFileNew(self):
        '''Process file new action'''
        if not self.prohibitChangeMapAction():
            return
        self.loadFile(None)

    def onFileOpen(self):
        '''Process file open action'''
        if not self.prohibitChangeMapAction():
            return
        fileName = QFileDialog.getOpenFileName(gui_factory.defaultParent(), self.tr('Open'),
                self.fileDirectory(), self.tr('Mind maps (*.mm);;All Files (*)'))
        if not fileName: return
        self.loadFile(unicode(fileName))

    def onFileSave(self):
        '''
        Process file save action
        @return: True if mind map was saved, False otherwise
        @rtype: bool
        '''
        if self.__fileName is None:
            return self.onFileSaveAs()
        else:
            try:
                parser.storeMindMap(self.__fileName, self.__mindMapScene.mindMap)
                self.__undoStack.setClean()
                self.__autosaved = False
                return True
            except IOError, args:
                QMessageBox.critical(gui_factory.defaultParent(),
                        self.tr('File save error'), unicode(args[1]))
                return False

    def onFileSaveAs(self):
        '''
        Process file save as action
        @return: True if mind map was saved False otherwise
        @rtype: bool
        '''
        fileName = self._getSaveFileName(self.tr('Save as'),
                [[self.tr('Mind maps (*.mm)'), '.*\.mm', '.mm']])
        if not fileName: return False
        try:
            parser.storeMindMap(fileName, self.__mindMapScene.mindMap)
            self._setFileName(fileName)
            self.__undoStack.setClean()
            self.__autosaved = False
            settings.addRecentFile(fileName)
            return True
        except IOError, args:
            QMessageBox.critical(gui_factory.defaultParent(), self.tr('File save error'),
                    unicode(args[1]))
            return False

    def executeCommandFromNetwork(self, command):
        '''
        Deserialize command and execute it
        @param command: serialized command
        @type command: str
        '''
        command = parser.deserializeCommand(command, self.__mindMapScene.mindMap)
        self.executeCommand(command)

    def executeCommand(self, command):
        '''
        Execute command, created outside main controller
        @param command: command to execute
        @type command: str
        '''
        if not isinstance(command, HivemindCommand):
            self.__undoStack.push(command)
        elif isinstance(command, ActionCommand):
            {'undo': self.__undoStack.undo, 'redo': self.__undoStack.redo}[command.action]()
        elif isinstance(command, CurrentNodeCommand):
            self.__mindMapScene.currentNode = command.node

    def _undoStackStateChanged(self, clean):
        '''
        Update internal's according to undo stack state
        @param clean: undo stack clean state indication
        @type clean: bool
        '''
        self.__actionBag.fileSaveAction.setDisabled(clean)

    def _showSaveChangesBox(self):
        '''
        Ask user to make a decision about unsaved changes

        @return: result of message box invocation
        @rtype: int
        '''
        msgBox = QMessageBox(QMessageBox.Question, self.tr('Save changes?'),
                 self.tr('MindMap has been modified'),
                 QMessageBox.Save | QMessageBox.Discard | QMessageBox.Cancel,
                 gui_factory.defaultParent())
        msgBox.setInformativeText(self.tr('Do you want to save changes?'))
        msgBox.setDefaultButton(QMessageBox.Save)
        return msgBox.exec_()

    def prohibitChangeMapAction(self):
        '''
        Ask user permition to forget mind map changes
        @return: False if user canceled action, True otherwise
        @rtype: bool
        '''
        if self.__undoStack.isClean() and not self.__autosaved:
            self._removeAutosave()
            return True
        decition = self._showSaveChangesBox()
        if decition == QMessageBox.Save:
            if self.onFileSave():
                self._removeAutosave()
                return True
            else:
                return False
        elif decition == QMessageBox.Cancel:
            return False
        self._removeAutosave()
        return True

    def _setFileName(self, fileName):
        '''Set __filename field and window header'''
        self.__fileName = fileName
        if fileName is not None:
            gui_factory.defaultParent().setWindowTitle(os.path.split(fileName)[1])
        else:
            gui_factory.defaultParent().setWindowTitle(None)

    def fileDirectory(self):
        '''
        @return: Directory of the oppenned file or home
        @rtype: str
        '''
        if not self.__fileName: return os.path.expanduser('~')
        return os.path.dirname(self.__fileName)

    @classmethod
    def _generateAutosaveFilename(cls, path):
        '''
        Generate autosave file name for custom file name
        @type path: unicode
        '''
        if path is not None:
            dirname, filename = os.path.split(path)
            return os.path.join(dirname, '.#' + filename + '#')
        return None

    def openRecentFile(self):
        '''
        Open file selected from list of recent files
        '''
        if not self.prohibitChangeMapAction():
            return
        action = QAction.sender(self)
        if action:
            self.loadFile(unicode(action.data().toString()))

    def loadFile(self, path):
        '''
        Load mind map file and show chosen map.
        @param path: path to mind map file
        @type path: unicode
        '''
        filePath = path
        if self._showRecoverDataDialog(filePath):
            self.__autosaved = True
            filePath = self._generateAutosaveFilename(path)
        else:
            self.__autosaved = False
        try:
            mindMap = parser.loadMindMap(filePath) if filePath is not None else MindMap()
        except Exception:
            QMessageBox.critical(gui_factory.defaultParent(), self.tr('File open error'),
                    unicode(self.tr('Cannot read mind map from file %s')) %
                    os.path.split(path)[1])
            return
        self._setFileName(path)
        self._loadMindMap(mindMap)
        settings.addRecentFile(path)

    def loadMindMapFromNetwork(self, mindMap):
        '''
        Deserialize received mind map and load it
        @param mindMap: Serialized mind map
        @type mindMap: str
        '''
        self._setFileName(None)
        self._loadMindMap(parser.mindMapFromString(mindMap))

    def _loadMindMap(self, mindMap):
        '''
        Load mindmap into scene
        @type mindMap: MindMap
        '''
        self.__initialMindMap = mindMap
        self.__undoStack.setClean()
        self.__undoStack.clear()
        self.__mindMapScene.setMap(mindMap.clone())

    def _showRecoverDataDialog(self, filename):
        '''
        Show dialog to the user asking him/her whether there is a need to recover
        autosaved data copy or not
        
        @param filename: name of the mind map file
        @type filename: str
        
        @return: True if user want to recover data, False otherwise
        @rtype: bool
        '''
        if filename is None: return False
        autosavedFilename = self._generateAutosaveFilename(filename)
        if os.path.isfile(autosavedFilename):
            if QMessageBox.question(gui_factory.defaultParent(),
                    self.tr('Autosaved data recovery'),
                    unicode(self.tr('File %s has automatically saved data. Recover it?')) %
                            os.path.split(filename)[1],
                    QMessageBox.Yes | QMessageBox.No) == QMessageBox.Yes:
                return True
        return False


def _printHelp():
    '''Print general information about the application'''
    print '''HiveMind - Cross-platform collaborative mind map editor
Copyright (C) 2010-2011 HiveMind developers

Usage: hivemind [filename]

Options:
    -h  Display this help message and exit
'''

def _configureApplication(controller):
    '''
    Parse command line arguments and configure application
    
    @param controller: the mind map controller controller
    @type controller: MindMapController
    '''
    options, arguments = getopt.getopt(sys.argv[1:], 'hl')
    # parsing options
    for option in options:
        if option[0] == '-h':
            _printHelp()
            sys.exit()
        elif option[0] == '-l':
            log.startLogging(sys.stdout)
    # parsing arguments
    controller.loadFile(arguments[0].decode('utf-8') if arguments else None)

def _getLocale():
    '''Get locale for the application'''
    locale = settings.get('locale')
    if locale == '':
        locale = QLocale.system().name()
        settings.set(locale = locale)
    return locale

def run():
    '''Run HiveMind application'''
    # translator installation
    locale = _getLocale()[0:2]
    translatorStd = QTranslator()
    if translatorStd.load("%s/qt_%s.qm" %
            (QLibraryInfo.location(QLibraryInfo.TranslationsPath), locale)):
        app.installTranslator(translatorStd)
    translator = QTranslator()
    if translator.load(sysconfig.TRANSLATION_PATH_TEMPLATE % locale):
        app.installTranslator(translator)
    MainWindowController()
    sys.exit(app.exec_())
