# -*- coding: utf-8 -*-
# HiveMind - Distributed mind map editor for Maemo 5 platform
# Copyright (C) 2010-2011 HiveMind developers
#
# HiveMind is the legal property of its developers, whose names are
# noticed in @author or @authors annotations at the beginning of each
# module or class.
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02111-1301  USA

'''
Main window class for Desktop platform
'''

import hivemind.resource as resource
from PyQt4.QtCore import SIGNAL, Qt
from PyQt4.QtGui import QMenu, QCursor, QMainWindow, QIcon
from hivemind.main_window import MainView, AbstractMainWindow
from hivemind.attribute import readable


class DesktopMainView(MainView):
    '''
    Mouse input filter object for the desktop
    @author: Andrew Vasilev
    '''

    __MIN_DRAG_PATH = 20
    '''Minimum drag path length for the desktop'''

    def __init__(self, scene, actionBag):
        '''
        @type scene: MindMapDelegate
        @type actionBag: ActionBag
        '''
        MainView.__init__(self, scene, actionBag)

    def mouseReleaseEvent(self, event):
        '''
        Mouse release event handler.
        Emits itemClicked, mapClicked signals.

        @type event: QMouseEvent
        '''
        MainView.mouseReleaseEvent(self, event)
        if self.draggedPath < DesktopMainView.__MIN_DRAG_PATH:
            item = self.scene().itemAt(self.mapToScene(event.pos()))
            parent, childLocation = self.scene().parentAt(self.mapToScene(event.pos()))
            if parent is not None:
                self.emit(SIGNAL('itemClicked'), event.button() == Qt.RightButton, item,
                        parent, childLocation)
            if item is None:
                self.emit(SIGNAL('mapClicked'))


class DesktopMainWindow(AbstractMainWindow):
    '''
    Main window of HiveMind program for Desktop platform

    @author: Alexander Kulikov
    '''

    readable('mindMapView')

    def __init__(self, actionBag, mindMapScene):
        '''Create new window'''
        AbstractMainWindow.__init__(self, actionBag, mindMapScene)
        self.setWindowIcon(QIcon(resource.getScalableImage('hivemind')))
        self.__mindMapView = DesktopMainView(mindMapScene, actionBag)
        self.setCentralWidget(self.__mindMapView)
        self.__contextMenu = self._createContextMenu()
        self._createMainMenu()
        self._createToolBars()

    def setWindowTitle(self, fileName):
        if fileName:
            QMainWindow.setWindowTitle(self, fileName + ' - HiveMind')
        else:
            QMainWindow.setWindowTitle(self, 'HiveMind')

    def _createContextMenu(self):
        '''Create context menu'''
        menu = QMenu(self)
        menu.addAction(self._actionBag.addNodeAction)
        menu.addAction(self._actionBag.addSiblingNodeAboveAction)
        menu.addAction(self._actionBag.addSiblingNodeBelowAction)
        menu.addSeparator()
        menu.addAction(self._actionBag.foldNodeAction)
        menu.addAction(self._actionBag.enterTransferModeAction)
        menu.addAction(self._actionBag.removeNodeAction)
        menu.addSeparator()
        menu.addAction(self._actionBag.editNodeAction)
        menu.addAction(self._actionBag.editEdgeAction)
        menu.addAction(self._actionBag.editLabelAction)
        menu.addAction(self._actionBag.editNodeIconsAction)
        menu.addAction(self._actionBag.editHyperlinkAction)
        menu.addAction(self._actionBag.openHyperlinkAction)
        return menu

    def _addEllipsesToActionText(self):
        '''Add ellipses to the text of the actions displaying dialogs'''
        actionsWithEllipses = [
                self._actionBag.fileOpenAction, self._actionBag.fileSaveAsAction,
                self._actionBag.settingsAction, self._actionBag.exportToImageAction,
                self._actionBag.addNodeAction, self._actionBag.addSiblingNodeBelowAction,
                self._actionBag.addSiblingNodeAboveAction, self._actionBag.editNodeAction,
                self._actionBag.editNodeIconsAction, self._actionBag.editLabelAction,
                self._actionBag.editEdgeAction, self._actionBag.editAffiliationAction,
                self._actionBag.startClientAction, self._actionBag.networkSettingsAction,
                self._actionBag.printAction, self._actionBag.findAction,
                self._actionBag.editHyperlinkAction, self._actionBag.startServiceAction]
        for action in actionsWithEllipses:
            action.setText(action.text() + '...')

    def _createMainMenu(self):
        '''Create main menu'''
        self._addEllipsesToActionText()
        menu = self.menuBar().addMenu(self.tr('&File'))
        menu.addAction(self._actionBag.fileNewAction)
        menu.addAction(self._actionBag.fileOpenAction)
        menu.addAction(self._actionBag.fileSaveAction)
        menu.addAction(self._actionBag.fileSaveAsAction)
        menu.addSeparator()
        menu.addAction(self._actionBag.exportToImageAction)
        menu.addAction(self._actionBag.printAction)
        menu.addSeparator()
        for action in self._actionBag.recentFileActions:
            menu.addAction(action)
        menu.addSeparator()
        menu.addAction(self._actionBag.exitAction)
        menu = self.menuBar().addMenu(self.tr('&Edit'))
        menu.addAction(self._actionBag.undoAction)
        menu.addAction(self._actionBag.redoAction)
        menu.addSeparator()
        menu.addAction(self._actionBag.findAction)
        menu.addSeparator()
        menu.addAction(self._actionBag.settingsAction)
        menu = self.menuBar().addMenu(self.tr('&Network'))
        menu.addAction(self._actionBag.startServiceAction)
        menu.addAction(self._actionBag.startClientAction)
        menu.addAction(self._actionBag.editAffiliationAction)
        menu.addAction(self._actionBag.stopProtocolAction)
        menu.addAction(self._actionBag.disconnectXMPPAction)
        menu.addSeparator()
        menu.addAction(self._actionBag.networkSettingsAction)
#        menu.addAction(self._actionBag.networkHelpAction)
        menu = self.menuBar().addMenu(self.tr('SmartConference'))
        menu.addAction(self._actionBag.startSmartConferenceRobot)
        menu.addAction(self._actionBag.stopSmartConferenceRobot)
        menu = self.menuBar().addMenu(self.tr('&Help'))
        menu.addAction(self._actionBag.aboutAction)

    def _createToolBars(self):
        '''Create tool bar'''
        toolbar = self.addToolBar(self.tr('File'))
        toolbar.addAction(self._actionBag.fileNewAction)
        toolbar.addAction(self._actionBag.fileOpenAction)
        toolbar.addAction(self._actionBag.fileSaveAction)
        toolbar.addAction(self._actionBag.fileSaveAsAction)
        toolbar = self.addToolBar(self.tr('Edit'))
        toolbar.addAction(self._actionBag.undoAction)
        toolbar.addAction(self._actionBag.redoAction)
        toolbar = self.addToolBar(self.tr('View'))
        toolbar.addAction(self._actionBag.zoomOutAction)
        toolbar.addAction(self._actionBag.zoomInAction)
        toolbar = self.addToolBar(self.tr('Connection status'))
        toolbar.addAction(self._actionBag.connectionStatusAction)

    def showMenu(self, rightButton, item):
        '''Show the node menu on right button click'''
        if rightButton:
            self.__contextMenu.exec_(QCursor.pos())
