# -*- coding: utf-8 -*-
# HiveMind - Distributed mind map editor for Maemo 5 platform
# Copyright (C) 2010 HiveMind developers
#
# HiveMind is the legal property of its developers, whose names are
# noticed in @author or @authors annotations at the beginning of each
# module or class.
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02111-1301  USA

'''
Helper widgets
'''

import hivemind.settings as settings
import hivemind.resource as resource
from PySide.QtGui import *
from hivemind.attribute import *
from PySide.QtCore import *


class SearchingTrMixin(object):
    '''
    Mixin, defining the substitution for QObject.tr() method which searches translation
    along the inheritance hierarchy

    @author: Ilya Paramonov
    '''

    def tr(self, text):
        '''
        Search the class and its ancestors for the translation of the given text
        '''
        for clazz in self.__class__.mro():
            translatedText = QApplication.translate(clazz.__name__, text)
            if translatedText != text: return translatedText
            if clazz.__name__ == 'SearchingTrMixin': return text
        return text

class SpecialTextEdit(QTextEdit):
    '''
    Special text edit widget which handles Enter as newline character insertion
    and emits accept event on Ctrl-Enter

    @author: Alexander Kulikov
    '''

    def __init__(self, parent = None):
        '''
        Initialize widget
        @type parent: QWidget
        '''
        QTextEdit.__init__(self, parent)

    def keyPressEvent(self, event):
        '''
        Key press event handler
        @type event: QKeyEvent
        '''
        if event.key() == Qt.Key_Return:
            if event.modifiers() == Qt.NoModifier:
                self.emit(SIGNAL('accept'))
                return
            if event.modifiers() & Qt.ControlModifier:
                event = QKeyEvent(QEvent.KeyPress, event.key(), Qt.NoModifier)
            QTextEdit.keyPressEvent(self, event)
        else:
            QTextEdit.keyPressEvent(self, event)


class IconItem(QGraphicsWidget):
    '''
    Graphics widget provides rendering icons

    @author: Alexander Kulikov
    '''

    readable('icon')
    def __init__(self, icon, size, iconSize, parent = None):
        '''
        Initialize widget with specified icon name
        @type icon: string
        @type size: int
        @type iconSize: int
        @type parent: QGraphicsItem
        '''
        QGraphicsWidget.__init__(self)
        self.__icon = icon
        self.__size = size
        self.__drawFrame = True
        self.__iconSize = iconSize
        self.resize(size, size)

    def setSize(self, size):
        self.__size = size

    def setIconSize(self, size):
        self.__iconSize = size

    def paint(self, paint, options, widget):
        '''Paint icon item'''
        delta = (self.__size - self.__iconSize) / 2
        if isinstance(paint, QPainter):
            paint.setRenderHint(QPainter.Antialiasing)
            paint.setBrush(self.palette().brush(widget.backgroundRole()))
            paint.drawRoundedRect(0, 0, self.__size, self.__size, 6, 6)
            paint.drawImage(QRectF(delta, delta,
                            self.__iconSize, self.__iconSize),
                            resource.getIconByName(self.__icon))


class ListWidgetItem(QListWidgetItem):
    '''
    List widget item for drawing icons in icon table
    @author: Alexander Kulikov
    '''
    readable('icon')
    def __init__(self, icon, parent = None):
        QListWidgetItem.__init__(self, parent)
        self.__icon = icon
        pixmap = QPixmap.fromImage(resource.getIconByName(icon).scaled(QSize(32, 32)))
        self.setIcon(QIcon(pixmap))

class IconTable(QListWidget):
    '''
    Icon table provides icon location in a grid
    '''
    def __init__(self, parent = None):
        '''
        Initialize icon table
        @type parent: QWidget
        '''
        QListWidget.__init__(self, parent)
        self.setViewMode(QListView.IconMode)
        self.setResizeMode(QListView.Adjust)
        self.setMovement(QListView.Static)
        self.setSelectionMode(QListView.NoSelection)
        for icon in settings.get("listIcons"):
            item = ListWidgetItem(icon)
            self.addItem(item)
        self.connect(SIGNAL('itemClicked(QListWidgetItem*)'), self._onClick)

    def _onClick(self, item):
        self.emit(SIGNAL('iconClicked'), item.icon)


class IconList(QGraphicsScene):
    '''
    Icon list provides icon location in an animated list

    @author: Alexander Kulikov
    '''

    DEFAULT_HORIZONTAL_SPACING = 10
    '''Default horizontal space between icon items'''

    DEFAULT_ICON_SIZE = 32
    '''Default icon size into item'''

    DEFAULT_ITEM_SIZE = 64
    '''Default item size'''

    __MIN_DRAG_PATH = 10
    '''Minimum drag path length'''

    MAX_ICON_COUNT = 10
    '''Maximum icon count'''

    def __init__(self, icons, parent = None):
        '''
        Initialize with specified icon list
        @type icons: list
        @type parent: QObject
        '''
        QGraphicsScene.__init__(self, parent)
        self.__horizontalSpacing = self.DEFAULT_HORIZONTAL_SPACING
        self.__itemSize = self.DEFAULT_ITEM_SIZE
        self.__iconSize = self.DEFAULT_ICON_SIZE
        self._updateSize()
        self.__items = []
        for icon in icons:
            self.addIcon(icon)

    def _updateSize(self):
        '''Update scene size'''
        self.setSceneRect(0, 0, (self.__itemSize + self.__horizontalSpacing) * \
                           self.MAX_ICON_COUNT - self.__horizontalSpacing, \
                           self.__itemSize)

    def setItemSize(self, itemSize):
        '''
        Set item size of current items
        @type itemSize: int
        '''
        self.__itemSize = itemSize
        self._updateSize()
        for item in self.items():
            if isinstance(item, IconItem):
                item.setSize(itemSize)

    def setHorizontalSpacing(self, spacing):
        self.__horizontalSpacing = spacing
        self._updateSize()
        i = 0
        for item in self.items():
            if isinstance(item, IconItem):
                left = i * (self.__itemSize + self.__horizontalSpacing)
                i += 1
                item.setPos(left, 0)

    def setIconSize(self, iconSize):
        '''
        Set icon size of current items
        @type iconSize: int
        '''
        self.__iconSize = iconSize
        self._updateSize()
        for item in self.items():
            if isinstance(item, IconItem):
                item.setIconSize(iconSize)

    def addIcon(self, icon):
        '''
        Add new icon in the icon list
        @type icon: string
        '''
        index = len(self.__items)
        if (index < self.MAX_ICON_COUNT):
            item = IconItem(icon, self.__itemSize, self.__iconSize, self)
            item.setZValue(len(self.__items))
            x = index * (item.rect().width() + self.__horizontalSpacing)
            item.setPos(x, 0)
            self.__items.append(item)
            self.addItem(item)

    def removeIcon(self, item):
        '''
        Remove icon widget from the icon list
        @type item: IconWidget
        '''
        self.removeItem(item)
        self.__items.remove(item)
        item.setParentItem(None)

    def getIcons(self):
        '''
        Get icon name list
        @rtype: list
        '''
        return [item.icon for item in self.__items]

    def mousePressEvent(self, event):
        '''
        Mouse press event handler
        @type event: QMouseEvent
        '''
        QGraphicsScene.mousePressEvent(self, event)
        self.__startPos = event.screenPos()
        self.__item = self.itemAt(event.scenePos())
        if self.__item is not None:
            self.__item.setZValue(len(self.__items))
        self.__draggedPath = 0

    def mouseMoveEvent(self, event):
        '''
        Mouse move event handler
        @type event: QMouseEvent
        '''
        delta = event.screenPos() - self.__startPos
        self.__draggedPath = max(self.__draggedPath, delta.manhattanLength())
        if self.__item is not None:
            self._moveItem(self.__item, event.scenePos() - event.lastScenePos())

    def mouseReleaseEvent(self, event):
        '''
        Mouse release event handler
        @type event: QMouseEvent
        '''
        QGraphicsScene.mouseReleaseEvent(self, event)
        if self.__item is not None:
            self.__item.setZValue(self.__items.index(self.__item))
        if self.__draggedPath < self.__MIN_DRAG_PATH:
            item = self.itemAt(event.scenePos())
            if item is not None:
                self.removeIcon(item)
        self.__item = None
        self._generateAnimation()

    def _moveItem(self, item, delta):
        '''
        Move icon widget in the icon list and on the scene
        @type item: IconWidget
        @type delta: QPoint
        '''
        delta.setY(0)
        item.setPos(item.pos() + delta)
        index = self.__items.index(item)
        max = len(self.__items) - 1
        left = (index - 1) * (self.__itemSize + self.__horizontalSpacing)
        right = (index + 1) * (self.__itemSize + self.__horizontalSpacing)
        if index > 0 and item.pos().x() < left + self.__itemSize / 2:
            self.__items[index - 1], self.__items[index] = \
            self.__items[index], self.__items[index - 1]
            self._generateAnimation([item])
        elif index < max and item.pos().x() > right - self.__itemSize / 2:
            self.__items[index + 1], self.__items[index] = \
            self.__items[index], self.__items[index + 1]
            self._generateAnimation([item])

    def _generateAnimation(self, exceptions = []):
        '''
        Create and start move animation for icon widgets
        @param exceptions: Widgets do not require animation
        @type exceptions: list
        '''
        group = QParallelAnimationGroup(self)
        for i in xrange(0, len(self.__items)):
            item = self.__items[i]
            if item in exceptions:
                continue
            left = i * (self.__itemSize + self.__horizontalSpacing)
            deltaX = left - item.pos().x()
            if deltaX != 0:
                animation = QPropertyAnimation(item, 'pos', self)
                animation.setDuration(abs(deltaX) * 3)
                animation.setStartValue(item.pos())
                animation.setEndValue(QPointF(left, 0))
                group.addAnimation(animation)
        group.start()


class ImageButton(QAbstractButton):
    '''
    Command widget, only image is displayed to show the state of the button
    
    @author: Andrew Vasilev
    '''

    def __init__(self, parent = None, baseImageName = ''):
        '''
        @param parent: parent widget object
        @type parent: QWidget
        
        @param baseImageName: base name of an images to load
        @type baseImageName: str
        '''
        QAbstractButton.__init__(self, parent)
        self.__maxImageSize = QSize()
        if baseImageName != '':
            self.setImage(baseImageName)

    __NORMAL_STATE_IMAGE_SUFFIX = '_normal'
    '''Suffix for image to be displayed in normal mode'''

    __SELECTED_STATE_IMAGE_SUFFIX = '_selected'
    '''Suffix for image to be displayed in selected mode'''

    def setImage(self, baseImageName):
        '''
        @param baseImageName: base name of an images to load
        @type baseImageName: str
        '''
        self.__normalModeImage = resource.getImage(baseImageName +

                                                   ImageButton.__NORMAL_STATE_IMAGE_SUFFIX)
        self.__selectedModeImage = resource.getImage(baseImageName +
                                                      ImageButton.__SELECTED_STATE_IMAGE_SUFFIX)
        normalModeImageSize = self.__normalModeImage.size()
        selectedModeImageSize = self.__selectedModeImage.size()
        if normalModeImageSize.width() > selectedModeImageSize.width():
            self.__maxImageSize = normalModeImageSize
        else:
            self.__maxImageSize = selectedModeImageSize

    def sizeHint(self):
        '''
        @return: size of the button, maximum icon size
        @rtype: QSize
        '''
        return self.__maxImageSize

    def paintEvent(self, paintEvent):
        '''Paint icon in corresponding state'''
        painter = QPainter(self)
        if self.isDown() or self.isChecked():
            painter.drawPixmap(self.rect(), self.__selectedModeImage,
                               self.__selectedModeImage.rect())
        else:
            painter.drawPixmap(self.rect(), self.__normalModeImage,
                               self.__normalModeImage.rect())

    def moveCenter(self, x, y):
        '''
        Move button center to specified coordinates
        
        @param x: x coordinate of button center
        @type x: int
        
        @param y: y coordinate of button center
        @type y: int
        '''
        self.move(x - self.__maxImageSize.width() / 2.0,
                  y - self.__maxImageSize.height() / 2.0)


class MaemoZoomWidget(QWidget):
    '''
    Widget for change map scaling

    @author: Alexander Kulikov
    '''
    def __init__(self, parent, zoomInAction, zoomOutAction):
        '''
        Initialize zoom widget
        @type parent: QWidget
        @type zoomInAction: QAction
        @type zoomOutAction: QAction
        '''
        QWidget.__init__(self, parent)
        self.__zoomInButton = ImageButton(self, 'zoom_in')
        self.__zoomInButton.setAutoRepeat(True)
        self.connect(self.__zoomInButton, SIGNAL('pressed()'), zoomInAction.trigger)
        self.__zoomOutButton = ImageButton(self, 'zoom_out')
        self.__zoomOutButton.setAutoRepeat(True)
        self.connect(self.__zoomOutButton, SIGNAL('pressed()'), zoomOutAction.trigger)
        layout = QVBoxLayout()
        layout.setContentsMargins(5, 20, 5, 5)
        layout.addWidget(self.__zoomInButton)
        layout.addWidget(self.__zoomOutButton)
        self.setLayout(layout)


class WidgetVisibilityAnimation(QAbstractAnimation):
    '''
    Widget visibility changer for the animation framework
    
    @author: Andrew Vasilev
    '''

    def __init__(self, parent, visibility):
        '''
        Initialize animation
        
        @param parent: target widget for visibility change
        @type parent: QWidget
        
        @param visibility: the visibility value
        @param visibility: bool
        '''
        QAbstractAnimation.__init__(self, parent)
        self.__widget = parent
        self.__visibility = visibility

    def updateCurrentTime(self, currentTime):
        '''Update widget property at passed time'''
        if self.__visibility:
            self.__widget.show()
        else:
            self.__widget.hide()

    def duration(self):
        '''Animation is instant'''
        return 0


class ActionModificationCommand(QUndoCommand):
    '''
    Undo stack common comand. Stores do and undo functions
    '''
    def __init__(self, do, undo):
        '''
        @param do: function that makes changes
        @param undo: function that withdarws changes
        '''
        QUndoCommand.__init__(self)
        self.__do = do
        self.__undo = undo

    def undo(self):
        '''Restore previous state by calling undo function'''
        self.__undo()

    def redo(self):
        '''Move to next state by calling do function'''
        self.__do()


class SettingsCreator(QObject):
    '''
    Create and layout application settings
    @author: Oleg Kandaurov
    '''
    def __init__(self, settings, parent = None):
        '''
        @param settings: Settings that are need to create
        @type settings: list
        @type parent: QWidget
        '''
        QObject.__init__(self, parent)
        self.__settings = settings
        self.__mainLayout = QFormLayout()
        self.__component = dict()
        self.__create = {'locale': self._createLocale,
            'confirmDelete': self._createConfirmDelete,
            'autoHideNodeMenuTimeout': self._createAutoHideNodeMenuTimeout,
            'showNodeMenuAnimationDuration': self._createShowNodeMenuAnimationDuration,
            'hideNodeMenuAnimationDuration': self._createHideNodeMenuAnimationDuration,
            'defaultFont': self._createDefaultFont,
            'defaultLabelFont': self._createDefaultLabelFont,
            'defaultNodeTextColor': self._createDefaultNodeTextColor,
            'defaultNodeBackgroundColor': self._createDefaultNodeBackgroundColor,
            'defaultSelectedNodeBackgroundColor': self._createDefaultSelectedNodeBackgroundColor,
            'defaultEdgeColor': self._createDefaultEdgeColor,
            'defaultLabelColor': self._createDefaultLabelColor,
            'defaultEdgeWidth': self._createDefaultEdgeWidth}
        self._createComponents()

    def getLayout(self):
        return self.__mainLayout

    def _createComponents(self):
        '''
        Create components
        '''
        for setting in self.__settings:
            self.__create[setting]()

    def retrieve(self):
        '''
        Retrieve settings
        '''
        newSettings = list()
        for setting in self.__settings:
            newSettings.append(self.__component[setting].retrieve())
        return newSettings

    def _createLocale(self):
        '''
        Create locale components
        '''
        languages = [{'English': QLocale(QLocale.English).name()},
                {'Russian': QLocale(QLocale.Russian).name()},
                {'Deutsch': QLocale(QLocale.German).name()}]
        binder = ComboBoxBinder('locale', settings.get('locale'), languages)
        self.__mainLayout.addRow(self.tr('Language'), binder)
        self.__component['locale'] = binder


    def _createConfirmDelete(self):
        '''
        Create confirmDelete components
        '''
        binder = CheckBoxBinder('confirmDelete', settings.get('confirmDelete'))
        binder.setText(self.tr('Confirm node deletion'))
        self.__mainLayout.addRow(binder)
        self.__component['confirmDelete'] = binder

    def _createAutoHideNodeMenuTimeout(self):
        '''
        Create autoHideNodeMenuTimeout components
        '''
        binder = SpinBoxBinder('autoHideNodeMenuTimeout',
                settings.get('autoHideNodeMenuTimeout'), 0, 10000)
        self.__mainLayout.addRow(self.tr('Menu hide timeout'), binder)
        self.__component['autoHideNodeMenuTimeout'] = binder

    def _createShowNodeMenuAnimationDuration(self):
        '''
        Create showNodeMenuAnimationDuration components
        '''
        binder = SpinBoxBinder('showNodeMenuAnimationDuration',
                settings.get('showNodeMenuAnimationDuration'), 0, 10000)
        self.__mainLayout.addRow(self.tr('Menu appearance duration'), binder)
        self.__component['showNodeMenuAnimationDuration'] = binder

    def _createHideNodeMenuAnimationDuration(self):
        '''
        Create hideNodeMenuAnimationDuration components
        '''
        binder = SpinBoxBinder('hideNodeMenuAnimationDuration',
                settings.get('hideNodeMenuAnimationDuration'), 0, 10000)
        self.__mainLayout.addRow(self.tr('Menu disappearance duration'), binder)
        self.__component['hideNodeMenuAnimationDuration'] = binder

    def _createDefaultFont(self):
        '''
        Create defaultFont components
        '''
        binder = FontBinder('defaultFont', settings.get('defaultFont'))
        layout = QHBoxLayout()
        layout.addWidget(binder.fontCombo)
        layout.addWidget(binder.sizeCombo)
        self.__mainLayout.addRow(self.tr('Text font'), layout)
        self.__component['defaultFont'] = binder

    def _createDefaultLabelFont(self):
        '''
        Create defaultLabelFont components
        '''
        binder = FontBinder('defaultLabelFont',
                settings.get('defaultLabelFont'))
        layout = QHBoxLayout()
        layout.addWidget(binder.fontCombo)
        layout.addWidget(binder.sizeCombo)
        self.__mainLayout.addRow(self.tr('Label font'), layout)
        self.__component['defaultLabelFont'] = binder

    def _createDefaultNodeTextColor(self):
        '''
        Create defaultNodeTextColor components
        '''
        binder = ColorLabelBinder('defaultNodeTextColor',
                settings.get('defaultNodeTextColor'))
        binderButton = QPushButton(self.tr('Choose color'))
        layout = QHBoxLayout()
        layout.addWidget(binder, Qt.AlignLeft)
        layout.addWidget(binderButton, Qt.AlignLeft)
        self.__mainLayout.addRow(self.tr('Text color'), layout)
        binderButton.connect(SIGNAL('clicked()'), binder.open)
        self.__component['defaultNodeTextColor'] = binder

    def _createDefaultNodeBackgroundColor(self):
        '''
        Create defaultNodeBackgroundColor components
        '''
        binder = ColorLabelBinder('defaultNodeBackgroundColor',
                settings.get('defaultNodeBackgroundColor'))
        binderButton = QPushButton(self.tr('Choose color'))
        layout = QHBoxLayout()
        layout.addWidget(binder, Qt.AlignLeft)
        layout.addWidget(binderButton, Qt.AlignLeft)
        self.__mainLayout.addRow(self.tr('Background color'), layout)
        binderButton.connect(SIGNAL('clicked()'), binder.open)
        self.__component['defaultNodeBackgroundColor'] = binder

    def _createDefaultEdgeColor(self):
        '''
        Create defaultEdgeColor components
        '''
        binder = ColorLabelBinder('defaultEdgeColor',
                settings.get('defaultEdgeColor'))
        binderButton = QPushButton(self.tr('Choose color'))
        layout = QHBoxLayout()
        layout.addWidget(binder, Qt.AlignLeft)
        layout.addWidget(binderButton, Qt.AlignLeft)
        self.__mainLayout.addRow(self.tr('Edge color'), layout)
        binderButton.connect(SIGNAL('clicked()'), binder.open)
        self.__component['defaultEdgeColor'] = binder

    def _createDefaultLabelColor(self):
        '''
        Create defaultLabelColor components
        '''
        binder = ColorLabelBinder('defaultLabelColor',
                settings.get('defaultLabelColor'))
        binderButton = QPushButton(self.tr('Choose color'))
        layout = QHBoxLayout()
        layout.addWidget(binder, Qt.AlignLeft)
        layout.addWidget(binderButton, Qt.AlignLeft)
        self.__mainLayout.addRow(self.tr('Label color'), layout)
        binderButton.connect(SIGNAL('clicked()'), binder.open)
        self.__component['defaultLabelColor'] = binder

    def _createDefaultEdgeWidth(self):
        '''
        Create defaultEdgeWidth components
        '''
        width = [{'1': 1}, {'2': 2}, {'4': 4}, {'8': 8}, {self.tr('Thin'): 11}]
        binder = ComboBoxBinder('defaultEdgeWidth',
                settings.get('defaultEdgeWidth'), width)
        self.__mainLayout.addRow(self.tr('Edge width'), binder)
        self.__component['defaultEdgeWidth'] = binder

    def _createDefaultSelectedNodeBackgroundColor(self):
        '''
        Create defaultSelectedNodeBackgroundColor component
        '''
        binder = ColorLabelBinder('defaultSelectedNodeBackgroundColor',
                settings.get('defaultSelectedNodeBackgroundColor'))
        binderButton = QPushButton(self.tr('Choose color'))
        layout = QHBoxLayout()
        layout.addWidget(binder, Qt.AlignLeft)
        layout.addWidget(binderButton, Qt.AlignLeft)
        self.__mainLayout.addRow(self.tr('Selected node background'), layout)
        binderButton.connect(SIGNAL('clicked()'), binder.open)
        self.__component['defaultSelectedNodeBackgroundColor'] = binder


class ComponentBinder(QObject):
    '''
    Abstract component binder
    @author: Oleg Kandaurov
    '''
    def __init__(self, paramName, parent = None):
        '''
        Initialize binder
        @param paramName: Parameter that is binded to component
        @type paramName: str
        @type parent: QWidget
        '''
        self._value = None
        self._paramName = paramName

    def isValid(self, value):
        '''
        @return: whether value correct or not
        @rtype: bool
        '''
        return True

    def retrieve(self):
        '''
        @return: pair parameter and value
        @rtype: dict
        '''
        if self.isValid(self._value):
            return {self._paramName: self._value}
        return None

    def _setValue(self, value):
        '''
        Sets new value
        @param value: new value
        '''
        if self.isValid(value):
            self._value = value


class LineEditBinder(QLineEdit, ComponentBinder):
    '''
    QLineEdit binder
    @author: Oleg Kandaurov
    '''
    def __init__(self, paramName, text, regexp = None, parent = None):
        '''
        Initialize binder
        @param paramName: Parameter that is binded to component
        @type paramName: str
        @param text: Initial text
        @type text: str
        @param regexp: Regular expression needed to inspect line edit values
        @type regexp: QRegExp
        @type parent: QWidget
        '''
        QLineEdit.__init__(self, parent)
        ComponentBinder.__init__(self, paramName, parent)
        self._value = text
        self.__regexp = QRegExp('.*') if regexp is None else regexp
        self.setText(text)
        SIGNAL('valid(bool)')
        self.textChanged.connect(self._setValue)

    def isValid(self, value = None):
        '''
        @return: whether value correct or not
        @rtype: bool
        '''
        if value is None: value = self.text()
        if self.__regexp.exactMatch(value):
            return True
        return False


class SpinBoxBinder(QSpinBox, ComponentBinder):
    '''
    QSpinBox binder
    @author: Oleg Kandaurov
    '''
    def __init__(self, paramName, value, min, max, parent = None):
        '''
        Initialize binder
        @param paramName: Parameter that is binded to component
        @type paramName: str
        @param value: Initial value
        @type value: int
        @param min: Minimum value
        @type min: int
        @param max: Maximum value
        @type max: int
        @type parent: Qwidget
        '''
        QSpinBox.__init__(self, parent)
        ComponentBinder.__init__(self, paramName, parent)
        self.setMinimum(min)
        self.setMaximum(max)
        self.setValue(value)
        self._value = value
        self.valueChanged.connect(self._setValue)


class CheckBoxBinder(QCheckBox, ComponentBinder):
    '''
    QCheckBox binder
    @author: Oleg Kandaurov
    '''
    def __init__(self, paramName, value, parent = None):
        '''
        Initialize binder
        @param paramName: Parameter that is binded to component
        @type paramName: str
        @param value: Initial state
        @type value: bool
        @type parent: Qwidget
        '''
        QCheckBox.__init__(self, parent)
        ComponentBinder.__init__(self, paramName, parent)
        self._value = value
        self.toggled.connect(self._setValue)
        self.setChecked(value)


class ComboBoxBinder(QComboBox, ComponentBinder):
    '''
    QComboBox binder
    @author: Oleg Kandaurov
    '''
    def __init__(self, paramName, value, data, parent = None):
        '''
        Initialize binder
        @param paramName: Parameter that is binded to component
        @type paramName: str
        @param value: Initial string
        @type value: str
        @param data: Possible values in QComboBox
        @type data: list of dicts
        @type parent: QWidget
        '''
        QComboBox.__init__(self, parent)
        ComponentBinder.__init__(self, paramName, parent)
        self.__data = data
        self._value = value
        for values in data:
            self.addItem(values.keys().pop(), values.values().pop())
        self.setCurrentIndex(self.findData(value))
        self.connect(self, SIGNAL('activated(int)'), self._setValue)

    def _setValue(self, index):
        '''
        Sets new value
        @param value: index in combo box
        @type value: int
        '''
        self._value = self.__data[index].values().pop()


class ColorLabelBinder(QLabel, ComponentBinder):
    '''
    QColorDialog binder
    @author: Oleg Kandaurov
    '''
    def __init__(self, paramName, color, parent = None):
        '''
        Initialize binder
        @param paramName: Parameter that is binded to component
        @type paramName: str
        @param color: Initial color
        @type color: QColor
        @type parent: QWidget
        '''
        QLabel.__init__(self, parent)
        ComponentBinder.__init__(self, paramName, parent)
        self.__palette = self.palette()
        self.__effect = QGraphicsBlurEffect()
        self.__effect.setBlurRadius(2)
        self.setGraphicsEffect(self.__effect)
        self.setAlignment(Qt.AlignCenter)
        self.setAutoFillBackground(True)
        self.setFrameStyle(QFrame.StyledPanel | QFrame.Raised)
        self._setValue(color)

    def open(self):
        '''
        Open color dialog and sets color
        '''
        color = QColorDialog.getColor(self._value, self)
        self._setValue(color)

    def _setLabelColor(self, color):
        '''
        Change color of the label
        @type color: QColor
        '''
        self.__palette.setColor(QPalette.Window, color)
        self.setPalette(self.__palette)

    def isValid(self, color):
        '''
        @return: whether color valid or not
        @rtype: bool
        '''
        return color.isValid()

    def _setValue(self, value):
        '''
        Sets new value
        @param value: new value
        '''
        if self.isValid(value):
            self._setLabelColor(value)
            self._value = value


class FontBinder(QWidget, ComponentBinder):
    '''
    Font binder
    @author: Oleg Kandaurov
    '''
    readable('fontCombo', 'sizeCombo')
    def __init__(self, paramName, font, parent = None):
        '''
        Initialize binder
        @param paramName: Parameter that is binded to component
        @type paramName: str
        @param font: initial font
        @type font: QFont
        @type parent: QWidget
        '''
        QWidget.__init__(self, parent)
        ComponentBinder.__init__(self, paramName, parent)
        self._value = font
        self.__font = font
        self.__size = font.pointSize()
        self.__sizeCombo = QComboBox()
        self.__sizeCombo.addItem(str(self.__size))
        self.__fontCombo = QFontComboBox()
        self.__fontCombo.setCurrentFont(font)
        self._setFont(font)
        self.__fontCombo.connect(SIGNAL('currentFontChanged(QFont)'), self._setFont)
        self.__sizeCombo.connect(SIGNAL('currentIndexChanged(int)'), self._setSize)

    def _setFont(self, font):
        '''
        Set font and update available sizes
        @type font: QFont
        '''
        self.__font = font
        self._fontSizes(font)
        self._setSizeIndex(self.__size)
        self.__font.setPointSize(self.__size)
        self._setValue(self.__font)

    def _setSize(self, index):
        '''
        Set size of font
        @type index: int
        '''
        if index == -1:
            self.__sizeCombo.addItem(str(self.__size))
            return
        self.__size = int(self.__sizeCombo.itemText(index))
        self.__font.setPointSize(self.__size)
        self._setValue(self.__font)

    def _setSizeIndex(self, size):
        '''
        Set font size in QComboBox
        @type size: int
        '''
        index = self.__sizeCombo.findText(str(size), Qt.MatchExactly)
        if index == -1:
            self.__sizeCombo.addItem(str(size))
        else:
            self.__sizeCombo.setCurrentIndex(index)

    def _fontSizes(self, font):
        '''
        Add to QComboBox sizes supported by current font
        @type font: QFont
        '''
        self.__sizeCombo.clear()
        self.__sizeCombo.addItems([str(value) for value in QFontDatabase().\
                                   pointSizes(font.family())])


class FullScreenExitButton(QToolButton):
    '''
    Full screen exit button for Maemo platform

    @author: Oleg Kandaurov
    '''
    def __init__(self, parent = None):
        '''
        @type parent: QWidget
        '''
        QToolButton.__init__(self, parent)
        self.setIcon(QIcon.fromTheme('general_fullsize'))
        self.setFixedSize(self.sizeHint())
        pal = self.palette()
        backgroundColor = pal.color(self.backgroundRole())
        backgroundColor.setAlpha(128)
        pal.setColor(self.backgroundRole(), backgroundColor)
        self.setPalette(pal)
        self.setAutoFillBackground(True)
        self.setVisible(False)
        self.connect(self, SIGNAL('clicked()'), parent, SLOT('showNormal()'));
        parent.installEventFilter(self);

    def eventFilter(self, obj, event):
        '''
        Catch Windows state changed and Resize events from parent object
        @type obj: QEvent
        @type event: QEvent
        '''
        parent = self.parentWidget()
        if obj != self.parent():
            return QToolButton.eventFilter(self, obj, event)
        isFullScreen = parent.windowState() & Qt.WindowFullScreen
        if event.type() == QEvent.WindowStateChange:
            if isFullScreen:
                self.setVisible(True)
                self.raise_()
            else:
                self.setVisible(False)
        elif event.type() == QEvent.Resize:
            if self.isVisible():
                self.move(parent.width() - self.width(), parent.height() - self.height())
        return QToolButton.eventFilter(self, obj, event)
