# -*- coding: utf-8 -*-
# HiveMind - Distributed mind map editor for Maemo 5 platform
# Copyright (C) 2010 HiveMind developers
#
# HiveMind is the legal property of its developers, whose names are
# noticed in @author or @authors annotations at the beginning of each
# module or class.
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02111-1301  USA

'''
View delegates for core objects
'''

import hivemind.settings as settings
import resource
import math
from hivemind.attribute import *
from hivemind.core import MindMap
from PySide.QtGui import *
from PySide.QtCore import *

class MindMapDelegate(QGraphicsScene):
    '''
    MindMap delegate
    
    Warning! C{currentNode} field is self-encapsulated!
    You must not use direct invocation (C{self.__currentNode})
    inside the class!
    
    @authors: Andrew Golovchenko, Andrew Vasilev, Alexander Kulikov
    '''

    readable('rootDelegate', 'mindMap')

    def _getCurrentNode(self):
        '''Get current node'''
        return self.__currentNode

    def _setCurrentNode(self, currentNode):
        '''
        Set the current node
        @type currentNode: Node
        '''
        if self.__nodeToDelegate.has_key(self.__currentNode):
            self.__nodeToDelegate[self.__currentNode].selected = False
        self.__currentNode = currentNode
        delegate = self.__nodeToDelegate[currentNode]
        delegate.selected = True
        delegate.ensureVisible(delegate.textBoundingRect())
        self.emit(SIGNAL('currentNodeChanged'), self.__currentNode)

    currentNode = property(_getCurrentNode, _setCurrentNode, None, 'Current node')

    def __init__(self):
        '''Initialize empty mind map'''
        QGraphicsScene.__init__(self)
        settings.connect(self._settingsChanged)
        self.__nodeToDelegate = {}
        self.__currentNode = None
        # used as 'switch' operator in processNotification() method
        self.__signalHandlers = {'childNodeAdded' : self.addNodeDelegate,
                                 'childNodeRemoved' : self.removeNodeDelegate,
                                 'nodeMoved' : self.relocateNodeDelegate,
                                 'nodeBoundsChanged' : self.nodeBoundsChanged,
                                 'localAttributeChanged' : self.repaintNode,
                                 'inheritedAttributeChanged' : self.update}
        self.setMap(MindMap())

    def addNodeDelegate(self, node):
        '''
        This method called, then signal for created delegate has come
        @param node: node was clicked
        @type node: Node
        '''
        self._addDelegatesForNodeSubtree(node)
        self.positionNodes()
        self.currentNode = node

    def _addDelegatesForNodeSubtree(self, root):
        '''
        Create delegates for all node descendants
        @param root: subtree root
        @type root: Node
        '''
        parentDelegate = self.__nodeToDelegate[root.parent]
        newDelegate = NodeDelegate(root, parentDelegate)
        root.addObserver(self)
        self.__nodeToDelegate[root] = newDelegate
        for child in root.children:
            self._addDelegatesForNodeSubtree(child)

    def nodeBoundsChanged(self, node):
        '''
        This method called, then signal for node bounds changed has come
        @param node: node was modify
        @type node: Node
        '''
        if node.children:
            visible = False if node.folded is not None else True
            if self.__nodeToDelegate[node.children[0]].isVisible() != visible:
                for child in node.children:
                    self.__nodeToDelegate[child].setVisible(visible)
        self.positionNodes(node)

    def _settingsChanged(self):
        '''Update and position all nodes on the mind map'''
        self.__mindMap.root.updateProperties()
        self.positionNodes()

    def processNotification(self, sender, *values, **args):
        '''
        Call handlers of notifications by type

        @param sender: source of notification
        @type sender: Observable
        '''
        notifyType = args.get('type')
        args = () if notifyType == 'inheritedAttributeChanged' else values
        self.__signalHandlers.get(notifyType)(*args)

    def removeNodeDelegate(self, node):
        '''
        Remove the delegate that contains the specified node
        @type node: Node
        '''
        delegate = self.__nodeToDelegate[node]
        if node.isAncestorOf(self.__currentNode):
            self.currentNode = delegate.parentItem().node
        self._removeNodeDelegateSubtree(node)
        self.removeItem(delegate)
        self.positionNodes()

    def _removeNodeDelegateSubtree(self, root):
        '''
        Remove the node and its children from the node-to-delegate dictionary
        @type root: Node
        '''
        for child in root.children:
            self._removeNodeDelegateSubtree(child)
        self.__nodeToDelegate.pop(root)

    def relocateNodeDelegate(self, node, dest):
        '''
        Relocate the delegate of the node to the destination

        @param node: transfered node
        @type node: Node
        
        @param dest: destination node
        @type dest: Node
        '''
        parentDelegate = self.__nodeToDelegate[dest]
        roamingDelegate = self.__nodeToDelegate[node]
        roamingDelegate.setParentItem(parentDelegate)
        if dest.folded: roamingDelegate.hide()
        # node relocation require recalculate position of all nodes
        self.positionNodes()

    def setMap(self, mindMap):
        '''
        Load new mind map and show it
        
        @param mindMap: mind map to be shown
        @type mindMap: MindMap
        '''
        # clear old values
        self.__nodeToDelegate.clear()
        for item in self.items(): self.removeItem(item)
        # set new values
        self.__mindMap = mindMap
        self.__rootDelegate = NodeDelegate(mindMap.root)
        self.__nodeToDelegate[mindMap.root] = self.__rootDelegate
        self.addItem(self.__rootDelegate)
        self.createNodeDelegates(self.__rootDelegate)
        self.positionNodes()
        self.currentNode = mindMap.root

    _MAP_MARGIN_WIDTH = 300
    ''' Width of mind map margin '''
    _MAP_MARGIN_HEIGHT = 150
    ''' Height of mind map margin '''

    def _updateSceneRectangle(self):
        '''Set new scene rectangle for the map. Add additional margin.'''
        sceneRect = self.itemsBoundingRect()
        sceneRect.adjust(-self._MAP_MARGIN_WIDTH, -self._MAP_MARGIN_HEIGHT, self._MAP_MARGIN_WIDTH,
                         self._MAP_MARGIN_HEIGHT)
        self.setSceneRect(sceneRect)

    def createNodeDelegates(self, parentNodeDelegate):
        '''
        Create node delegates from parent
        
        @type parentNodeDelegate: NodeDelegate
        '''
        parentNodeDelegate.node.addObserver(self)
        childNodes = parentNodeDelegate.node.children
        for node in childNodes:
            delegate = NodeDelegate(node, parentNodeDelegate)
            self.__nodeToDelegate[node] = delegate
            if parentNodeDelegate.node.folded:
                delegate.hide()
            self.createNodeDelegates(delegate)

    def positionNodes(self, node = None):
        '''
        Position all nodes on the mind map

        @param node: changed node
        @type node: Node
        '''
        self.__rootDelegate.positionTree()
        self.update()
        self._updateSceneRectangle()

    def repaintNode(self, node):
        '''Repaint node bounding area'''
        self.__nodeToDelegate[node].update()

    def getDelegate(self, node):
        '''
        @return: delegate for the passed Node or None if no delegate found
        @rtype: NodeDelegate
        '''
        return self.__nodeToDelegate.get(node)

    def itemAt(self, position):
        '''
        @type position: QPointF
        @return: item at specified coordinates or None if no item was found.
        Uses textBoundingRect for accurate choice.
        @rtype: NodeDelegate
        '''
        for item in self.items(position):
            if item.textRectOnScene().contains(position):
                return item
        return None

    def parentAt(self, position):
        '''
        Return parent at specified coordinates and location relative to the parent.

        @type position: QPointF
        @rtype: NodeDelegate, NodeLocation
        '''
        parent = self._getParent(position)
        if parent is None: return None, None
        above, below = self._neighbourDelegates(position, parent)
        right = self._isPositionToTheRight(position, self.__rootDelegate)
        if self.__rootDelegate.textRectOnScene().contains(position):
            return parent, NodeLocation(right, self._isPositionAbove(position, self.__rootDelegate), None)
        nearest = self._nearestDelegateByOrdinate(position, above, below)
        leftSideOfTheNode = (position.x() < nearest.textRectOnScene().center().x())
        if leftSideOfTheNode and right or not leftSideOfTheNode and not right:
            if below is None:
                insertPos = parent.node.getChildPosition(above.node) + 1
            else:
                insertPos = parent.node.getChildPosition(below.node)
            return parent, NodeLocation(right, None, insertPos)
        else:
            return nearest, NodeLocation(right, self._isPositionAbove(position, nearest), None)

    def _neighbourDelegates(self, position, parent):
        '''
        Return two children of the parent that are neighbor to the position by y-coordinate

        @type position: QPointF
        @type parent: NodeDelegate
        @rtype: NodeDelegate, NodeDelegate
        '''
        if parent is None: return None, None
        prev = None
        for node in parent.node.children:
            if node.left is self._isPositionToTheRight(position, self.__rootDelegate):
                continue
            child = self.getDelegate(node)
            if prev is None:
                if position.y() < child.textRectOnScene().center().y():
                    return None, child
            elif position.y() > prev.textRectOnScene().center().y() and \
                    position.y() < child.textRectOnScene().center().y():
                return prev, child
            prev = child
        return prev, None

    def _nearestDelegateByOrdinate(self, position, above, below):
        '''
        Return one of the delegates that are closer to the position by y-coordinate

        @type position: QPointF
        @type above: NodeDelegate
        @type below: NodeDelegate
        @rtype: NodeDelegate
        '''
        if above is None:
            return below
        if below is None:
            return above
        return above if above.yDistanceTo(position) < below.yDistanceTo(position) else below

    def _nearestDelegateByAbscissa(self, position, delegates):
        '''
        Return one of the delegates that are closer to the position by x-coordinate

        @type position: QPointF
        @param delegates: list of the delegates
        @type delegates: list
        @rtype: NodeDelegate
        '''
        if delegates == []: return None
        nearest = delegates[0]
        for item in delegates:
            if item is None: continue
            if nearest.xDistanceTo(position) > item.xDistanceTo(position):
                nearest = item
        return nearest

    def _getParent(self, position):
        '''
        Return parent for the position on scene

        @type position: QPointF
        @rtype: NodeDelegate
        '''
        children = self.items(position)
        if children is None: return None
        parents = set()
        for child in children:
            if (child.parentItem()) is not None:
                parents.add(child.parentItem())
            else:
                parents.add(self.__rootDelegate)
        parent = self._nearestDelegateByAbscissa(position, list(parents))
        return parent

    def _isPositionToTheRight(self, position, item):
        '''
        Determine if the given position is to the right from the item

        @type position: QPointF
        @type item: NodeDelegate
        @rtype: bool
        '''
        return position.x() > item.textRectOnScene().center().x()

    def _isPositionAbove(self, position, item):
        '''
        Determine if the given position is above the item

        @type position: QPointF
        @type item: NodeDelegate
        @rtype: bool
        '''
        return position.y() < item.textRectOnScene().center().y()


class NodeDelegate(QGraphicsItem):
    '''
    Node delegate
    
    @authors: Andrew Golovchenko, Andrew Vasilev, Alexander Kulikov
    '''
    readable('node', 'verticalSpace')

    def _isSelected(self):
        '''Get selected flag'''
        return self.__selected

    def _setSelected(self, selected):
        '''
        Set the selected flag
        @type selected: boolean
        '''
        self.__selected = selected
        self.update()

    selected = property(_isSelected, _setSelected, None, 'Selected flag')

    def __init__(self, node, parent = None):
        '''
        Initialize delegate with specified node
        
        @param node: delegated Node object
        @type node: Node
        @param parent: parent object of this delegate 
        @type parent: NodeDelegate
        '''
        self.__node = node
        self.__verticalSpace = 0
        self.__selected = False
        QGraphicsItem.__init__(self, parent)

    __NODE_VERTICAL_SPACING = 4
    '''Minimal vertical space required for node decorations and spacing'''

    __NODE_HORIZONTAL_SPACING = 50
    '''Space between the node and its parent'''

    __ICON_SIZE = 16
    '''Icon width and height'''

    __TEXT_MARGIN = 4.0
    '''Default node content margin. It is never changed manually'''

    __FOLD_MARK_SIZE = 4.0
    '''Radius of round node folded mark'''

    __LABEL_MARGIN = 4.0
    '''Horisontal margin of edge label'''

    __MAX_ROOT_ECCENTRICITY_VERTICALLY = 0.98
    '''Maximum eccentricity of root node if it is stretched vertically'''

    __MAX_ROOT_ECCENTRICITY_HORIZONTALLY = 0.95
    '''Maximum eccentricity of root node if it is stretched horizontally'''

    def positionTree(self):
        '''
        Position all subtree nodes of the current node according to their requirements
        '''
        self._computeSoloVerticalSpace()
        for leftSubtree in [True, False]:
            if self.__node.folded: continue
            subtree = self._subtree(leftSubtree)
            if not subtree: continue
            for child in subtree:
                child.positionTree()
            self._positionChildNodes(subtree)

    def _subtree(self, left):
        '''
        Get list of child node delegates that belong to specified tree.
        @param left: subtree position
        @type left: bool
        @return: list of child delegates that belong to specified tree
        @rtype: list
        '''
        subtree = [item for item in self.childItems()
                       if item.node.left is left]
        subtree.sort(key = lambda delegate: self.__node.children.index(delegate.node))
        return subtree

    def _computeSoloVerticalSpace(self):
        '''
        Compute minimum vertical space needed for the current node
        '''
        self.__verticalSpace = self.textBoundingRect().height() + self._labelOverhead() + \
            NodeDelegate.__NODE_VERTICAL_SPACING
        if self.__node.folded and self.__node.effectiveStyle == 'fork':
            self.__verticalSpace += NodeDelegate.__FOLD_MARK_SIZE

    def _computeVerticalSpace(self, subtree):
        '''
        Calculate vertical space needed for current node specified subtree

        @param subtree: left or right subtree of the node
        @type subtree: list

        @return: required vertical space for the current node subtree
        @rtype: float
        '''
        subtreeSpace = 0.0
        for child in subtree:
            subtreeSpace += child.verticalSpace
        return subtreeSpace

    def _childHorizontalPosition(self, child):
        '''
        Calculate horizontal position of the child

        @param child: child node of current parent
        @type child: NodeDelegate

        @return: horizontal position of the passed delegate
        @rtype: float
        '''
        if child.node.left:
            return - NodeDelegate.__NODE_HORIZONTAL_SPACING \
                - child.textBoundingRect().width() - child.linkPointOffset()
        else:
            return NodeDelegate.__NODE_HORIZONTAL_SPACING \
                + self.textBoundingRect().width() - child.linkPointOffset()

    def _positionChildNodes(self, nodes):
        '''
        Position child node delegates of current node delegate

        @param nodes: child nodes to position
        @type nodes: list of NodeDelgates
        '''
        baseVerticalPosition = (self.textBoundingRect().height() - \
            self._computeVerticalSpace(nodes)) / 2.0
        initialVerticalPosition = baseVerticalPosition
        for child in nodes:
            verticalPosition = baseVerticalPosition + (child.verticalSpace - \
                child.textBoundingRect().height() + child._labelOverhead()) / 2.0
            horizontalPosition = self._childHorizontalPosition(child)
            child.setPos(horizontalPosition, verticalPosition)
            baseVerticalPosition += child.verticalSpace
#           FIXME: requires to extract in separate method,
#                  but change the local variable 'baseVerticalPosition' ...
            offsetPoint = child._delegateOffsetPoint()
            if offsetPoint is not None:
                if offsetPoint.y() > 0.0:
                    child.setPos(child.pos() + offsetPoint)
                    baseVerticalPosition += abs(offsetPoint.y())
                else:
                    child.setX(child.x() + offsetPoint.x())
                    edgeLine = QLineF(self.mapFromItem(child, child._parentLinkPoint()),
                                      self.mapFromItem(child, child.linkPoint))
                    hPosition = horizontalPosition
                    if child.node.left: hPosition += child.textBoundingRect().width()
                    horizontalPositionLine = QLineF(hPosition, -50.0, hPosition, 50.0)
                    intersectionType, intersectionPoint = horizontalPositionLine.intersect(edgeLine)
                    if baseVerticalPosition < intersectionPoint.y():
                        baseVerticalPosition += abs(intersectionPoint.y() - baseVerticalPosition) \
                                + NodeDelegate.__NODE_VERTICAL_SPACING
        # FIXME: remove this line with the refactoring of the label draw algorithm
        self._computeSoloVerticalSpace()
        self.__verticalSpace = max(self.__verticalSpace, baseVerticalPosition -
            initialVerticalPosition)

    @property
    def linkPoint(self):
        '''
        Return connection point for joint link

        @rtype: QPointF
        '''
        if self.__node.parent is None: return None
        linkPoint = QPointF()
        if self.__node.effectiveStyle == 'bubble':
            linkPoint.setY(self.textBoundingRect().height() / 2.0)
        else:
            linkPoint.setY(self.textBoundingRect().height() -
                           self.__node.effectiveEdgeWidth / 2.0)
        if self.__node.left:
            linkPoint.setX(self.textBoundingRect().width())
        return linkPoint

    def paint(self, painter, option, widget):
        '''Paint node content and its joint link'''
        # calculate bounding rect (without joint link)
        boundRect = self.textBoundingRect()
        # join link color
        edgeColor = self.__node.effectiveEdgeColor
        # join link width
        edgeWidth = self.__node.effectiveEdgeWidth
        # join link shape style
        edgeStyle = self.__node.effectiveEdgeStyle
        # node background color
        bgColor = self.__node.background_color
        if not bgColor: bgColor = settings.get('defaultNodeBackgroundColor')
        if self.__selected:
            bgColor = settings.get('defaultSelectedNodeBackgroundColor')
        # TODO user color value from settings instead of Qt.darkGray
        painter.setPen(QPen(Qt.darkGray, 1.0))
        painter.setBrush(bgColor)
        # if root node
        if self.__node.parent is None:
            self._paintRoot(painter, boundRect)
            return
        # calculate self link point and paint shape
        if self.__node.effectiveStyle == 'bubble':
            painter.drawRoundedRect(boundRect, 5.0, 5.0)
        else:
            boundRect.setHeight(boundRect.height() - edgeWidth / 2.0)
            painter.fillRect(boundRect, bgColor)
            # some decorative setting
            if edgeStyle.startswith('sharp'): edgeWidth = 0.5
            painter.setPen(QPen(edgeColor, edgeWidth, Qt.SolidLine, Qt.RoundCap))
            painter.drawLine(boundRect.bottomLeft(), boundRect.bottomRight())
        # paint edge label
        if self.__node.labelText:
            self._paintLabel(painter)
        # disable brush to prevent color filling
        painter.setBrush(Qt.NoBrush)
        # paint icons and text content
        self._paintNodeContent(painter)
        # paint join link from parent to self
        self._paintEdge(painter, edgeStyle)
        # paint folded mark (circle)
        if self.__node.folded:
            self._paintFoldedMark(painter, boundRect)

    def _paintLabel(self, painter):
        ''' Paint text label of node'''
        painter.setFont(self.__node.effectiveLabelFont)
        painter.setPen(self.__node.effectiveLabelColor)
        if self.__node.effectiveEdgeStyle.endswith('bezier'):
            labelPoint = self.linkPoint
            if self.__node.left:
                labelPoint.setX(labelPoint.x() + NodeDelegate.__LABEL_MARGIN)
            else:
                labelPoint.setX(labelPoint.x() - self.__node.labelSize.width() - NodeDelegate.__LABEL_MARGIN)
            labelPoint.setY(labelPoint.y() - self.__node.effectiveEdgeWidth / 2.0 - 2.0)
            painter.drawText(labelPoint, self.__node.labelText)
            painter.setPen(QPen(self.__node.effectiveEdgeColor, self.__node.effectiveEdgeWidth,
                    Qt.SolidLine, Qt.FlatCap))
            linkPoint = self.linkPoint
            painter.drawLine(linkPoint, QPointF(linkPoint.x() + self.linkPointOffset(), linkPoint.y()))
        else: # linear or sharp linear
            painter.save()
            edgeLine = QLineF(self._parentLinkPoint(), self.linkPoint)
            painter.translate(self.linkPoint)
            labelPoint = QPointF()
            if self.__node.left:
                labelPoint.setX(self._nodeToLabelOffset())
            else:
                labelPoint.setX(-self._nodeToLabelOffset() - self.__node.labelSize.width())
            labelPoint.setY(labelPoint.y() - self.__node.effectiveEdgeWidth / 2.0 - 2.0)
            if edgeLine.dx() * edgeLine.dy() > 0.0:
                painter.rotate(90.0 - math.fmod(edgeLine.angle(), 90.0))
            else:
                painter.rotate(-math.fmod(edgeLine.angle(), 90.0))
            painter.setFont(self.__node.effectiveLabelFont)
            painter.drawText(labelPoint, self.__node.labelText)
            painter.restore()

    def _paintRoot(self, painter, boundRect):
        '''
        Paint root node

        @type painter: QPanter

        @param boundRect: bounding rect of root node
        @type boundRect: QRectF
        '''
        painter.drawEllipse(boundRect)
        if self.__node.folded:
            self._paintFoldedMark(painter, boundRect)
        self._paintNodeContent(painter)

    def _paintFoldedMark(self, painter, boundRect):
        painter.setPen(QPen(Qt.darkGray, 1.0))
        painter.setBrush(Qt.white)
        markPoint = self.linkPoint
        if markPoint is None: # root
            markPoint = QPointF()
            markPoint.setY(boundRect.height() / 2.0)
            if any(not node.left for node in self.__node.children):
                markPoint.setX(boundRect.width())
                painter.drawEllipse(markPoint, 4.0, 4.0)
            if any(node.left for node in self.__node.children):
                markPoint.setX(0.0)
        else: # child node
            markPoint.setX(0.0 if markPoint.x() else boundRect.width())
        painter.drawEllipse(markPoint, 4.0, 4.0)

    def _rootBoundingRect(self, size):
        '''
        Calculate bounding rectangle of the root node

        @type size: QSize

        @return: Root's bounding rect
        @rtype: QRectF
        '''
        # 'a' and 'b' - sides of ellipse; 'e' - eccentricity
        b = size.height()
        a = size.width()
        if b > a:
            e = math.sqrt(1 - pow(a / b, 2))
            if e > self.__MAX_ROOT_ECCENTRICITY_VERTICALLY:
                e = self.__MAX_ROOT_ECCENTRICITY_VERTICALLY
            yAxis = math.sqrt(b * b + a * a / (1 - e * e))
            xAxis = yAxis * math.sqrt(1 - e * e)
        else:
            e = math.sqrt(1 - pow(b / a, 2))
            if e > self.__MAX_ROOT_ECCENTRICITY_HORIZONTALLY:
                e = self.__MAX_ROOT_ECCENTRICITY_HORIZONTALLY
            xAxis = math.sqrt(a * a + b * b / (1 - e * e))
            yAxis = xAxis * math.sqrt(1 - e * e)
        return QRectF(0.0, 0.0, xAxis, yAxis)

    def _rootTextPosition(self):
        '''
        Calculate the start position of the root text

        @rtype: QPointF
        '''
        boundRect = self.textBoundingRect()
        return QPointF((boundRect.width() -
                        len(self.__node.icons) * NodeDelegate.__ICON_SIZE -
                        self.__node.textSize().width()) / 2.0,
                        (boundRect.height() - self.__node.textSize().height()) / 2.0)

    def _paintNodeContent(self, painter):
        '''Paint internal content of the node'''
        painter.save()
        if not self.parentItem(): painter.translate(self._rootTextPosition())
        # paint icons
        xIconPos = NodeDelegate.__TEXT_MARGIN
        yIconPos = (self.__node.textSize().height() - NodeDelegate.__ICON_SIZE) / 2.0
        for icon in self.__node.icons:
            painter.drawImage(QPointF(xIconPos, yIconPos), resource.getIconByName(icon))
            xIconPos += NodeDelegate.__ICON_SIZE
        # folded mark offset
        if self.__node.folded and self.__node.left:
            painter.translate(NodeDelegate.__FOLD_MARK_SIZE, 0)
        # paint text content
        if xIconPos > NodeDelegate.__TEXT_MARGIN:
            painter.translate(xIconPos, 0)
        self.__node.paint(painter)
        painter.restore()

    def textBoundingRect(self):
        '''Calculate bounding region only for node without link to the parent'''
        size = self.__node.textSize()
        if self.__node.icons:
            size.setWidth(size.width() + NodeDelegate.__TEXT_MARGIN
                    + len(self.__node.icons) * NodeDelegate.__ICON_SIZE)
            size.setHeight(max(size.height(), NodeDelegate.__ICON_SIZE))
        if not self.parentItem(): return self._rootBoundingRect(size)
        if self.__node.folded: size.setWidth(size.width() + NodeDelegate.__FOLD_MARK_SIZE)
        boundRect = QRectF(0.0, 0.0, size.width(), size.height())
        if self.__node.effectiveStyle == 'fork':
            boundRect.setHeight(boundRect.height() + self.__node.effectiveEdgeWidth / 2.0)
        return boundRect

    def boundingRect(self):
        '''Calculate bounding rectangle'''
        if self.parentItem() is None:
            return self.textBoundingRect()
        linkPoint = self._parentLinkPoint()
        textBounds = self.textBoundingRect()
        left = min(linkPoint.x(), textBounds.x())
        top = min(linkPoint.y(), textBounds.y(), -self._labelOverhead())
        if not self.__node.left:
            width = textBounds.width() + math.fabs(linkPoint.x()) + textBounds.x()
        else:
            width = linkPoint.x() - textBounds.x()
            if self.__node.folded: left -= NodeDelegate.__FOLD_MARK_SIZE
        if linkPoint.y() < textBounds.y():
            height = textBounds.height() + math.fabs(top) + textBounds.y()
        else:
            height = max(math.fabs(top) + textBounds.height(), linkPoint.y() - textBounds.y())
        if self.__node.folded:
            width += NodeDelegate.__FOLD_MARK_SIZE
            if self.__node.effectiveStyle == 'fork': height += NodeDelegate.__FOLD_MARK_SIZE
        return QRectF(left, top, width, height)

    def textRectOnScene(self):
        '''
        Return text bounding rectangle in scene coordinates

        @rtype: QRectF
        '''
        return self.mapRectToScene(self.textBoundingRect())

    def xDistanceTo(self, position):
        '''
        Return absolute distance by x-coordinate from center of the delegate
        to the position on scene

        @type position: QPointF
        @rtype: float
        '''
        return abs(self.textRectOnScene().center().x() - position.x())

    def yDistanceTo(self, position):
        '''
        Return absolute distance by y-coordinate from center of the delegate
        to the position on scene

        @type position: QPointF
        @rtype: float
        '''
        return abs(self.textRectOnScene().center().y() - position.y())

    def boundingRectOnScene(self):
        '''
        Return bounding rectangle in scene coordinates

        @rtype: QRectF
        '''
        return self.mapRectToScene(self.boundingRect())

    def _delegateOffsetPoint(self):
        '''
        @return: additional offset point of node delegate specified by label size.
                 If no offset required, None value will be returned
        @rtype: QPointF
        '''
        if self.__node.labelText is None or self.__node.effectiveEdgeStyle.endswith('bezier'):
            return None
        edgeLine = QLineF(self._parentLinkPoint(), self.linkPoint)
        requiredLength = self.__node.labelSize.width() + self._nodeToLabelOffset() \
                + self._neighborEdgeToLabelOffset()
        if requiredLength > edgeLine.length():
            if edgeLine.angle() < 180.0: # for top half of tree used only horizontal offset
                horisontalOffset = math.sqrt(requiredLength * requiredLength \
                        - edgeLine.dy() * edgeLine.dy())
                horisontalOffset -= abs(edgeLine.dx())
                if self.__node.left: horisontalOffset = -horisontalOffset
                return QPointF(horisontalOffset, 0.0)
            else: # offset along current edge line
                lineDifference = abs(requiredLength - edgeLine.length())
                edgeLine.setLength(lineDifference)
                return QPointF(edgeLine.dx(), edgeLine.dy())
        return None

    def _nodeToLabelOffset(self):
        '''
        @return: offset specified of angle between edge and node area.

        @rtype: float
        '''
        edgeLine = QLineF(self._parentLinkPoint(), self.linkPoint)
        angle = edgeLine.angle()
        if angle > 180.0: # only lower half of tree
            offset = min(self.textBoundingRect().height() / 2.0, self.__node.labelSize.height())
#            if edgeLine.dx() * edgeLine.dy() > 0.0:
#                angle = math.radians(math.fmod(angle, 90.0))
#            else:
#                angle = math.radians(90.0 - math.fmod(angle, 90.0))
#            oppositeCathet = self.__node.labelSize.height() + self.__node.effectiveEdgeWidth / 2.0
#            offset = min(abs(oppositeCathet / math.tan(angle)), self.textBoundingRect().height() / 2.0)
            return offset + NodeDelegate.__LABEL_MARGIN
        return NodeDelegate.__LABEL_MARGIN

    def _neighborEdgeToLabelOffset(self):
        '''
        @return: offset specified of angle between intersected neighbor edges.
                 If offset is very large, label height-specified constant will be returned
        @rtype: float
        '''
        edgeLine = QLineF(self._parentLinkPoint(), self.linkPoint)
        topNeighborNode = self._nearestNeighborDelegate(True)
        if topNeighborNode is None:
            if self.__node.left: # use random line with 45.0 angle value
                topNeighborEdge = QLineF(0.0, 0.0, 30.0, -30.0)
            else: # use random line with 135.0 angle value
                topNeighborEdge = QLineF(0.0, 0.0, -30.0, -30.0)
        else: # use edge of top neighbor node
#           TODO: may be use linkPoint + linkPoint offset
            topNeighborEdge = QLineF(edgeLine.p1(),
                    self.mapFromItem(topNeighborNode, topNeighborNode.linkPoint))
        if self.__node.left:
            angle = topNeighborEdge.angleTo(edgeLine)
        else:
            angle = edgeLine.angleTo(topNeighborEdge)
        if angle < 90.0:
            angle = math.radians(angle)
            oppositeCathet = self.__node.labelSize.height() \
                    + self.__node.effectiveEdgeWidth / 2.0
            if topNeighborNode is not None:
                oppositeCathet += topNeighborNode.node.effectiveEdgeWidth / 2.0
            return min(abs(oppositeCathet / math.tan(angle)),
                       self.__node.labelSize.height() * 2.0) + NodeDelegate.__LABEL_MARGIN
        return NodeDelegate.__LABEL_MARGIN

    def _nearestNeighborDelegate(self, upward = False):
        '''
        Search upper or under delegate in similar subtree

        @param upward: direction of search: if True, upper delegate 
                       will be returned, otherwise under delegate
        @type upward: bool

        @return: nearest delegate in a specified direction or None
        @rtype: NodeDelegate
        '''
        children = self.__node.parent.children
        step = -1 if upward else 1
        stop = -1 if upward else len(children)
        index = children.index(self.__node) + step
        while index != stop:
            if children[index].left is self.__node.left:
                return self.scene().getDelegate(children[index])
            index += step
        return None

    def _labelOverhead(self):
        '''
        @return: difference between top points value of node (text bounding rect) and its label
        @rtype: float
        '''
#       TODO: consider angle of label on inclined edges
        if not self.__node.labelText: return 0.0
        textBounds = self.textBoundingRect()
        # 2.0 is a decorative space between label and edge
        labelTop = textBounds.height() - self.__node.labelSize.height() - \
                self.__node.effectiveEdgeWidth / 2.0 - 2.0
        if self.__node.effectiveStyle == 'bubble':
            labelTop -= textBounds.height() / 2.0
        return - labelTop if labelTop < 0.0 else 0.0

    def _parentLinkPoint(self):
        '''
        Calculate connection point to parent node
        
        @return: connection point in current node coordinates
        @rtype: QPointF
        '''
        parentLinkPoint = QPointF()
        parentRect = self.parentItem().textBoundingRect()
        if not self.__node.left:
            parentLinkPoint.setX(parentRect.width())
        else:
            parentLinkPoint.setX(0.0)
        parentNode = self.parentItem().node
        # root node never has fork style and only root has no parent
        if parentNode.effectiveStyle == 'bubble' or parentNode.parent is None:
            parentLinkPoint.setY(parentRect.height() / 2.0)
        else:
            parentRect.setHeight(parentRect.height() - self.__node.effectiveEdgeWidth / 2.0)
            parentLinkPoint.setY(parentRect.height())
        return self.mapFromParent(parentLinkPoint)

    def _paintEdge(self, painter, style):
        '''
        Paint join link from parent node to current

        @type painter: QPanter

        @param style: style of node join link
        @type style: string
        '''
        painter.setPen(QPen(self.__node.effectiveEdgeColor, self.__node.effectiveEdgeWidth,
                Qt.SolidLine, Qt.FlatCap, Qt.RoundJoin))
        startPoint = self._parentLinkPoint()
        endPoint = self.linkPoint
        endPoint.setX(endPoint.x() + self.linkPointOffset())
        curvePath = QPainterPath()
        if style == 'bezier':
            curveRect = QRectF(startPoint, endPoint)
            p1 = curveRect.topRight()
            p1.setX(curveRect.topRight().x() - curveRect.width() / 2.0)
            p2 = curveRect.bottomLeft()
            p2.setX(curveRect.bottomLeft().x() + curveRect.width() / 2.0)
            curvePath.moveTo(startPoint)
            curvePath.cubicTo(p1, p2, endPoint)
            painter.drawPath(curvePath)
        elif style == 'sharp_bezier':
            painter.setPen(QPen(self.__node.effectiveEdgeColor, 0.5))
            curveRect = QRectF(startPoint, endPoint)
            p1 = curveRect.topRight()
            p1.setX(curveRect.topRight().x() - curveRect.width() / 2.0)
            p2 = curveRect.bottomLeft()
            p2.setX(curveRect.bottomLeft().x() + curveRect.width() / 2.0)
            curvePath.moveTo(startPoint.x(), startPoint.y() + self.__node.effectiveEdgeWidth / 2.0)
            curvePath.cubicTo(p1, p2, endPoint)
            startPoint.setY(startPoint.y() - self.__node.effectiveEdgeWidth)
            curvePath.cubicTo(p2, p1, startPoint)
            painter.drawPath(curvePath)
            painter.fillPath(curvePath, self.__node.effectiveEdgeColor)
        elif style == 'linear':
            if self.__node.left:
                startPoint, endPoint = endPoint, startPoint
            points = QPolygonF()
            points.append(startPoint)
            points.append(QPointF(startPoint.x() + self.__node.effectiveEdgeWidth / 2.0, startPoint.y()))
            points.append(QPointF(endPoint.x() - self.__node.effectiveEdgeWidth / 2.0, endPoint.y()))
            points.append(endPoint)
            painter.drawPolyline(points)
        elif style == 'sharp_linear':
            painter.setPen(QPen(self.__node.effectiveEdgeColor, 0.1))
            curvePath.moveTo(startPoint.x(), startPoint.y() + self.__node.effectiveEdgeWidth / 2.0)
            curvePath.lineTo(endPoint)
            curvePath.lineTo(startPoint.x(), startPoint.y() - self.__node.effectiveEdgeWidth / 2.0)
            painter.drawPath(curvePath)
            painter.fillPath(curvePath, self.__node.effectiveEdgeColor)

    def linkPointOffset(self):
        '''
        @return: offset of edge link point in pixels specified
                 by label width and its margins
        @rtype: float
        '''
        labelWidth = self.__node.labelSize.width()
        if labelWidth == 0.0 or self.__node.effectiveEdgeStyle.endswith('linear'):
            return 0.0
        offset = labelWidth + NodeDelegate.__LABEL_MARGIN * 2.0
        return offset if self.__node.left else - offset

class NodeLocation(object):
    '''
    Class for holding position of a node

    @author: Oleg Kandaurov
    '''
    readable('right', 'top', 'position')
    writable('right', 'top', 'position')

    def __init__(self, right = None, top = None, position = None):
        '''
        @param right: horizontal position. True if right position, otherwise False
        @type right: bool

        @param top: vertical position. True if top position, otherwise False
        @type top: bool

        @param position: vertical order
        @type position: int
        '''
        self.__right = right
        self.__top = top
        self.__position = position

    @property
    def valid(self):
        '''
        True if all the data necessary to use this object is valid, otherwise False.

        @rtype: bool
        '''
        return self.__position is not None or self.__right is not None

