# -*- coding: utf-8 -*-
# HiveMind - Distributed mind map editor for Maemo 5 platform
# Copyright (C) 2010 HiveMind developers
#
# HiveMind is the legal property of its developers, whose names are
# noticed in @author or @authors annotations at the beginning of each
# module or class.
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02111-1301  USA

'''
Main window class for Desktop platform
'''

from hivemind.main_window import *

class DesktopMainWindow(AbstractMainWindow):
    '''
    Main window of HiveMind program for Desktop platform

    @author: Alexander Kulikov
    '''
    def __init__(self, controller, mindMapScene):
        '''Create new window'''
        AbstractMainWindow.__init__(self, controller, mindMapScene)
        self._createContextMenu(controller)
        self._createMainMenu(controller)
        self._createToolBars(controller)

    def _createContextMenu(self, controller):
        '''Create context menu'''
        self.__menu = QMenu(self)
        self.__menu.addAction(controller.addNodeAction)
        self.__menu.addAction(controller.editNodeAction)
        self.__menu.addAction(controller.foldNodeAction)
        self.__menu.addAction(controller.moveNodeAction)
        self.__menu.addAction(controller.removeNodeAction)
        self.__menu.addSeparator()
        self.__menu.addAction(controller.editNodeIconsAction)
        edgeSubmenu = QMenu(self)
        edgeSubmenu.setTitle(self.tr('Edge parameters'))
        edgeSubmenu.addAction(controller.editEdgeAction)
        edgeSubmenu.addAction(controller.editLabelAction)
        edgeSubmenu.addSeparator()
        edgeSubmenu.addAction(controller.parentWidthAction)
        edgeSubmenu.addAction(controller.thinWidthAction)
        edgeSubmenu.addAction(controller.oneWidthAction)
        edgeSubmenu.addAction(controller.twoWidthAction)
        edgeSubmenu.addAction(controller.fourWidthAction)
        edgeSubmenu.addAction(controller.eightWidthAction)
        edgeSubmenu.addSeparator()
        edgeSubmenu.addAction(controller.parentStyleAction)
        edgeSubmenu.addAction(controller.linearStyleAction)
        edgeSubmenu.addAction(controller.bezierStyleAction)
        edgeSubmenu.addAction(controller.sharpLinearStyleAction)
        edgeSubmenu.addAction(controller.sharpBezierStyleAction)
        self.__menu.addMenu(edgeSubmenu)

    def _addEllipsesToActionText(self, controller):
        '''Add ellipses to the text of the actions displaying dialogs'''
        actionsWithEllipses = [controller.fileOpenAction, controller.fileSaveAsAction,
                controller.settingsAction, controller.startNetworkServerAction,
                controller.startNetworkClientAction, controller.aboutAction,
                controller.addNodeAction, controller.editNodeAction,
                controller.editNodeIconsAction, controller.editLabelAction,
                controller.editEdgeAction]
        for action in actionsWithEllipses:
            action.setText(action.text() + '...')

    def _createMainMenu(self, controller):
        '''
        Create main menu
        @type controller: MainWindowController
        '''
        self._addEllipsesToActionText(controller)
        menu = self.menuBar().addMenu(self.tr('&File'))
        menu.addAction(controller.fileNewAction)
        menu.addAction(controller.fileOpenAction)
        menu.addAction(controller.fileSaveAction)
        menu.addAction(controller.fileSaveAsAction)
        menu.addSeparator()
        menu.addAction(controller.exitAction)
        menu = self.menuBar().addMenu(self.tr('&Edit'))
        menu.addAction(controller.undoAction)
        menu.addAction(controller.redoAction)
        menu.addSeparator()
        menu.addAction(controller.settingsAction)
        menu = self.menuBar().addMenu(self.tr('&Collaboration'))
        menu.addAction(controller.startNetworkServerAction)
        menu.addAction(controller.startNetworkClientAction)
        menu.addAction(controller.stopNetworkAction)
        menu = self.menuBar().addMenu(self.tr('&Help'))
        menu.addAction(controller.aboutAction)

    def _createToolBars(self, controller):
        '''
        Create tool bar
        @type controller: MainWindowController
        '''
        toolbar = self.addToolBar(self.tr('File'))
        toolbar.addAction(controller.fileNewAction)
        toolbar.addAction(controller.fileOpenAction)
        toolbar.addAction(controller.fileSaveAction)
        toolbar.addAction(controller.fileSaveAsAction)
        toolbar = self.addToolBar(self.tr('Edit'))
        toolbar.addAction(controller.undoAction)
        toolbar.addAction(controller.redoAction)
        toolbar = self.addToolBar(self.tr('View'))
        toolbar.addAction(controller.zoomOutAction)
        toolbar.addAction(controller.zoomInAction)

    def showMenu(self, rightButton, item):
        '''Show the node menu on right button click'''
        if rightButton:
            self.__menu.exec_(QCursor.pos())
