# -*- coding: utf-8 -*-
# HiveMind - Distributed mind map editor for Maemo 5 platform
# Copyright (C) 2010-2011 HiveMind developers
#
# HiveMind is the legal property of its developers, whose names are
# noticed in  or  annotations at the beginning of each
# module or class.
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02111-1301  USA

'''
Network subsystem of the application
'''
import idavoll
from idavoll.backend import PubSubServiceFromBackend, BackendService
from idavoll.memory_storage import Storage, LeafNode
from wokkel.xmppim import RosterClientProtocol
from wokkel.pubsub import PubSubClient, Subscription, NS_PUBSUB_ERRORS
from wokkel.client import HybridAuthenticator, XMPPClientConnector
from wokkel.ping import PingHandler, NS_PING
from wokkel.generic import stripNamespace
from wokkel.subprotocols import StreamManager, XMPPHandler
from twisted.words.protocols import jabber
from twisted.words.xish.xmlstream import STREAM_ERROR_EVENT
from twisted.words.xish.domish import Element
from twisted import application
from twisted.internet.task import LoopingCall
from twisted import internet
from twisted.python import log
from hivemind.attribute import Observable, writable, readable
from hivemind.network.network_models import AffiliationModel, RosterModel
from hivemind.network import error_messages
from hivemind.gui_main import reactor
import hivemind.gui_factory as gui_factory
import hivemind
from PySide.QtCore import QObject
import uuid
#pylint: disable=C0103

''' XMPP PubSub node name '''
XMPP_NODE_NAME = unicode('HiveMind')
''' Time in seconds we must wait for unsubscribe '''
UNSUBSCRIBE_TIMEOUT = 2
''' Cold shutdown of network if timeout expires'''
SHUTDOWN_TIMEOUT = 4


class NetworkController(QObject):
    '''
    Asynchronous command processor. Sends commands into shared space, listens to incoming
    messages and sends correct commands to mind map controller.
    @author: Andrew Vasilev, Oleg Kandaurov
    '''

    def __init__(self, mindMapController, actionBag):
        '''
        Constructor
        @type mindMapController: MindMapController
        @type actionBag: ActionBag
        '''
        QObject.__init__(self)
        self.__mindMapController = mindMapController
        self.__actionBag = actionBag
        self.__affiliationModel = AffiliationModel()
        self.__rosterModel = RosterModel()
        self.__streamMessages = error_messages.StreamErrorMessages()
        self.__saslMessages = error_messages.SASLErrorMessages()
        self.__stanzaMessages = error_messages.StanzaErrorMessages()
        self.__pubsubMessages = error_messages.PubSubErrorMessages()
        self.__xmppClient = None
        self.__protocol = None
        self.__roster = None
        self.__pingHandler = None
        self.__storage = None
        self.__affiliationDialog = None
        self.__networkDialog = None
        self.__clientDialog = None
        self.__serviceDialog = None
        self.__afterNetworkDialog = None
        self.__errorDialog = None
        self.__pingManager = None
        self.__mindmap = None
        self._initDialogs()
        self._connectActions()
        self._initActions(transport = False, protocol = False)

    def _initDialogs(self):
        '''initialize network dialogs'''
        def affiliationAccepted():
            '''Process accepted signal from affiliation dialog'''
            self.__storage.rootNode.affiliations = self.__affiliationModel.affiliations

        def networkAccepted():
            '''Process accepted signal from network dialog'''
            self._startTransport(self.__networkDialog.attributes['userJid'] + '/hivemind',
                    self.__networkDialog.attributes['password'])
            if self.__afterNetworkDialog is not None:
                self.__afterNetworkDialog.exec_()
            self.__afterNetworkDialog = None

        def clientAccepted():
            '''Process accepted signal from client dialog'''
            self.__protocol = HivemindClient(jabber.jid.internJID(
                    self.__clientDialog.attributes['serviceJid'] + '/hivemind'))
            self._startProtocol()

        def serviceAccepted():
            '''Process accepted signal from service dialog'''
            self._configureService(self.__mindMapController.initialMindMap,
                    self.__mindMapController.undoStack,
                    self.__serviceDialog.attributes['readOnly'],
                    self.__serviceDialog.attributes['accessModel'])
            self._startProtocol()

        self.__networkDialog = gui_factory.createNetworkConnectionDialog()
        self.__networkDialog.accepted.connect(networkAccepted)
        self.__clientDialog = gui_factory.createNetworkClientDialog(self.__rosterModel)
        self.__clientDialog.accepted.connect(clientAccepted)
        self.__serviceDialog = gui_factory.createNetworkServiceDialog()
        self.__serviceDialog.accepted.connect(serviceAccepted)
        self.__affiliationDialog = gui_factory.createAffiliationDialog(self.__affiliationModel)
        self.__affiliationDialog.accepted.connect(affiliationAccepted)
        self.__errorDialog = gui_factory.createNetworkErrorDialog()

    def _connectActions(self):
        '''Connect actions'''
        def showStartServiceDialog():
            '''Show service dialog after network dialog'''
            self.__afterNetworkDialog = self.__serviceDialog
            self.__networkDialog.exec_()

        def showStartClientDialog():
            '''Show client dialog after network dialog'''
            self.__afterNetworkDialog = self.__clientDialog
            self.__networkDialog.exec_()

        self.__actionBag.connectXMPPAction.triggered.connect(self.__networkDialog.exec_)
        self.__actionBag.disconnectXMPPAction.triggered.connect(self._shutdownCommunication)
        self.__actionBag.createServiceAction.triggered.connect(self.__serviceDialog.exec_)
        self.__actionBag.createClientAction.triggered.connect(self.__clientDialog.exec_)
        self.__actionBag.stopProtocolAction.triggered.connect(self._closeProtocol)
        self.__actionBag.startServiceAction.triggered.connect(showStartServiceDialog)
        self.__actionBag.startClientAction.triggered.connect(showStartClientDialog)
        self.__actionBag.editAffiliationAction.triggered.connect(
                self.__affiliationDialog.exec_)

    def _initActions(self, transport, protocol):
        '''
        Initialize actions
        @param transport: Whether transport connected or not
        @type transport: bool
        @param protocol: Whether protocol connected or not
        @type protocol: bool
        '''
        self.__actionBag.connectXMPPAction.setDisabled(transport)
        self.__actionBag.disconnectXMPPAction.setEnabled(transport)
        self.__actionBag.createServiceAction.setEnabled(transport and not protocol)
        self.__actionBag.createClientAction.setEnabled(transport and not protocol)
        self.__actionBag.stopProtocolAction.setEnabled(transport and protocol)
        self.__actionBag.startServiceAction.setDisabled(transport)
        self.__actionBag.startClientAction.setDisabled(transport)
        self.__actionBag.editAffiliationAction.setEnabled(transport and protocol and
                self._isServiceRole())

    def _isServiceRole(self):
        '''
        Return True if protocol acts as service, otherwise False
        @rtype: bool
        '''
        if isinstance(self.__protocol, HivemindService):
            return True
        return False

    def publishCommand(self, command):
        '''
        Publish command in common space
        @param command: command to publish in shared space
        '''
        data = hivemind.parser.serializeCommand(command)
        if self.__protocol:
            self.__protocol.proposeChangeset(Changeset(data, 'command'))
        else:
            self.__mindMapController.executeCommand(command)

    def _startTransport(self, userJid, password):
        '''
        Start XMPP connection
        @param userJid: JID for XMPP connection
        @type userJid: str
        @param password: password for XMPP connection
        @type password: str
        '''
        authenticator = HybridAuthenticator(jabber.jid.internJID(userJid), password)
        factory = HivemindXmlStreamFactory(authenticator)
        factory.addObserver(self)
        self.__xmppClient = HivemindXMPPClient(authenticator, factory)
        self.__xmppClient.addObserver(self)
        self.__xmppClient.logTraffic = True
        self.__xmppClient.startService()

    def _configureService(self, mindmap, undoStack, readOnly, accessModel):
        '''
        Configure HiveMind service
        @param mindmap: current mind map to publish first
        @type mindmap: MindMap
        @param undoStack: Undo stack associated with mindmap
        @type undoStack: QUndoStack
        @param readOnly: specifies whether mind map read only or not
        @type readOnly: bool
        @param accessModel: Access model of service
        @type accessModel: str
        '''
        self.__storage = HivemindStorage(self)
        config = Storage.defaultConfig['leaf']
        config['pubsub#node_type'] = 'leaf'
        config['pubsub#send_last_published_item'] = 'never'
        config['pubsub#access_model'] = accessModel
        config['pubsub#publish_model'] = 'subscribers'
        if readOnly:
            config['pubsub#publish_model'] = 'publishers'
        self.__storage.createRootNode(self.__xmppClient.jid, config)
        self.__storage.rootNode.roster = self.__rosterModel.roster
        self.__protocol = HivemindService(hivemind.parser.mindMapToString(mindmap, False),
                hivemind.parser.serializeCommandStack(undoStack),
                HivemindBackendService(self.__storage))

    def _startProtocol(self):
        '''Start Hivemind protocol'''
        self.__protocol.addObserver(self)
        self.__protocol.setHandlerParent(self.__xmppClient)

    def _startTransportHelpers(self):
        '''Start transport helpers'''
        self.__roster = HivemindRosterClientProtocol()
        self.__roster.addObserver(self)
        self.__roster.setHandlerParent(self.__xmppClient)
        self.__pingManager = PingManager(self, self.__xmppClient)
        self.__pingManager.addPinger(self.__xmppClient.server,
                hivemind.settings.get('transportPingInterval'),
                hivemind.settings.get('transportPingMaxCount'), greedy = True)
        self.__pingManager.startPing(self.__xmppClient.server)

    def _stopTransportHelpers(self):
        '''Stop transport helpers'''
        if self.__roster is not None:
            self.__roster.disownHandlerParent(self.__xmppClient)
            self.__roster.deleteObserver(self)
            self.__roster = None
        if self.__pingManager is not None:
            self.__pingManager.stopPing(self.__xmppClient.server)
            self.__pingManager.removeAll()
            self.__pingManager = None

    def _startProtocolHelpers(self):
        '''Start protocol helpers'''
        self.__pingHandler = PingHandler()
        self.__pingHandler.setHandlerParent(self.__xmppClient)

    def _stopProtocolHelpers(self):
        '''Stop protocol helpers'''
        if self.__pingHandler is not None:
            self.__pingHandler.disownHandlerParent(self.__xmppClient)
            self.__pingHandler = None
        if self.__pingManager is not None:
            self.__pingManager.removeAll([self.__xmppClient.server])

    def shutdownNetwork(self):
        '''Shutdown network communication asynchronously'''
        def stopReactor(result):
            '''Stop reactor'''
            log.msg('Network closed')
            reactor.stop()

        if not self.__mindMapController.prohibitChangeMapAction():
            return
        defer = self._shutdownCommunication()
        defer.addCallback(stopReactor)
        reactor.callLater(SHUTDOWN_TIMEOUT, stopReactor, self)

    def _shutdownCommunication(self):
        '''Shutdown communication with the XMPP network'''
        defer = self._closeProtocol()
        defer.addCallback(lambda success: self._stopTransport())
        return defer

    def _stopProtocol(self):
        '''Stop protocol'''
        log.msg('Protocol stopped')
        self._stopProtocolHelpers()
        if self.__protocol is not None:
            self.__protocol.disownHandlerParent(self.__xmppClient)
            self.__protocol.deleteObserver(self)
            self.__protocol = None

    def _shutdownProtocol(self):
        '''Shutdown protocol'''
        defer = internet.defer.Deferred()
        if self.__protocol is not None:
            defer = self.__protocol.shutdown()
        else:
            defer.callback([])
        return defer

    def _closeProtocol(self):
        '''Close protocol'''
        defer = self._shutdownProtocol()
        defer.addCallback(lambda success: self._stopProtocol())
        defer.addCallback(lambda success: log.msg('Protocol closed'))
        return defer

    def _stopTransport(self):
        '''Stop transport'''
        log.msg('Transport stopped')
        self._processTransportState('disconnected')
        self._stopTransportHelpers()
        if self.__xmppClient is not None:
            self.__xmppClient.stopService()
            self.__xmppClient = None
        self.__mindmap = None

    def processNotification(self, *values, **kvargs):
        '''Process notification'''
        if 'transport' in kvargs:
            self._processTransportState(kvargs['transport'])
        if 'protocol' in kvargs:
            self._processProtocolEvent(kvargs['protocol'])
        elif 'error' in kvargs:
            self._processError(kvargs['error'])
        elif 'changeset' in kvargs:
            self._addChangeset(kvargs['changeset'])
        elif 'roster' in kvargs:
            self._processRosterEvent(kvargs['roster'])
        elif 'ping' in kvargs:
            self._processPingEvent(kvargs['ping'])
        elif 'affiliation' in kvargs:
            entity, affiliation = kvargs['affiliation']
            self.__affiliationModel.setAffiliation(entity, affiliation)

    def _processTransportPingEvent(self, event):
        '''Process ping event from transport'''
        if event['state'] == 'alive':
            self._processTransportState('alive')
        elif event['state'] == 'timeout':
            self.__pingManager.stopPing(event['entity'])
            self.__xmppClient.factory.resetDelay()
            self.__xmppClient.factory.retry(self.__xmppClient.connection)

    def _processProtocolPingEvent(self, event):
        '''Process ping event from protocol'''
        if event['state'] == 'timeout':
            self.__protocol.unsubscribe(event['entity'], self.__xmppClient.jid,
                    XMPP_NODE_NAME, event['entity'])
        self.__affiliationModel.updateState(event['entity'],
                'subscribed' if event['state'] == 'alive' else 'unsubscribed',)

    def _processPingEvent(self, event):
        '''Process ping event'''
        if event['entity'] == self.__xmppClient.server:
            self._processTransportPingEvent(event)
        else:
            self._processProtocolPingEvent(event)

    def _processRosterEvent(self, event):
        '''Process event from roster'''
        if event['type'] == 'roster':
            self.__rosterModel.roster = event['data']
        elif event['type'] == 'remove':
            self.__rosterModel.removeContact(event['data'])
        elif event['type'] == 'set':
            self.__rosterModel.setContact(event['data'])

    def _handleSASLError(self, failure):
        '''
        Handle sasl error
        @type failure: Failure
        '''
        self._shutdownCommunication()
        self.__errorDialog.showError(self.tr('Authentication failed.'),
                self.__saslMessages.condition[failure.value.condition],
                failure.getTraceback(0, 'verbose'))

    def _handleStanzaError(self, failure):
        '''
        Handle stanza error
        @type failure: Failure
        '''
        self._closeProtocol()
        self.__errorDialog.showError(self.tr('Error from protocol level has ' +
                            'been occured. Disconnecting.'),
                self.__stanzaMessages.condition[failure.value.condition],
                failure.getTraceback(0, 'verbose'))

    def _handleStreamError(self, failure):
        '''
        Handle stream error
        @type failure: Failure
        '''
        self._shutdownCommunication()
        self.__errorDialog.showError(self.tr('Stream error has been occured. ' +
                            'The error is unrecoverable, disconnecting'),
                self.__streamMessages.condition[failure.value.condition],
                failure.getTraceback(0, 'verbose'))

    def _handlePubSubError(self, failure):
        '''
        Handle pubsub error
        @type failure: Failure
        '''
        condition, feature = _pubsubConditionFromStanzaError(failure.value)
        message = self.__stanzaMessages.condition[failure.value.condition] + \
                '\n\nPublish-Subscribe specific condition is: ' + \
                self.__pubsubMessages.condition[condition]
        if feature is not None:
            message += '\n\nFor feature: ' + self.__pubsubMessages.feature[feature]
        self.__errorDialog.showError(self.tr('Publish-Subscribe error has been occured'),
                message,
                failure.getTraceback(0, 'verbose'))

    def _processError(self, failure):
        ''''
        Process network errors
        @type failure: Failure
        '''
        error = failure.trap(jabber.sasl.SASLError, jabber.error.StanzaError,
                jabber.error.StreamError, internet.error.ConnectionLost)
        if error == jabber.sasl.SASLError:
            self._handleSASLError(failure)
        elif error == jabber.error.StreamError:
            self._handleStreamError(failure)
        elif error == jabber.error.StanzaError:
            if _isPubSubError(failure.value):
                self._handlePubSubError(failure)
            else:
                self._handleStanzaError(failure)

    def _processProtocolEvent(self, event):
        '''
        Process protocol events
        @type event: dict
        '''
        self.__affiliationModel.updateState(event['entity'], event['state'])
        if event['state'] == 'subscribed':
            self._startProtocolHelpers()
            self.__actionBag.subscribeStatusAction.setStatus('yes')
            if event['entity'] != self.__protocol.parent.jid:
                self.__pingManager.addPinger(event['entity'],
                        hivemind.settings.get('protocolPingInterval'),
                        hivemind.settings.get('protocolPingMaxCount'), greedy = False)
                self.__pingManager.startPing(event['entity'])
        elif event['state'] == 'unsubscribed':
            self.__pingManager.removePinger(event['entity'])
            self.__actionBag.subscribeStatusAction.setStatus('no')
        self._initActions(self.__actionBag.connectionStatusAction.status != 'no',
            self.__actionBag.subscribeStatusAction.status != 'no')

    def _processTransportState(self, state):
        '''
        Process transport state
        @param state: Current transport state
        @type state: str
        '''
        if state == 'initialized':
            self._startTransportHelpers()
            self.__actionBag.connectionStatusAction.setStatus('established')
        elif state == 'alive':
            self.__actionBag.connectionStatusAction.setStatus('established')
        elif state == 'connecting':
            self.__actionBag.connectionStatusAction.setStatus('connecting')
        elif state == 'disconnected':
            self._stopTransportHelpers()
            self.__actionBag.connectionStatusAction.setStatus('no')
            self.__actionBag.subscribeStatusAction.setStatus('no')
        self._initActions(self.__actionBag.connectionStatusAction.status != 'no',
                self.__actionBag.subscribeStatusAction.status != 'no')

    def _addChangeset(self, changeset):
        '''
        Add changeset received from hivemind network
        @type changeset: Changeset
        '''
        if changeset.type == 'mindmap':
            self.__mindmap = hivemind.parser.mindMapFromString(changeset.data)
            self.__mindMapController.loadMindMap(self.__mindmap)
        elif changeset.type == 'command':
            self.__mindMapController.executeCommand(hivemind.parser.deserializeCommand(
                    changeset.data, self.__mindmap))
        elif changeset.type == 'undostack':
            self.__mindMapController.undoStack = hivemind.parser.deserializeCommandStack(
                    changeset.data, self.__mindmap)


class HivemindXmlStreamFactory(jabber.xmlstream.XmlStreamFactory, Observable):
    '''
    Factory for Jabber XmlStream objects as a reconnecting client.
    @author: Oleg Kandaurov
    '''

    def __init__(self, authenticator):
        jabber.xmlstream.XmlStreamFactory.__init__(self, authenticator)
        Observable.__init__(self)
        self.initialDelay = 1.0
        self.factor = 1.6180339887498948
        self.maxDelay = 40

    def startedConnecting(self, connector):
        '''
        Called when a connection has been started.
        @param connector: a Connector object.
        '''
        log.msg('Connection started at %s' % self.__class__.__name__)
        jabber.xmlstream.XmlStreamFactory.startedConnecting(self, connector)
        self.notifyObservers(transport = 'connecting')

    def clientConnectionFailed(self, connector, reason):
        '''
        Called when a connection has failed to connect.
        @type reason: Failure
        '''
        log.msg('Connection failed with %s at %s' % (reason, self.__class__.__name__))
        jabber.xmlstream.XmlStreamFactory.clientConnectionFailed(self, connector, reason)

    def clientConnectionLost(self, connector, reason):
        '''
        Called when an established connection is lost.
        @type reason: Failure
        '''
        log.msg('Connection lost with %s at %s' % (reason, self.__class__.__name__))
        jabber.xmlstream.XmlStreamFactory.clientConnectionLost(self, connector, reason)


class HivemindPingClientProtocol(XMPPHandler, Observable):
    '''
    Hivemind ping client
    @author: Oleg Kandaurov
    '''

    def __init__(self, entity, interval, maxCount, suppressErrors):
        '''
        Constructor
        @param entity: Entity to be pinged
        @type entity: str
        @param interval: Interval between sending ping
        @type interval: int
        @param maxCount: if maxCount pings without response has been occured, connection is
        considered lost
        @type maxCount: int
        @param suppressErrors: If True then we treat errors from server as successful pongs
        @type suppressErrors: bool
        '''
        Observable.__init__(self)
        XMPPHandler.__init__(self)
        self.__entity = entity
        self.__interval = interval
        self.__maxCount = maxCount
        self.__suppressErrors = suppressErrors
        self.__pingLoop = None
        self.__count = 0

    def ping(self, entity, sender = None):
        '''
        Send out a ping request and wait for a response
        @param entity: Entity to be pinged
        @type entity: L{jid.JID}
        @return: A deferred that fires upon receiving a response.
        @rtype: L{defer.Deferred}
        @param sender: Optional sender address.
        @type sender: L{jid.JID}
        '''
        request = jabber.xmlstream.IQ(self.xmlstream, 'get')
        request.addElement((NS_PING, 'ping'))
        if sender is not None:
            request['from'] = unicode(sender)
        defer = request.send(entity.full())
        return defer

    def _suppressError(self, failure): #pylint: disable=R0201
        '''Suppress error responses from server'''
        failure.trap(jabber.error.StanzaError)
        exc = failure.value
        if exc.condition in ['service-unavailable', 'feature-not-implemented']:
            return None
        return failure

    def _onSuccess(self, result):
        '''Handler of alive event'''
        #log.msg('Connection to %s is alive' % (self.__entity))
        self.notifyObservers(ping = {'state': 'alive', 'entity': self.__entity})
        self.__count = 0

    def _sendPing(self):
        '''Send ping to entity and wait for response'''
        if self.__count > self.__maxCount:
            self.notifyObservers(ping = {'state': 'timeout', 'entity': self.__entity})
        self.__count += 1
        defer = self.ping(self.__entity)
        if self.__suppressErrors:
            defer.addErrback(self._suppressError)
        defer.addCallback(self._onSuccess)
        defer.addErrback(self._suppressError)
        defer.addErrback(_notifyError, self)

    def stopPing(self):
        '''Stop ping loop'''
        if self.__pingLoop is not None and self.__pingLoop.running:
            self.__pingLoop.stop()
            self.__pingLoop = None
            log.msg('Ping to %s stopped' % (self.__entity))

    def startPing(self):
        '''Start ping loop'''
        self.__count = 0
        self.__pingLoop = LoopingCall(self._sendPing)
        self.__pingLoop.start(self.__interval, now = False)
        log.msg('Ping to %s started' % (self.__entity))


class PingManager(object):
    '''
    Manages multiple ping protocols
    @author: Oleg Kandaurov
    '''

    def __init__(self, parent, transport):
        '''
        Constructor
        @param parent: Object that handles events from ping protocols
        @param transport: Ping protocols will sent ping through transport 
        '''
        object.__init__(self)
        self.__parent = parent
        self.__transport = transport
        self.__pinger = dict()

    def startPing(self, entity):
        '''
        Start ping to entity
        @type entity: JID
        '''
        if not self.__pinger.has_key(entity):
            return
        self.__pinger[entity].startPing()

    def stopPing(self, entity):
        '''
        Stop ping to entity
        @type entity: JID
        '''
        if not self.__pinger.has_key(entity):
            return
        self.__pinger[entity].stopPing()

    def addPinger(self, entity, interval, maxCount, greedy):
        '''
        Add pinger to manager
        @param entity: Entity to be pinged
        @type entity: JID
        @param interval: Interval between sending ping
        @type interval: int
        @param maxCount: if maxCount pings without response has been occured, connection is
        considered lost
        @type maxCount: int
        @param greedy: True means that we treat all responses from pinged entity as pongs
        @param greedy: bool
        '''
        if self.__pinger.has_key(entity):
            return
        self.__pinger[entity] = HivemindPingClientProtocol(entity, interval, maxCount, greedy)
        self.__pinger[entity].addObserver(self.__parent)
        self.__pinger[entity].setHandlerParent(self.__transport)

    def removePinger(self, entity):
        '''
        Remove specific pinger from manager
        @type entity: JID
        '''
        if not self.__pinger.has_key(entity):
            return
        self.stopPing(entity)
        self.__pinger[entity].disownHandlerParent(self.__transport)
        self.__pinger[entity].deleteObserver(self.__parent)
        del self.__pinger[entity]

    def removeAll(self, exclude = None):
        '''
        Remove all pingers from manager except entities from exclude list
        @param exclude: exclude entities
        @type exclude: list
        '''
        for entity in self.__pinger.keys():
            if exclude is not None and entity not in exclude:
                self.removePinger(entity)


class HivemindXMPPClient(StreamManager, application.service.Service, Observable):
    '''
    Service that initiates an XMPP client connection
    @author: Oleg Kandaurov
    '''

    readable('server')
    def __init__(self, authenticator, factory, host = None, port = 5222):
        '''
        @type authenticator: HybridAuthenticator
        @type factory: XmlStreamFactory
        @type host: str
        @type port: int
        '''
        self.jid = authenticator.jid
        self.domain = authenticator.jid.host
        self.__server = jabber.jid.internJID(self.domain)
        self.host = host
        self.port = port
        self._connection = None
        StreamManager.__init__(self, factory)
        Observable.__init__(self)

    def startService(self):
        '''Start service'''
        application.service.Service.startService(self)
        self._connection = self._createConnector()

    def stopService(self):
        '''Stop service'''
        application.service.Service.stopService(self)
        self.factory.stopTrying()
        self._connection.disconnect()

    def initializationFailed(self, reason):
        '''
        Called when stream initialization has failed
        @param reason: A failure instance indicating why stream initialization failed
        @type reason: Failure
        '''
        log.msg('Initialization failed at %s' % (self.__class__.__name__))
        _notifyError(reason, self)

    def _authd(self, xs):
        '''
        Called when the stream has been initialized.
        Save the JID that we were assigned by the server, as the resource might
        differ from the JID we asked for. This is stored on the authenticator
        by its constituent initializers.
        '''
        StreamManager._authd(self, xs)
        self.jid = self.factory.authenticator.jid
        log.msg('Connection initialized at %s' % (self.__class__.__name__))
        self.notifyObservers(transport = 'initialized')

    def _connected(self, xs):
        '''Called when the transport connection has been established'''
        def _onError(error):
            '''Cause i can not add "self" argument to addObserver'''
            _notifyError(error, self)
        StreamManager._connected(self, xs)
        log.msg('Transport connected at %s' % (self.__class__.__name__))
        self.notifyObservers(transport = 'connected')
        xs.addObserver(STREAM_ERROR_EVENT, _onError)

    def _disconnected(self, reason):
        '''Called when the stream has been closed'''
        StreamManager._disconnected(self, reason)
        log.msg('Stream closed at %s' % (self.__class__.__name__))
        self.notifyObservers(transport = 'disconnected')

    def _createConnector(self):
        '''Create connector instance'''
        connector = XMPPClientConnector(reactor, self.domain, self.factory)
        connector.connect()
        return connector

    def _getConnection(self):
        '''Return connector instance'''
        return self._connection

    connection = property(_getConnection, None, None, 'Client connector')


class HivemindRosterClientProtocol(RosterClientProtocol, Observable):
    '''
    Client side roster protocol for Hivemind
    @author: Oleg Kandaurov
    '''
    def __init__(self):
        '''Constructor'''
        RosterClientProtocol.__init__(self)
        Observable.__init__(self)

    def connectionInitialized(self):
        '''Called when connection has been initialized'''
        RosterClientProtocol.connectionInitialized(self)
        defer = self.getRoster()
        defer.addCallback(lambda roster: self.notifyObservers(roster = {'type': 'roster',
                'data': roster}))
        defer.addErrback(_notifyError, self)

    def onRosterRemove(self, entity):
        '''
        Called when a roster push for the removal of an item was received
        @param entity: The entity for which the roster item has been removed
        @type entity: JID
        '''
        self.notifyObservers(roster = {'type': 'remove', 'data': entity})

    def onRosterSet(self, item):
        '''
        Called when a roster push for a new or update item was received
        @param item: The pushed roster item
        @type item: RosterItem
        '''
        self.notifyObservers(roster = {'type': 'set', 'data': item})


class HivemindService(PubSubServiceFromBackend, Observable):
    '''
    Pubsub service for hivemind implementation
    @author: Andrew Vasilev, Oleg Kandaurov
    '''

    def __init__(self, mindMap, undoStack, backend):
        '''
        Constructor
        @param mindMap: initial record in the service with whole mind map
        @type mindMap: str
        @param undoStack: undostack associated with mindMap
        @type undoStack: str
        @param backend: Generic publish - subscribe backend service
        @type backend: BackendService
        '''
        PubSubServiceFromBackend.__init__(self, backend)
        Observable.__init__(self)
        self._errorMap[NotAllowedSubscriber] = ('not-allowed', 'closed-node', None)
        self._errorMap[NotInRosterSubscriber] = ('not-authorized', 'not-in-roster-group', None)
        rootNode = self.backend.storage.rootNode
        self.serviceJID = rootNode.owner
        self.discoIdentity = {'category' : 'pubsub', 'type' : 'generic', 'name' : 'Hivemind'}
        mindMapChangeset = Changeset(mindMap, 'mindmap')
        undoStackChangeset = Changeset(undoStack, 'undostack')
        rootNode.storeChangeset(mindMapChangeset, rootNode.owner)
        rootNode.storeChangeset(undoStackChangeset, rootNode.owner)

    def connectionInitialized(self):
        '''The XML stream has been initialized'''
        self.notifyObservers(protocol = {'state': 'subscribed', 'entity': self.parent.jid})

    def shutdown(self):
        '''Shutdown service'''
        self.notifyObservers(protocol = {'state': 'unsubscribed', 'entity': self.parent.jid})
        return internet.defer.succeed(None)

    def subscribe(self, requestor, service, nodeIdentifier, subscriber):
        '''Called when a subscribe request has been received'''
        def subscribeSucceded(result):
            '''Notify observers about subscription state'''
            self.notifyObservers(protocol = {'state': 'subscribed', 'entity': subscriber})
            return result
        defer = self.backend.subscribe(nodeIdentifier, subscriber, requestor)
        defer.addCallback(subscribeSucceded)
        return defer.addErrback(self._mapErrors)

    def unsubscribe(self, requestor, service, nodeIdentifier, subscriber):
        '''Called when a unsubscribe request has been received'''
        def unsubscribeSucceded(result):
            '''Notify observers about subscription state'''
            self.notifyObservers(protocol = {'state': 'unsubscribed', 'entity': subscriber})
            return result
        defer = self.backend.unsubscribe(nodeIdentifier, subscriber, requestor)
        defer.addCallback(unsubscribeSucceded)
        return defer.addErrback(self._mapErrors)

    def proposeChangeset(self, changeset):
        '''
        Propose changeset to the HiveMind network
        @param changeset: changeset to publish inside network
        @type changeset: Changeset
        '''
        rootNode = self.backend.storage.rootNode
        def cb(subscriptions):
            '''Notify all subscribers'''
            for subscription in subscriptions:
                self._notify({'items': rootNode.stack.getItems(1),
                        'nodeIdentifier': rootNode.nodeIdentifier,
                        'subscription': subscription})
        rootNode.storeChangeset(changeset, rootNode.owner)
        defer = rootNode.getSubscriptions()
        defer.addCallback(cb)


class HivemindClient(PubSubClient, Observable):
    '''
    Pubsub client for HiveMind service
    @author: Andrew Vasilev
    '''

    def __init__(self, hivemindService):
        '''
        @param hivemindService: jid of hivemind service
        @type hivemindService: JID
        '''
        PubSubClient.__init__(self)
        Observable.__init__(self)
        self.__service = hivemindService

    def connectionInitialized(self):
        '''The XML stream has been initialized.'''
        PubSubClient.connectionInitialized(self)
        defer = self.subscribe(self.__service, XMPP_NODE_NAME, self.parent.jid)
        defer.addCallback(self.retrieveItems)
        defer.addCallback(lambda success: self.notifyObservers(protocol =
                    {'state': 'subscribed', 'entity': self.parent.jid}))
        defer.addErrback(_notifyError, self)

    def shutdown(self):
        '''Shutdown client'''
        defer = self._unsubscribe()
        return defer

    def _unsubscribe(self):
        '''Unsubscribe from the service'''
        def notifyUnsubscription(result):
            ''' Notify observers about unsubscribe event '''
            log.msg('Unsubscribed from %s' % self.__service.full())
            if unsubscribeTimer.active():
                unsubscribeTimer.cancel()
            self.notifyObservers(protocol =
                    {'state': 'unsubscribed', 'entity': self.parent.jid})

        unsubscribeTimer = reactor.callLater(UNSUBSCRIBE_TIMEOUT, notifyUnsubscription, self)
        defer = self.unsubscribe(self.__service, XMPP_NODE_NAME, self.parent.jid)
        defer.addCallback(notifyUnsubscription)
        defer.addErrback(_notifyError, self)
        return defer

    def retrieveItems(self, result = None):
        '''Retrieve all items from the server'''
        defer = self.items(self.__service, XMPP_NODE_NAME)
        defer.addCallback(self._parseInitItems)
        defer.addErrback(_notifyError, self)

    def _parseInitItems(self, items):
        '''
        Parse items on initial request
        @param items: a set of items retrieved from the service
        @type items: list
        '''
        for item in items:
            changeset = Changeset.fromElement(item)
            self.notifyObservers(changeset = changeset)

    def itemsReceived(self, event):
        '''
        Called when an items notification has been received for a node.
        @param event: The items event.
        @type event: ItemsEvent
        '''
        if event.sender != self.__service or event.nodeIdentifier != XMPP_NODE_NAME:
            return
        self._parseInitItems(event.items)

    def proposeChangeset(self, changeset):
        '''
        Propose changeset to the hivemind network
        @param changeset: changeset to send to the controller
        @type changeset: str
        '''
        changeset.publisher = self.parent.jid
        defer = self.publish(self.__service, XMPP_NODE_NAME, [changeset.toElement()])
        defer.addErrback(_notifyError, self)


class HivemindBackendService(BackendService):
    '''
    Hivemind backend service
    @author: Oleg Kandaurov
    '''

    BackendService.nodeOptions['pubsub#access_model'] = \
            {'type': 'list-single',
            'label': 'Who may subscribe and retrieve items',
            'options': {
                'authorize': 'Subscription requests must be approved and only ' +
                    'subscribers may retrieve items',
                'open': 'Anyone may subscribe and retrieve items',
                'presence': 'Anyone with a presence subscription of both or from  ' +
                        'may subscribe and retrieve items',
                'roster': 'Anyone in the specified roster group(s) may subscribe ' +
                            'and retrieve items',
                'whitelist':'Only those on a whitelist may subscribe and retrieve items'}}
    BackendService.nodeOptions['pubsub#publish_model'] = \
            {'type': 'list-single',
            'label': 'Specify the publisher model',
            'options': {
                'publishers': 'Only publishers may publish',
                'subscribers': 'Subscribers may publish',
                'open': 'Anyone may publish'}},

    def __init__(self, storage):
        '''
        Initialize backend service
        @param storage: Storage that associated with service
        @type storage: Storage
        '''
        BackendService.__init__(self, storage)

    def _checkAuth(self, node, requestor):
        '''
        Check if entity is authorized to publish to node
        @type node: Node
        @type requestor: JID
        '''
        def check(affiliation, node):
            '''Check affiliation for node'''
            if affiliation == 'outcast':
                raise idavoll.error.Forbidden()
            elif affiliation == 'member':
                raise idavoll.error.ItemForbidden()
            return node
        defer = node.getAffiliation(requestor)
        defer.addCallback(check, node)
        return defer

    def publish(self, nodeIdentifier, items, requestor):
        '''Publish particular items to the node'''
        def checkSubscription(node):
            '''Check requestor subscription for the current node'''
            if not node.isSubscribed(requestor):
                raise idavoll.error.NotSubscribed()
            return node
        defer = self.storage.getNode(nodeIdentifier)
        defer.addCallback(checkSubscription)
        defer.addCallback(self._checkAuth, requestor)
        defer.addCallback(self._doPublish, items, requestor)
        return defer


class HivemindStorage(Storage):
    '''
    Hivemind storage
    @author: Oleg Kandaurov
    '''

    readable('rootNode')
    def __init__(self, parent):
        '''
        Initialize storage
        '''
        Storage.__init__(self)
        self.__parent = parent
        self.__rootNode = None

    def createRootNode(self, owner, config):
        '''
        Creates root node of the storage
        @param owner: Owner of the node
        @type owner: JID
        @param config: Node configuration
        @type config: dict
        '''
        node = HivemindNode(XMPP_NODE_NAME, owner, config)
        self._nodes[XMPP_NODE_NAME] = node
        node.addObserver(self.__parent)
        self.__rootNode = node
        return internet.defer.succeed(None)


class HivemindNode(LeafNode, Observable, object):
    '''
    Hivemind node
    @author: Oleg Kandaurov
    '''

    readable('owner', 'stack')
    writable('roster')
    def __init__(self, nodeIdentifier, owner, config):
        '''
        @param nodeIdentifier: Name of the node
        @type nodeIdentifier: str
        @param owner: Owner of the current node
        @type owner: JID
        @param config: Node configuration
        @type config: dict
        '''
        Observable.__init__(self)
        LeafNode.__init__(self, nodeIdentifier, owner, config)
        self.__stack = ChangesetStack()
        self.__roster = list()
        self.__owner = owner

    def storeItems(self, items, publisher):
        '''
        Store items in persistent storage for later retrieval
        @param items: The list of items to be stored
        @type items: list of Element
        @param publisher: JID of the publishing entity
        @type publisher: JID
        @return: deferred that fires upon success
        '''
        for element in items:
            element.attributes['publisher'] = publisher.full()
            self.__stack.addUnsignedChangeset(Changeset.fromElement(element))
            self.notifyObservers(changeset = Changeset.fromElement(element))
        return internet.defer.succeed(None)

    def storeChangeset(self, changeset, publisher):
        '''
        Store changeset in persistent storage for later retrieval
        @param items: The changeset to be stored
        @type items: Changeset
        @param publisher: JID of the publishing entity
        @type publisher: JID
        '''
        changeset.publisher = publisher
        self.__stack.addUnsignedChangeset(changeset)
        self.notifyObservers(changeset = changeset)

    def getItems(self, maxItems = None):
        '''
        Get items.
        If maxItems is not given, all items in the node are returned.
        Otherwise, maxItems limits the returned items to a maximum
        of that number of most recently published items.
        @param maxItems: if given, a natural number (> 0) that limits the
                            returned number of items.
        @return: deferred that fires with a list of found items.
        '''
        return internet.defer.succeed(self.__stack.getItems(maxItems))

    def getItemsById(self, itemIdentifiers):
        '''
        Get items by item id.
        Each item in the returned list is a xml Element
        @param itemIdentifiers: list of item ids.
        @return: deferred that fires with a list of found items.
        '''
        items = []
        for itemIdentifier in itemIdentifiers:
            items.append(self.__stack.getItemById(itemIdentifier))
        return internet.defer.succeed(items)

    def addSubscription(self, subscriber, state, options):
        '''Add new subscription to this node with given state'''
        def setSubscription(subscription):
            '''Just sets new subscription'''
            self._subscriptions[subscriber.full()] = subscription

        if self._subscriptions.get(subscriber.full()):
            return internet.defer.fail(idavoll.error.SubscriptionExists())
        defer = self._checkAccess(subscriber)
        defer.addCallback(lambda success: self.assignAffiliation(subscriber))
        defer.addCallback(lambda success: setSubscription(Subscription(
                self.nodeIdentifier, subscriber, state, options)))
        return defer

    def _checkAccess(self, entity):
        '''
        Check if entity are allowed to subscribe to current node
        '''
        if  self._config['pubsub#access_model'] == 'presence':
            return internet.defer.fail(idavoll.error.InvalidConfigurationOption())
        elif self._config['pubsub#access_model'] == 'roster':
            for contact in self.__roster:
                if contact == entity.userhost():
                    return internet.defer.succeed(None)
            return internet.defer.fail(NotInRosterSubscriber())
        elif self._config['pubsub#access_model'] == 'whitelist':
            if self._affiliations.get(entity.userhost()) is None:
                return internet.defer.fail(NotAllowedSubscriber())
        elif self._config['pubsub#access_model'] == 'authorize':
            msgBox = gui_factory.createNetworkAuthorizationDialog(entity.userhost())
            msgBox.exec_()
            result = msgBox.mapButton[msgBox.clickedButton()]
            if result == 'block' :
                self.setAffiliation(entity, 'outcast')
            if result != 'accept':
                return internet.defer.fail(NotAllowedSubscriber())
        return internet.defer.succeed(None)

    def assignAffiliation(self, entity):
        '''
        Assign affiliation to entity accrording to publish model
        @type entity: JID
        '''
        if entity.userhost() in self._affiliations:
            return
        affiliation = 'publisher'
        if self._config['pubsub#publish_model'] == 'publishers':
            affiliation = 'member'
        self.setAffiliation(entity, affiliation)

    def getAffiliation(self, entity):
        '''
        Get affiliations of entities with this node according to XEP - 0060
        All affiliations MUST be based on a bare JID instead of a full JID
        Therefore implementation in Idavoll is wrong
        @type entity: JID
        @return: affiliation associated with entity
        @rtype: one of ['owner', publisher', 'outcast', 'member']
        '''
        return internet.defer.succeed(self._affiliations.get(entity.userhost()))

    def setAffiliation(self, entity, affiliation):
        '''
        Set affilation of entity
        @type entity: JID
        @param affiliation: One of ['owner', publisher', 'outcast', 'member']
        @type affiliation: str
        '''
        self._affiliations[entity.userhost()] = affiliation
        self.notifyObservers(affiliation = (entity.userhostJID(), affiliation))
        return internet.defer.succeed(None)

    def getAffiliations(self):
        '''
        Get all affiliations
        @rtype: dict of str, str
        '''
        return self._affiliations

    def setAffiliations(self, affiliations):
        '''
        Set all affiliations
        @param affiliations: Dictionary of JIDs and affiliations
        @type affiliations: dict of str, str
        '''
        self._affiliations = affiliations

    affiliations = property(getAffiliations, setAffiliations, None, 'Affiliations property')

    def purge(self):
        raise Exception('Not implemented')

    def removeItems(self, itemIdentifiers):
        raise Exception('Not implemented')


class Changeset(object):
    '''
    Changeset representation
    @author: Andrew Vasilev
    '''

    writable('id', 'number', 'data', 'type', 'publisher')
    def __init__(self, data = None, changesetType = None, publisher = None,
            changesetId = None, number = None):
        '''
        Initialize object from the several components
        @param data: xml-serialized data of the changeset
        @type data: str
        @param changesetType: type of the changeset
        @type changesetType: str
        @param changesetId: unique id of the changeset
        @type changesetId: str
        @param publisher: publisher of the changeset
        @type publisher: JID
        @param number: the number of the changeset
        @type number: int
        '''
        #pylint: disable=R0913
        self.__id = changesetId
        self.__number = number
        self.__data = data
        self.__type = changesetType
        self.__publisher = publisher

    def __str__(self):
        return str(self.__number) + ' : ' + self.__publisher.full() + ' : ' + \
                str(self.__id) + ' : ' + str(self.__type) + ' : ' + str(self.__data)

    def toElement(self):
        '''
        Convert this changeset into domish xml Element
        @return: Element representation of current object
        @rtype: Element
        '''
        element = Element((None, 'item'))
        if self.__id: element.attributes['id'] = self.__id
        if self.__number: element.attributes['number'] = str(self.__number)
        if self.__type: element.attributes['type'] = self.__type
        if self.__publisher: element.attributes['publisher'] = self.__publisher.full()
        element.addRawXml(self.__data)
        return element

    def isValid(self, fieldList):
        '''
        Check fields of the changeset to be valid
        @param fieldList: fields of the changeset to test for None
        @type fieldList: list of str
        @rtype: bool
        '''
        for field in fieldList:
            if self.__getattribute__(field) is None:
                raise ChangesetException('Field ' + field + ' cannot be None')
        return True

    @staticmethod
    def fromElement(element):
        '''
        Create Changeset object from xml Element
        @param element: xml representation of the object
        @type element: Element
        '''
        xmlChangeset = element.firstChildElement()
        stripNamespace(xmlChangeset)
        changeset = xmlChangeset.toXml()
        return Changeset(changeset, element.getAttribute('type'),
                jabber.jid.JID(element.getAttribute('publisher')), element.getAttribute('id'),
                element.getAttribute('number'))


class ChangesetStack(object):
    '''
    Stack of the changesets
    @author: Andrew Vasilev, Oleg Kandaurov
    '''

    def __init__(self):
        self.__changesets = []
        self.__elements = []

    def getItemById(self, itemId):
        '''
        Get item by item id. Item is a xml Element
        @param itemId: Item id
        @type itemId: str
        @rtype: Element
        '''
        for element in self.__elements:
            if element.getAttribute('id') == itemId:
                return element
        return None

    def getItems(self, maxItems = None):
        '''
        Get items from stack
        @param maxItems: number of recently published items or all item if None
        @type maxItems: int
        @rtype: list of Element
        '''
        if maxItems:
            return self.__elements[-maxItems:]
        else:
            return self.__elements

    def addChangeset(self, changeset):
        '''
        Add changeset to the stack
        @param changeset: fully-functional changeset
        @type changeset: Changeset
        @raise ChangesetError: if passed changeset is invalid
        @raise ChangesetNumberError: if changeset has invalid humber
        @raise ChangesetIdError: if changeset has invalid id
        '''
        changeset.isValid(['id', 'number', 'data', 'type', 'publisher'])
        self._checkChangesetNumber(changeset)
        if not self._isUniqueId(changeset.id):
            raise ChangesetIdException('Changeset has not unique id')
        self.__changesets.append(changeset)
        self.__elements.append(changeset.toElement())

    def addUnsignedChangeset(self, changeset):
        '''
        Add unsigned changeset to the stack. Warning! Changes passed object!
        @param changeset: changeset with data and type fields present
        @type changeset: Changeset
        @raise ChangesetError: if passed changeset is invalid
        '''
        self.signChangeset(changeset)
        self.addChangeset(changeset)

    def signChangeset(self, changeset):
        '''
        Set appropriate number and hash for the changeset if needed
        @type changeset: Changeset
        '''
        changeset.number = len(self.__changesets)
        if not self._isUniqueId(changeset.id):
            changeset.id = self._generateUniqueId()

    def _checkChangesetNumber(self, changeset):
        '''
        Check number of the changeset to have next number in the list
        @type changeset: Changeset
        @raise ChangesetException: if changeset has wrong number
        '''
        if changeset.number != len(self.__changesets):
            raise ChangesetNumberException('Wrong number of the changeset.' +
                    '%d expected, but %d got' % (len(self.__changesets), changeset.number))

    def _generateUniqueId(self):
        '''
        Generate unique identifier for the changeset
        @return: new unique identifier for the changeset
        @rtype: str
        '''
        while True:
            newId = str(uuid.uuid4())
            if not self._isUniqueId(newId): continue
            return newId

    def _isUniqueId(self, changesetId):
        '''
        Check for the uniqueness of the id
        @type id: str
        @return: False if there is a changeset with the same id
        @rtype: bool
        '''
        if changesetId is None or changesetId is '': return False
        for changeset in self.__changesets:
            if changeset.id == changesetId:
                return False
        return True

    def __len__(self):
        '''Add support for getting number of changesets via len()'''
        return len(self.__changesets)

    def lastId(self):
        '''
        Get the id of the last changeset we are holding
        @rtype: str
        '''
        if len(self.__changesets) == 0: return None
        return self.__changesets[-1].id

    def items(self, changesetId = None):
        '''
        Retrieve the list of last items in the stack. If no id is passed,
        then return all available items
        @param id: id of the changeset known to the user
        @type id: str
        @rtype: list of Element
        '''
        if changesetId is None: return self.__elements
        index = 0
        for changeset in self.__changesets:
            index += 1
            if changeset.id == changesetId:
                return self.__elements[index:]
        return self.__elements

    def itemsByNumber(self, number = 1):
        '''
        Retrieve the list of last items in the stack.
        @param number: maximum number of items to recieve
        @type number: int
        @rtype: list of Element
        '''
        if number < 0 or number is None:
            return self.__elements
        count = min(number, len(self.__changesets))
        return self.__elements[-count:]


class ChangesetException(Exception):
    '''
    Base exception class for all exception that can be raised by ChangesetStack
    @author: Andrew Vasilev
    '''

    readable('reason')
    def __init__(self, reason):
        '''
        @param reason: reason for the exception to be raised
        @type reason: str
        '''
        Exception.__init__(self)
        self.__reason = reason

    def __str__(self):
        return self.__reason


class ChangesetNumberException(ChangesetException):
    '''
    Exception indicates that something is wrong with the number of the changeset
    @author: Andrew Vasilev
    '''

    def __init__(self, reason):
        ChangesetException.__init__(self, reason)


class ChangesetIdException(ChangesetException):
    '''
    Exception indicates that something is wrong with the identification of the changeset
    @author: Andrew Vasilev
    '''

    def __init__(self, reason):
        ChangesetException.__init__(self, reason)


class NotAllowedSubscriber(idavoll.error.Error):
    '''Subcriber is not in whitelist'''
    pass


class NotInRosterSubscriber(idavoll.error.Error):
    '''
    Subcriber is not in roster
    '''
    pass


def _notifyError(failure, observable):
    '''
    Log error and notify observers
    @param observable: Object that generates error.
    @type failure: Failure
    '''
    log.msg('Failure %s at %s' % (failure, object.__class__.__name__))
    observable.notifyObservers(error = failure)

def _isPubSubError(error):
    '''
    Check if error is PubSubError
    @type error: StanzaError
    @rtype bool
    '''
    if error.appCondition is None or error.appCondition.uri != NS_PUBSUB_ERRORS:
        return False
    return True

def _pubsubConditionFromStanzaError(error):
    '''
    Return pubsub specific condition and feature from stanza error
    @type error: StanzaError
    @rtype: tuple of str, str
    '''
    if not _isPubSubError(error):
        return None, None
    return error.appCondition.name, error.appCondition.getAttribute('feature')

