# -*- coding: utf-8 -*-
# HiveMind - Distributed mind map editor for Maemo 5 platform
# Copyright (C) 2010-2011 HiveMind developers
#
# HiveMind is the legal property of its developers, whose names are
# noticed in @author or @authors annotations at the beginning of each
# module or class.
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02111-1301  USA

'''
Dialogs for Maemo platform
'''

import hivemind.resource as resource
from PySide.QtGui import QFrame, QWidget, QGridLayout, QStackedWidget, QVBoxLayout, \
QRadioButton, QHBoxLayout, QToolBar, QIcon, QPalette, QPushButton, QLabel, \
QTextBrowser, QAbstractSlider, QTabWidget, QDialog, QFont, QDesktopServices
from PySide.QtCore import Qt, SIGNAL, QSize, QTimer, QUrl
from hivemind.dialogs import EditNodeDialog, EditLabelDialog, IconDialog, \
AboutDialog, EditEdgeDialog, SettingsDialog, AffiliationDialog, NetworkServiceDialog, \
NetworkClientDialog, NetworkConnectionDialog
from hivemind.gui_widgets import ScalingWidget, SettingsCreator, ActionImageButton
from hivemind.attribute import adaptCallable
import hivemind.__version__ as __version__


class MaemoDialogMixin:
    '''
    Mixin for Maemo dialogs

    @author: Alexander Kulikov
    '''

    def __init__(self):
        '''Initialize mixin'''
        self.setWindowFlags(Qt.Window)
        self.setAttribute(Qt.WA_Maemo5StackedWindow) #pylint: disable = E1103

    def exec_(self):
        '''This method is overriden to call show() as it is necessary on Maemo'''
        self.show()


class MaemoEditNodeDialog(MaemoDialogMixin, EditNodeDialog):
    '''
    Edit node dialog for Maemo platform

    @author: Alexander Kulikov
    '''

    def __init__(self, node, parent = None):
        '''
        Initialize dialog with specific node
        @type node: Node
        @type parent: QWidget
        '''
        EditNodeDialog.__init__(self, node, parent)
        MaemoDialogMixin.__init__(self)
        self._initComponents()
        self._showTextSourceButtons(self._isHtml)

    def _initComponents(self):
        '''Form initialization'''
        #pylint: disable = W0201
        toolbar = self._createToolbar()
        #create edit widget
        scalingWidget = ScalingWidget(self._editor, 2.0)
        self._editor.setFrameShape(QFrame.Box)
        editWidget = QWidget()
        editLayout = QGridLayout()
        editWidget.setLayout(editLayout)
        editLayout.addWidget(toolbar, 0, 0, 1, 6)
        editLayout.addWidget(scalingWidget, 1, 0, 1, 6)
        #create html widget
        scalingWidget = ScalingWidget(self._htmlEditor, 1.5)
        self._htmlEditor.setFrameShape(QFrame.Box)
        htmlWidget = QWidget()
        htmlLayout = QGridLayout()
        htmlWidget.setLayout(htmlLayout)
        htmlLayout.addWidget(scalingWidget, 0, 0)
        #create stacked widget
        self.__stackedWidget = QStackedWidget()
        self.__textIndex = self.__stackedWidget.addWidget(editWidget)
        self.__htmlIndex = self.__stackedWidget.addWidget(htmlWidget)
        mainLayout = QVBoxLayout()
        self.__showText = QRadioButton(self.tr('Text'))
        self.__showHtml = QRadioButton(self.tr('Source'))
        self.__showText.setChecked(True)
        mainLayout.addWidget(self.__stackedWidget)
        bottomLayout = QHBoxLayout()
        bottomLayout.addWidget(self.__showText)
        bottomLayout.addWidget(self.__showHtml)
        bottomLayout.addWidget(self._propagateFormatting)
        bottomLayout.addStretch()
        bottomLayout.addWidget(self._buttonBox)
        mainLayout.addLayout(bottomLayout)
        self._editor.setFocus()
        self.setLayout(mainLayout)
        self.__showHtml.connect(SIGNAL('toggled(bool)'), adaptCallable(self._changedMode))
        self._textHtmlAction.connect(SIGNAL('toggled(bool)'),
                                     adaptCallable(self._showTextSourceButtons))

    def _createToolbar(self):
        '''Create editor toolbar'''
        toolbar = QToolBar(self)
        self._boldAction.setIcon(QIcon(resource.getImage('node_editor_bold')))
        toolbar.addAction(self._boldAction)
        self._italicAction.setIcon(QIcon(resource.getImage('node_editor_italic')))
        toolbar.addAction(self._italicAction)
        self._underlineAction.setIcon(QIcon(resource.getImage('node_editor_underline')))
        toolbar.addAction(self._underlineAction)
        self._textFontAction.setIcon(QIcon(resource.getImage('node_editor_font')))
        toolbar.addAction(self._textFontAction)
        self._textHtmlAction.setIcon(QIcon(resource.getImage('node_editor_html')))
        toolbar.addAction(self._textHtmlAction)
        self._colorAction.setIcon(QIcon(resource.getImage('node_editor_font_color')))
        toolbar.addAction(self._colorAction)
        self._backgroundAction.setIcon(QIcon(
                        resource.getImage('node_editor_background_color')))
        toolbar.addAction(self._backgroundAction)
        self._clearFormatAction.setIcon(QIcon(resource.getImage('node_editor_clear')))
        toolbar.addAction(self._clearFormatAction)
        return toolbar

    def _changedMode(self, isSourceMode):
        '''
        Modificate dialog when html source was selected
        @type isSourceMode: bool
        '''
        self._isHtmlSourceMode = (isSourceMode == self.__htmlIndex)
        if isSourceMode == self.__htmlIndex:
            self._htmlEditor.setPlainText(self._editor.toHtml())
            self.__stackedWidget.setCurrentIndex(self.__htmlIndex)
        else:
            self._editor.setHtml(self._htmlEditor.toPlainText())
            self.__stackedWidget.setCurrentIndex(self.__textIndex)

    def _showTextSourceButtons(self, isHtml):
        '''
        Show switch buttons of text/source when html mode was selected
        @type isHtml: bool
        '''
        self.__showText.setVisible(isHtml)
        self.__showHtml.setVisible(isHtml)
        self._propagateFormatting.setVisible(not isHtml)

class MaemoEditLabelDialog(MaemoDialogMixin, EditLabelDialog):
    '''
    Edit label dialog for Maemo platform

    @author: Oleg Kandaurov
    '''
    def __init__(self, node, parent = None):
        '''
        Initialize dialog with specific node
        @type node: Node
        @type parent: QWidget
        '''
        EditLabelDialog.__init__(self, node, parent)
        MaemoDialogMixin.__init__(self)
        self._initComponents()

    def _initComponents(self):
        '''Form initialization'''
        #create toolbar
        toolbar = QToolBar(self)
        self._boldAction.setIcon(QIcon(resource.getImage('node_editor_bold')))
        toolbar.addAction(self._boldAction)
        self._italicAction.setIcon(QIcon(resource.getImage('node_editor_italic')))
        toolbar.addAction(self._italicAction)
        self._underlineAction.setIcon(QIcon(resource.getImage('node_editor_underline')))
        toolbar.addAction(self._underlineAction)
        secondaryLayout = QHBoxLayout()
        secondaryLayout.addWidget(self._fontParent)
        secondaryLayout.addSpacing(20)
        secondaryLayout.addWidget(self._colorParent)
        secondaryLayout.addWidget(self._colorButton)
        scalingWidget = ScalingWidget(self._editor, 2.0)
        self._editor.setFrameShape(QFrame.Box)
        mainLayout = QGridLayout()
        mainLayout.addWidget(toolbar, 0, 0, 1, 2)
        mainLayout.addWidget(self._comboFont, 0, 2, 1, 5)
        mainLayout.addWidget(self._comboSize, 0, 7, 1, 3)
        mainLayout.addWidget(scalingWidget, 1, 0, 1, 10)
        mainLayout.addLayout(secondaryLayout, 2, 0, 1, 10)
        bottomLayout = QHBoxLayout()
        bottomLayout.addWidget(self._propagateFormatting)
        bottomLayout.addStretch()
        bottomLayout.addWidget(self._buttonBox)
        mainLayout.addLayout(bottomLayout, 3, 0, 1, 10)
        self.setLayout(mainLayout)
        self._editor.setFocus()
        QWidget.setTabOrder(self._editor, self._buttonBox)
        QWidget.setTabOrder(self._buttonBox, self._comboFont)
        QWidget.setTabOrder(self._comboFont, self._comboSize)


class MaemoIconDialog(MaemoDialogMixin, IconDialog):
    '''
    Icon selection dialog for Maemo platform

    @author: Alexander Kulikov
    '''

    def __init__(self, node, parent = None):
        '''
        Initialize with specific node
        @type node: Node
        @type parent: QWidget
        '''
        IconDialog.__init__(self, node, parent)
        MaemoDialogMixin.__init__(self)
        self._initComponents()

    def _initComponents(self):
        '''Form initialization'''
        layout = QGridLayout()
        self._iconList.setBackgroundRole(QPalette.Window)
        self._iconList.setStyleSheet('''QGraphicsView {border: 1px solid gray;
                            border-radius: 5px; padding: 1 1px;}''')
        self._iconTable.setStyleSheet('''QListWidget {border: 1px solid gray;
                            border-radius: 5px; padding: 1 1px;}''')
        self._iconTable.setGridSize(QSize(68, 68))
        self._iconList.setMinimumHeight(self._listScene.height() + 8)
        self._iconList.setMaximumHeight(self._listScene.height() + 8)
        layout.addWidget(self._iconTable, 0, 0, 2, 4)
        layout.addWidget(self._iconList, 2, 0, 1, 4)
        layout.addWidget(self._buttonBox, 3, 0, 1, 4)
        self.setLayout(layout)


class MaemoAboutDialog(AboutDialog):
    '''
    About dialog for Maemo platform

    @author: Oleg Kandaurov, Alexander Kulikov
    '''
    def __init__(self, parent = None):
        AboutDialog.__init__(self, parent)
        self._createComponents()
        self._initComponents()

    def _createComponents(self):
        '''Create components'''
        #pylint: disable = W0201
        AboutDialog._createComponents(self)
        self._logo.setPixmap(self._logoImage)
        self._visitWebsite = QPushButton(self.tr("Visit website"))
        self._reportBug = QPushButton(self.tr("Report bug"))
        self._name = QLabel("<h2>HiveMind %s</h2>" % __version__.VERSION)
        self._about = QTextBrowser()
        self._about.setHtml(self.tr('''<br><br><br><br><br><br><br><br><br><center>
                            This application  is licensed under the terms <br>of the
                            <a href="http://www.gnu.org/licenses/gpl-3.0.html">
                            GNU General Public License v3</a>
                            <p><br>Copyright &copy; 2010-2011 HiveMind developers</center>
                            <ul style="list-style-type: square;margin: 0px 0px 0px 40px">
                            <li><a href="mailto:golovchenkoaa@gmail.com">Andrey Golovchenko</a>, core developer
                            <li><a href="mailto:vamonster@gmail.com">Andrey Vasilev</a>, developer/designer
                            <li><a href="mailto:asgard1988@yandex.ru">Alexander Kulikov</a>, developer
                            <li><a href="mailto:kandaurov.oleg@gmail.com">Oleg Kandaurov</a>, developer
                            <li><a href="mailto:ivparamonov@gmail.com">Ilya Paramonov</a>, technical manager</ul>
                            <br><center>HiveMind development is supported by<br>
                            <a href="http://fruct.org/">
                            Open Innovations Framework Program FRUCT</a>
                            <br><br>HiveMind logo by <a href="http://nicubunu.ro">Nicu Buculei</a> (nicubunu)
                            <br>Notification icons by <a href="http://www.mentalrey.it/">Mentalrey</a>
                            <br><br><br><br><br><br><br><br><br>
                            </center>'''))
        self._about.setVerticalScrollBarPolicy(Qt.ScrollBarAlwaysOff)
        self._about.setHorizontalScrollBarPolicy(Qt.ScrollBarAlwaysOff)
        self._about.setOpenExternalLinks(True)
        self._scrollBar = self._about.verticalScrollBar()
        self._scrollBar.setSingleStep(1)
        self._scrollBar.setRepeatAction(QAbstractSlider.SliderSingleStepAdd, 0, 70)
        self._scrollBar.connect(SIGNAL('actionTriggered(int)'),
                                 adaptCallable(self._scroll))
        self._visitWebsite.connect(SIGNAL('clicked()'), openWebsite)
        self._reportBug.connect(SIGNAL('clicked()'), openBugReport)

    def _initComponents(self):
        '''Form initialization'''
        mainLayout = QGridLayout()
        mainLayout.addWidget(self._logo, 0, 14, 16, 6)
        mainLayout.addWidget(self._name, 0, 0, 3, 14, Qt.AlignCenter)
        mainLayout.addWidget(self._info, 2, 0, 3, 14, Qt.AlignCenter)
        mainLayout.addWidget(self._about, 5, 0, 15, 14)
        buttonsLayout = QVBoxLayout()
        buttonsLayout.addWidget(self._visitWebsite)
        buttonsLayout.addWidget(self._reportBug)
        mainLayout.addLayout(buttonsLayout, 16, 14, 4, 6)
        self.setLayout(mainLayout)

    def _scroll(self, action):
        '''
        Handle action from vertical scroll bar

        @type action: int
        '''
        if action == QAbstractSlider.SliderSingleStepAdd:
            position = self._scrollBar.sliderPosition()
            self._scrollBar.setSliderPosition(position + self._scrollBar.singleStep())
            if position == self._scrollBar.sliderPosition():
                self._scrollBar.triggerAction(QAbstractSlider.SliderToMinimum)


class MaemoEditEdgeDialog(MaemoDialogMixin, EditEdgeDialog):
    '''
    Dialog for editing decoration and style of a joint link for Maemo platform
    @author: Oleg Kandaurov
    '''
    def __init__(self, node = None, parent = None):
        '''
        Initialize dialog with specific node
        @type node: Node
        '''
        EditEdgeDialog.__init__(self, node, parent)
        MaemoDialogMixin.__init__(self)
        self._initComponents()

    def _initComponents(self):
        '''Form initialization'''
        colorLayout = QHBoxLayout()
        colorLayout.addWidget(self._colorParent)
        colorLayout.addWidget(self._colorButton)
        mainLayout = QGridLayout()
        mainLayout.addWidget(self._widthLabel, 0, 0)
        mainLayout.addWidget(self._styleLabel, 0, 1)
        mainLayout.addWidget(self._edgeWidthComboBox, 1, 0)
        mainLayout.addWidget(self._edgeStyleComboBox, 1, 1)
        mainLayout.addWidget(self._nodeStyleLabel, 2, 0, 1, 2)
        mainLayout.addWidget(self._nodeStyleComboBox, 3, 0, 1, 2)
        mainLayout.addLayout(colorLayout, 4, 0, 1, 2)
        mainLayout.addWidget(self._buttonBox, 5, 0, 1, 2)
        self.setLayout(mainLayout)


class MaemoSettingsDialog(MaemoDialogMixin, SettingsDialog):
    '''
    Settings dialog for Maemo platform
    @author: Oleg Kandaurov
    '''
    def __init__(self, parent = None):
        '''
        @type parent: QWidget
        '''
        SettingsDialog.__init__(self, parent)
        MaemoDialogMixin.__init__(self)
        tabWidget = QTabWidget()
        self._general = SettingsCreator(['locale', 'confirmDelete', 'useExtendedMenu'])
        generalTab = QWidget()
        generalTab.setLayout(self._general.getLayout())
        self._node = SettingsCreator(['defaultFont',
                'defaultNodeTextColor',
                'defaultNodeBackgroundColor',
                'defaultSelectedNodeBackgroundColor'])
        nodeTab = QWidget()
        nodeTab.setLayout(self._node.getLayout())
        self._edge = SettingsCreator(['defaultEdgeWidth', 'defaultEdgeColor'])
        edgeTab = QWidget()
        edgeTab.setLayout(self._edge.getLayout())
        self._edgeLabel = SettingsCreator(['defaultLabelFont', 'defaultLabelColor'])
        edgeLabelTab = QWidget()
        edgeLabelTab.setLayout(self._edgeLabel.getLayout())
        self._animation = SettingsCreator(['autoHideNodeMenuTimeout',
                'showNodeMenuAnimationDuration',
                'hideNodeMenuAnimationDuration'])
        animationTab = QWidget()
        animationTab.setLayout(self._animation.getLayout())
        self._autosave = SettingsCreator(['autosaveEnabled', 'autosaveInterval'])
        autosaveTab = QWidget()
        autosaveTab.setLayout(self._autosave.getLayout())
        self._maxTextWidth = SettingsCreator(['defaultZoomLevel','maxNodeTextWidth'])
        otherTab = QWidget()
        otherTab.setLayout(self._maxTextWidth.getLayout())
        tabWidget.addTab(generalTab, self.tr('General'))
        tabWidget.addTab(nodeTab, self.tr('Node'))
        tabWidget.addTab(edgeTab, self.tr('Edge'))
        tabWidget.addTab(edgeLabelTab, self.tr('Edge Label'))
        tabWidget.addTab(animationTab, self.tr('Animation'))
        tabWidget.addTab(autosaveTab, self.tr('Autosave'))
        tabWidget.addTab(otherTab, self.tr('View'))
        mainLayout = QVBoxLayout()
        mainLayout.addWidget(tabWidget)
        mainLayout.addWidget(self._buttonBox)
        self.setLayout(mainLayout)

    def _accept(self):
        '''
        Retrieve new settings
        '''
        self._newSettings.append(self._general.retrieve())
        self._newSettings.append(self._node.retrieve())
        self._newSettings.append(self._edge.retrieve())
        self._newSettings.append(self._edgeLabel.retrieve())
        self._newSettings.append(self._animation.retrieve())
        self._newSettings.append(self._autosave.retrieve())
        self._newSettings.append(self._maxTextWidth.retrieve())
        SettingsDialog._accept(self)


class MaemoAffiliationDialog(MaemoDialogMixin, AffiliationDialog):
    '''
    Affiliation dialog for Maemo platform
    @author: Oleg Kandaurov
    '''
    def __init__(self, model, parent = None):
        AffiliationDialog.__init__(self, model, parent)
        MaemoDialogMixin.__init__(self)


class MaemoExtendedMenuDialog(QDialog):
    '''
    Extended menu dialog for Maemo platform
    @author: Alexander Kulikov
    '''
    def __init__(self, actionBag, freezeTime = 0, parent = None):
        '''
        @type actionBag: ActionBag
        @param freezeTime: Time interval during which dialog can not be rejected
        @type freezeTime: int
        @type parent: None
        '''
        QDialog.__init__(self, parent)
        self.setWindowTitle(self.tr('Extended menu'))
        self.__freezed = freezeTime != 0
        QTimer.singleShot(freezeTime, self._unfreeze)
        self._createComponents(actionBag)

    def _createComponents(self, actionBag):
        '''
        Create dialog widgets
        @type actionBag: ActionBag
        '''
        layout = QGridLayout()
        font = QFont()
        font.setPixelSize(28)
        singleClickLabel = QLabel(self.tr('Single-click on the node to show the main menu:'))
        singleClickLabel.setFont(font)
        doubleClickLabel = QLabel(
                self.tr('Double-click on the node to show the additional menu:'))
        doubleClickLabel.setFont(font)
        layout.addWidget(singleClickLabel, 0, 0, 1, 4, Qt.AlignCenter)
        layout.addWidget(doubleClickLabel, 3, 0, 1, 4, Qt.AlignCenter)
        layout.addWidget(self._createActionWidget(actionBag.addNodeAction), 1, 0,
                Qt.AlignCenter)
        layout.addWidget(QLabel(self.tr('Add node')), 2, 0, Qt.AlignHCenter | Qt.AlignTop)
        layout.addWidget(self._createActionWidget(actionBag.editNodeAction), 1, 1,
                Qt.AlignCenter)
        layout.addWidget(QLabel(self.tr('Edit node')), 2, 1, Qt.AlignHCenter |
                Qt.AlignTop)
        layout.addWidget(self._createActionWidget(actionBag.removeNodeAction), 1, 2,
                Qt.AlignCenter)
        layout.addWidget(QLabel(self.tr('Remove node')), 2, 2, Qt.AlignHCenter |
                Qt.AlignTop)
        layout.addWidget(self._createActionWidget(actionBag.foldNodeAction), 1, 3,
                Qt.AlignCenter)
        layout.addWidget(QLabel(self.tr('<center>Fold/unfold<br>node subtree</center>')),
                2, 3, Qt.AlignHCenter | Qt.AlignTop)
        layout.addWidget(self._createActionWidget(actionBag.editNodeIconsAction), 4, 0,
                Qt.AlignCenter)
        layout.addWidget(QLabel(self.tr('Select icons')), 5, 0, Qt.AlignHCenter |
                Qt.AlignTop)
        layout.addWidget(self._createActionWidget(actionBag.editEdgeAction), 4, 1,
                Qt.AlignCenter)
        layout.addWidget(QLabel(self.tr('Edit edge')), 5, 1, Qt.AlignHCenter |
                Qt.AlignTop)
        layout.addWidget(self._createActionWidget(actionBag.enterTransferModeAction), 4, 2,
                Qt.AlignCenter)
        layout.addWidget(QLabel(self.tr('Move node')), 5, 2, Qt.AlignHCenter |
                Qt.AlignTop)
        layout.addWidget(self._createActionWidget(actionBag.editLabelAction), 4, 3,
                Qt.AlignCenter)
        layout.addWidget(QLabel(self.tr('Edit edge label')), 5, 3, Qt.AlignHCenter |
                Qt.AlignTop)
        self.setLayout(layout)

    def _unfreeze(self):
        '''Unfreeze reject functionality'''
        self.__freezed = False

    def reject(self):
        if self.__freezed == False:
            QDialog.reject(self)

    def _createActionWidget(self, action):
        '''
        Create action widget
        @type action: QAction
        '''
        button = ActionImageButton(action)
        button.connect(SIGNAL('clicked()'), self.accept)
        return button


class MaemoNetworkServiceDialog(MaemoDialogMixin, NetworkServiceDialog):
    '''
    Service dialog for Maemo platfrom
    @author: Oleg Kandaurov
    '''

    def __init__(self, parent = None):
        NetworkServiceDialog.__init__(self, parent)
        MaemoDialogMixin.__init__(self)


class MaemoNetworkClientDialog(MaemoDialogMixin, NetworkClientDialog):
    '''
    Client dialog for Maemo platfrom
    @author: Oleg Kandaurov
    '''

    def __init__(self, rosterModel, parent = None):
        NetworkClientDialog.__init__(self, rosterModel, parent)
        MaemoDialogMixin.__init__(self)


class MaemoNetworkConnectionDialog(MaemoDialogMixin, NetworkConnectionDialog):
    '''
    Connection dialog for Maemo platfrom
    @author: Oleg Kandaurov
    '''

    def __init__(self, parent = None):
        NetworkConnectionDialog.__init__(self, parent)
        MaemoDialogMixin.__init__(self)


def openWebsite():
    '''Open project web page'''
    QDesktopServices.openUrl(QUrl(
            'https://linuxlab.corp7.uniyar.ac.ru/projects/hivemind'))

def openBugReport():
    '''Open bug report page'''
    QDesktopServices.openUrl(QUrl(
            'https://linuxlab.corp7.uniyar.ac.ru/projects/hivemind/issues/new'))
