# -*- coding: utf-8 -*-
# HiveMind - Distributed mind map editor for Maemo 5 platform
# Copyright (C) 2010-2011 HiveMind developers
#
# HiveMind is the legal property of its developers, whose names are
# noticed in @author or @authors annotations at the beginning of each
# module or class.
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02111-1301  USA

'''
Helper widgets
'''

import hivemind.settings as settings
import hivemind.resource as resource
from PySide.QtGui import QGraphicsView, QTextEdit, QGraphicsWidget, \
QListWidgetItem, QListWidget, QGraphicsScene, QAbstractButton, QWidget, \
QAction, QUndoCommand, QLineEdit, QSpinBox, QCheckBox, QComboBox, QLabel, \
QToolButton, QIcon, QApplication, QFormLayout, QFontComboBox, QFontDatabase, \
QHBoxLayout, QGraphicsBlurEffect, QFrame, QPalette, QPushButton, \
QColorDialog, QPainter, QKeyEvent, QPixmap, QListView, QVBoxLayout, QGridLayout
from hivemind.attribute import readable, adaptCallable
from PySide.QtCore import QAbstractAnimation, QObject, SIGNAL, QLocale, Qt, \
QRect, QEvent, QRectF, QSize, QParallelAnimationGroup, QPropertyAnimation, \
QPointF, QRegExp, SLOT, QTimeLine


class SearchingTrMixin(object):
    '''
    Mixin, defining the substitution for QObject.tr() method which searches translation
    along the inheritance hierarchy

    @author: Ilya Paramonov
    '''

    def tr(self, text):
        '''
        Search the class and its ancestors for the translation of the given text
        '''
        for clazz in self.__class__.mro():
            translatedText = QApplication.translate(clazz.__name__, text)
            if translatedText != text: return translatedText
            if clazz.__name__ == 'SearchingTrMixin': return text
        return text


class ScalingWidget(QGraphicsView):
    '''
    Special widget for widgets scaling

    @author: Alexander Kulikov
    '''
    def __init__(self, widget, scale, parent = None):
        '''
        Initialize widget

        @type widget: QWidget
        @type scale: int
        @type parent: QWidget
        '''
        QGraphicsView.__init__(self, parent)
        self.__widget = widget
        self.__scene = QGraphicsScene()
        self.setScene(self.__scene)
        self.__scene.addWidget(widget)
        self.__scale = scale
        self.scale(self.__scale, self.__scale)
        self.setRenderHints(QPainter.Antialiasing)

    def resizeEvent(self, event):
        '''
        Resize event handler

        @type event: QResizeEvent
        '''
        width = event.size().width() / self.__scale
        height = event.size().height() / self.__scale
        self.__scene.setSceneRect(QRect(0, 0, width, height))
        self.__widget.resize(width, height)
        QGraphicsView.resizeEvent(self, event)


class SpecialTextEdit(QTextEdit):
    '''
    Special text edit widget which handles Enter as newline character insertion
    and emits accept event on Ctrl-Enter

    @author: Alexander Kulikov
    '''

    def __init__(self, parent = None):
        '''
        Initialize widget
        @type parent: QWidget
        '''
        QTextEdit.__init__(self, parent)

    def keyPressEvent(self, event):
        '''
        Key press event handler
        @type event: QKeyEvent
        '''
        if event.key() == Qt.Key_Return:
            if event.modifiers() == Qt.NoModifier:
                self.emit(SIGNAL('accept'))
                return
            if event.modifiers() & Qt.ControlModifier:
                event = QKeyEvent(QEvent.KeyPress, event.key(), Qt.NoModifier)
            QTextEdit.keyPressEvent(self, event)
        else:
            QTextEdit.keyPressEvent(self, event)


class IconItem(QGraphicsWidget):
    '''
    Graphics widget provides rendering icons

    @author: Alexander Kulikov
    '''

    readable('icon')
    def __init__(self, icon, size, iconSize, parent = None):
        '''
        Initialize widget with specified icon name
        @type icon: string
        @type size: int
        @type iconSize: int
        @type parent: QGraphicsItem
        '''
        QGraphicsWidget.__init__(self)
        self.__icon = icon
        self.__size = size
        self.__drawFrame = True
        self.__iconSize = iconSize
        self.resize(size, size)

    def setSize(self, size):
        self.__size = size

    def setIconSize(self, size):
        self.__iconSize = size

    def paint(self, paint, options, widget):
        '''Paint icon item'''
        delta = (self.__size - self.__iconSize) / 2
        if isinstance(paint, QPainter):
            paint.setRenderHint(QPainter.Antialiasing)
            paint.setBrush(self.palette().brush(widget.backgroundRole()))
            paint.drawRoundedRect(0, 0, self.__size, self.__size, 6, 6)
            paint.drawImage(QRectF(delta, delta,
                            self.__iconSize, self.__iconSize),
                            resource.getIconByName(self.__icon))


class ListWidgetItem(QListWidgetItem):
    '''
    List widget item for drawing icons in icon table
    @author: Alexander Kulikov
    '''
    readable('icon')
    def __init__(self, icon, parent = None):
        QListWidgetItem.__init__(self, parent)
        self.__icon = icon
        pixmap = QPixmap.fromImage(resource.getIconByName(icon).scaled(QSize(32, 32)))
        self.setIcon(QIcon(pixmap))

class IconTable(QListWidget):
    '''
    Icon table provides icon location in a grid
    '''
    def __init__(self, parent = None):
        '''
        Initialize icon table
        @type parent: QWidget
        '''
        QListWidget.__init__(self, parent)
        self.setViewMode(QListView.IconMode)
        self.setResizeMode(QListView.Adjust)
        self.setMovement(QListView.Static)
        self.setSelectionMode(QListView.NoSelection)
        for icon in settings.get("listIcons"):
            item = ListWidgetItem(icon)
            self.addItem(item)
        self.connect(SIGNAL('itemClicked(QListWidgetItem*)'), self._onClick)

    def _onClick(self, item):
        self.emit(SIGNAL('iconClicked'), item.icon)


class IconList(QGraphicsScene):
    '''
    Icon list provides icon location in an animated list

    @author: Alexander Kulikov
    '''

    DEFAULT_HORIZONTAL_SPACING = 10
    '''Default horizontal space between icon items'''

    DEFAULT_ICON_SIZE = 32
    '''Default icon size into item'''

    DEFAULT_ITEM_SIZE = 64
    '''Default item size'''

    __MIN_DRAG_PATH = 10
    '''Minimum drag path length'''

    MAX_ICON_COUNT = 10
    '''Maximum icon count'''

    def __init__(self, icons, parent = None):
        '''
        Initialize with specified icon list
        @type icons: list
        @type parent: QObject
        '''
        QGraphicsScene.__init__(self, parent)
        self.__horizontalSpacing = self.DEFAULT_HORIZONTAL_SPACING
        self.__itemSize = self.DEFAULT_ITEM_SIZE
        self.__iconSize = self.DEFAULT_ICON_SIZE
        self._updateSize()
        self.__items = []
        for icon in icons:
            self.addIcon(icon)

    def _updateSize(self):
        '''Update scene size'''
        self.setSceneRect(0, 0, (self.__itemSize + self.__horizontalSpacing) * \
                           self.MAX_ICON_COUNT - self.__horizontalSpacing, \
                           self.__itemSize)

    def _updatePosition(self):
        '''Calculate new position for icon items'''
        for i in xrange(0, len(self.__items)):
            item = self.__items[i]
            if isinstance(item, IconItem):
                x = i * (self.__itemSize + self.__horizontalSpacing)
                item.setPos(x, 0)

    def setItemSize(self, itemSize):
        '''
        Set item size of current items
        @type itemSize: int
        '''
        self.__itemSize = itemSize
        self._updateSize()
        for item in self.items():
            if isinstance(item, IconItem):
                item.setSize(itemSize)
        self._updatePosition()

    def setHorizontalSpacing(self, spacing):
        '''
        Set horizontal spacing beetwing icon items
        @type spacing: int
        '''
        self.__horizontalSpacing = spacing
        self._updateSize()
        self._updatePosition()

    def setIconSize(self, iconSize):
        '''
        Set icon size of current items
        @type iconSize: int
        '''
        self.__iconSize = iconSize
        self._updateSize()
        for item in self.items():
            if isinstance(item, IconItem):
                item.setIconSize(iconSize)

    def addIcon(self, icon):
        '''
        Add new icon in the icon list
        @type icon: string
        '''
        index = len(self.__items)
        if (index < self.MAX_ICON_COUNT):
            item = IconItem(icon, self.__itemSize, self.__iconSize, self)
            item.setZValue(len(self.__items))
            x = index * (item.rect().width() + self.__horizontalSpacing)
            item.setPos(x, 0)
            self.__items.append(item)
            self.addItem(item)

    def removeIcon(self, item):
        '''
        Remove icon widget from the icon list
        @type item: IconWidget
        '''
        self.removeItem(item)
        self.__items.remove(item)
        item.setParentItem(None)

    def getIcons(self):
        '''
        Get icon name list
        @rtype: list
        '''
        return [item.icon for item in self.__items]

    def mousePressEvent(self, event):
        '''
        Mouse press event handler
        @type event: QMouseEvent
        '''
        QGraphicsScene.mousePressEvent(self, event)
        self.__startPos = event.screenPos()
        self.__item = self.itemAt(event.scenePos())
        if self.__item is not None:
            self.__item.setZValue(len(self.__items))
        self.__draggedPath = 0

    def mouseMoveEvent(self, event):
        '''
        Mouse move event handler
        @type event: QMouseEvent
        '''
        delta = event.screenPos() - self.__startPos
        self.__draggedPath = max(self.__draggedPath, delta.manhattanLength())
        if self.__item is not None:
            self._moveItem(self.__item, event.scenePos() - event.lastScenePos())

    def mouseReleaseEvent(self, event):
        '''
        Mouse release event handler
        @type event: QMouseEvent
        '''
        QGraphicsScene.mouseReleaseEvent(self, event)
        if self.__item is not None:
            self.__item.setZValue(self.__items.index(self.__item))
        if self.__draggedPath < self.__MIN_DRAG_PATH:
            item = self.itemAt(event.scenePos())
            if item is not None:
                self.removeIcon(item)
        self.__item = None
        self._generateAnimation()

    def _moveItem(self, item, delta):
        '''
        Move icon widget in the icon list and on the scene
        @type item: IconWidget
        @type delta: QPoint
        '''
        delta.setY(0)
        item.setPos(item.pos() + delta)
        index = self.__items.index(item)
        max = len(self.__items) - 1
        left = (index - 1) * (self.__itemSize + self.__horizontalSpacing)
        right = (index + 1) * (self.__itemSize + self.__horizontalSpacing)
        if index > 0 and item.pos().x() < left + self.__itemSize / 2:
            self.__items[index - 1], self.__items[index] = \
            self.__items[index], self.__items[index - 1]
            self._generateAnimation([item])
        elif index < max and item.pos().x() > right - self.__itemSize / 2:
            self.__items[index + 1], self.__items[index] = \
            self.__items[index], self.__items[index + 1]
            self._generateAnimation([item])

    def _generateAnimation(self, exceptions = []):
        '''
        Create and start move animation for icon widgets
        @param exceptions: Widgets do not require animation
        @type exceptions: list
        '''
        group = QParallelAnimationGroup(self)
        for i in xrange(0, len(self.__items)):
            item = self.__items[i]
            if item in exceptions:
                continue
            left = i * (self.__itemSize + self.__horizontalSpacing)
            deltaX = left - item.pos().x()
            if deltaX != 0:
                animation = QPropertyAnimation(item, 'pos', self)
                animation.setDuration(abs(deltaX) * 3)
                animation.setStartValue(item.pos())
                animation.setEndValue(QPointF(left, 0))
                group.addAnimation(animation)
        group.start()


class ActionImageButton(QAbstractButton):
    '''
    Command widget that only shows the icon of the action and tightly assosiated with it.
    
    @author: Andrew Vasilev
    '''

    __MAX_ICON_HEIGHT = 64
    '''Maximum height and width of the icon'''

    def __init__(self, action, parent = None, sizeHint = None):
        '''
        @param action: action to bound state of the button
        @type action: QAction
        @param parent: parent widget to paint button on
        @type parent: QWidget
        @param sizeHint: maximum height\width of the icon to draw
        @type sizeHint: int
        '''
        QAbstractButton.__init__(self, parent)
        self.__action = action
        self._syncProperties()
        self._connectToAction()
        self._setupSizeHint(sizeHint)

    def _syncProperties(self):
        '''Synchronize action properties with the button properties and update the view'''
        self.setAutoRepeat(self.__action.autoRepeat())
        self.setCheckable(self.__action.isCheckable())
        self.setEnabled(self.__action.isEnabled())
        self.update()

    def _connectToAction(self):
        '''Create connections to the action state and property changes and enable acction
        trigger on button click'''
        self.__action.connect(SIGNAL('changed()'), self._syncProperties)
        self.__action.connect(SIGNAL('toggled(bool)'), self.setChecked)
        self.connect(SIGNAL('clicked()'), self.__action.trigger)

    def _setupSizeHint(self, sizeHint):
        '''
        Setup size hint attribute for the button. Cannot be more than 
        ActionImageButton.__MAX_ICON_HEIGHT and the size of the icon.
        @param sizeHint: proposed height/width of the button
        @type sizeHint: int
        '''
        defaultSize = ActionImageButton.__MAX_ICON_HEIGHT if sizeHint is None else sizeHint
        self.__sizeHint = QSize(defaultSize, defaultSize)
        newSize = self.__action.icon().actualSize(self.__sizeHint)
        if newSize.height() < self.__sizeHint.height():
            self.__sizeHint = newSize

    def sizeHint(self):
        '''
        @return: recomendet size for the widget
        @rtype: QSize
        '''
        return self.__sizeHint

    def paintEvent(self, paintEvent):
        '''
        Paint action icon in the corresponding state
        @param paintEvent: paint event representation
        @type paintEvent: QPaintEvent 
        '''
        painter = QPainter(self)
        if not self.__action.isEnabled(): mode = QIcon.Disabled
        elif self.isDown() or self.__action.isChecked(): mode = QIcon.Selected
        else: mode = QIcon.Normal
        self.__action.icon().paint(painter, self.rect(), Qt.AlignCenter, mode)


class MaemoConnectionStatusWidget(QWidget):
    '''
    Widget shows status of connection

    @author: Oleg Kandaurov
    '''
    def __init__(self, parent, connectionAction, subscribeAction):
        '''
        Initialize widget
        @type parent: QWidget
        @type connectionAction: StatusAction
        @type subscribeAction: StatusAction
        '''
        QWidget.__init__(self, parent)
        self.__connectionButton = ActionImageButton(connectionAction, parent, 48)
        self.__subscribeButton = ActionImageButton(subscribeAction, parent, 48)
        layout = QVBoxLayout()
        layout.setContentsMargins(5, 5, 5, 5)
        layout.addWidget(self.__connectionButton)
        layout.addWidget(self.__subscribeButton)
        self.setLayout(layout)


class StatusAction(QAction):
    '''
    Action with support for statuses

    @author: Oleg Kandaurov
    '''
    readable('image')

    def __init__(self, text, type, status, parent = None):
        '''
        Constructs action
        @param text: Text of the action
        @type text: str
        @param type: Type of the action
        @type type: str
        @param status: Initial status of the action
        @type status: str
        @param parent: Parent
        @type parent: QObject
        '''
        QAction.__init__(self, text, parent)
        SIGNAL('imageChanged(QImage)')
        SIGNAL('statusChanged(str)')
        self.__type = type
        self.setStatus(status)
        self.setEnabled(False)

    def setStatus(self, status):
        '''
        Set status of the action and change icon according to status
        '''
        self.__status = status
        self.__image = resource.getImage(str(self.__type) + '_' + str(self.__status))
        icon = QIcon()
        icon.addPixmap(self.__image, QIcon.Disabled)
        self.setIcon(icon)
        self.emit(SIGNAL('statusChanged'), self.__status)

    def getStatus(self):
        '''
        Get status of the action
        '''
        return self.__status

    status = property(getStatus, setStatus, None, 'Status of the Action')


class MaemoZoomWidget(QWidget):
    '''
    Widget for change map scaling

    @author: Alexander Kulikov
    '''
    def __init__(self, parent, zoomInAction, zoomOutAction):
        '''
        Initialize zoom widget
        @type parent: QWidget
        @type zoomInAction: QAction
        @type zoomOutAction: QAction
        '''
        QWidget.__init__(self, parent)
        zoomInAction.setIcon(resource.getIcon('zoom_in'))
        zoomInAction.setAutoRepeat(True)
        self.__zoomInButton = ActionImageButton(zoomInAction, parent)
        zoomOutAction.setIcon(resource.getIcon('zoom_out'))
        zoomOutAction.setAutoRepeat(True)
        self.__zoomOutButton = ActionImageButton(zoomOutAction, parent)
        layout = QVBoxLayout()
        layout.setContentsMargins(5, 20, 5, 5)
        layout.addWidget(self.__zoomInButton)
        layout.addWidget(self.__zoomOutButton)
        self.setLayout(layout)


class WidgetVisibilityAnimation(QAbstractAnimation):
    '''
    Widget visibility changer for the animation framework
    
    @author: Andrew Vasilev
    '''

    def __init__(self, parent, visibility):
        '''
        Initialize animation
        
        @param parent: target widget for visibility change
        @type parent: QWidget
        
        @param visibility: the visibility value
        @param visibility: bool
        '''
        QAbstractAnimation.__init__(self, parent)
        self.__widget = parent
        self.__visibility = visibility

    def updateCurrentTime(self, currentTime):
        '''Update widget property at passed time'''
        if self.__visibility:
            self.__widget.show()
        else:
            self.__widget.hide()

    def duration(self):
        '''Animation is instant'''
        return 0


class ActionModificationCommand(QUndoCommand):
    '''
    Undo stack common comand. Stores do and undo functions
    '''
    def __init__(self, do, undo):
        '''
        @param do: function that makes changes
        @param undo: function that withdarws changes
        '''
        QUndoCommand.__init__(self)
        self.__do = do
        self.__undo = undo

    def undo(self):
        '''Restore previous state by calling undo function'''
        self.__undo()

    def redo(self):
        '''Move to next state by calling do function'''
        self.__do()


class SettingsCreator(QObject):
    '''
    Create and layout application settings
    @author: Oleg Kandaurov
    '''
    def __init__(self, settings, parent = None):
        '''
        @param settings: Settings that are need to create
        @type settings: list
        @type parent: QWidget
        '''
        QObject.__init__(self, parent)
        self.__settings = settings
        self.__mainLayout = QFormLayout()
        self.__component = dict()
        self._createDecorations()

    def getLayout(self):
        return self.__mainLayout

    def _createDecorations(self):
        '''
        Create components
        '''
        for setting in self.__settings:
            getattr(self, '_create%s' % setting)()

    def retrieve(self):
        '''
        Retrieve settings
        '''
        newSettings = list()
        for setting in self.__settings:
            newSettings.append(self.__component[setting].retrieve())
        return newSettings

    def _createlocale(self):
        '''
        Create locale components
        '''
        languages = [{'English': QLocale(QLocale.English).name()},
                {'Russian': QLocale(QLocale.Russian).name()},
                {'Deutsch': QLocale(QLocale.German).name()}]
        binder = ComboBoxBinder('locale', settings.get('locale'), languages)
        self.__mainLayout.addRow(self.tr('Language'), binder)
        self.__component['locale'] = binder

    def _createdefaultZoomLevel(self):
        '''
        Create defaultZoomLevel components
        '''
        binder = SpinBoxBinder('defaultZoomLevel',
                settings.get('defaultZoomLevel'), -5, 5)
        self.__mainLayout.addRow(self.tr('Default zoom level'), binder)
        self.__component['defaultZoomLevel'] = binder

    def _createconfirmDelete(self):
        '''
        Create confirmDelete components
        '''
        binder = CheckBoxBinder('confirmDelete', settings.get('confirmDelete'))
        binder.setText(self.tr('Confirm node deletion'))
        self.__mainLayout.addRow(binder)
        self.__component['confirmDelete'] = binder

    def _createautoHideNodeMenuTimeout(self):
        '''
        Create autoHideNodeMenuTimeout components
        '''
        binder = SpinBoxBinder('autoHideNodeMenuTimeout',
                settings.get('autoHideNodeMenuTimeout'), 0, 10000)
        self.__mainLayout.addRow(self.tr('Menu hide timeout'), binder)
        self.__component['autoHideNodeMenuTimeout'] = binder

    def _createshowNodeMenuAnimationDuration(self):
        '''
        Create showNodeMenuAnimationDuration components
        '''
        binder = SpinBoxBinder('showNodeMenuAnimationDuration',
                settings.get('showNodeMenuAnimationDuration'), 0, 10000)
        self.__mainLayout.addRow(self.tr('Menu appearance duration'), binder)
        self.__component['showNodeMenuAnimationDuration'] = binder

    def _createhideNodeMenuAnimationDuration(self):
        '''
        Create hideNodeMenuAnimationDuration components
        '''
        binder = SpinBoxBinder('hideNodeMenuAnimationDuration',
                settings.get('hideNodeMenuAnimationDuration'), 0, 10000)
        self.__mainLayout.addRow(self.tr('Menu disappearance duration'), binder)
        self.__component['hideNodeMenuAnimationDuration'] = binder

    def _createdefaultFont(self):
        '''
        Create defaultFont components
        '''
        binder = FontBinder('defaultFont', settings.get('defaultFont'))
        layout = QHBoxLayout()
        layout.addWidget(binder.fontCombo)
        layout.addWidget(binder.sizeCombo)
        self.__mainLayout.addRow(self.tr('Text font'), layout)
        self.__component['defaultFont'] = binder

    def _createdefaultLabelFont(self):
        '''
        Create defaultLabelFont components
        '''
        binder = FontBinder('defaultLabelFont',
                settings.get('defaultLabelFont'))
        layout = QHBoxLayout()
        layout.addWidget(binder.fontCombo)
        layout.addWidget(binder.sizeCombo)
        self.__mainLayout.addRow(self.tr('Label font'), layout)
        self.__component['defaultLabelFont'] = binder

    def _createdefaultNodeTextColor(self):
        '''
        Create defaultNodeTextColor components
        '''
        binder = ColorLabelBinder('defaultNodeTextColor',
                settings.get('defaultNodeTextColor'))
        binderButton = QPushButton(self.tr('Choose color'))
        layout = QHBoxLayout()
        layout.addWidget(binder, Qt.AlignLeft)
        layout.addWidget(binderButton, Qt.AlignLeft)
        self.__mainLayout.addRow(self.tr('Text color'), layout)
        binderButton.connect(SIGNAL('clicked()'), binder.open)
        self.__component['defaultNodeTextColor'] = binder

    def _createdefaultNodeBackgroundColor(self):
        '''
        Create defaultNodeBackgroundColor components
        '''
        binder = ColorLabelBinder('defaultNodeBackgroundColor',
                settings.get('defaultNodeBackgroundColor'))
        binderButton = QPushButton(self.tr('Choose color'))
        layout = QHBoxLayout()
        layout.addWidget(binder, Qt.AlignLeft)
        layout.addWidget(binderButton, Qt.AlignLeft)
        self.__mainLayout.addRow(self.tr('Background color'), layout)
        binderButton.connect(SIGNAL('clicked()'), binder.open)
        self.__component['defaultNodeBackgroundColor'] = binder

    def _createdefaultEdgeColor(self):
        '''
        Create defaultEdgeColor components
        '''
        binder = ColorLabelBinder('defaultEdgeColor',
                settings.get('defaultEdgeColor'))
        binderButton = QPushButton(self.tr('Choose color'))
        layout = QHBoxLayout()
        layout.addWidget(binder, Qt.AlignLeft)
        layout.addWidget(binderButton, Qt.AlignLeft)
        self.__mainLayout.addRow(self.tr('Edge color'), layout)
        binderButton.connect(SIGNAL('clicked()'), binder.open)
        self.__component['defaultEdgeColor'] = binder

    def _createdefaultLabelColor(self):
        '''
        Create defaultLabelColor components
        '''
        binder = ColorLabelBinder('defaultLabelColor',
                settings.get('defaultLabelColor'))
        binderButton = QPushButton(self.tr('Choose color'))
        layout = QHBoxLayout()
        layout.addWidget(binder, Qt.AlignLeft)
        layout.addWidget(binderButton, Qt.AlignLeft)
        self.__mainLayout.addRow(self.tr('Label color'), layout)
        binderButton.connect(SIGNAL('clicked()'), binder.open)
        self.__component['defaultLabelColor'] = binder

    def _createdefaultEdgeWidth(self):
        '''
        Create defaultEdgeWidth components
        '''
        width = [{'1': 1}, {'2': 2}, {'4': 4}, {'8': 8}, {self.tr('Thin'): 11}]
        binder = ComboBoxBinder('defaultEdgeWidth',
                settings.get('defaultEdgeWidth'), width)
        self.__mainLayout.addRow(self.tr('Edge width'), binder)
        self.__component['defaultEdgeWidth'] = binder

    def _createdefaultSelectedNodeBackgroundColor(self):
        '''
        Create defaultSelectedNodeBackgroundColor component
        '''
        binder = ColorLabelBinder('defaultSelectedNodeBackgroundColor',
                settings.get('defaultSelectedNodeBackgroundColor'))
        binderButton = QPushButton(self.tr('Choose color'))
        layout = QHBoxLayout()
        layout.addWidget(binder, Qt.AlignLeft)
        layout.addWidget(binderButton, Qt.AlignLeft)
        self.__mainLayout.addRow(self.tr('Selected node background'), layout)
        binderButton.connect(SIGNAL('clicked()'), binder.open)
        self.__component['defaultSelectedNodeBackgroundColor'] = binder

    def _createautosaveEnabled(self):
        '''
        Create autosaveEnabled component
        '''
        binder = CheckBoxBinder('autosaveEnabled', settings.get('autosaveEnabled'))
        binder.setText(self.tr('Enable autosave of mind map'))
        self.__mainLayout.addRow(binder)
        self.__component['autosaveEnabled'] = binder

    def _createautosaveInterval(self):
        '''
        Create autosaveInterval component
        '''
        binder = SpinBoxBinder('autosaveInterval',
                settings.get('autosaveInterval'), 1, 600)
        binder.setSuffix(self.tr(' sec'))
        self.__mainLayout.addRow(self.tr('Autosave interval'), binder)
        self.__component['autosaveInterval'] = binder

    def _createuseExtendedMenu(self):
        '''
        Create useExtendedMenu component
        '''
        binder = CheckBoxBinder('useExtendedMenu', settings.get('useExtendedMenu'))
        binder.setText(self.tr('Use extended menu'))
        self.__mainLayout.addRow(binder)
        self.__component['useExtendedMenu'] = binder

    def _createmaxNodeTextWidth(self):
        '''
        Create maxNodeTextWidth component
        '''
        binder = SpinBoxBinder('maxNodeTextWidth',
                settings.get('maxNodeTextWidth'), 50, 1000)
        binder.setSuffix(self.tr(' px'))
        self.__mainLayout.addRow(self.tr('Maximum width of node text'), binder)
        self.__component['maxNodeTextWidth'] = binder


class ComponentBinder(QObject):
    '''
    Abstract component binder
    @author: Oleg Kandaurov
    '''
    def __init__(self, paramName, parent = None):
        '''
        Initialize binder
        @param paramName: Parameter that is binded to component
        @type paramName: str
        @type parent: QWidget
        '''
        self._value = None
        self._paramName = paramName

    def isValid(self, value):
        '''
        @return: whether value correct or not
        @rtype: bool
        '''
        return True

    def retrieve(self):
        '''
        @return: pair parameter and value
        @rtype: dict
        '''
        if self.isValid(self._value):
            return {self._paramName: self._value}
        return None

    def _getName(self):
        '''Retrieve name of parameter'''
        return self._paramName

    def _getValue(self):
        '''Retrieve current value'''
        if self.isValid(self._value):
            return self._value

    def _setValue(self, value):
        '''
        Sets new value
        @param value: new value
        '''
        if self.isValid(value):
            self._value = value

    name = property(_getName, None, None, 'Parameter name')
    value = property(_getValue, None, None, 'Binder value')


class LineEditBinder(QLineEdit, ComponentBinder):
    '''
    QLineEdit binder
    @author: Oleg Kandaurov
    '''
    def __init__(self, paramName, text, regexp = None, parent = None):
        '''
        Initialize binder
        @param paramName: Parameter that is binded to component
        @type paramName: str
        @param text: Initial text
        @type text: str
        @param regexp: Regular expression needed to inspect line edit values
        @type regexp: QRegExp
        @type parent: QWidget
        '''
        QLineEdit.__init__(self, parent)
        ComponentBinder.__init__(self, paramName, parent)
        self._value = text
        self.__regexp = QRegExp('.*') if regexp is None else regexp
        self.setText(text)
        SIGNAL('valid(bool)')
        self.textChanged.connect(self._setValue)

    def isValid(self, value = None):
        '''
        @return: whether value correct or not
        @rtype: bool
        '''
        if value is None: value = self.text()
        if self.__regexp.exactMatch(value):
            return True
        return False


class SpinBoxBinder(QSpinBox, ComponentBinder):
    '''
    QSpinBox binder
    @author: Oleg Kandaurov
    '''
    def __init__(self, paramName, value, min, max, parent = None): #pylint: disable=R0913
        '''
        Initialize binder
        @param paramName: Parameter that is binded to component
        @type paramName: str
        @param value: Initial value
        @type value: int
        @param min: Minimum value
        @type min: int
        @param max: Maximum value
        @type max: int
        @type parent: Qwidget
        '''
        QSpinBox.__init__(self, parent)
        ComponentBinder.__init__(self, paramName, parent)
        self.setMinimum(min)
        self.setMaximum(max)
        self.setValue(value)
        self._value = value
        self.valueChanged[int].connect(self._setValue)


class CheckBoxBinder(QCheckBox, ComponentBinder):
    '''
    QCheckBox binder
    @author: Oleg Kandaurov
    '''
    def __init__(self, paramName, value, parent = None):
        '''
        Initialize binder
        @param paramName: Parameter that is binded to component
        @type paramName: str
        @param value: Initial state
        @type value: bool
        @type parent: Qwidget
        '''
        QCheckBox.__init__(self, parent)
        ComponentBinder.__init__(self, paramName, parent)
        self._value = value
        self.toggled.connect(self._setValue)
        self.setChecked(value)


class ComboBoxBinder(QComboBox, ComponentBinder):
    '''
    QComboBox binder
    @author: Oleg Kandaurov
    '''
    def __init__(self, paramName, value, data, parent = None):
        '''
        Initialize binder
        @param paramName: Parameter that is binded to component
        @type paramName: str
        @param value: Initial string
        @type value: str
        @param data: Possible values in QComboBox
        @type data: list of dicts
        @type parent: QWidget
        '''
        QComboBox.__init__(self, parent)
        ComponentBinder.__init__(self, paramName, parent)
        self.__data = data
        self._value = value
        for values in data:
            self.addItem(values.keys().pop(), values.values().pop())
        self.setCurrentIndex(self.findData(value))
        self.connect(self, SIGNAL('activated(int)'), self._setValue)

    def _setValue(self, index):
        '''
        Sets new value
        @param value: index in combo box
        @type value: int
        '''
        self._value = self.__data[index].values().pop()


class ColorLabelBinder(QLabel, ComponentBinder):
    '''
    QColorDialog binder
    @author: Oleg Kandaurov
    '''
    def __init__(self, paramName, color, parent = None):
        '''
        Initialize binder
        @param paramName: Parameter that is binded to component
        @type paramName: str
        @param color: Initial color
        @type color: QColor
        @type parent: QWidget
        '''
        QLabel.__init__(self, parent)
        ComponentBinder.__init__(self, paramName, parent)
        self.__palette = self.palette()
        self.__effect = QGraphicsBlurEffect()
        self.__effect.setBlurRadius(2)
        self.setGraphicsEffect(self.__effect)
        self.setAlignment(Qt.AlignCenter)
        self.setAutoFillBackground(True)
        self.setFrameStyle(QFrame.StyledPanel | QFrame.Raised)
        self._setValue(color)

    def open(self):
        '''
        Open color dialog and sets color
        '''
        color = QColorDialog.getColor(self._value, self)
        self._setValue(color)

    def _setLabelColor(self, color):
        '''
        Change color of the label
        @type color: QColor
        '''
        self.__palette.setColor(QPalette.Window, color)
        self.setPalette(self.__palette)

    def isValid(self, color):
        '''
        @return: whether color valid or not
        @rtype: bool
        '''
        return color.isValid()

    def _setValue(self, value):
        '''
        Sets new value
        @param value: new value
        '''
        if self.isValid(value):
            self._setLabelColor(value)
            self._value = value


class FontBinder(QWidget, ComponentBinder):
    '''
    Font binder
    @author: Oleg Kandaurov
    '''
    readable('fontCombo', 'sizeCombo')
    def __init__(self, paramName, font, parent = None):
        '''
        Initialize binder
        @param paramName: Parameter that is binded to component
        @type paramName: str
        @param font: initial font
        @type font: QFont
        @type parent: QWidget
        '''
        QWidget.__init__(self, parent)
        ComponentBinder.__init__(self, paramName, parent)
        self._value = font
        self.__font = font
        self.__size = font.pointSize()
        self.__sizeCombo = QComboBox()
        self.__sizeCombo.addItem(str(self.__size))
        self.__sizeCombo.setEditable(True)
        self.__fontCombo = QFontComboBox()
        self.__fontCombo.setCurrentFont(font)
        self._setFont(font)
        self.__fontCombo.connect(SIGNAL('currentFontChanged(QFont)'), self._setFont)
        self.__sizeCombo.connect(SIGNAL('currentIndexChanged(int)'), self._setSize)

    def _setFont(self, font):
        '''
        Set font and update available sizes
        @type font: QFont
        '''
        self.__font = font
        self._fontSizes(font)
        self._setSizeIndex(self.__size)
        self.__font.setPointSize(self.__size)
        self._setValue(self.__font)

    def _setSize(self, index):
        '''
        Set size of font
        @type index: int
        '''
        if index == -1:
            self.__sizeCombo.addItem(str(self.__size))
            return
        self.__size = int(self.__sizeCombo.itemText(index))
        self.__font.setPointSize(self.__size)
        self._setValue(self.__font)

    def _setSizeIndex(self, size):
        '''
        Set font size in QComboBox
        @type size: int
        '''
        index = self.__sizeCombo.findText(str(size), Qt.MatchExactly)
        if index == -1:
            self.__sizeCombo.addItem(str(size))
        else:
            self.__sizeCombo.setCurrentIndex(index)

    def _fontSizes(self, font):
        '''
        Add to QComboBox sizes supported by current font
        @type font: QFont
        '''
        self.__sizeCombo.clear()
        sizes = [str(value) for value in QFontDatabase().pointSizes(font.family())]
        if str(self.__size) not in sizes:
            sizes.append(str(self.__size))
        self.__sizeCombo.addItems(sizes)


class FullScreenExitButton(QToolButton):
    '''
    Full screen exit button for Maemo platform

    @author: Oleg Kandaurov
    '''
    def __init__(self, parent = None):
        '''
        @type parent: QWidget
        '''
        QToolButton.__init__(self, parent)
        self.setIcon(QIcon.fromTheme('general_fullsize'))
        self.setFixedSize(self.sizeHint())
        pal = self.palette()
        backgroundColor = pal.color(self.backgroundRole())
        backgroundColor.setAlpha(128)
        pal.setColor(self.backgroundRole(), backgroundColor)
        self.setPalette(pal)
        self.setAutoFillBackground(True)
        self.setVisible(False)
        self.connect(self, SIGNAL('clicked()'), parent, SLOT('showNormal()'))
        parent.installEventFilter(self)

    def eventFilter(self, obj, event):
        '''
        Catch Windows state changed and Resize events from parent object
        @type obj: QEvent
        @type event: QEvent
        '''
        parent = self.parentWidget()
        if obj != self.parent():
            return QToolButton.eventFilter(self, obj, event)
        isFullScreen = parent.windowState() & Qt.WindowFullScreen
        if event.type() == QEvent.WindowStateChange:
            if isFullScreen:
                self.setVisible(True)
                self.raise_()
            else:
                self.setVisible(False)
        elif event.type() == QEvent.Resize:
            if self.isVisible():
                self.move(parent.width() - self.width(), parent.height() - self.height())
        return QToolButton.eventFilter(self, obj, event)


class SlideWidget(QWidget):
    '''
    Slide widget
    @author: Alexander Kulikov
    '''

    __FRAMES = 50
    '''Count of animation frames'''
    __DURATION = 200
    '''Animation duration'''

    __MINIMUM_VISIBLE_WIDTH = 24
    '''Width of part widget which will be visible always'''

    __MIN_DRAG_PATH = 25
    '''Minimum drag path length for the Maemo platform'''

    LEFT = 'left'
    '''Widget is attached to the left side of the parent'''

    def __init__(self, dockArea, parent):
        '''
        Initialize widget
        @param dockArea: side of parent widget to which widget will be attached
        @type dockArea: string
        @type parent: MainView
        '''
        QWidget.__init__(self, parent)
        self.__dockArea = dockArea
        self.__timeLine = QTimeLine(self.__DURATION, self)
        self.__timeLine.setFrameRange(0, self.__FRAMES)
        self.__timeLine.connect(SIGNAL('frameChanged(int)'),
                adaptCallable(self._updateFrame))
        self.__direction = self.__lastClick = None
        self.__draggedPath = 0
        self.__animationLock = False
        palette = QPalette()
        backgroundColor = self.palette().color(QPalette.Window)
        backgroundColor.setAlpha(64)
        palette.setColor(QPalette.Base, backgroundColor)
        self.setPalette(palette)
        self.setAutoFillBackground(True)
        self.connect(parent, SIGNAL('resized'), adaptCallable(self._updatePosition))
        self._updateFrame(0)
        self._createDecorations()

    def _createDecorations(self):
        '''Create and place decoration components'''
        mainLayout = QGridLayout()
        QWidget.setLayout(self, mainLayout)
        mainLayout.setContentsMargins(0, 0, 0, 0)
        mainLayout.setSpacing(0)
        if self.__dockArea == self.LEFT:
            btn = QLabel()
            btn.setPixmap(resource.getImage('slide_button'))
            btn.setFixedWidth(self.__MINIMUM_VISIBLE_WIDTH)
            mainLayout.addWidget(btn, 1, 2)
            side = QLabel()
            side.setPixmap(resource.getImage('slide_side'))
            btn.setFixedWidth(self.__MINIMUM_VISIBLE_WIDTH)
            side.setScaledContents(True)
            mainLayout.addWidget(side, 0, 2)
            mainLayout.setRowStretch(0, 2)
            side = QLabel()
            side.setPixmap(resource.getImage('slide_side'))
            btn.setFixedWidth(self.__MINIMUM_VISIBLE_WIDTH)
            side.setScaledContents(True)
            mainLayout.addWidget(side, 2, 2)
            mainLayout.setRowStretch(2, 2)

    def _updateFrame(self, value):
        '''
        Update state of widget
        @type value: int
        '''
        width = self.width() - self.__MINIMUM_VISIBLE_WIDTH
        if self.__dockArea == self.LEFT:
            value = width * (self.__FRAMES - value) / self.__FRAMES
        if self.__dockArea == self.LEFT:
            self.move(-value, 0)

    def _getCurrentFrame(self):
        '''
        Get current frame of animation
        @rtype: int
        '''
        width = float(self.width() - self.__MINIMUM_VISIBLE_WIDTH)
        if self.__dockArea == self.LEFT:
            value = self.pos().x() / width + 1.0
        return int(value * self.__FRAMES)

    def slideOpen(self):
        '''Open slide widget'''
        self.__timeLine.setFrameRange(self._getCurrentFrame(), self.__FRAMES)
        self.__timeLine.setDirection(QTimeLine.Forward)
        self.__timeLine.start()

    def slideClose(self):
        '''Close slide widget'''
        self.__timeLine.setDirection(QTimeLine.Backward)
        self.__timeLine.setFrameRange(0, self._getCurrentFrame())
        self.__timeLine.start()

    def _updatePosition(self, size):
        '''
        Update position on the parent area
        @type size: QSize
        '''
        if self.__dockArea == self.LEFT:
            self.setFixedHeight(size.height())
        self._updateFrame(self.__timeLine.currentFrame())

    def mousePressEvent(self, event):
        '''
        Mouse press event handler
        @type event: QMouseEvent
        '''
        QWidget.mousePressEvent(self, event)
        self.__direction = None
        self.__lastClick = event.pos()
        self.__animationLock = False
        self.__draggedPath = 0

    def _startAnimation(self, direction):
        '''
        Configure animation timeline according to the chosen direction
        @type direction: QTimeLine.Direction
        '''
        self.__timeLine.setDirection(direction)
        if direction == QTimeLine.Forward:
            self.__timeLine.setFrameRange(self._getCurrentFrame(), self.__FRAMES)
        else:
            self.__timeLine.setFrameRange(0, self._getCurrentFrame())
        self.__timeLine.start()

    def mouseMoveEvent(self, event):
        '''
        Mouse move event handerl
        @type event: QMouseEvent
        '''
        QWidget.mouseMoveEvent(self, event)
        if event.buttons() == Qt.LeftButton and not self.__animationLock:
            newX = event.pos().x() - self.__lastClick.x() + self.pos().x()
            self.__draggedPath += event.posF().x() - self.__lastClick.x()
            if self.__dockArea == self.LEFT:
                if event.pos().x() - self.__lastClick.x() > 0:
                    self.__direction = QTimeLine.Forward
                elif event.pos().x() - self.__lastClick.x() < 0:
                    self.__direction = QTimeLine.Backward
                if abs(self.__draggedPath) > self.__MIN_DRAG_PATH and \
                not self.__animationLock:
                    self.__animationLock = True
                    self._startAnimation(self.__direction)
                newX = min(0, max(newX, -(self.width() - self.__MINIMUM_VISIBLE_WIDTH)))
                self.move(newX, 0)

    def setLayout(self, layout):
        '''
        Set layout
        @type layout: QLayout
        '''
        self.layout().addLayout(layout, 0, 0, 3, 1)

    def mouseReleaseEvent(self, event):
        '''
        Mouse release event handler
        @type event: QMouseEvent
        '''
        QWidget.mouseReleaseEvent(self, event)
        if self.__direction is not None and not self.__animationLock:
            self._startAnimation(self.__direction)
