# -*- coding: utf-8 -*-
# HiveMind - Distributed mind map editor for Maemo 5 platform
# Copyright (C) 2010-2011 HiveMind developers
#
# HiveMind is the legal property of its developers, whose names are
# noticed in @author or @authors annotations at the beginning of each
# module or class.
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02111-1301  USA


'''
Main window classes
'''

import getopt
import os
import re
from twisted.python import log
from PySide.QtGui import QActionGroup, QGraphicsBlurEffect, QMessageBox, QUndoStack, \
QFileDialog, QApplication, QUndoCommand
from PySide.QtCore import QObject, SIGNAL, QTimer, QLocale, QTranslator, QLibraryInfo
import hivemind.gui_factory as gui_factory
import hivemind.parser as parser
import hivemind.settings as settings
import hivemind.sysconfig as sysconfig
from hivemind.action_bag import ActionBag
from hivemind.attribute import writable, adaptCallable, sys
from hivemind.commands import AddNodeCommand, StateModificationCommand, RemoveNodeCommand, \
MoveNodeCommand
from hivemind.core import MindMap
from hivemind.gui_delegates import MindMapDelegate, NodeLocation
from hivemind.gui_widgets import readable

# Install qt4reactor as main twisted framework reactor
from twisted.internet import main
from qt4reactor.qt4reactor import QTReactor
app = QApplication(sys.argv)
global reactor
reactor = QTReactor()
main.installReactor(reactor)

from hivemind.network.network_core import NetworkController

class MainWindowController(QObject):
    '''
    Base controller class for main window widgets
    @author: Andrew Vasilev, Alexander Kulikov, Ilya Paramonov
    '''

    writable('mode')

    MODE_NORMAL = 'normal'
    '''All actions are parsed as normal'''

    MODE_TRANSFER = 'transfer'
    '''Map is in transfer mode'''

    def __init__(self):
        QObject.__init__(self)
        '''Init widgets and configure application'''
        self.__actionBag = ActionBag(self)
        self.__mode = self.MODE_NORMAL
        self._createActions()
        self._createStateModificationActions()
        self.__mindMapScene = MindMapDelegate()
        self.connect(self.__mindMapScene, SIGNAL('currentNodeChanged'),
                     adaptCallable(self._setActionsState))
        self.__mainWindow = gui_factory.createMainWindow(self.__actionBag, self.__mindMapScene)
        mindMapController = MindMapController(self.__mindMapScene, self.__actionBag)
        self.__networkController = NetworkController(mindMapController, self.__actionBag)
        self.connect(self.__mainWindow.mindMapView, SIGNAL('itemClicked'),
            adaptCallable(self.__mindMapScene.onItemClicked))
        self.connect(self.__mainWindow.mindMapView, SIGNAL('itemClicked'),
            adaptCallable(self.onItemClicked))
        self.connect(self.__mainWindow, SIGNAL('closeEvent'), self._closeEvent)
        _configureApplication(mindMapController)
        self.__mainWindow.show()

    def _setActionsState(self, currentNode):
        '''
        Set properties of the actions according to the current node selection
        @type currentNode: Node
        '''
        self.__widthActions[currentNode.edgeWidth].setChecked(True)
        self.__styleActions[currentNode.edgeStyle].setChecked(True)
        self.__actionBag.foldNodeAction.setEnabled(len(currentNode.children) > 0)
        self.__actionBag.foldNodeAction.setText(
               self.tr('Unfold node') if currentNode.folded else self.tr('Fold node'))
        self.__actionBag.foldNodeAction.setChecked(currentNode.folded is not None)
        self.__actionBag.removeNodeAction.setEnabled(
                currentNode.parent is not None and self.__mode == self.MODE_NORMAL)
        self.__actionBag.editLabelAction.setEnabled(
                currentNode.parent is not None and self.__mode == self.MODE_NORMAL)
        self.__actionBag.enterTransferModeAction.setEnabled(
                currentNode.parent is not None and self.__mode == self.MODE_NORMAL)
        self.__actionBag.addSiblingNodeBelowAction.setEnabled(
                currentNode.parent is not None and self.__mode == self.MODE_NORMAL)
        self.__actionBag.addSiblingNodeAboveAction.setEnabled(
                currentNode.parent is not None and self.__mode == self.MODE_NORMAL)
        self.__actionBag.editEdgeAction.setEnabled(self.__mode == self.MODE_NORMAL)
        self.__actionBag.addNodeAction.setEnabled(self.__mode == self.MODE_NORMAL)
        self.__actionBag.editNodeIconsAction.setEnabled(self.__mode == self.MODE_NORMAL)
        self.__actionBag.editNodeAction.setEnabled(self.__mode == self.MODE_NORMAL)
        self.__mindMapScene.selectRootNodeAction.setEnabled(self.__mode == self.MODE_NORMAL)
        self.__actionBag.cancelTransferModeAction.setEnabled(self.__mode == self.MODE_TRANSFER)
        self.__actionBag.putNodeAction.setEnabled(self.__mode == self.MODE_TRANSFER)
        self.__actionBag.putNodeBelowAction.setEnabled(self.__mode == self.MODE_TRANSFER)
        self.__actionBag.putNodeAboveAction.setEnabled(self.__mode == self.MODE_TRANSFER)

    def _connectStateModificationAction(self, action, alteredAttributes, checkable = True):
        '''
        Connect action to a handler which alters given attributes
        @type action: QAction
        @param alteredAttributes: attributes to be altered by the handler
        @type alteredAttributes: dict
        @type checkable: bool
        '''
        def handler():
            self.createStateModificationCommand(
                    self.__mindMapScene.currentNode, alteredAttributes)
        action.connect(SIGNAL('triggered()'), handler)
        action.setCheckable(checkable)

    def _createStateModificationActions(self):
        '''Create state modification actions'''
        self._connectStateModificationAction(self.__actionBag.parentWidthAction,
            {'edgeWidth': None})
        self._connectStateModificationAction(self.__actionBag.thinWidthAction,
            {'edgeWidth': 11})
        self._connectStateModificationAction(self.__actionBag.oneWidthAction,
            {'edgeWidth': 1})
        self._connectStateModificationAction(self.__actionBag.twoWidthAction,
            {'edgeWidth': 2})
        self._connectStateModificationAction(self.__actionBag.fourWidthAction,
            {'edgeWidth': 4})
        self._connectStateModificationAction(self.__actionBag.eightWidthAction,
            {'edgeWidth': 8})
        self._connectStateModificationAction(self.__actionBag.parentStyleAction,
            {'edgeStyle': None})
        self._connectStateModificationAction(self.__actionBag.linearStyleAction,
            {'edgeStyle': 'linear'})
        self._connectStateModificationAction(self.__actionBag.bezierStyleAction,
            {'edgeStyle': 'bezier'})
        self._connectStateModificationAction(self.__actionBag.sharpLinearStyleAction,
            {'edgeStyle': 'sharp_linear'})
        self._connectStateModificationAction(self.__actionBag.sharpBezierStyleAction,
            {'edgeStyle': 'sharp_bezier'})
        self.__widthActions = {None : self.__actionBag.parentWidthAction,
                               11 : self.__actionBag.thinWidthAction,
                               1 : self.__actionBag.oneWidthAction,
                               2 : self.__actionBag.twoWidthAction,
                               4 : self.__actionBag.fourWidthAction,
                               8 : self.__actionBag.eightWidthAction}
        self.__styleActions = {'linear' : self.__actionBag.linearStyleAction,
                               'bezier' : self.__actionBag.bezierStyleAction,
                               'sharp_linear' : self.__actionBag.sharpLinearStyleAction,
                               'sharp_bezier' : self.__actionBag.sharpBezierStyleAction,
                               None : self.__actionBag.parentStyleAction}
        group = QActionGroup(self)
        for action in self.__widthActions.values():
            group.addAction(action)
        group = QActionGroup(self)
        for action in self.__styleActions.values():
            group.addAction(action)

    def _createActions(self):
        '''Create actions'''
        self.__actionBag.editEdgeAction.connect(SIGNAL('triggered()'), self.editEdge)
        self.__actionBag.editLabelAction.connect(SIGNAL('triggered()'), self.editLabel)
        self.__actionBag.editNodeAction.connect(SIGNAL('triggered()'), self.editNode)
        self.__actionBag.editNodeIconsAction.connect(SIGNAL('triggered()'), self.editNodeIcons)
        self.__actionBag.foldNodeAction.connect(SIGNAL('triggered()'), self.foldNode)
        self.__actionBag.addNodeAction.connect(SIGNAL('triggered()'), self.addNode)
        self.__actionBag.addSiblingNodeBelowAction.connect(SIGNAL('triggered()'),
            self._addSiblingNodeBelowAction)
        self.__actionBag.addSiblingNodeAboveAction.connect(SIGNAL('triggered()'),
            self._addSiblingNodeAboveAction)
        self.__actionBag.removeNodeAction.connect(SIGNAL('triggered()'), self.removeNode)
        self.__actionBag.enterTransferModeAction.connect(SIGNAL('triggered()'),
                self._startNodeMoving)
        self.__actionBag.cancelTransferModeAction.connect(SIGNAL('triggered()'),
                self._cancelNodeMoving)
        self.__actionBag.putNodeAction.connect(SIGNAL('triggered()'), self._putNode)
        self.__actionBag.putNodeBelowAction.connect(SIGNAL('triggered()'),
                self._putNodeBelow)
        self.__actionBag.putNodeAboveAction.connect(SIGNAL('triggered()'),
                self._putNodeAbove)
        self.__actionBag.undoAction.connect(SIGNAL('triggered()'), self._doUndo)
        self.__actionBag.redoAction.connect(SIGNAL('triggered()'), self._doRedo)
        self.__actionBag.aboutAction.connect(SIGNAL('triggered()'), self.onAbout)
        self.__actionBag.exitAction.connect(SIGNAL('triggered()'), self._closeMainWindow)
        self.__actionBag.settingsAction.connect(SIGNAL('triggered()'), self.onSettings)

    def _doUndo(self):
        '''
        Record undo action
        '''
        self._recordCommand('undo')

    def _doRedo(self):
        '''
        Record redo action
        '''
        self._recordCommand('redo')

    def createStateModificationCommand(self, selectedNode, alteredAttributes):
        '''
        Create modification command using modified attributes on selected node

        @param selectedNode: the object of modification
        @type selectedNode: Node

        @param alteredAttributes: modified attributes
        @type alteredAttributes: Hash
        '''
        initialAttributeValues = selectedNode.getAttributes(getAll = True)
        unmodifiedAttributes = {}
        for key in alteredAttributes.keys():
            unmodifiedAttributes[key] = initialAttributeValues[key]
        self._recordCommand(StateModificationCommand(selectedNode, unmodifiedAttributes,
                                                     alteredAttributes))

    def _recordStateModificationCommand(self, node, dialog):
        '''
        Store node modifications made by dialog in the undo stack

        @param node: the object of modification
        @type node: Node

        @param dialog: dialog that performs change, must emit accepted() signal
                       after node modification
        @type dialog: QDialog
        '''
        initialAttributeValues = node.getAttributes(getAll = True)
        def _accept():
            alteredAttributes = dialog.attributes()
            # need to fix it, must use initialAttributeValues
            unmodifiedAttributes = {}
            for key in alteredAttributes.keys():
                if initialAttributeValues.has_key(key):
                    unmodifiedAttributes[key] = initialAttributeValues[key]
            self._recordCommand(StateModificationCommand(node, unmodifiedAttributes,
                    alteredAttributes))
        dialog.connect(SIGNAL('accepted()'), _accept)
        dialog.exec_()

    def _recordCommand(self, command):
        '''
        Process created command
        
        @param command: undo stack command to 
        '''
        self.__networkController.publishCommand(command)

    def onItemClicked(self, rightButton, item, parent, childLocation):
        if self.__mode == MainWindowController.MODE_NORMAL:
            if item is not None:
                self.__mainWindow.showMenu(rightButton, item)
        else:
            if parent.node.folded is None:
                self._finishNodeMoving(parent, childLocation)
            else:
                self.__mindMapScene.currentNode = item.node
                self.foldNode()

    def _addSiblingNodeBelowAction(self):
        '''Create and edit a sibling node, add it under selected node'''
        if self.__mindMapScene.currentNode.parent is None:
            return
        def _accept():
            '''Process accept action'''
            newNode.setAttributes(left = self.__mindMapScene.currentNode.left)
            newNode.setAttributes(**editor.attributes())
            parent = self.__mindMapScene.currentNode.parent
            index = parent.children.index(self.__mindMapScene.currentNode)
            self._recordCommand(AddNodeCommand(parent, newNode, index + 1))
        newNode = self.__mindMapScene.currentNode.parent.createChildPrototype()
        editor = gui_factory.createEditNodeDialog(newNode)
        editor.connect(SIGNAL('accepted()'), _accept)
        editor.show()

    def _addSiblingNodeAboveAction(self):
        '''Create and edit a sibling node, add it over selected node'''
        if self.__mindMapScene.currentNode.parent is None:
            return
        def _accept():
            '''Process accept action'''
            newNode.setAttributes(left = self.__mindMapScene.currentNode.left)
            newNode.setAttributes(**editor.attributes())
            parent = self.__mindMapScene.currentNode.parent
            index = parent.children.index(self.__mindMapScene.currentNode)
            self._recordCommand(AddNodeCommand(parent, newNode, index))
        newNode = self.__mindMapScene.currentNode.parent.createChildPrototype()
        editor = gui_factory.createEditNodeDialog(newNode)
        editor.connect(SIGNAL('accepted()'), _accept)
        editor.show()

    def _startNodeMoving(self):
        '''
        Switch controller and view to transfer mode
        '''
        self.__mode = self.MODE_TRANSFER
        self.__inactiveEffect = QGraphicsBlurEffect()
        self.__inactiveEffect.setBlurRadius(3)
        self.__movingNode = self.__mindMapScene.currentNode
        self.__mindMapScene.getDelegate(self.__movingNode) \
                .setGraphicsEffect(self.__inactiveEffect)
        self._setActionsState(self.__mindMapScene.currentNode)

    def _cancelNodeMoving(self):
        '''Cancel transfer mode. Switch controller and view to normal mode'''
        self.__mode = self.MODE_NORMAL
        self.__mindMapScene.getDelegate(self.__movingNode).setGraphicsEffect(None)
        self._setActionsState(self.__mindMapScene.currentNode)

    def _putNode(self):
        '''Move relocable node from old position to new parent'''
        if self.__mindMapScene.currentNode.folded:
            self.foldNode()
        else:
            if self.__mindMapScene.currentNode.parent is None:
                dialog = gui_factory.createSideSelectionDialog()
                dialog.exec_()
                if dialog.leftSide is None:
                    self._cancelNodeMoving()
                    return
                location = NodeLocation(not dialog.leftSide)
            else:
                location = NodeLocation()
            self._finishNodeMoving(self.__mindMapScene.getDelegate(
                    self.__mindMapScene.currentNode), location)

    def _putNodeBelow(self):
        '''Move relocable node as sibling below relative to current node'''
        parent = self.__mindMapScene.currentNode.parent
        if parent is not None:
            location = NodeLocation(not self.__mindMapScene.currentNode.left, None,
                    parent.getChildPosition(self.__mindMapScene.currentNode) + 1)
            self._finishNodeMoving(self.__mindMapScene.getDelegate(parent),
                    location)

    def _putNodeAbove(self):
        '''Move relocable node as sibling above relative to current node'''
        parent = self.__mindMapScene.currentNode.parent
        if parent is not None:
            location = NodeLocation(not self.__mindMapScene.currentNode.left, None,
                    parent.getChildPosition(self.__mindMapScene.currentNode))
            self._finishNodeMoving(self.__mindMapScene.getDelegate(parent),
                    location)

    def _finishNodeMoving(self, parent, location):
        '''
        Switch controller to normal mode
        @type parent: NodeDelegate
        @type location: NodeLocation
        '''
        self.__mode = self.MODE_NORMAL
        self.__mindMapScene.getDelegate(self.__movingNode).setGraphicsEffect(None)
        if parent is None: return
        if self.__movingNode == parent.node: return
        self.moveNode(self.__movingNode, parent.node, location)
        self._setActionsState(self.__mindMapScene.currentNode)

    def onAbout(self):
        '''
        Process about action
        '''
        gui_factory.createAboutDialog().exec_()

    def onSettings(self):
        '''
        Process settings action
        '''
        gui_factory.createSettingsDialog().exec_()

    def editEdge(self):
        '''Edit node's edge decorations and style'''
        dialog = gui_factory.createEditEdgeDialog(self.__mindMapScene.currentNode)
        self._recordStateModificationCommand(self.__mindMapScene.currentNode, dialog)

    def editNode(self):
        '''Edit selected node'''
        dialog = gui_factory.createEditNodeDialog(self.__mindMapScene.currentNode)
        self._recordStateModificationCommand(self.__mindMapScene.currentNode, dialog)

    def editNodeIcons(self):
        '''Edit set of icons for selected node'''
        dialog = gui_factory.createIconDialog(self.__mindMapScene.currentNode)
        self._recordStateModificationCommand(self.__mindMapScene.currentNode, dialog)

    def foldNode(self):
        '''Fold node subtree'''
        unmodifiedAttributes = {}
        alteredAttributes = {}
        unmodifiedAttributes['folded'] = self.__mindMapScene.currentNode.folded
        if self.__mindMapScene.currentNode.folded is None:
            alteredAttributes['folded'] = True
        else:
            alteredAttributes['folded'] = None
        self._recordCommand(StateModificationCommand(self.__mindMapScene.currentNode,
                unmodifiedAttributes, alteredAttributes))

    def addNode(self):
        '''
        Create and edit a child node, add it to selected node
        '''
        def _accept():
            '''Process accept action'''
            if self.__mindMapScene.currentNode.parent is None:
                dialog = gui_factory.createSideSelectionDialog()
                dialog.exec_()
                if dialog.leftSide is None:
                    # User doesn't accept dialog
                    return
                childNode.setAttributes(left = dialog.leftSide)
            if self.__mindMapScene.currentNode.folded:
                self.foldNode()
            childNode.setAttributes(**editor.attributes())
            self._recordCommand(AddNodeCommand(self.__mindMapScene.currentNode, childNode))
        childNode = self.__mindMapScene.currentNode.createChildPrototype()
        editor = gui_factory.createEditNodeDialog(childNode)
        editor.connect(SIGNAL('accepted()'), _accept)
        editor.show()

    def removeNode(self):
        '''
        Remove selected node
        '''
        def _recordCommand():
            self._recordCommand(RemoveNodeCommand(self.__mindMapScene.currentNode))
        if settings.get('confirmDelete'):
            ret = QMessageBox.question(self.__mainWindow, self.tr('Confirm delete'),
                  self.tr('Do you really want to delete the node?'),
                  QMessageBox.Yes, QMessageBox.No)
            if ret == QMessageBox.Yes:
                _recordCommand()
        else:
            _recordCommand()

    def moveNode(self, roamingNode, parentNode, location):
        '''
        Move node from current position to the new ancestor
        @param roamingNode: node to move
        @type roamingNode: Node
        @param parentNode: new parent for selected node
        @type parentNode: Node
        @param location: location relative to parent
        @type location: NodeLocation
        '''
        newPosition = location.position
        if newPosition is None:
            if location.top:
                newPosition = 0
            else:
                newPosition = len(parentNode.children)
        newSide = None
        if parentNode.parent is None:
            newSide = not location.right
        self._recordCommand(MoveNodeCommand(roamingNode, parentNode, newPosition, newSide))

    def editLabel(self):
        '''Edit edge label'''
        dialog = gui_factory.createEditLabelDialog(self.__mindMapScene.currentNode)
        self._recordStateModificationCommand(self.__mindMapScene.currentNode, dialog)

    def _closeMainWindow(self):
        '''Close main window of the application'''
        self.__mainWindow.close()

    def _closeEvent(self):
        '''
        Shutdown application
        '''
        self.__networkController.shutdownNetwork()


class MindMapController(QObject):
    '''
    Mind map controller class for save and load mind maps
    @author: Andrew Vasilev, Alexander Kulikov
    '''

    readable('initialMindMap')

    def _getUndoStack(self):
        '''Get undo stack'''
        return self.__undoStack

    def _setUndoStack(self, undoStack):
        '''
        Load new undo stack
        @param undoStack: UndoStack to be loaded
        @type undoStack: QUndoStack
        '''
        self.__undoStack = undoStack
        self.__actionBag.undoAction.setEnabled(undoStack.canRedo())
        self.__actionBag.redoAction.setEnabled(undoStack.canUndo())
        self.__undoStack.connect(SIGNAL('canUndoChanged(bool)'),
            adaptCallable(self.__actionBag.undoAction.setEnabled))
        self.__undoStack.connect(SIGNAL('canRedoChanged(bool)'),
            adaptCallable(self.__actionBag.redoAction.setEnabled))
        self.__undoStack.connect(SIGNAL('cleanChanged(bool)'),
            adaptCallable(self._undoStackStateChanged))

    undoStack = property(_getUndoStack, _setUndoStack, None, 'Undo stack')

    def __init__(self, mindMapScene, actionBag):
        '''
        @type mindMapScene: MindMapDelegate
        @type actionBag: ActionBag
        '''
        QObject.__init__(self)
        self.__mindMapScene = mindMapScene
        self.__actionBag = actionBag
        self.undoStack = QUndoStack(self)
        self.__fileName = None
        self.__autosaveTimer = QTimer(self)
        self.__autosaveTimer.setSingleShot(False)
        self.__autosaveTimer.connect(SIGNAL('timeout()'), self._backupMindMap)
        settings.connect(self._updateAutosave)
        self._updateAutosave()
        self.__actionBag.fileNewAction.connect(SIGNAL('triggered()'), self.onFileNew)
        self.__actionBag.fileOpenAction.connect(SIGNAL('triggered()'), self.onFileOpen)
        self.__actionBag.fileSaveAction.connect(SIGNAL('triggered()'), self.onFileSave)
        self.__actionBag.fileSaveAsAction.connect(SIGNAL('triggered()'), self.onFileSaveAs)

    def _backupMindMap(self):
        '''Save backup of current map'''
        if self.__undoStack.canUndo():
            if self.__fileName:
                path = self._getBackupFilename(self.__fileName)
                parser.storeMindMap(path, self.__mindMapScene.mindMap)

    def _removeBackup(self):
        '''Remove backup of current map'''
        backup = self._getBackupFilename(self.__fileName)
        if backup is not None:
            if os.path.isfile(backup):
                os.remove(backup)

    def _updateAutosave(self):
        '''Update autosave timer'''
        if settings.get('autosaveEnabled') == True:
            if settings.get('autosaveInterval') * 1000 != self.__autosaveTimer.interval():
                self.__autosaveTimer.start(settings.get('autosaveInterval') * 1000)
        else:
            self.__autosaveTimer.stop()

    def onFileNew(self):
        '''Process file new action'''
        if not self.prohibitChangeMapAction():
            return
        self.loadFile(None)

    def onFileOpen(self):
        '''Process file open action'''
        if not self.prohibitChangeMapAction():
            return
        fileName = QFileDialog.getOpenFileName(gui_factory.defaultParent(), self.tr('Open'),
                self._fileDirectory(), self.tr('Mind maps (*.mm);;All Files (*)'))[0]
        if not fileName: return
        self.loadFile(fileName)

    def onFileSave(self):
        '''
        Process file save action
        @return: True if mind map was saved, False otherwise
        @rtype: bool
        '''
        if self.__fileName is None:
            return self.onFileSaveAs()
        else:
            try:
                parser.storeMindMap(self.__fileName, self.__mindMapScene.mindMap)
                self.__undoStack.setClean()
                return True
            except IOError, args:
                QMessageBox.critical(gui_factory.defaultParent(),
                        self.tr('File save error'), unicode(args[1]))
                return False

    def onFileSaveAs(self):
        '''
        Process file save as action
        @return: True if mind map was saved False otherwise
        @rtype: bool
        '''
        fileName = QFileDialog.getSaveFileName(gui_factory.defaultParent(),
                self.tr('Save as'), self._fileDirectory(),
                self.tr('Mind maps (*.mm);;All Files (*)'))[0]
        if not fileName: return False
        if not re.match('.*\.mm', fileName): fileName += '.mm'
        try:
            parser.storeMindMap(fileName, self.__mindMapScene.mindMap)
            self._setFileName(fileName)
            self.__undoStack.setClean()
            return True
        except IOError, args:
            QMessageBox.critical(gui_factory.defaultParent(), self.tr('File save error'),
                    unicode(args[1]))
            return False

    def executeCommand(self, command):
        '''
        Execute command, created outside main controller
        @param command: command to execute
        '''
        if isinstance(command, QUndoCommand):
            self.__undoStack.push(command)
        elif command == 'undo':
            self.__undoStack.undo()
        elif command == 'redo':
            self.__undoStack.redo()

    def _undoStackStateChanged(self, clean):
        '''
        Update internal's according to undo stack state
        @param clean: undo stack clean state indication
        @type clean: bool
        '''
        self.__actionBag.fileSaveAction.setDisabled(clean)

    def _showSaveChangesBox(self):
        '''
        Ask user to make a decision about unsaved changes

        @return: result of message box invocation
        @rtype: int
        '''
        msgBox = QMessageBox(QMessageBox.Question, self.tr('Save changes?'),
                 self.tr('MindMap has been modified'),
                 QMessageBox.Save | QMessageBox.Discard | QMessageBox.Cancel,
                 gui_factory.defaultParent())
        msgBox.setInformativeText(self.tr('Do you want to save changes?'))
        msgBox.setDefaultButton(QMessageBox.Save)
        return msgBox.exec_()

    def prohibitChangeMapAction(self):
        '''
        Ask user permition to forget mind map changes
        @return: False if user canceled action, True otherwise
        @rtype: bool
        '''
        if self.__undoStack.isClean():
            self._removeBackup()
            return True
        decition = self._showSaveChangesBox()
        if decition == QMessageBox.Save:
            return self.onFileSave()
        elif decition == QMessageBox.Cancel:
            return False
        self._removeBackup()
        return True

    def _setFileName(self, fileName):
        '''Set __filename field and window header'''
        self.__fileName = fileName
        if fileName is not None:
            gui_factory.defaultParent().setWindowTitle(os.path.split(fileName)[1])
        else:
            gui_factory.defaultParent().setWindowTitle(None)

    def _fileDirectory(self):
        '''
        @return: Directory of the oppenned file or home
        @rtype: str
        '''
        if not self.__fileName: return os.path.expanduser('~')
        return os.path.dirname(self.__fileName)

    def _getBackupFilename(self, path):
        '''
        Generate backup file name for custom file name
        @type path: unicode
        '''
        if path is not None:
            dirname, filename = os.path.split(path)
            return os.path.join(dirname, '.#' + filename + '#')
        return None

    def _getMindMap(self, path):
        '''
        Get mindmap from file
        If path is empty brand new mind map is created
        @param path: path to mind map file
        @type path: unicode
        @rtype: MindMap
        '''
        if path is not None:
            return parser.loadMindMap(path)
        else:
            return MindMap()

    def loadFile(self, path):
        '''
        Load mind map file and show chosen map.
        @param path: path to mind map file
        @type path: unicode
        '''
        filePath = path
        backup = self._getBackupFilename(path)
        if self._showRecoverDataDialog(backup):
            filePath = backup
        try:
            mindMap = self._getMindMap(filePath)
            self.__initialMindMap = mindMap
            self.__undoStack.setClean()
            self.loadMindMap(self._getMindMap(filePath), path)
        except Exception:
            QMessageBox.critical(gui_factory.defaultParent(), self.tr('File open error'),
                    self.tr('Cannot read mind map from file %s') % os.path.split(path)[1])

    def _showRecoverDataDialog(self, filename):
        '''
        Show dialog to the user asking him/her whether there is a need to recover
        autosaved data copy or not
        
        @param filename: name of the backup file
        @type filename: str
        
        @return: True if user want to recover data, False otherwise
        @rtype: bool
        '''
        if filename is None: return False
        if os.path.isfile(filename):
            if QMessageBox.question(gui_factory.defaultParent(),
                    self.tr('Autosaved data recovery'),
                    self.tr('File %s has automatically saved data. Recover it?') %
                            os.path.split(filename)[1][:-1],
                    QMessageBox.Yes | QMessageBox.No) == QMessageBox.Yes:
                return True
        return False

    def loadMindMap(self, mindMap, path = None):
        '''
        Load externally created mind map

        @param mindMap: mind map object
        @type mindMap: MindMap

        @param path: path to the file to store
        @type path: sting
        '''
        self._setFileName(path)
        # need to store a copy in __initalMindMap
        self.__mindMapScene.setMap(mindMap)
        self.__undoStack.clear()


def _printHelp():
    '''Print general information about the application'''
    print '''HiveMind - Cross-platform collaborative mind map editor
Copyright (C) 2010-2011 HiveMind developers

Usage: hivemind [filename]

Options:
    -h  Display this help message and exit
'''

def _configureApplication(controller):
    '''
    Parse command line arguments and configure application
    
    @param controller: the mind map controller controller
    @type controller: MindMapController
    '''
    options, arguments = getopt.getopt(sys.argv[1:], 'hl')
    # parsing options
    for option in options:
        if option[0] == '-h':
            _printHelp()
            sys.exit()
        elif option[0] == '-l':
            log.startLogging(sys.stdout)
    # parsing arguments
    controller.loadFile(arguments[0] if arguments else None)

def _getLocale():
    '''Get locale for the application'''
    locale = settings.get('locale')
    if locale == '':
        locale = QLocale.system().name()
        settings.set(locale = locale)
    return locale

def run():
    '''Run HiveMind application'''
    app = QApplication.instance()
    # translator installation
    locale = _getLocale()[0:2]
    translatorStd = QTranslator()
    if translatorStd.load("%s/qt_%s.qm" %
            (QLibraryInfo.location(QLibraryInfo.TranslationsPath), locale)):
        app.installTranslator(translatorStd)
    translator = QTranslator()
    if translator.load(sysconfig.TRANSLATION_PATH_TEMPLATE % locale):
        app.installTranslator(translator)
    # launch of reactor and controller
    global reactor
    reactor.runReturn()
    # stopping twisted event shuts down application
    reactor.addSystemEventTrigger('after', 'shutdown', app.quit)
    # exit from twisted when last window is closed
    app.connect(app, SIGNAL('lastWindowClosed()'), reactor.stop)
    MainWindowController()
    sys.exit(app.exec_())
