# -*- coding: utf-8 -*-
# HiveMind - Distributed mind map editor for Maemo 5 platform
# Copyright (C) 2010-2011 HiveMind developers
#
# HiveMind is the legal property of its developers, whose names are
# noticed in @author or @authors annotations at the beginning of each
# module or class.
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02111-1301  USA


'''
Abstract dialog classes
'''

import hivemind.settings as settings
import hivemind.resource as resource
from PySide.QtGui import QDialog, QMessageBox, QDialogButtonBox, QLabel, QLineEdit, \
QVBoxLayout, QFormLayout, QAbstractItemView, QTableView, QHeaderView, QHBoxLayout, \
QRadioButton, QButtonGroup, QComboBox, QPushButton, QToolBar, QFontComboBox, QAction, \
QFontDatabase, QColorDialog, QTextCharFormat, QFont, QPalette, QBrush, QTextCursor, \
QGridLayout, QCheckBox, QGraphicsView
from PySide.QtCore import SIGNAL, QRegExp, Qt, QSize, SLOT
from hivemind.attribute import readable, adaptCallable
from hivemind.gui_widgets import SearchingTrMixin, SpecialTextEdit, \
ScalingWidget, IconList, IconTable, LineEditBinder, CheckBoxBinder, \
SettingsCreator
from hivemind.network.network_models import AffiliationDelegate
from twisted.words.protocols.jabber import jid

'''Regexp for jid validation'''
VALID_JID = '[A-Za-z0-9._%+-]+@(?:[A-Za-z0-9-]+\\.)+[A-Za-z]{2,6}'

class CommonDialog(SearchingTrMixin, QDialog):
    '''
    Abstract dialog containing OK and Cancel buttons

    @author: Alexander Kulikov
    '''
    def __init__(self, parent = None):
        '''
        Create OK and Cancel buttons and connect them to appropriate handlers
        '''
        QDialog.__init__(self, parent)
        SearchingTrMixin.__init__(self)
        self._buttonBox = QDialogButtonBox()
        self._okButton = self._buttonBox.addButton(QDialogButtonBox.Ok)
        self._cancelButton = self._buttonBox.addButton(QDialogButtonBox.Cancel)
        self._buttonBox.connect(SIGNAL('accepted()'), self._accept)
        self._buttonBox.connect(SIGNAL('rejected()'), self.close)

    def _accept(self):
        '''
        Some dialogs doesn't need to have _accept method
        '''

class EditNodeDialog(CommonDialog):
    '''
    Main dialog for node editing

    @author: Alexander Kulikov
    '''

    def __init__(self, node = None, parent = None):
        '''
        Initialize dialog with specific node
        @type node: Node
        '''
        CommonDialog.__init__(self, parent)
        self.setWindowTitle(self.tr('Node editor'))
        self.__node = node
        self.__background = self.__font = self.__color = None
        self._isHtml = self.__node.isHtml()
        self.__newAttributes = None
        self._isHtmlSourceMode = False
        self._createComponents()
        self._initEditor()

    def _createComponents(self):
        '''Create components and actions'''
        #pylint: disable = W0201
        # init edit widget
        self._editor = SpecialTextEdit()
        self.connect(self._editor, SIGNAL('accept'), self._accept)
        # init html widget
        self._htmlEditor = SpecialTextEdit()
        # create font toolbar
        self._fontToolbar = QToolBar()
        self.__comboFont = QFontComboBox()
        self.__comboSize = QComboBox()
        self.__comboSize.setEditable(True)
        self.__comboFont.connect(SIGNAL('currentFontChanged(QFont)'), self._buildFont)
        self.__comboSize.connect(SIGNAL('currentIndexChanged(int)'), self._buildFont)
        self._fontToolbar.addWidget(self.__comboFont)
        self._fontToolbar.addWidget(self.__comboSize)
        # text format actions
        self._boldAction = QAction(self)
        self._boldAction.setCheckable(True)
        self._italicAction = QAction(self)
        self._italicAction.setCheckable(True)
        self._underlineAction = QAction(self)
        self._underlineAction.setCheckable(True)
        # font action
        self._textFontAction = QAction(self)
        # HTML action
        self._textHtmlAction = QAction(self)
        self._textHtmlAction.setCheckable(True)
        # color actions
        self._colorAction = QAction(self)
        self._backgroundAction = QAction(self)
        # adhoc formatting propagation checkbox
        self._propagateFormatting = QCheckBox(self.tr('Propagate formatting to child nodes'))
        # clear format action
        self._clearFormatAction = QAction(self)
        # actions signals connection
        self._boldAction.connect(SIGNAL('triggered(bool)'), self._setBold)
        self._italicAction.connect(SIGNAL('triggered(bool)'), self._setItalic)
        self._underlineAction.connect(SIGNAL('triggered(bool)'), self._setUnderline)
        self._colorAction.connect(SIGNAL('triggered()'), self._textColor)
        self._backgroundAction.connect(SIGNAL('triggered()'), self._textBackground)
        self._textHtmlAction.connect(SIGNAL('triggered()'), self._textHtml)
        self._clearFormatAction.connect(SIGNAL('triggered()'), self._clearFormat)
        self._textFontAction.connect(SIGNAL('triggered()'), self._textFont)
        self._editor.connect(SIGNAL('currentCharFormatChanged(QTextCharFormat)'),
                             self._currentCharFormatChanged)

    def _currentCharFormatChanged(self, textFormat):
        '''
        Update tool bar if the current character format has changed
        @type textFormat: QTextCharFormat
        '''
        self._fontChanged(textFormat.font())

    def _fontChanged(self, font):
        '''
        Update buttons if the current character format has changed
        @type font: QFont
        '''
        self._boldAction.setChecked(font.bold())
        self._italicAction.setChecked(font.italic())
        self._underlineAction.setChecked(font.underline())
        self.__comboFont.blockSignals(True)
        self.__comboSize.blockSignals(True)
        self.__comboFont.setCurrentFont(font)
        self.__comboSize.clear()
        self.__comboSize.addItems([str(value) for value in QFontDatabase().\
                                   pointSizes(self.__comboFont.currentFont().family())])
        index = self.__comboSize.findText(str(font.pointSize()), Qt.MatchExactly)
        if index == -1:
            self.__comboSize.setEditText(str(font.pointSize()))
        else: self.__comboSize.setCurrentIndex(index)
        self.__comboFont.blockSignals(False)
        self.__comboSize.blockSignals(False)

    def _accept(self):
        '''
        Hide the dialog, update attributes of the Node
        and set result code to Accepted
        '''
        storeAttrs = {}
        if self._isHtml:
            if self._isHtmlSourceMode:
                storeAttrs['html'] = self._htmlEditor.toPlainText()
            else:
                storeAttrs['html'] = self._editor.toHtml()
        else:
            storeAttrs['font'] = self.__font
            storeAttrs['text'] = self._editor.toPlainText()
        storeAttrs['color'] = self.__color
        storeAttrs['backgroundColor'] = self.__background
        storeAttrs['propagateNodeFormatting'] = self._propagateFormatting.isChecked()
        self.__newAttributes = storeAttrs
        QDialog.accept(self)

    def attributes(self):
        '''
        Get dict of modified attributes with new values
        @return: new attribute values
        @rtype: dict
        '''
        return self.__newAttributes

    def _initEditor(self):
        '''Initialize text editor'''
        self._textHtmlAction.setChecked(self._isHtml)
        self._propagateFormatting.setChecked(self.__node.propagateNodeFormatting)
        if self.__node.isHtml():
            self._editor.setHtml(self.__node.text)
            self.__color = self.__font = None
        else:
            self._editor.setPlainText(self.__node.text)
            self._setFont(self.__node.font)
            self._setColor(self.__node.color)
        self._editor.document().setDefaultFont(settings.get('defaultFont'))
        self._fontChanged(self._editor.textCursor().charFormat().font())
        self._setBackgroundColor(self.__node.backgroundColor)
        self._editor.selectAll()

    def _setHtmlSourceMode(self, isHtmlSourceMode):
        '''
        Set html source mode
        @type isHtmlSourceMode: bool
        '''
        self._isHtmlSourceMode = isHtmlSourceMode

    def _textColor(self):
        '''Choose font color'''
        color = QColorDialog.getColor(settings.get('defaultNodeTextColor') \
                                      if self.__color is None else self.__color,
                                      self)
        if color.isValid():
            self._setColor(color)

    def _textBackground(self):
        '''Choose background text color'''
        color = QColorDialog.getColor(settings.get('defaultNodeBackgroundColor') \
                                      if self.__background is None else self.__background,
                                      self)
        if color.isValid():
            self._setBackgroundColor(color)

    def _textFont(self):
        '''Edit current font'''
        fontDialog = FontDialog(settings.get('defaultFont') \
                                if self.__font is None else self.__font, self)
        if fontDialog.exec_() == QDialog.Accepted:
            self._setFont(fontDialog.font)

    def _textHtml(self):
        '''Switch between html editor and simple editor'''
        self._isHtml = not self._isHtml
        if not self._isHtml:
            self._editor.setPlainText(self._editor.toPlainText())
            if self.__font is not None:
                self.__font = self._editor.currentFont()

    def _setBold(self, checked):
        '''
        Update text editor if Bold button has triggered
        '''
        if self.__font is None:
            self.__font = self._editor.currentFont()
        self.__font.setBold(checked)
        fmt = QTextCharFormat()
        fmt.setFontWeight(QFont.Bold if checked else QFont.Normal)
        self._mergeFormat(fmt)

    def _setItalic(self, checked):
        '''
        Update text editor if Italic button has triggered
        '''
        if self.__font is None:
            self.__font = self._editor.currentFont()
        self.__font.setItalic(checked)
        fmt = QTextCharFormat()
        fmt.setFontItalic(checked)
        self._mergeFormat(fmt)

    def _setUnderline(self, checked):
        '''
        Update text editor if Underline button has triggered
        '''
        if self.__font is None:
            self.__font = self._editor.currentFont()
        self.__font.setUnderline(checked)
        fmt = QTextCharFormat()
        fmt.setFontUnderline(checked)
        self._mergeFormat(fmt)

    def _setBackgroundColor(self, color):
        '''
        Set node's background color
        @type color: QColor
        '''
        palette = QPalette(self._editor.palette())
        self.__background = color
        palette.setColor(QPalette.Base, settings.get('defaultNodeBackgroundColor') \
                         if color is None else color)
        self._editor.setPalette(palette)

    def _setColor(self, color):
        '''
        Set node text color
        @type color: QColor
        '''
        self.__color = color
        fmt = QTextCharFormat()
        fmt.setForeground(QBrush(settings.get('defaultNodeTextColor') \
                                 if color is None else color))
        self._mergeFormat(fmt)

    def _setFont(self, font):
        '''
        Set node text font
        @type font: QFont
        '''
        self.__font = font
        fmt = QTextCharFormat()
        fmt.setFont(settings.get('defaultFont') if font is None else font)
        self._mergeFormat(fmt)

    def _buildFont(self, value):
        '''Build font'''
        font = self.__comboFont.currentFont()
        font.setPointSize(int(self.__comboSize.currentText()))
        font.setBold(self._boldAction.isChecked())
        font.setItalic(self._italicAction.isChecked())
        font.setUnderline(self._underlineAction.isChecked())
        self._setFont(font)

    def _clearFormat(self):
        '''Clear font, color and background of the node'''
        self._setFont(None)
        self._setColor(None)
        self._setBackgroundColor(None)

    def _mergeFormat(self, textFormat):
        '''
        Merge current format in text editor and new format
        @type format: QTextCharFormat
        '''
        cursor = self._editor.textCursor()
        if not self._isHtml:
            cursor.select(QTextCursor.Document)
        elif not cursor.hasSelection():
            cursor.select(QTextCursor.WordUnderCursor)
        cursor.mergeCharFormat(textFormat)
        self._editor.mergeCurrentCharFormat(textFormat)


class FontDialog(CommonDialog):
    '''
    Font selection dialog

    @author: Alexander Kulikov
    '''
    readable('font')
    def __init__(self, font, parent = None):
        '''
        Initialize widget with specified font
        @type font: QFont
        @type parent: QWidget
        '''
        CommonDialog.__init__(self, parent)
        self.setWindowFlags(Qt.Window)
        self.setAttribute(Qt.WA_Maemo5StackedWindow) #pylint: disable=E1101
        self.__font = font
        self._initComponents(font)

    def _initComponents(self, font):
        '''Init components'''
        #pylint: disable = W0201
        self.__comboFont = QFontComboBox()
        self.__comboFont.setCurrentFont(font)
        self.__comboSize = QComboBox()
        self.__comboSize.addItems([str(value) for value in \
                QFontDatabase().pointSizes(self.__comboFont.currentFont().family())])
        self.__comboSize.setEditable(True)
        index = self.__comboSize.findText(str(font.pointSize()), Qt.MatchExactly)
        if index == -1:
            self.__comboSize.setEditText(str(font.pointSize()))
        else: self.__comboSize.setCurrentIndex(index)
        self.__sampleText = QLineEdit(self.tr('ABCDE abcde'))
        self.__sampleText.setFrame(False)
        self.__sampleText.setFont(QFont(self.__comboFont.currentFont().family(),
                int(self.__comboSize.currentText())))
        self.__sampleText.setAlignment(Qt.AlignHCenter | Qt.AlignVCenter)
        layout = QGridLayout()
        layout.addWidget(self.__comboFont, 0, 0, 1, 3)
        layout.addWidget(self.__comboSize, 0, 3)
        scalingWidget = ScalingWidget(self.__sampleText, 2.0)
        layout.addWidget(scalingWidget, 1, 0, 1, 4)
        layout.addWidget(self._buttonBox, 2, 0, 1, 4)
        self.setLayout(layout)
        self.__comboFont.connect(SIGNAL('currentFontChanged(QFont)'), self._setLabelFont)
        self.__comboSize.connect(SIGNAL('currentIndexChanged(int)'), self._setLabelSize)
        self._setLabelFormat()

    def _accept(self):
        '''
        Hide the dialog and set result code to Accepted
        '''
        QDialog.accept(self)

    def _setLabelFormat(self):
        '''Update sample text font'''
        self.__font = self.__comboFont.currentFont()
        self.__font.setPointSize(int(self.__comboSize.currentText()))
        self.__sampleText.setFont(self.__font)

    def _setLabelFont(self, font):
        '''
        Update sample text font
        @type font: QFont
        '''
        self._setLabelFormat()

    def _setLabelSize(self, index):
        '''
        Update sample text font
        @type index : int
        '''
        self._setLabelFormat()


class EditLabelDialog(CommonDialog):
    '''
    Dialog for edge label editing

    @author: Oleg Kandaurov
    '''
    def __init__(self, node = None, parent = None):
        '''
        Initialize dialog with specific node

        @type node: Node
        '''
        CommonDialog.__init__(self, parent)
        self.setWindowTitle(self.tr('Edit Label'))
        self.__node = node
        self.__color = None
        self.__font = None
        self.__newAttributes = {}
        self._createComponents()
        self._initEditor()

    def _createComponents(self):
        '''Create components and actions'''
        #pylint: disable = W0201
        self._editor = SpecialTextEdit()
        self._boldAction = QAction(self)
        self._boldAction.setCheckable(True)
        self._italicAction = QAction(self)
        self._italicAction.setCheckable(True)
        self._underlineAction = QAction(self)
        self._underlineAction.setCheckable(True)
        self._comboFont = QFontComboBox()
        self._comboSize = QComboBox()
        self.connect(self._editor, SIGNAL('accept'), self._accept)
        self._comboFont.connect(SIGNAL('currentFontChanged(QFont)'), self._setFont)
        self._comboSize.connect(SIGNAL('currentIndexChanged(int)'), self._setSize)
        self._colorParent = QCheckBox(self.tr("Parent's color"))
        self._colorButton = QPushButton(self.tr('Choose color'))
        self._fontParent = QCheckBox(self.tr("Parent's font"))
        self._propagateFormatting = QCheckBox(self.tr('Propagate formatting to child nodes'))
        self._boldAction.connect(SIGNAL('triggered(bool)'), self._setBold)
        self._italicAction.connect(SIGNAL('triggered(bool)'), self._setItalic)
        self._underlineAction.connect(SIGNAL('triggered(bool)'), self._setUnderline)
        self._colorButton.connect(SIGNAL('clicked()'), self._textColor)
        self._colorParent.connect(SIGNAL('toggled(bool)'), self._colorButton,
                                  SLOT('setDisabled(bool)'))
        self._colorParent.connect(SIGNAL('toggled(bool)'), self._updateEditor)
        self._fontParent.connect(SIGNAL('toggled(bool)'), self._comboFont,
                                 SLOT('setDisabled(bool)'))
        self._fontParent.connect(SIGNAL('toggled(bool)'), self._comboSize,
                                 SLOT('setDisabled(bool)'))
        self._fontParent.connect(SIGNAL('toggled(bool)'), self._boldAction,
                                 SLOT('setDisabled(bool)'))
        self._fontParent.connect(SIGNAL('toggled(bool)'), self._italicAction,
                                 SLOT('setDisabled(bool)'))
        self._fontParent.connect(SIGNAL('toggled(bool)'), self._underlineAction,
                                 SLOT('setDisabled(bool)'))
        self._fontParent.connect(SIGNAL('toggled(bool)'), self._updateEditor)

    def _initEditor(self):
        '''Initialize editor'''
        self._editor.setPlainText(self.__node.labelText)
        self._editor.document().setDefaultFont(settings.get('defaultLabelFont'))
        self.__font = self.__node.labelFont
        self._setEditorFont(self.__node.labelFont)
        self.__color = self.__node.labelColor
        self._setEditorColor(self.__node.labelColor)
        if self.__node.labelColor is None:
            self._colorParent.setChecked(True)
        if self.__node.labelFont is None:
            self._fontParent.setChecked(True)
        self._comboSize.setEditable(True)
        font = self.__node.effectiveLabelFont
        self._comboFont.setCurrentFont(font)
        self._propagateFormatting.setChecked(self.__node.propagateLabelFormatting)
        self._editor.selectAll()

    def _fontSizes(self, font):
        '''
        Add to comboSize sizes supported by current font

        @type font: QFont
        '''
        self._comboSize.clear()
        self._comboSize.addItems([str(value) for value in QFontDatabase().\
                                   pointSizes(font.family())])

    def _setSizeIndex(self, font):
        '''
        Set font size in comboSize

        @type font: QFont
        '''
        index = self._comboSize.findText(str(font.pointSize()), Qt.MatchExactly)
        if index == -1:
            self._comboSize.setEditText(str(font.pointSize()))
        else:
            self._comboSize.setCurrentIndex(index)

    def _updateButtonsState(self, font):
        '''
        Update state of buttons according to font

        @type font: QFont
        '''
        self._boldAction.setChecked(font.bold())
        self._italicAction.setChecked(font.italic())
        self._underlineAction.setChecked(font.underline())

    def _setFont(self, font):
        '''
        Set and update font

        @type font: QFont
        '''
        if font is None:
            font = self.__node.effectiveLabelFont
        if self._comboSize.currentText() != '':
            font.setPointSize(int(self._comboSize.currentText()))
        self._fontSizes(font)
        self._setSizeIndex(font)
        self._setEditorFont(font)
        self._updateButtonsState(font)
        # don't save current formatting if parent button checked
        if not self._fontParent.isChecked():
            self.__font = font

    def _setSize(self, index):
        '''
        Set and update size of font

        @type index: int
        '''
        font = self._comboFont.currentFont()
        if index != -1:
            font.setPointSize(int(self._comboSize.itemText(index)))
        self._setEditorFont(font)
        self._updateButtonsState(font)
        # don't save current formatting if parent button checked
        if not self._fontParent.isChecked():
            self.__font = font

    def _setBold(self, checked):
        '''
        Update text editor if Bold button has triggered
        '''
        fmt = QTextCharFormat()
        fmt.setFontWeight(QFont.Bold if checked else QFont.Normal)
        self._mergeFormat(fmt)
        # don't save current formatting if parent button checked
        if not self._fontParent.isChecked():
            self.__font.setBold(checked)

    def _setItalic(self, checked):
        '''
        Update text editor if Italic button has triggered
        '''
        fmt = QTextCharFormat()
        fmt.setFontItalic(checked)
        self._mergeFormat(fmt)
        # don't save current formatting if parent button checked
        if not self._fontParent.isChecked():
            self.__font.setItalic(checked)

    def _setUnderline(self, checked):
        '''
        Update text editor if Underline button has triggered
        '''
        fmt = QTextCharFormat()
        fmt.setFontUnderline(checked)
        self._mergeFormat(fmt)
        # don't save current formatting if parent button checked
        if not self._fontParent.isChecked():
            self.__font.setUnderline(checked)

    def _textColor(self):
        '''Choose font color'''
        color = QColorDialog.getColor(self.__node.effectiveLabelColor, self)
        if color.isValid():
            self.__color = color
            self._setEditorColor(color)

    def _updateEditor(self):
        '''
        Update text editor according to "parent" buttons
        '''
        if self._colorParent.isChecked():
            self._setEditorColor(None)
        else:
            self._setEditorColor(self.__color)
        if self._fontParent.isChecked():
            font = self.__node.effectiveLabelFont
        else:
            font = self.__font
            if font is None:
                font = self.__node.effectiveLabelFont
        self._fontSizes(font)
        self._setSizeIndex(font)
        self._comboFont.setCurrentFont(font)

    def _setEditorColor(self, color):
        '''
        Update text color of text editor

        @type color: QColor
        '''
        fmt = QTextCharFormat()
        newColor = color if color else self.__node.effectiveLabelColor
        fmt.setForeground(QBrush(newColor))
        self._mergeFormat(fmt)

    def _setEditorFont(self, font):
        '''
        Update text font of text editor

        @type font: QFont
        '''
        fmt = QTextCharFormat()
        newFont = font if font else self.__node.effectiveLabelFont
        fmt.setFont(newFont)
        self._mergeFormat(fmt)

    def _mergeFormat(self, textFormat):
        '''
        Merge current format in text editor and new format

        @type textFormat: QTextCharFormat
        '''
        cursor = self._editor.textCursor()
        cursor.select(QTextCursor.Document)
        cursor.mergeCharFormat(textFormat)
        self._editor.mergeCurrentCharFormat(textFormat)

    def _accept(self):
        '''
        Hide the dialog, update attributes of the label
        and set result code to Accepted
        '''
        modified = {}
        if self._colorParent.isChecked():
            self.__color = None
        else:
            if self.__color is None:
                self.__color = self.__node.effectiveLabelColor
        if self._fontParent.isChecked():
            self.__font = None
        else:
            if self.__font is None:
                self.__font = self.__node.effectiveLabelFont
        modified['labelColor'] = self.__color
        modified['labelFont'] = self.__font
        modified['labelText'] = self._editor.toPlainText().replace('\n', ' ')
        modified['propagateLabelFormatting'] = self._propagateFormatting.isChecked()
        self.__newAttributes = modified
        QDialog.accept(self)

    def attributes(self):
        '''
        Get dict of modified attributes with new values

        @return: new attribute values
        @rtype: dict
        '''
        return self.__newAttributes


class IconDialog(CommonDialog):
    '''
    Abstract class for icon selection dialog

    @author: Alexander Kulikov
    '''

    def __init__(self, node, parent = None):
        '''
        Initialize with specific node
        @type node: Node
        @type parent: QWidget
        '''
        CommonDialog.__init__(self, parent)
        self.setWindowTitle(self.tr('Icon selector'))
        self.__node = node
        self.__newAttributes = {}
        self._createComponents()

    def _createComponents(self):
        '''Create components'''
        #pylint: disable = W0201
        if len(self.__node.icons) > IconList.MAX_ICON_COUNT:
            self._listScene = IconList(self.__node.icons[:10], self)
        else: self._listScene = IconList(self.__node.icons, self)
        self._iconTable = IconTable(self)
        self._iconTable.setIconSize(QSize(64, 64))
        self._iconList = QGraphicsView(self._listScene, self)
        self._iconList.setHorizontalScrollBarPolicy(Qt.ScrollBarAlwaysOff)
        self._iconList.setVerticalScrollBarPolicy(Qt.ScrollBarAlwaysOff)
        self._iconTable.connect(SIGNAL('iconClicked'), \
                                 adaptCallable(self._listScene.addIcon))

    def _accept(self):
        '''
        Hide the dialog, update attributes of the Node
        and set result code to Accepted
        '''
        icons = self._iconList.scene().getIcons()
        if len(self.__node.icons) > IconList.MAX_ICON_COUNT:
            icons.extend(self.__node.icons[10:])
        self.__newAttributes = {'icons' : icons}
        QDialog.accept(self)

    def attributes(self):
        '''
        Get dict of modified attributes with new values
        @return: new attribute values
        @rtype: dict
        '''
        return self.__newAttributes


class AboutDialog(CommonDialog):
    '''
    Abstract class for about dialog

    @author: Oleg Kandaurov, Alexander Kulikov
    '''
    def __init__(self, parent = None):
        '''
        Constructor
        @type parent: QWidget
        '''
        CommonDialog.__init__(self, parent)
        self.setWindowTitle(self.tr('About HiveMind'))
        self._createComponents()

    def _createComponents(self):
        '''Create components'''
        #pylint: disable = W0201
        self._logo = QLabel()
        self._logoImage = resource.getImage('logo')
        self._info = QLabel(self.tr("Collaborative mind map editor"))


class EditEdgeDialog(CommonDialog):

    '''
    Abstract class for edit edge dialog
    @author: Oleg Kandaurov
    '''
    def __init__(self, node = None, parent = None):
        '''
        Initialize dialog with specific node
        @type node: Node
        '''
        CommonDialog.__init__(self, parent)
        self.setWindowTitle(self.tr('Edge editor'))
        self.__node = node
        self.__newAttributes = {}
        self.__color = None
        self.__widthToButton = None
        self.__styleToButton = None
        self._createComponents()
        self._initEditor()

    def _createComponents(self):
        '''Create components'''
        #pylint: disable = W0201
        # Setting up width view
        self._widthLabel = QLabel(self.tr('Edge width'))
        self._edgeWidthComboBox = QComboBox()
        self._edgeWidthComboBox.addItem(self.tr('Parent'), None)
        self._edgeWidthComboBox.addItem(self.tr('Thin'), 11)
        self._edgeWidthComboBox.addItem('1', 1)
        self._edgeWidthComboBox.addItem('2', 2)
        self._edgeWidthComboBox.addItem('4', 4)
        self._edgeWidthComboBox.addItem('8', 8)
        # Setting up style view
        self._styleLabel = QLabel(self.tr('Edge style'))
        self._edgeStyleComboBox = QComboBox()
        self._edgeStyleComboBox.addItem(self.tr('Parent'), None)
        self._edgeStyleComboBox.addItem(self.tr('Linear'), 'linear')
        self._edgeStyleComboBox.addItem(self.tr('Bezier'), 'bezier')
        self._edgeStyleComboBox.addItem(self.tr('Sharp linear'), 'sharp_linear')
        self._edgeStyleComboBox.addItem(self.tr('Sharp bezier'), 'sharp_bezier')
        # Setting up node style view
        self._nodeStyleLabel = QLabel(self.tr('Node style'))
        self._nodeStyleComboBox = QComboBox()
        self._nodeStyleComboBox.addItem(self.tr('Parent'), None)
        self._nodeStyleComboBox.addItem(self.tr('Bubble'), 'bubble')
        self._nodeStyleComboBox.addItem(self.tr('Fork'), 'fork')
        # Setting up color view
        self._colorLabel = QLabel(self.tr('Color'))
        self._colorParent = QCheckBox(self.tr('Parent'))
        self._colorButton = QPushButton(self.tr('Choose color'))
        self._colorButton.connect(SIGNAL('clicked()'), self._edgeColor)
        self._colorParent.connect(SIGNAL('toggled(bool)'), self._colorButton,
                                    SLOT('setDisabled(bool)'))

    def _initEditor(self):
        '''Initialize editor'''
        # Initialize color of the Edge
        self.__color = self.__node.edgeColor
        if self.__color is None:
            self._colorParent.setChecked(True)
        self._edgeWidthComboBox.setCurrentIndex(self._edgeWidthComboBox.findData(
                self.__node.edgeWidth))
        self._edgeStyleComboBox.setCurrentIndex(self._edgeStyleComboBox.findData(
                self.__node.edgeStyle))
        self._nodeStyleComboBox.setCurrentIndex(self._nodeStyleComboBox.findData(
                self.__node.style))

    def _edgeColor(self):
        '''Choose color of the Edge'''
        color = QColorDialog.getColor(self.__node.effectiveEdgeColor, self)
        if color.isValid():
            self.__color = color

    def _accept(self):
        '''
        Hide the dialog, update attributes of the Edge
        and set result code to Accepted
        '''
        modified = {}
        if self._colorParent.isChecked():
            self.__color = None
        elif self.__color is None:
            self.__color = self.__node.effectiveEdgeColor
        modified['edgeColor'] = self.__color
        modified['edgeWidth'] = self._edgeWidthComboBox.itemData(
                self._edgeWidthComboBox.currentIndex())
        modified['edgeStyle'] = self._edgeStyleComboBox.itemData(
                self._edgeStyleComboBox.currentIndex())
        modified['style'] = self._nodeStyleComboBox.itemData(
                self._nodeStyleComboBox.currentIndex())
        self.__newAttributes = modified
        QDialog.accept(self)

    def attributes(self):
        '''
        Get dict of modified attributes with new values
        @return: new attribute values
        @rtype: dict
        '''
        return self.__newAttributes

class SettingsDialog(CommonDialog):
    '''
    Abstract class for settings dialog
    @author: Oleg Kandaurov
    '''
    def __init__(self, parent = None):
        '''
        @type parent: QWidget
        '''
        CommonDialog.__init__(self, parent)
        self.setWindowTitle(self.tr('Settings'))
        self._newSettings = list()

    def _accept(self):
        '''
        Apply new settings
        '''
        oldLocale = settings.get('locale')
        newSettings = {}
        for group in self._newSettings:
            for setting in group:
                if setting: newSettings.update(setting)
        settings.set(**newSettings)
        if oldLocale != settings.get('locale'):
            QMessageBox.information(self, self.tr('Settings changed'),
                    self.tr('The language settings will be applied on restart \
                    of the application!'))
        QDialog.accept(self)


class SideSelectionDialog(CommonDialog):
    '''
    Side selection dialog
    @author: Oleg Kandaurov
    '''

    readable('leftSide')

    def __init__(self, parent = None):
        '''
        @type parent: QWidget
        '''
        CommonDialog.__init__(self, parent)
        self.setWindowTitle(self.tr('Node Position'))
        self.__leftSide = None
        self._createComponents()
        self._initComponents()

    def _createComponents(self):
        '''
        Create components
        '''
        #pylint: disable = W0201
        self.__label = QLabel(self.tr('Select node position'))
        self.__left = QPushButton(self.tr('Left'))
        self.__right = QPushButton(self.tr('Right'))
        self.connect(self.__left, SIGNAL('clicked()'), self._accept)
        self.connect(self.__right, SIGNAL('clicked()'), self._accept)

    def _initComponents(self):
        '''
        Init components
        '''
        buttonLayout = QHBoxLayout()
        buttonLayout.addWidget(self.__left)
        buttonLayout.addWidget(self.__right)
        mainLayout = QVBoxLayout()
        self.__label.setAlignment(Qt.AlignCenter)
        mainLayout.addWidget(self.__label)
        mainLayout.addLayout(buttonLayout)
        self.setLayout(mainLayout)

    def _accept(self):
        '''
        Hide the dialog and sets side
        '''
        self.__leftSide = True if self.sender() == self.__left else False
        QDialog.accept(self)


class AffiliationDialog(CommonDialog):
    '''
    Abstract class for affiliation dialog
    @author: Oleg Kandaurov
    '''

    def __init__(self, model, parent = None):
        '''Constructor'''
        CommonDialog.__init__(self, parent)
        self.__model = model
        self.setWindowTitle(self.tr('Edit affiliations'))
        self._createComponents()
        self._initComponents()

    def _createComponents(self):
        '''Create components'''
        #pylint: disable = W0201
        self.__delegate = AffiliationDelegate()
        self.__view = QTableView()
        self.__view.setModel(self.__model)
        self.__view.setItemDelegateForColumn(1, self.__delegate)
        self.__view.setEditTriggers(QAbstractItemView.AllEditTriggers)
        horizontalHeader = self.__view.horizontalHeader()
        horizontalHeader.setResizeMode(QHeaderView.Stretch)
        horizontalHeader.setStretchLastSection(True)
        self.__jidLabel = QLabel(self.tr('JID'))
        self.__jidEdit = QLineEdit()
        self.__affiliationLabel = QLabel(self.tr('Affiliation'))
        self.__affiliationCombo = QComboBox()
        for data, name in self.__model.mapAffiliations.iteritems():
            self.__affiliationCombo.addItem(name, data)
        self.__additionButton = QPushButton(self.tr('Add participant'))
        self.__additionButton.clicked.connect(self._addParticipant)

    def _initComponents(self):
        '''Initialize components'''
        jidLayout = QVBoxLayout()
        self.__jidLabel.setAlignment(Qt.AlignCenter)
        jidLayout.addWidget(self.__jidLabel)
        jidLayout.addWidget(self.__jidEdit)
        affilationLayout = QVBoxLayout()
        self.__affiliationLabel.setAlignment(Qt.AlignCenter)
        affilationLayout.addWidget(self.__affiliationLabel)
        affilationLayout.addWidget(self.__affiliationCombo)
        buttonLayout = QVBoxLayout()
        buttonLayout.addWidget(QLabel(''))
        buttonLayout.addWidget(self.__additionButton)
        additionLayout = QHBoxLayout()
        additionLayout.setAlignment(Qt.AlignBottom)
        additionLayout.addLayout(jidLayout)
        additionLayout.addLayout(affilationLayout)
        additionLayout.addLayout(buttonLayout)
        mainLayout = QVBoxLayout()
        mainLayout.addWidget(self.__view)
        mainLayout.addLayout(additionLayout)
        mainLayout.addWidget(self._buttonBox)
        self.setLayout(mainLayout)

    def _addParticipant(self):
        '''Retrieve info from components and create participant with given access'''
        jidRegexp = QRegExp(VALID_JID)
        if not jidRegexp.exactMatch(self.__jidEdit.text()):
            QMessageBox.critical(self, self.tr('Incorrect JID!'),
                    self.tr('Please, enter correct jabber id.'))
            return
        self.__model.setAffiliation(jid.internJID(self.__jidEdit.text()),
                self.__affiliationCombo.itemData(self.__affiliationCombo.currentIndex()))
        self.__jidEdit.clear()

    def _accept(self):
        '''
        Hide the dialog
        '''
        QDialog.accept(self)


class NetworkAuthorizationDialog(QMessageBox):
    '''
    Authorization dialog
    @author: Oleg Kandaurov
    '''

    readable('mapButton')
    def __init__(self, requestor, parent = None):
        '''
        @type parent: QWidget
        @type requestor: str
        '''
        QMessageBox.__init__(self, parent)
        self.setIcon(QMessageBox.Question)
        self.setWindowTitle(self.tr('Authorization request'))
        self.setText(self.tr('Authorization request from %s') % requestor)
        self.setInformativeText(self.tr('Do you want to accept authorization request?'))
        acceptButton = self.addButton(self.tr('Accept'), QMessageBox.AcceptRole)
        denyButton = self.addButton(self.tr('Deny'), QMessageBox.RejectRole)
        blockButton = self.addButton(self.tr('Block'), QMessageBox.ActionRole)
        self.setDefaultButton(acceptButton)
        self.setEscapeButton(denyButton)
        self.__mapButton = {acceptButton: 'accept', denyButton: 'deny', blockButton: 'block'}


class NetworkServiceDialog(CommonDialog):
    '''
    Abstract class for service dialog
    @author: Oleg Kandaurov
    '''
    readable('attributes')
    def __init__(self, parent = None):
        CommonDialog.__init__(self, parent)
        self.setWindowTitle(self.tr('Create service'))
        self.__attributes = dict()
        self._createComponents()
        self._initComponents()

    def _createComponents(self):
        '''Create components'''
        #pylint: disable = W0201
        self.__openModelRadio = QRadioButton(self.tr('Open'))
        self.__authModelRadio = QRadioButton(self.tr('Authorize'))
        self.__rosterModelRadio = QRadioButton(self.tr('Roster'))
        self.__whitelistModelRadio = QRadioButton(self.tr('Whitelist'))
        self.__accessModelGroup = QButtonGroup()
        self.__accessModelGroup.addButton(self.__openModelRadio)
        self.__accessModelGroup.addButton(self.__rosterModelRadio)
        self.__accessModelGroup.addButton(self.__authModelRadio)
        self.__accessModelGroup.addButton(self.__whitelistModelRadio)
        self.__mapAccessModels = {'open': self.__openModelRadio,
                'roster': self.__rosterModelRadio, 'authorize': self.__authModelRadio,
                'whitelist': self.__whitelistModelRadio}
        self.__mapAccessModels[settings.get('accessModel')].setChecked(True)
        self.__readOnlyBinder = CheckBoxBinder('readOnly', settings.get('readOnly'))
        self.__readOnlyBinder.setText(self.tr('Read only'))

    def _initComponents(self):
        '''Create components'''
        mainLayout = QVBoxLayout()
        firstGroupLayout = QHBoxLayout()
        firstGroupLayout.addWidget(self.__openModelRadio)
        firstGroupLayout.addWidget(self.__rosterModelRadio)
        secondGroupLayout = QHBoxLayout()
        secondGroupLayout.addWidget(self.__authModelRadio)
        secondGroupLayout.addWidget(self.__whitelistModelRadio)
        mainLayout.addLayout(firstGroupLayout)
        mainLayout.addLayout(secondGroupLayout)
        mainLayout.addWidget(self.__readOnlyBinder)
        mainLayout.addWidget(self._buttonBox)
        self.setLayout(mainLayout)

    def _accept(self):
        '''Hide the dialog, update attributes'''
        self.__attributes = self.__readOnlyBinder.retrieve()
        for model, button in self.__mapAccessModels.iteritems():
            if button.isChecked():
                self.__attributes['accessModel'] = model
        settings.set(**self.__attributes)
        QDialog.accept(self)


class NetworkClientDialog(CommonDialog):
    '''
    Abstract class for client dialog
    @author: Oleg Kandaurov
    '''

    readable('attributes')
    def __init__(self, rosterModel, parent = None):
        CommonDialog.__init__(self, parent)
        self.setWindowTitle(self.tr('Create client'))
        self.__model = rosterModel
        self.__attributes = dict()
        self._createComponents()
        self._initComponents()

    def _createComponents(self):
        '''Create components'''
        #pylint: disable = W0201
        self.__view = QTableView()
        horizontalHeader = self.__view.horizontalHeader()
        horizontalHeader.setResizeMode(QHeaderView.Stretch)
        horizontalHeader.setStretchLastSection(True)
        self.__view.setModel(self.__model)
        self.__serviceBinder = LineEditBinder('serviceJid', settings.get('serviceJid'),
                QRegExp(VALID_JID))
        self.__serviceLabel = QLabel(self.tr('Service JID:'))
        self.__view.clicked.connect(self._itemClicked)

    def _initComponents(self):
        '''Create components'''
        mainLayout = QVBoxLayout()
        firstLayout = QHBoxLayout()
        firstLayout.addWidget(self.__serviceLabel)
        firstLayout.addWidget(self.__serviceBinder)
        mainLayout.addLayout(firstLayout)
        mainLayout.addWidget(self.__view)
        mainLayout.addWidget(self._buttonBox)
        self.setLayout(mainLayout)

    def _itemClicked(self, modelIndex):
        '''Process clicked signal from view'''
        self.__serviceBinder.setText(modelIndex.data(Qt.EditRole))

    def _accept(self):
        '''Hide the dialog, update attributes'''
        if not self.__serviceBinder.isValid():
            QMessageBox.critical(self, self.tr('Incorrect JID!'),
                    self.tr('Please, enter correct jabber id.'))
            return
        self.__attributes = self.__serviceBinder.retrieve()
        settings.set(**self.__attributes)
        QDialog.accept(self)


class NetworkConnectionDialog(CommonDialog):
    '''
    Abstract class for connection dialog
    @author: Oleg Kandaurov
    '''

    readable('attributes')
    def __init__(self, parent = None):
        CommonDialog.__init__(self, parent)
        self.setWindowTitle(self.tr('Connect to XMPP'))
        self.__attributes = dict()
        self._createComponents()
        self._initComponents()

    def _createComponents(self):
        '''Create components'''
        #pylint: disable = W0201
        self.__userBinder = LineEditBinder('userJid', settings.get('userJid'),
                QRegExp(VALID_JID))
        self.__userLabel = QLabel(self.tr('User JID:'))
        self.__passwordEdit = QLineEdit()
        self.__passwordEdit.setEchoMode(QLineEdit.Password)
        password = settings.get('password')
        if password is not None:
            self.__passwordEdit.setText(password)
        self.__passwordLabel = QLabel(self.tr('Password:'))

    def _initComponents(self):
        '''Create components'''
        mainLayout = QVBoxLayout()
        formLayout = QFormLayout()
        formLayout.addRow(self.__userLabel, self.__userBinder)
        formLayout.addRow(self.__passwordLabel, self.__passwordEdit)
        mainLayout.addLayout(formLayout)
        mainLayout.addWidget(self._buttonBox)
        self.setLayout(mainLayout)

    def _accept(self):
        '''Hide the dialog, update attributes'''
        if not self.__userBinder.isValid():
            QMessageBox.critical(self, self.tr('Incorrect JID!'),
                    self.tr('Please, enter correct jabber id.'))
            return
        self.__attributes = self.__userBinder.retrieve()
        settings.set(**self.__attributes)
        self.__attributes['password'] = self.__passwordEdit.text()
        QDialog.accept(self)


class NetworkErrorDialog(QMessageBox):
    '''
    Dialog that shows network errors
    @author: Oleg Kandaurov
    '''

    def __init__(self, parent = None):
        '''
        Constructor
        @type parent: QWidget
        '''
        QMessageBox.__init__(self, parent)
        self.setIcon(QMessageBox.Warning)
        self.setWindowTitle(self.tr('Network error'))
        self.setStandardButtons(QMessageBox.Ok);
        self.setDefaultButton(QMessageBox.Ok);

    def showError(self, info, reason, details):
        '''
        Shows specific error information
        @param info: Short information about error type
        @type info: str
        @param reason: Possible reason of error
        @type reason: str
        @param details: Technical details
        @type details: str
        '''
        self.setText(info)
        self.setInformativeText(reason)
        self.setDetailedText(details)
        self.exec_()

