# -*- coding: utf-8 -*-
# HiveMind - Distributed mind map editor for Maemo 5 platform
# Copyright (C) 2010-2011 HiveMind developers
#
# HiveMind is the legal property of its developers, whose names are
# noticed in @author or @authors annotations at the beginning of each
# module or class.
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02111-1301  USA

'''
Main window class for Desktop platform
'''

from PySide.QtCore import SIGNAL, Qt
from PySide.QtGui import QMenu, QCursor, QMainWindow
from hivemind.main_window import MainView, AbstractMainWindow


class DesktopMainView(MainView):
    '''
    Mouse input filter object for the desktop
    @author: Andrew Vasilev
    '''

    __MIN_DRAG_PATH = 20
    '''Minimum drag path length for the desktop'''

    def __init__(self, scene, actionBag):
        '''
        @type scene: MindMapDelegate
        @type actionBag: ActionBag
        '''
        MainView.__init__(self, scene, actionBag)

    def mouseReleaseEvent(self, event):
        '''
        Mouse release event handler.
        Emits itemClicked, mapClicked signals.

        @type event: QMouseEvent
        '''
        MainView.mouseReleaseEvent(self, event)
        if self.draggedPath < DesktopMainView.__MIN_DRAG_PATH:
            item = self.scene().itemAt(self.mapToScene(event.pos()))
            parent, childLocation = self.scene().parentAt(self.mapToScene(event.pos()))
            if parent is not None:
                self.emit(SIGNAL('itemClicked'), event.button() == Qt.RightButton, item, parent,
                        childLocation)
            if item is None:
                self.emit(SIGNAL('mapClicked'))


class DesktopMainWindow(AbstractMainWindow):
    '''
    Main window of HiveMind program for Desktop platform

    @author: Alexander Kulikov
    '''
    def __init__(self, actionBag, mindMapScene):
        '''Create new window'''
        AbstractMainWindow.__init__(self, actionBag, mindMapScene)
        self._mindMapView = DesktopMainView(mindMapScene, actionBag)
        self.setCentralWidget(self._mindMapView)
        self._createContextMenu()
        self._createMainMenu()
        self._createToolBars()

    def setWindowTitle(self, fileName):
        if fileName:
            QMainWindow.setWindowTitle(self, fileName + ' - HiveMind')
        else:
            QMainWindow.setWindowTitle(self, 'HiveMind')

    def _createContextMenu(self):
        '''Create context menu'''
        self.__menu = QMenu(self)
        self.__menu.addAction(self._actionBag.addNodeAction)
        self.__menu.addAction(self._actionBag.addSiblingNodeAboveAction)
        self.__menu.addAction(self._actionBag.addSiblingNodeBelowAction)
        self.__menu.addSeparator()
        self.__menu.addAction(self._actionBag.foldNodeAction)
        self.__menu.addAction(self._actionBag.enterTransferModeAction)
        self.__menu.addAction(self._actionBag.removeNodeAction)
        self.__menu.addSeparator()
        self.__menu.addAction(self._actionBag.editNodeAction)
        self.__menu.addAction(self._actionBag.editEdgeAction)
        self.__menu.addAction(self._actionBag.editLabelAction)
        self.__menu.addAction(self._actionBag.editNodeIconsAction)

    def _addEllipsesToActionText(self):
        '''Add ellipses to the text of the actions displaying dialogs'''
        actionsWithEllipses = [
                self._actionBag.fileOpenAction, self._actionBag.fileSaveAsAction,
                self._actionBag.settingsAction, self._actionBag.aboutAction,
                self._actionBag.addNodeAction, self._actionBag.addSiblingNodeBelowAction,
                self._actionBag.addSiblingNodeAboveAction, self._actionBag.editNodeAction,
                self._actionBag.editNodeIconsAction, self._actionBag.editLabelAction,
                self._actionBag.editEdgeAction, self._actionBag.editAffiliationAction,
                self._actionBag.connectXMPPAction, self._actionBag.createServiceAction,
                self._actionBag.createClientAction, self._actionBag.startServiceAction,
                self._actionBag.startClientAction]
        for action in actionsWithEllipses:
            action.setText(action.text() + '...')

    def _createMainMenu(self):
        '''Create main menu'''
        self._addEllipsesToActionText()
        menu = self.menuBar().addMenu(self.tr('&File'))
        menu.addAction(self._actionBag.fileNewAction)
        menu.addAction(self._actionBag.fileOpenAction)
        menu.addAction(self._actionBag.fileSaveAction)
        menu.addAction(self._actionBag.fileSaveAsAction)
        menu.addSeparator()
        menu.addAction(self._actionBag.exitAction)
        menu = self.menuBar().addMenu(self.tr('&Edit'))
        menu.addAction(self._actionBag.undoAction)
        menu.addAction(self._actionBag.redoAction)
        menu.addSeparator()
        menu.addAction(self._actionBag.settingsAction)
        menu = self.menuBar().addMenu(self.tr('&Network'))
        advMenu = QMenu(self.tr('&Advanced'))
        advMenu.addAction(self._actionBag.connectXMPPAction)
        advMenu.addAction(self._actionBag.createServiceAction)
        advMenu.addAction(self._actionBag.createClientAction)
        advMenu.addAction(self._actionBag.stopProtocolAction)
        menu.addMenu(advMenu)
        menu.addAction(self._actionBag.startServiceAction)
        menu.addAction(self._actionBag.startClientAction)
        menu.addAction(self._actionBag.disconnectXMPPAction)
        menu.addAction(self._actionBag.editAffiliationAction)
        menu = self.menuBar().addMenu(self.tr('&Help'))
        menu.addAction(self._actionBag.aboutAction)

    def _createToolBars(self):
        '''Create tool bar'''
        toolbar = self.addToolBar(self.tr('File'))
        toolbar.addAction(self._actionBag.fileNewAction)
        toolbar.addAction(self._actionBag.fileOpenAction)
        toolbar.addAction(self._actionBag.fileSaveAction)
        toolbar.addAction(self._actionBag.fileSaveAsAction)
        toolbar = self.addToolBar(self.tr('Edit'))
        toolbar.addAction(self._actionBag.undoAction)
        toolbar.addAction(self._actionBag.redoAction)
        toolbar = self.addToolBar(self.tr('View'))
        toolbar.addAction(self._actionBag.zoomOutAction)
        toolbar.addAction(self._actionBag.zoomInAction)
        toolbar = self.addToolBar(self.tr('Connection status'))
        toolbar.addAction(self._actionBag.connectionStatusAction)
        toolbar.addAction(self._actionBag.subscribeStatusAction)

    def showMenu(self, rightButton, item):
        '''Show the node menu on right button click'''
        if rightButton:
            self.__menu.exec_(QCursor.pos())
