# -*- coding: utf-8 -*-
# HiveMind - Distributed mind map editor for Maemo 5 platform
# Copyright (C) 2010-2011 HiveMind developers
#
# HiveMind is the legal property of its developers, whose names are
# noticed in @author or @authors annotations at the beginning of each
# module or class.
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02111-1301  USA

'''
Dialogs for Desktop platform
'''

from PySide.QtGui import QToolBar, QIcon, QTabWidget, QGridLayout, \
QHBoxLayout, QPalette, QTextBrowser, QFrame, QLabel, QDialogButtonBox, \
QVBoxLayout, QGroupBox, QWidget
from PySide.QtCore import SIGNAL, QSize, Qt
from hivemind.dialogs import EditNodeDialog, EditLabelDialog, IconDialog, \
AboutDialog, EditEdgeDialog, SettingsDialog, NetworkConnectionDialog, \
AffiliationDialog, SettingsCreator, NetworkServiceDialog, NetworkClientDialog
from hivemind.attribute import adaptCallable
import hivemind.__version__ as __version__


class DesktopEditNodeDialog(EditNodeDialog):
    '''
    Edit node dialog for Desktop platform

    @author: Alexander Kulikov
    '''
    def __init__(self, node, parent = None):
        '''
        Initialize dialog with specific node
        @type node: Node
        @type parent: QWidget
        '''
        EditNodeDialog.__init__(self, node, parent)
        self._initComponents()
        self._enableSourceTab(self._isHtml)

    def _initComponents(self):
        '''Form initialization'''
        #pylint: disable = W0201
        toolbar = QToolBar(self)
        self._boldAction.setToolTip(self.tr('Bold'))
        self._boldAction.setIcon(QIcon.fromTheme('format-text-bold'))
        toolbar.addAction(self._boldAction)
        self._italicAction.setToolTip(self.tr('Italic'))
        self._italicAction.setIcon(QIcon.fromTheme('format-text-italic'))
        toolbar.addAction(self._italicAction)
        self._underlineAction.setToolTip(self.tr('Underline'))
        self._underlineAction.setIcon(QIcon.fromTheme('format-text-underline'))
        toolbar.addAction(self._underlineAction)
        self._textHtmlAction.setToolTip(self.tr('HTML'))
        self._textHtmlAction.setIcon(QIcon.fromTheme('text-html'))
        toolbar.addAction(self._textHtmlAction)
        self._colorAction.setToolTip(self.tr('Color'))
        self._colorAction.setText(self.tr('Color'))
        toolbar.addAction(self._colorAction)
        self._backgroundAction.setToolTip(self.tr('Background'))
        self._backgroundAction.setText(self.tr('Background'))
        toolbar.addAction(self._backgroundAction)
        self._clearFormatAction.setToolTip(self.tr('Clear format'))
        self._clearFormatAction.setIcon(QIcon.fromTheme('edit-clear'))
        toolbar.addAction(self._clearFormatAction)
        self.__tabWidget = QTabWidget(self)
        self.__textIndex = self.__tabWidget.addTab(self._editor, self.tr('Text'))
        self.__htmlIndex = self.__tabWidget.addTab(self._htmlEditor, 'Html')
        gridLayout = QGridLayout(self)
        gridLayout.addWidget(self._fontToolbar, 0, 0, 1, 2)
        gridLayout.addWidget(toolbar, 0, 2, 1, 2)
        gridLayout.addWidget(self.__tabWidget, 1, 0, 1, 4)
        lastLineLayout = QHBoxLayout()
        lastLineLayout.addWidget(self._propagateFormatting)
        lastLineLayout.addStretch()
        lastLineLayout.addWidget(self._buttonBox)
        gridLayout.addLayout(lastLineLayout, 2, 0, 1, 4)
        QWidget.setTabOrder(self._editor, self.__tabWidget)
        self._textHtmlAction.connect(SIGNAL('toggled(bool)'),
                adaptCallable(self._enableSourceTab))
        self.__tabWidget.connect(SIGNAL('currentChanged(int)'),
                adaptCallable(self._changedMode))

    def _changedMode(self, isSourceMode):
        '''
        Modificate dialog when html source was selected
        @type isSourceMode: bool
        '''
        self._boldAction.setDisabled(isSourceMode == self.__htmlIndex)
        self._italicAction.setDisabled(isSourceMode == self.__htmlIndex)
        self._underlineAction.setDisabled(isSourceMode == self.__htmlIndex)
        self._colorAction.setDisabled(isSourceMode == self.__htmlIndex)
        self._backgroundAction.setDisabled(isSourceMode == self.__htmlIndex)
        self._clearFormatAction.setDisabled(isSourceMode == self.__htmlIndex)
        self._isHtmlSourceMode = (isSourceMode == self.__htmlIndex)
        if isSourceMode == self.__htmlIndex:
            self._htmlEditor.setPlainText(self._editor.toHtml())
        else:
            self._editor.setHtml(self._htmlEditor.toPlainText())

    def _enableSourceTab(self, isHtml):
        '''
        Enable html source tab
        @type isHtml: bool
        '''
        self.__tabWidget.setTabEnabled(self.__htmlIndex, isHtml)

class DesktopEditLabelDialog(EditLabelDialog):
    '''
    Edit label dialog for Desktop platform

    @author: Oleg Kandaurov
    '''
    def __init__(self, node, parent = None):
        '''
        Initialize dialog with specific node
        @type node: Node
        @type parent: QWidget
        '''
        EditLabelDialog.__init__(self, node, parent)
        self._initComponents()

    def _initComponents(self):
        '''Form initialization'''
        #create toolbar
        toolbar = QToolBar(self)
        self._boldAction.setToolTip(self.tr('Bold'))
        self._boldAction.setIcon(QIcon.fromTheme('format-text-bold'))
        toolbar.addAction(self._boldAction)
        self._italicAction.setToolTip(self.tr('Italic'))
        self._italicAction.setIcon(QIcon.fromTheme('format-text-italic'))
        toolbar.addAction(self._italicAction)
        self._underlineAction.setToolTip(self.tr('Underline'))
        self._underlineAction.setIcon(QIcon.fromTheme('format-text-underline'))
        toolbar.addAction(self._underlineAction)
        secondaryLayout = QHBoxLayout()
        secondaryLayout.addWidget(self._fontParent)
        secondaryLayout.addSpacing(20)
        secondaryLayout.addWidget(self._colorParent)
        secondaryLayout.addWidget(self._colorButton)
        mainLayout = QGridLayout()
        mainLayout.addWidget(toolbar, 0, 0, 1, 2)
        mainLayout.addWidget(self._comboFont, 0, 2, 1, 5)
        mainLayout.addWidget(self._comboSize, 0, 7, 1, 3)
        mainLayout.addWidget(self._editor, 1, 0, 1, 10)
        mainLayout.addLayout(secondaryLayout, 2, 0, 1, 10)
        bottomLayout = QHBoxLayout()
        bottomLayout.addWidget(self._propagateFormatting)
        bottomLayout.addStretch()
        bottomLayout.addWidget(self._buttonBox)
        mainLayout.addLayout(bottomLayout, 3, 0, 1, 10)
        self.setLayout(mainLayout)
        QWidget.setTabOrder(self._editor, self._buttonBox)
        QWidget.setTabOrder(self._buttonBox, self._comboFont)
        QWidget.setTabOrder(self._comboFont, self._comboSize)


class DesktopIconDialog(IconDialog):
    '''
    Icon selection dialog for Desktop platform

    @author: Alexander Kulikov
    '''

    def __init__(self, node, parent = None):
        '''
        Initialize with specific node
        @type node: Node
        @type parent: QWidget
        '''
        IconDialog.__init__(self, node, parent)
        self._initComponents()

    def _initComponents(self):
        '''Form initialization'''
        self.setMinimumWidth(410)
        self._iconTable.setIconSize(QSize(16, 16))
        # setup icon list scene
        self._listScene.setItemSize(32)
        self._listScene.setIconSize(16)
        self._listScene.setHorizontalSpacing(4)
        self._iconList.setMinimumHeight(self._listScene.height() + 8)
        self._iconList.setMaximumHeight(self._listScene.height() + 8)
        groupList = QGroupBox(self.tr('Selected icons'))
        groupTable = QGroupBox(self.tr('Available icons'))
        layout = QGridLayout()
        layout.addWidget(self._iconList, 0, 0)
        groupList.setLayout(layout)
        layout = QGridLayout()
        layout.addWidget(self._iconTable, 0, 0)
        groupTable.setLayout(layout)
        layout = QGridLayout()
        layout.addWidget(groupList, 0, 0, 1, 6)
        layout.addWidget(groupTable, 1, 0, 2, 6)
        layout.addWidget(self._buttonBox, 3, 0, 1, 6)
        self.setLayout(layout)


class DesktopAboutDialog(AboutDialog):
    '''
    About dialog for Desktop platform

    @author: Oleg Kandaurov, Alexander Kulikov
    '''
    def __init__(self, parent = None):
        AboutDialog.__init__(self, parent)
        self._createComponents()
        self._initComponents()

    def _createComponents(self):
        '''Create components'''
        #pylint: disable = W0201
        AboutDialog._createComponents(self)
        self._logo.setPixmap(self._logoImage.scaled(QSize(72, 72)))
        palette = self.palette()
        palette.setColor(QPalette.Base, self.palette().color(QPalette.Window))
        self.setPalette(palette)
        self._tabWidget = QTabWidget()
        about = QTextBrowser()
        about.setFrameStyle(QFrame.NoFrame)
        about.setOpenExternalLinks(True)
        about.setHtml(self.tr('''HiveMind - Collaborative mind map editor<br>
            Copyright &copy; 2010-2011 HiveMind developers<br><br>
            License: <a href="http://www.gnu.org/licenses/gpl-3.0.html">
            GNU General Public License Version 3</a><br><br>
            HiveMind development is supported by<br>
            <a href="http://fruct.org/"> Open Innovations Framework Program FRUCT</a>
            '''))
        self._tabWidget.addTab(about, '&About')
        authors = QTextBrowser()
        authors.setFrameStyle(QFrame.NoFrame)
        authors.setOpenExternalLinks(True)
        authors.setHtml(self.tr('''
            Please use <a href="https://linuxlab.corp7.uniyar.ac.ru/projects/hivemind/issues/new">
            https://linuxlab.corp7.uniyar.ac.ru/</a> to report bugs.<br><br>
            Andrey Golovchenko<br>
            &nbsp;&nbsp;<a href="mailto:golovchenkoaa@gmail.com">golovchenkoaa@gmail.com</a><br>
            &nbsp;&nbsp;core developer<br>
            <br>
            Andrey Vasilev<br>
            &nbsp;&nbsp;<a href="mailto:vamonster@gmail.com">vamonster@gmail.com</a><br>
            &nbsp;&nbsp;developer/designer<br>
            <br>
            Alexander Kulikov<br>
            &nbsp;&nbsp;<a href="mailto:asgard1988@yandex.ru">asgard1988@yandex.ru</a><br>
            &nbsp;&nbsp;developer<br>
            <br>
            Oleg Kandaurov<br>
            &nbsp;&nbsp;<a href="mailto:kandaurov.oleg@gmail.com">kandaurov.oleg@gmail.com</a><br>
            &nbsp;&nbsp;developer<br>
            <br>
            Ilya Paramonov<br>
            &nbsp;&nbsp;<a href="mailto:ivparamonov@gmail.com">ivparamonov@gmail.com</a><br>
            &nbsp;&nbsp;technical manager<br>
            '''))
        self._tabWidget.addTab(authors, 'A&uthors')
        thanks = QTextBrowser()
        thanks.setFrameStyle(QFrame.NoFrame)
        thanks.setOpenExternalLinks(True)
        thanks.setHtml(self.tr('''
            Nicu Buculei<br>
            &nbsp;&nbsp;<a href="http://nicubunu.ro">nicubunu.ro</a><br>
            &nbsp;&nbsp;HiveMind logo<br><br>
            Mentalrey<br>
            &nbsp;&nbsp;<a href="http://www.mentalrey.it/">mentalrey.it</a><br>
            &nbsp;&nbsp;Notification icons<br>'''))
        self._tabWidget.addTab(thanks, '&Thanks To')
        self._name = QLabel('<h2>HiveMind</h2>')
        self._version = QLabel(self.tr('<h4>Version %s</h4>') % __version__.VERSION)

    def _initComponents(self):
        '''Form initialization'''
        self.setMinimumWidth(500)
        self.setMinimumHeight(300)
        close = QDialogButtonBox(QDialogButtonBox.Close)
        headerInfoLayout = QVBoxLayout()
        headerInfoLayout.addWidget(self._name)
        headerInfoLayout.addWidget(self._version)
        headerInfoLayout.addWidget(self._info)
        headerInfoLayout.setAlignment(Qt.AlignTop)
        headerLayout = QHBoxLayout()
        headerLayout.addWidget(self._logo)
        headerLayout.addLayout(headerInfoLayout)
        headerLayout.setAlignment(Qt.AlignLeft)
        mainLayout = QVBoxLayout()
        mainLayout.addLayout(headerLayout)
        mainLayout.addWidget(self._tabWidget)
        mainLayout.addWidget(close)
        self.setLayout(mainLayout)
        close.connect(SIGNAL('rejected()'), self.close)


class DesktopEditEdgeDialog(EditEdgeDialog):
    '''
    Dialog for editing decoration and style of a joint link for Desktop platform
    @author: Oleg Kandaurov
    '''
    def __init__(self, node = None, parent = None):
        '''
        Initialize dialog with specific node
        @type node: Node
        '''
        EditEdgeDialog.__init__(self, node, parent)
        self._initComponents()

    def _initComponents(self):
        '''Form initialization'''
        colorLayout = QHBoxLayout()
        colorLayout.addWidget(self._colorParent)
        colorLayout.addWidget(self._colorButton)
        mainLayout = QGridLayout()
        mainLayout.addWidget(self._widthLabel, 0, 0)
        mainLayout.addWidget(self._styleLabel, 0, 1)
        mainLayout.addWidget(self._edgeWidthComboBox, 1, 0)
        mainLayout.addWidget(self._edgeStyleComboBox, 1, 1)
        mainLayout.addWidget(self._nodeStyleLabel, 2, 0, 1, 2)
        mainLayout.addWidget(self._nodeStyleComboBox, 3, 0, 1, 2)
        mainLayout.addLayout(colorLayout, 4, 0, 1, 2)
        mainLayout.addWidget(self._buttonBox, 5, 0, 1, 2)
        self.setLayout(mainLayout)


class DesktopSettingsDialog(SettingsDialog):
    '''
    Settings dialog for Desktop platform
    @author: Oleg Kandaurov
    '''
    def __init__(self, parent = None):
        '''
        @type parent: QWidget
        '''
        SettingsDialog.__init__(self, parent)
        self._general = SettingsCreator(['locale', 'defaultZoomLevel', 'confirmDelete'])
        generalGroup = QGroupBox(self.tr('General settings'))
        generalGroup.setLayout(self._general.getLayout())
        self._node = SettingsCreator(['defaultFont', 'defaultNodeTextColor',
                'defaultNodeBackgroundColor', 'defaultSelectedNodeBackgroundColor',
                'maxNodeTextWidth'])
        nodeGroup = QGroupBox(self.tr('Node settings'))
        nodeGroup.setLayout(self._node.getLayout())
        self._edge = SettingsCreator(['defaultLabelFont', 'defaultLabelColor',
                'defaultEdgeWidth', 'defaultEdgeColor'])
        edgeGroup = QGroupBox(self.tr('Edge && label settings'))
        edgeGroup.setLayout(self._edge.getLayout())
        self._autosave = SettingsCreator(['autosaveEnabled', 'autosaveInterval'])
        autosaveGroup = QGroupBox(self.tr('Autosave settings'))
        autosaveGroup.setLayout(self._autosave.getLayout())
        mainLayout = QVBoxLayout()
        mainLayout.addWidget(generalGroup)
        mainLayout.addWidget(nodeGroup)
        mainLayout.addWidget(edgeGroup)
        mainLayout.addWidget(autosaveGroup)
        mainLayout.addWidget(self._buttonBox)
        self.setLayout(mainLayout)

    def _accept(self):
        '''
        Retrieve new settings
        '''
        self._newSettings.append(self._general.retrieve())
        self._newSettings.append(self._node.retrieve())
        self._newSettings.append(self._edge.retrieve())
        self._newSettings.append(self._autosave.retrieve())
        SettingsDialog._accept(self)


class DesktopAffiliationDialog(AffiliationDialog):
    '''
    Affiliation dialog for Desktop platform
    @author: Oleg Kandaurov
    '''
    def __init__(self, model, parent = None):
        AffiliationDialog.__init__(self, model, parent)


class DesktopNetworkServiceDialog(NetworkServiceDialog):
    '''
    Service dialog for Desktop platfrom
    @author: Oleg Kandaurov
    '''

    def __init__(self, parent = None):
        NetworkServiceDialog.__init__(self, parent)


class DesktopNetworkClientDialog(NetworkClientDialog):
    '''
    Client dialog for Desktop platfrom
    @author: Oleg Kandaurov
    '''

    def __init__(self, rosterModel, parent = None):
        NetworkClientDialog.__init__(self, rosterModel, parent)


class DesktopNetworkConnectionDialog(NetworkConnectionDialog):
    '''
    Connection dialog for Desktop platfrom
    @author: Oleg Kandaurov
    '''

    def __init__(self, parent = None):
        NetworkConnectionDialog.__init__(self, parent)
