# -*- coding: utf-8 -*-
# HiveMind - Distributed mind map editor for Maemo 5 platform
# Copyright (C) 2010-2011 HiveMind developers
#
# HiveMind is the legal property of its developers, whose names are
# noticed in  or  annotations at the beginning of each
# module or class.
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02111-1301  USA
#pylint: disable=W0603

'''
Application settings manager

@author: Ilya Paramonov
'''

from PySide.QtCore import *
from PySide.QtGui import *
import os

import hivemind.__dirs__ as __dirs__

__INT_PARAMETERS = ('defaultEdgeWidth', 'autoHideNodeMenuTimeout',
        'showNodeMenuAnimationDuration', 'hideNodeMenuAnimationDuration',
        'xmppKeepAliveInterval', 'xmppKeepAliveCount', 'autosaveInterval')
'''
Integer parameters names
'''

__DOUBLE_PARAMETERS = ()
'''
Float parameters names
'''

__FONT_PARAMETERS = ('defaultFont', 'defaultLabelFont')
'''
Font parameters names
'''

__COLOR_PARAMETERS = ('defaultNodeTextColor', 'defaultNodeBackgroundColor',
                      'defaultEdgeColor', 'defaultLabelColor',
                      'defaultSelectedNodeBackgroundColor')
'''
Font parameters names
'''

__BOOLEAN_PARAMETERS = ('confirmDelete', 'autosaveEnabled', 'readOnly', 'firstClick',
                        'useExtendedMenu')
'''
Boolean parameters names
'''

def translationDir():
    '''Return directory where application translations reside'''
    return "%s/translations" % __dirs__.BASEDIR

def get(parameter):
    '''
    Get parameter by name
    
    @return: parameter value
    @rtype: according to parameter type
    '''
    value = __settings.value(parameter)
    if value is None:
        raise Exception('Invalid parameter %s' % parameter)
    if parameter in __INT_PARAMETERS:
        return int(value)
    elif parameter in __DOUBLE_PARAMETERS:
        return value
    elif parameter in __FONT_PARAMETERS:
        font = QFont()
        font.fromString(value)
        return font
    elif parameter in __COLOR_PARAMETERS:
        return QColor(value)
    elif parameter in __BOOLEAN_PARAMETERS:
        if value is True or value is False:
            return value
        return {'true': True, 'false': False}.get(value.lower())
    else:
        return value

def set(**parameters): #pylint: disable=W0622
    '''
    Set parameters
    '''
    for paramName, paramValue in parameters.iteritems():
        if paramName in __FONT_PARAMETERS:
            variant = paramValue.toString()
        elif paramName in __COLOR_PARAMETERS:
            variant = paramValue.name()
        else:
            variant = paramValue
        __settings.setValue(paramName, variant)
        __signalizer.emit(SIGNAL('settingsChanged'))

def connect(receiver):
    '''
    Subscribe receiver to be called when settings changes
    
    @param receiver: callable object
    '''
    __signalizer.connect(SIGNAL('settingsChanged'), receiver)

def _reinitialize(path):
    '''
    Reinitialize __settings object
    
    We use '/' as a path in order to prevent using of user scope
    configuration file in unit test mode
    
    @param path: path to the user scope configuration file
    '''
    QSettings.setPath(QSettings.NativeFormat, QSettings.UserScope, path)
    global __settings
    __settings = QSettings(QSettings.NativeFormat, QSettings.UserScope,
                           'hivemind', 'hivemind')


# Initialization
QSettings.setPath(QSettings.NativeFormat, QSettings.SystemScope,
                  os.path.join(__dirs__.BASEDIR, 'config')) # default config path
__settings = None
'''Qt settings manager'''
__signalizer = QObject()
'''Object used for emitting signals when settings changed'''
if 'UNITTEST' in os.environ:
    _reinitialize('/') # in order to prevert reading/writing
    def reinitialize(path = '/'):
        '''Reinitialize __settings object for unit testing case'''
        _reinitialize(path)
else:
    _reinitialize('%s/.config' % os.path.expanduser('~'))
