# -*- coding: utf-8 -*-
# HiveMind - Distributed mind map editor for Maemo 5 platform
# Copyright (C) 2010-2011 HiveMind developers
#
# HiveMind is the legal property of its developers, whose names are
# noticed in  or  annotations at the beginning of each
# module or class.
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02111-1301  USA

'''
Network subsystem of the application
'''

from idavoll.backend import PubSubServiceFromBackend, BackendService
from idavoll.memory_storage import Storage, LeafNode
from wokkel.pubsub import PubSubClient
from wokkel.client import HybridAuthenticator, XMPPClientConnector
from wokkel.ping import PingClientProtocol
from wokkel.generic import stripNamespace
from wokkel.subprotocols import StreamManager
from twisted.words.protocols import jabber
from twisted.words.xish.domish import Element
from twisted.internet.defer import Deferred
from twisted.internet.task import LoopingCall
from twisted import application
from twisted import internet
from twisted.python import log
from twisted.python.failure import Failure
from hivemind import parser
from hivemind.attribute import Observable, writable, readable
from hivemind.gui_main import reactor
import hivemind
import PySide
import uuid
#pylint: disable=C0103

''' XMPP PubSub node name '''
XMPP_NODE_NAME = unicode('HiveMind')
''' Time in seconds we must wait for unsubscribe '''
UNSUBSCRIBE_TIMEOUT = 2
''' Cold shutdown of network if timeout expires'''
SHUTDOWN_TIMEOUT = 4


class NetworkController(PySide.QtCore.QObject):
    '''
    Asynchronous command processor. Sends commands into shared space, listens to incoming
    messages and sends correct commands to mind map controller.
    @author: Andrew Vasilev, Oleg Kandaurov
    '''

    affiliationChanged = PySide.QtCore.Signal(str, str, name = 'affiliationChanged')
    subscriptionChanged = PySide.QtCore.Signal(str, str, name = 'subscriptionChanged')

    def __init__(self, mindMapController, actionBag):
        '''
        Constructor
        @type mindMapController: MindMapController
        @type actionBag: ActionBag
        '''
        PySide.QtCore.QObject.__init__(self)
        self.__mindMapController = mindMapController
        self.__actionBag = actionBag
        self.__xmppClient = None
        self.__protocol = None
        self.__storage = None
        self.__affiliationDialog = None
        self.__mindmap = None
        self._createActions()
        self._initActions(transport = False, protocol = False)

    def _createActions(self):
        '''
        Create network related actions
        '''
        self.__actionBag.startNetworkServerAction.triggered.connect(self._showServiceDialog)
        self.__actionBag.startNetworkClientAction.triggered.connect(self._showClientDialog)
        self.__actionBag.stopNetworkAction.triggered.connect(self._shutdownCommunication)
        self.__actionBag.connectXMPPAction.triggered.connect(self._showXMPPDialog)
        self.__actionBag.startPubSubServiceAction.triggered.connect(\
                self._showPubSubServiceDialog)
        self.__actionBag.startPubSubClientAction.triggered.connect(\
                self._showPubSubClientDialog)
        self.__actionBag.stopProtocolAction.triggered.connect(self._closeProtocol)

    def _initActions(self, transport, protocol):
        '''
        Initialize actions
        @param transport: Whether transport connected or not
        @type transport: bool
        @param protocol: Whether protocol connected or not
        @type protocol: bool
        '''
        self.__actionBag.startNetworkClientAction.setEnabled(not transport)
        self.__actionBag.startNetworkServerAction.setEnabled(not transport)
        self.__actionBag.editAffiliationAction.setEnabled(protocol)
        self.__actionBag.stopNetworkAction.setEnabled(transport)
        self.__actionBag.connectXMPPAction.setEnabled(not transport)
        self.__actionBag.startPubSubServiceAction.setEnabled(transport and not protocol)
        self.__actionBag.startPubSubClientAction.setEnabled(transport and not protocol)
        self.__actionBag.stopProtocolAction.setEnabled(protocol)

    def _showNetworkConnectionDialog(self, credentials, client):
        '''
        Show network dialog according to configuration
        @param credentials: Show XMPP credentials
        @type credentials: bool
        @param client: Show client or server dialogs
        @type client: bool
        '''
        def _accept():
            '''Process accepted signal from dialog'''
            settings = dialog.attributes()
            if credentials:
                self._startXMPPconnection(settings['userJid'] + '/hivemind',
                        settings['password'])
            if client == True:
                self._startHiveMindClient(settings['serviceJid'] + '/hivemind')
            elif client == False:
                self._startHiveMindService(settings['readOnly'],
                    self.__mindMapController.initialMindMap,
                    self.__mindMapController.undoStack)

        if client:
            if not self.__mindMapController.prohibitChangeMapAction():
                return
        dialog = hivemind.gui_factory.createNetworkConnectionDialog(credentials, client)
        dialog.accepted.connect(_accept)
        dialog.exec_()

    def _showXMPPDialog(self):
        '''
        Show xmpp network dialog
        '''
        self._showNetworkConnectionDialog(credentials = True, client = None)

    def _showPubSubClientDialog(self):
        '''
        Show pubsub network dialog for client
        '''
        self._showNetworkConnectionDialog(credentials = False, client = True)

    def _showClientDialog(self):
        '''
        Show network dialog for client
        '''
        self._showNetworkConnectionDialog(credentials = True, client = True)

    def _showPubSubServiceDialog(self):
        '''
        Show pubsub network dialog for server
        '''
        self._showNetworkConnectionDialog(credentials = False, client = False)

    def _showServiceDialog(self):
        '''
        Show network dialog for server
        '''
        self._showNetworkConnectionDialog(credentials = True, client = False)

    def _initAffiliationDialog(self):
        '''
        Initialize dialog that presents role of every connected participant
        '''
        def _accept():
            '''Process accepted signal from dialog'''
            self.__storage.rootNode.affiliations = self.__affiliationDialog.getAffiliations()

        self.__affiliationDialog = hivemind.gui_factory.createAffiliationDialog()
        self.__affiliationDialog.accepted.connect(_accept)
        self.affiliationChanged.connect(self.__affiliationDialog.setAffiliation)
        self.subscriptionChanged.connect(self.__affiliationDialog.updateStatus)
        self.__actionBag.editAffiliationAction.triggered.connect(
                self.__affiliationDialog.exec_)

    def publishCommand(self, command):
        '''
        Publish command in common space
        @param command: command to publish in shared space
        '''
        data = parser.serializeCommand(command)
        if self.__protocol:
            self.__protocol.proposeChangeset(Changeset(data, 'command'))
        else:
            self.__mindMapController.executeCommand(command)

    def _startHiveMindService(self, readOnly, mindmap, undoStack):
        '''
        Start HiveMind service
        @param readOnly: specifies whether mind map read only or not
        @type readOnly: bool
        @param mindmap: current mind map to publish first
        @type mindmap: MindMap
        @param undoStack: Undo stack associated with mindmap
        @type undoStack: QUndoStack
        '''
        self._initAffiliationDialog()
        config = Storage.defaultConfig['leaf']
        # Default config doesn't contain node type
        config['pubsub#node_type'] = 'leaf'
        config['pubsub#send_last_published_item'] = 'never'
        self.__storage = HivemindStorage()
        self.__storage.addObserver(self)
        self.__storage.createRootNode(self.__xmppClient.jid, readOnly, config)
        self.__protocol = HivemindService(parser.mindMapToString(mindmap, False),
                parser.serializeCommandStack(undoStack),
                HivemindBackendService(self.__storage))
        self.__protocol.addObserver(self)
        self.__protocol.setHandlerParent(self.__xmppClient)

    def _startXMPPconnection(self, userJid, password):
        '''
        Start XMPP connection
        @param userJid: JID for XMPP connection
        @type userJid: str
        @param password: password for XMPP connection
        @type password: str
        '''
        authenticator = HybridAuthenticator(jabber.jid.internJID(userJid), password)
        factory = HivemindXmlStreamFactory(authenticator)
        factory.addObserver(self)
        self.__xmppClient = HivemindXMPPClient(authenticator, factory)
        self.__xmppClient.addObserver(self)
        self.__xmppClient.logTraffic = True
        self.__xmppClient.startService()

    def _startHiveMindClient(self, serviceJid):
        '''
        Start HiveMind client, connect to specified service
        @param serviceJid: HiveMind service JID
        @type serviceJid: str
        '''
        self.__protocol = HivemindClient(jabber.jid.internJID(serviceJid))
        self.__protocol.setHandlerParent(self.__xmppClient)
        self.__protocol.addObserver(self)

    def shutdownNetwork(self):
        '''Shutdown network communication asynchronously'''
        def stopReactor(result):
            '''Stop reactor'''
            log.msg('Network closed')
            reactor.stop()

        if not self.__mindMapController.prohibitChangeMapAction():
            return
        defer = self._shutdownCommunication()
        defer.addCallback(stopReactor)
        reactor.callLater(SHUTDOWN_TIMEOUT, stopReactor, self)

    def _shutdownCommunication(self):
        '''Shutdown communication with the XMPP network'''
        defer = self._closeProtocol()
        defer.addCallback(lambda success: self._stopTransport())
        return defer

    def _stopProtocol(self):
        '''Stop protocol'''
        log.msg('Protocol stopped')
        if self.__protocol is not None:
            self.__protocol.disownHandlerParent(self.__xmppClient)
            self.__protocol.deleteObserver(self)
            self.__protocol = None

    def _shutdownProtocol(self):
        '''Shutdown protocol'''
        defer = Deferred()
        if self.__protocol is not None:
            defer = self.__protocol.shutdown()
        else:
            defer.callback([])
        return defer

    def _closeProtocol(self):
        '''Close protocol'''
        defer = self._shutdownProtocol()
        defer.addCallback(lambda success: self._stopProtocol())
        defer.addCallback(lambda success: log.msg('Protocol closed'))
        return defer

    def _stopTransport(self):
        '''Stop transport'''
        log.msg('Transport stopped')
        if self.__xmppClient is not None:
            self.__xmppClient.stopService()
            self.__xmppClient = None
        self.__mindmap = None
        self._processTransportState('offline')

    def processNotification(self, *values, **kvargs):
        '''Process notification'''
        if 'transport' in kvargs:
            self._processTransportState(kvargs['transport'])
        if 'protocol' in kvargs:
            self._processProtocolEvent(kvargs['protocol'])
        elif 'error' in kvargs:
            self._processError(kvargs['error'])
        elif 'changeset' in kvargs:
            self._addChangeset(kvargs['changeset'])
        elif 'affiliation' in kvargs:
            jid, affiliation = kvargs['affiliation']
            self.affiliationChanged.emit(jid, affiliation)

    def _processError(self, failure):
        ''' Process various errors '''
        log.msg('Failure %s occured at %s' % (failure, self.__class__.__name__))
        if isinstance(failure, Failure):
            error = failure.trap(jabber.sasl.SASLError, jabber.error.StanzaError)
            if error == jabber.sasl.SASLError:
                self._stopProtocol()
                self._stopTransport()
                PySide.QtGui.QMessageBox.information(hivemind.gui_factory.defaultParent(),
                        self.tr('Authorization failed'),
                        self.tr('Incorrect login/password. You are disconnected'))
            elif error == jabber.error.StanzaError:
                if failure.value.condition == 'forbidden':
                    return
                self._closeProtocol()
                PySide.QtGui.QMessageBox.information(hivemind.gui_factory.defaultParent(),
                        self.tr('Subscription failed'),
                        self.tr('Subscription error. You are disconnected.'))

    def _processProtocolEvent(self, event):
        '''
        Process events from protocol level
        @param event: Event from protocol level
        @type event: dict
        '''
        if event['type'] == 'subscription':
            bareJid = event['subscriber'].userhost()
            self.subscriptionChanged.emit(event['state'], bareJid)
            if event['state'] == 'subscribed':
                self.__actionBag.subscribeStatusAction.setStatus('yes')
            elif event['state'] == 'unsubscribed':
                self.__actionBag.subscribeStatusAction.setStatus('no')
            self._initActions(self.__actionBag.connectionStatusAction.status != 'no',
                self.__actionBag.subscribeStatusAction.status != 'no')

    def _processTransportState(self, state):
        '''
        Process transport state
        @param state: Current transport state
        @type state: str
        '''
        if (state == 'initialized') | (state == 'alive'):
            self.__actionBag.connectionStatusAction.setStatus('established')
        elif state == 'connecting':
            self.__actionBag.connectionStatusAction.setStatus('connecting')
        elif (state == 'offline') | (state == 'lost') | (state == 'disconnected'):
            self.__actionBag.connectionStatusAction.setStatus('no')
            self.__actionBag.subscribeStatusAction.setStatus('no')
        self._initActions(self.__actionBag.connectionStatusAction.status != 'no',
                self.__actionBag.subscribeStatusAction.status != 'no')

    def _addChangeset(self, changeset):
        '''
        Add changeset received from hivemind network
        @type changeset: Changeset
        '''
        if changeset.type == 'mindmap':
            self.__mindmap = parser.mindMapFromString(changeset.data)
            self.__mindMapController.loadMindMap(self.__mindmap)
        elif changeset.type == 'command':
            self.__mindMapController.executeCommand(parser.deserializeCommand(
                    changeset.data, self.__mindmap))
        elif changeset.type == 'undostack':
            self.__mindMapController.undoStack = parser.deserializeCommandStack(
                    changeset.data, self.__mindmap)


class HivemindXmlStreamFactory(jabber.xmlstream.XmlStreamFactory, Observable):
    '''
    Factory for Jabber XmlStream objects as a reconnecting client.
    @author: Oleg Kandaurov
    '''

    def __init__(self, authenticator):
        jabber.xmlstream.XmlStreamFactory.__init__(self, authenticator)
        Observable.__init__(self)
        self.initialDelay = 1.0
        self.factor = 1.6180339887498948
        self.maxDelay = 40

    def startedConnecting(self, connector):
        '''
        Called when a connection has been started.
        @param connector: a Connector object.
        '''
        log.msg('Connection started at %s' % self.__class__.__name__)
        jabber.xmlstream.XmlStreamFactory.startedConnecting(self, connector)
        self.notifyObservers(transport = 'connecting')

    def clientConnectionFailed(self, connector, reason):
        '''
        Called when a connection has failed to connect.
        @type reason: Failure
        '''
        log.msg('Connection failed with %s at %s' % (reason, self.__class__.__name__))
        jabber.xmlstream.XmlStreamFactory.clientConnectionFailed(self, connector, reason)

    def clientConnectionLost(self, connector, reason):
        '''
        Called when an established connection is lost.
        @type reason: Failure
        '''
        log.msg('Connection lost with %s at %s' % (reason, self.__class__.__name__))
        jabber.xmlstream.XmlStreamFactory.clientConnectionLost(self, connector, reason)


class HivemindXMPPClient(StreamManager, application.service.Service, PingClientProtocol,
        Observable):
    '''
    Service that initiates an XMPP client connection
    @author: Oleg Kandaurov
    '''

    def __init__(self, authenticator, factory, host = None, port = 5222):
        '''
        @type authenticator: HybridAuthenticator
        @type factory: XmlStreamFactory
        @type host: str
        @type port: int
        '''
        self.jid = authenticator.jid
        self.domain = authenticator.jid.host
        self.host = host
        self.port = port
        self.__keepaliveLoop = None
        self.__pingCount = 0
        self.__pingWOResponse = hivemind.settings.get('xmppKeepAliveCount')
        self._connection = None
        StreamManager.__init__(self, factory)
        Observable.__init__(self)
        PingClientProtocol.__init__(self)

    def startService(self):
        '''
        Starts service
        '''
        application.service.Service.startService(self)
        self._connection = self._getConnection()

    def stopService(self):
        '''
        Stop service
        '''
        application.service.Service.stopService(self)
        self.factory.stopTrying()
        self._connection.disconnect()

    def _alive(self, result = None):
        '''
        Handler of alive event
        '''
        #log.msg('Connection alive at %s' % self.__class__.__name__)
        self.notifyObservers(transport = 'alive')
        self.__pingCount = 0

    def _handleServerError(self, failure):
        '''
        Handle errors from protocol level
        '''
        err = failure.trap(jabber.error.StanzaError)
        if err == jabber.error.StanzaError:
            # StanzaError means that server responds to us
            self._alive()
            return None
        return failure

    def _lost(self, failure):
        '''
        Handle errors from transport level
        '''
        log.msg('Failure %s at %s' % (failure, self.__class__.__name__))
        if failure.check(internet.error.ConnectionLost):
            return None
        return failure

    def _sendPing(self):
        '''
        Send ping to XMPP server and wait for reply
        '''
        if self.__pingCount > self.__pingWOResponse:
            self.notifyObservers(transport = 'lost')
            self._stopPing()
            self.factory.resetDelay()
            self.factory.retry(self._connection)
        self.__pingCount += 1
        defer = self.ping(jabber.jid.JID(self.domain))
        defer.addCallback(self._alive)
        defer.addErrback(self._handleServerError)
        defer.addErrback(self._lost)

    def _stopPing(self):
        '''
        Stop ping loop
        '''
        log.msg('Ping stopped at % s' % (self.__class__.__name__))
        if self.__keepaliveLoop is not None and self.__keepaliveLoop.running:
            self.__keepaliveLoop.stop()
            self.__keepaliveLoop = None

    def _startPing(self):
        '''
        Start ping loop
        '''
        log.msg('Ping started at %s' % (self.__class__.__name__))
        self.__pingCount = 0
        self.__keepaliveLoop = LoopingCall(self._sendPing)
        self.__keepaliveLoop.start(hivemind.settings.get('xmppKeepAliveInterval'), now = False)

    def initializationFailed(self, reason):
        '''
        Called when stream initialization has failed
        @param reason: A failure instance indicating why stream initialization failed
        @type reason: Failure
        '''
        log.msg('Initialization failed %s at %s' % (reason, self.__class__.__name__))
        self.notifyObservers(error = reason)

    def _authd(self, xs):
        '''
        Called when the stream has been initialized.
        Save the JID that we were assigned by the server, as the resource might
        differ from the JID we asked for. This is stored on the authenticator
        by its constituent initializers.
        '''
        log.msg('Connection initialized at %s' % (self.__class__.__name__))
        self._startPing()
        self.notifyObservers(transport = 'initialized')
        self.jid = self.factory.authenticator.jid
        StreamManager._authd(self, xs)

    def _connected(self, xs):
        '''
        Called when the transport connection has been established.
        '''
        log.msg('Transport connected at %s' % (self.__class__.__name__))
        self.notifyObservers(transport = 'connected')
        StreamManager._connected(self, xs)

    def _disconnected(self, reason):
        '''
        Called when the stream has been closed.
        '''
        log.msg('Stream closed at %s' % (self.__class__.__name__))
        self._stopPing()
        self.notifyObservers(transport = 'disconnected')
        StreamManager._disconnected(self, reason)

    def _getConnection(self):
        '''
        Create connector instance
        '''
        if self.host:
            return reactor.connectTCP(self.host, self.port, self.factory)
        else:
            connector = XMPPClientConnector(reactor, self.domain, self.factory)
            connector.connect()
            return connector


class HivemindService(PubSubServiceFromBackend, Observable):
    '''
    Pubsub service for hivemind implementation
    @author: Andrew Vasilev, Oleg Kandaurov
    '''

    def __init__(self, mindMap, undoStack, backend):
        '''
        Constructor
        @param mindMap: initial record in the service with whole mind map
        @type mindMap: str
        @param undoStack: undostack associated with mindMap
        @type undoStack: str
        @param backend: Generic publish - subscribe backend service
        @type backend: BackendService
        '''
        PubSubServiceFromBackend.__init__(self, backend)
        Observable.__init__(self)
        rootNode = self.backend.storage.rootNode
        self.serviceJID = rootNode.owner
        self.discoIdentity = {'category' : 'pubsub', 'type' : 'generic', 'name' : 'Hivemind'}
        mindMapChangeset = Changeset(mindMap, 'mindmap')
        undoStackChangeset = Changeset(undoStack, 'undostack')
        rootNode.storeChangeset(mindMapChangeset, rootNode.owner)
        rootNode.storeChangeset(undoStackChangeset, rootNode.owner)

    def connectionInitialized(self):
        '''The XML stream has been initialized'''
        PubSubServiceFromBackend.connectionInitialized(self)
        self.notifyObservers(protocol = {'type': 'subscription', 'state': 'subscribed',
                        'subscriber': self.parent.jid})

    def shutdown(self):
        '''Shutdown service'''
        self.notifyObservers(protocol = {'type': 'subscription', 'state': 'unsubscribed',
                        'subscriber': self.parent.jid})
        return internet.defer.succeed(None)

    def subscribe(self, requestor, service, nodeIdentifier, subscriber):
        '''Called when a subscribe request has been received'''
        defer = PubSubServiceFromBackend.subscribe(self,
                requestor, service, nodeIdentifier, subscriber)
        defer.addCallback(lambda succeded: self.notifyObservers(protocol =
                {'type': 'subscription', 'state': 'subscribed', 'subscriber': subscriber}))
        return defer

    def unsubscribe(self, requestor, service, nodeIdentifier, subscriber):
        '''Called when a unsubscribe request has been received'''
        defer = PubSubServiceFromBackend.unsubscribe(self,
                requestor, service, nodeIdentifier, subscriber)
        defer.addCallback(lambda succeded: self.notifyObservers(protocol =
                {'type': 'subscription', 'state': 'unsubscribed', 'subscriber': subscriber}))
        return defer

    def proposeChangeset(self, changeset):
        '''
        Propose changeset to the HiveMind network
        @param changeset: changeset to publish inside network
        @type changeset: Changeset
        '''
        rootNode = self.backend.storage.rootNode
        def cb(subscriptions):
            '''Notify all subscribers'''
            for subscription in subscriptions:
                self._notify({'items': rootNode.stack.getItems(1),
                        'nodeIdentifier': rootNode.nodeIdentifier,
                        'subscription': subscription})

        rootNode.storeChangeset(changeset, rootNode.owner)
        defer = rootNode.getSubscriptions()
        defer.addCallback(cb)


class HivemindClient(PubSubClient, Observable):
    '''
    Pubsub client for HiveMind service
    @author: Andrew Vasilev
    '''

    def __init__(self, hivemindService):
        '''
        @param hivemindService: jid of hivemind service
        @type hivemindService: JID
        '''
        PubSubClient.__init__(self)
        Observable.__init__(self)
        self.__service = hivemindService

    def _errorHandler(self, error):
        '''
        Handles various errors
        '''
        self.notifyObservers(error = error)

    def connectionInitialized(self):
        '''
        The XML stream has been initialized.
        '''
        PubSubClient.connectionInitialized(self)
        defer = self.subscribe(self.__service, XMPP_NODE_NAME, self.parent.jid)
        defer.addCallback(self.retrieveItems)
        defer.addCallback(lambda success: self.notifyObservers(protocol =
                {'type': 'subscription', 'state': 'subscribed',
                        'subscriber': self.parent.jid}))
        defer.addErrback(self._errorHandler)

    def shutdown(self):
        '''
        Shutdown client
        '''
        defer = self._unsubscribe()
        return defer

    def _unsubscribe(self):
        '''
        Unsubscribe from the node
        '''
        def notifyUnsubscription(result):
            ''' Notify observers about unsubscribe event '''
            log.msg('Unsubscribed from %s' % self.__service.full())
            if unsubscribeTimer.active():
                unsubscribeTimer.cancel()
            self.notifyObservers(protocol = {'type': 'subscription', 'state': 'unsubscribed',
                    'subscriber': self.parent.jid})

        unsubscribeTimer = reactor.callLater(UNSUBSCRIBE_TIMEOUT, notifyUnsubscription, self)
        defer = self.unsubscribe(self.__service, XMPP_NODE_NAME, self.parent.jid)
        defer.addCallback(notifyUnsubscription)
        defer.addErrback(self._errorHandler)
        return defer

    def retrieveItems(self, result = None):
        '''
        Retrieve all items from the server
        '''
        defer = self.items(self.__service, XMPP_NODE_NAME)
        defer.addCallback(self._parseInitItems)
        defer.addErrback(self._errorHandler)

    def _parseInitItems(self, items):
        '''
        Parse items on initial request
        @param items: a set of items retrieved from the service
        @type items: list
        '''
        for item in items:
            changeset = Changeset.fromElement(item)
            self.notifyObservers(changeset = changeset)

    def itemsReceived(self, event):
        '''
        Called when an items notification has been received for a node.
        @param event: The items event.
        @type event: ItemsEvent
        '''
        if event.sender != self.__service or event.nodeIdentifier != XMPP_NODE_NAME:
            return
        self._parseInitItems(event.items)

    def proposeChangeset(self, changeset):
        '''
        Propose changeset to the hivemind network
        @param changeset: changeset to send to the controller
        @type changeset: str
        '''
        changeset.publisher = self.parent.jid
        defer = self.publish(self.__service, XMPP_NODE_NAME, [changeset.toElement()])
        defer.addErrback(self._errorHandler)


class HivemindBackendService(BackendService):
    '''
    Hivemind backend service
    @author: Oleg Kandaurov
    '''

    def __init__(self, storage):
        '''
        Initialize backend service
        @param storage: Storage that associated with service
        @type storage: Storage
        '''
        BackendService.__init__(self, storage)


class HivemindStorage(Storage, Observable):
    '''
    Hivemind storage
    @author: Oleg Kandaurov
    '''

    readable('rootNode')
    def __init__(self):
        '''
        Initialize storage
        '''
        Storage.__init__(self)
        Observable.__init__(self)
        self.__rootNode = None

    def createRootNode(self, owner, readOnly, config):
        '''
        Creates root node of the storage
        @param owner: Owner of the node
        @type owner: JID
        @param readOnly: Whether node read only or not
        @type readOnly: bool
        @param config: Node configuration
        @type config: dict
        '''
        node = HivemindNode(XMPP_NODE_NAME, owner, readOnly, config)
        self._nodes[XMPP_NODE_NAME] = node
        self.__rootNode = node
        node.addObserver(self)
        for jid, affiliation in node.affiliations.iteritems():
            self.notifyObservers(affiliation = (jid, affiliation))
        return internet.defer.succeed(None)

    def processNotification(self, *values, **kvargs):
        '''Process notification'''
        if 'affiliation' in kvargs:
            self.notifyObservers(affiliation = kvargs['affiliation'])
        elif 'changeset' in kvargs:
            self.notifyObservers(changeset = kvargs['changeset'])


class HivemindNode(LeafNode, Observable, object):
    '''
    Hivemind node
    @author: Oleg Kandaurov
    '''

    readable('owner', 'stack')
    def __init__(self, nodeIdentifier, owner, readOnly, config):
        '''
        @param nodeIdentifier: Name of the node
        @type nodeIdentifier: str
        @param owner: Owner of the current node
        @type owner: JID
        @param config: Node configuration
        @type config: dict
        '''
        Observable.__init__(self)
        LeafNode.__init__(self, nodeIdentifier, owner, config)
        self.__stack = ChangesetStack()
        self.__readOnly = readOnly
        self.__owner = owner

    def storeItems(self, items, publisher):
        '''
        Store items in persistent storage for later retrieval
        @param items: The list of items to be stored
        @type items: list of Element
        @param publisher: JID of the publishing entity
        @type publisher: JID
        @return: deferred that fires upon success
        '''
        for element in items:
            element.attributes['publisher'] = publisher.full()
            self.__stack.addUnsignedChangeset(Changeset.fromElement(element))
            self.notifyObservers(changeset = Changeset.fromElement(element))
        return internet.defer.succeed(None)

    def storeChangeset(self, changeset, publisher):
        '''
        Store changeset in persistent storage for later retrieval
        @param items: The changeset to be stored
        @type items: Changeset
        @param publisher: JID of the publishing entity
        @type publisher: JID
        '''
        changeset.publisher = publisher
        self.__stack.addUnsignedChangeset(changeset)
        self.notifyObservers(changeset = changeset)

    def getItems(self, maxItems = None):
        '''
        Get items.
        If maxItems is not given, all items in the node are returned.
        Otherwise, maxItems limits the returned items to a maximum
        of that number of most recently published items.
        @param maxItems: if given, a natural number (> 0) that limits the
                            returned number of items.
        @return: deferred that fires with a list of found items.
        '''
        return internet.defer.succeed(self.__stack.getItems(maxItems))

    def getItemsById(self, itemIdentifiers):
        '''
        Get items by item id.
        Each item in the returned list is a xml Element
        @param itemIdentifiers: list of item ids.
        @return: deferred that fires with a list of found items.
        '''
        items = []
        for itemIdentifier in itemIdentifiers:
            items.append(self.__stack.getItemById(itemIdentifier))
        return internet.defer.succeed(items)

    def setAffiliation(self, entity, affiliation):
        '''
        Set affilation of entity
        @type entity: JID
        @param affiliation: One of ['owner', publisher', 'outcast', 'member']
        @type affiliation: str
        '''
        self._affiliations[entity.userhost()] = affiliation
        self.notifyObservers(affiliation = (entity.userhost(), affiliation))
        return internet.defer.succeed(None)

    def getAffiliation(self, entity):
        '''
        Get affiliations of entities with this node.
        @type entity: JID
        @return: deferred that returns a list of tuples (jid, affiliation),where jid is a 
        JID and affiliation is one of ['owner', publisher', 'outcast', 'member']
        '''
        if entity.userhost() in self._affiliations:
            return internet.defer.succeed(self._affiliations.get(entity.userhost()))
        affiliation = 'publisher'
        if self.__readOnly:
            affiliation = 'member'
        self.setAffiliation(entity, affiliation)
        return internet.defer.succeed(affiliation)

    def getAffiliations(self):
        '''
        Get all affiliations
        @rtype: dict of str, str
        '''
        return self._affiliations

    def setAffiliations(self, affiliations):
        '''
        Set all affiliations
        @param affiliations: Dictionary of JIDs and affiliations
        @type affiliations: dict of str, str
        '''
        self._affiliations = affiliations

    affiliations = property(getAffiliations, setAffiliations, None, 'Affiliations property')

    def purge(self):
        raise Exception('Not implemented')

    def removeItems(self, itemIdentifiers):
        raise Exception('Not implemented')


class Changeset(object):
    '''
    Changeset representation
    @author: Andrew Vasilev
    '''

    writable('id', 'number', 'data', 'type', 'publisher')
    def __init__(self, data = None, changesetType = None, publisher = None,
            changesetId = None, number = None):
        '''
        Initialize object from the several components
        @param data: xml-serialized data of the changeset
        @type data: str
        @param changesetType: type of the changeset
        @type changesetType: str
        @param changesetId: unique id of the changeset
        @type changesetId: str
        @param publisher: publisher of the changeset
        @type publisher: JID
        @param number: the number of the changeset
        @type number: int
        '''
        #pylint: disable=R0913
        self.__id = changesetId
        self.__number = number
        self.__data = data
        self.__type = changesetType
        self.__publisher = publisher

    def __str__(self):
        return str(self.__number) + ' : ' + self.__publisher.full() + ' : ' + \
                str(self.__id) + ' : ' + str(self.__type) + ' : ' + str(self.__data)

    def toElement(self):
        '''
        Convert this changeset into domish xml Element
        @return: Element representation of current object
        @rtype: Element
        '''
        element = Element((None, 'item'))
        if self.__id: element.attributes['id'] = self.__id
        if self.__number: element.attributes['number'] = str(self.__number)
        if self.__type: element.attributes['type'] = self.__type
        if self.__publisher: element.attributes['publisher'] = self.__publisher.full()
        element.addRawXml(self.__data)
        return element

    def isValid(self, fieldList):
        '''
        Check fields of the changeset to be valid
        @param fieldList: fields of the changeset to test for None
        @type fieldList: list of str
        @rtype: bool
        '''
        for field in fieldList:
            if self.__getattribute__(field) is None:
                raise ChangesetException('Field ' + field + ' cannot be None')
        return True

    @staticmethod
    def fromElement(element):
        '''
        Create Changeset object from xml Element
        @param element: xml representation of the object
        @type element: Element
        '''
        xmlChangeset = element.firstChildElement()
        stripNamespace(xmlChangeset)
        changeset = xmlChangeset.toXml()
        return Changeset(changeset, element.getAttribute('type'),
                jabber.jid.JID(element.getAttribute('publisher')), element.getAttribute('id'),
                element.getAttribute('number'))


class ChangesetStack(object):
    '''
    Stack of the changesets
    @author: Andrew Vasilev, Oleg Kandaurov
    '''

    def __init__(self):
        self.__changesets = []
        self.__elements = []

    def getItemById(self, itemId):
        '''
        Get item by item id. Item is a xml Element
        @param itemId: Item id
        @type itemId: str
        @rtype: Element
        '''
        for element in self.__elements:
            if element.getAttribute('id') == itemId:
                return element
        return None

    def getItems(self, maxItems = None):
        '''
        Get items from stack
        @param maxItems: number of recently published items or all item if None
        @type maxItems: int
        @rtype: list of Element
        '''
        if maxItems:
            return self.__elements[-maxItems:]
        else:
            return self.__elements

    def addChangeset(self, changeset):
        '''
        Add changeset to the stack
        @param changeset: fully-functional changeset
        @type changeset: Changeset
        @raise ChangesetError: if passed changeset is invalid
        @raise ChangesetNumberError: if changeset has invalid humber
        @raise ChangesetIdError: if changeset has invalid id
        '''
        changeset.isValid(['id', 'number', 'data', 'type', 'publisher'])
        self._checkChangesetNumber(changeset)
        if not self._isUniqueId(changeset.id):
            raise ChangesetIdException('Changeset has not unique id')
        self.__changesets.append(changeset)
        self.__elements.append(changeset.toElement())

    def addUnsignedChangeset(self, changeset):
        '''
        Add unsigned changeset to the stack. Warning! Changes passed object!
        @param changeset: changeset with data and type fields present
        @type changeset: Changeset
        @raise ChangesetError: if passed changeset is invalid
        '''
        self.signChangeset(changeset)
        self.addChangeset(changeset)

    def signChangeset(self, changeset):
        '''
        Set appropriate number and hash for the changeset if needed
        @type changeset: Changeset
        '''
        changeset.number = len(self.__changesets)
        if not self._isUniqueId(changeset.id):
            changeset.id = self._generateUniqueId()

    def _checkChangesetNumber(self, changeset):
        '''
        Check number of the changeset to have next number in the list
        @type changeset: Changeset
        @raise ChangesetException: if changeset has wrong number
        '''
        if changeset.number != len(self.__changesets):
            raise ChangesetNumberException('Wrong number of the changeset.' +
                    '%d expected, but %d got' % (len(self.__changesets), changeset.number))

    def _generateUniqueId(self):
        '''
        Generate unique identifier for the changeset
        @return: new unique identifier for the changeset
        @rtype: str
        '''
        while True:
            newId = str(uuid.uuid4())
            if not self._isUniqueId(newId): continue
            return newId

    def _isUniqueId(self, changesetId):
        '''
        Check for the uniqueness of the id
        @type id: str
        @return: False if there is a changeset with the same id
        @rtype: bool
        '''
        if changesetId is None or changesetId is '': return False
        for changeset in self.__changesets:
            if changeset.id == changesetId:
                return False
        return True

    def __len__(self):
        '''Add support for getting number of changesets via len()'''
        return len(self.__changesets)

    def lastId(self):
        '''
        Get the id of the last changeset we are holding
        @rtype: str
        '''
        if len(self.__changesets) == 0: return None
        return self.__changesets[-1].id

    def items(self, changesetId = None):
        '''
        Retrieve the list of last items in the stack. If no id is passed,
        then return all available items
        @param id: id of the changeset known to the user
        @type id: str
        @rtype: list of Element
        '''
        if changesetId is None: return self.__elements
        index = 0
        for changeset in self.__changesets:
            index += 1
            if changeset.id == changesetId:
                return self.__elements[index:]
        return self.__elements

    def itemsByNumber(self, number = 1):
        '''
        Retrieve the list of last items in the stack.
        @param number: maximum number of items to recieve
        @type number: int
        @rtype: list of Element
        '''
        if number < 0 or number is None:
            return self.__elements
        count = min(number, len(self.__changesets))
        return self.__elements[-count:]


class ChangesetException(Exception):
    '''
    Base exception class for all exception that can be raised by ChangesetStack
    @author: Andrew Vasilev
    '''

    readable('reason')
    def __init__(self, reason):
        '''
        @param reason: reason for the exception to be raised
        @type reason: str
        '''
        Exception.__init__(self)
        self.__reason = reason

    def __str__(self):
        return self.__reason


class ChangesetNumberException(ChangesetException):
    '''
    Exception indicates that something is wrong with the number of the changeset
    @author: Andrew Vasilev
    '''

    def __init__(self, reason):
        ChangesetException.__init__(self, reason)


class ChangesetIdException(ChangesetException):
    '''
    Exception indicates that something is wrong with the identification of the changeset
    @author: Andrew Vasilev
    '''

    def __init__(self, reason):
        ChangesetException.__init__(self, reason)
