# -*- coding: utf-8 -*-
# HiveMind - Distributed mind map editor for Maemo 5 platform
# Copyright (C) 2010-2011 HiveMind developers
#
# HiveMind is the legal property of its developers, whose names are
# noticed in @author or @authors annotations at the beginning of each
# module or class.
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02111-1301  USA

'''
Abstract main window class
'''
from PySide.QtGui import *
from PySide.QtCore import *
from hivemind.gui_widgets import *
from hivemind.gui_delegates import *

class MainView(QGraphicsView):
    '''
    Main view of the application displaying mind map.
    
    @author: Andrew Vasilev
    '''

    readable('draggedPath')

    def __init__(self, scene, actionBag):
        QGraphicsView.__init__(self, scene)
        self.setRenderHints(QPainter.Antialiasing)
        self.setDragMode(QGraphicsView.ScrollHandDrag)
        self.setResizeAnchor(QGraphicsView.AnchorViewCenter)
        palette = self.palette()
        palette.setColor(QPalette.Base, Qt.white)
        self.setPalette(palette)
        self.__draggedPath = 0
        self.__actionBag = actionBag
        self.__zoomLevel = 0
        self.__actionBag.zoomInAction.connect(SIGNAL('triggered()'), self._zoomIn)
        self.__actionBag.zoomOutAction.connect(SIGNAL('triggered()'), self._zoomOut)
        self.connect(scene, SIGNAL('mapChanged'), self._resetView)

    def scrollContentsBy(self, dx, dy):
        '''
        Scrolls scene contents

        @type dx: int
        @type dy: int
        '''
        QGraphicsView.scrollContentsBy(self, dx, dy)
        self.__draggedPath += abs(dx) + abs(dy)

    def mousePressEvent(self, event):
        '''
        Mouse press event handler

        @type event: QMouseEvent
        '''
        QGraphicsView.mousePressEvent(self, event)
        self.__draggedPath = 0
        self.__startPoint = event.pos()

    def mouseReleaseEvent(self, event):
        '''
        Mouse release event handler.
        Emits itemClicked, mapClicked signals.

        @type event: QMouseEvent
        '''
        QGraphicsView.mouseReleaseEvent(self, event)
        self.__draggedPath += self._length(self.__startPoint, event.pos())

    def _length(self, point1, point2):
        '''
        Calculates approximate path between passed points

        @type point1: QPoint
        @type point2: QPoint

        @rtype: int
        '''
        diff = point2 - point1
        return abs(diff.x()) + abs(diff.y())

    __ZOOM_IN_COEFFICIENT = 1.25
    '''View scale coefficient for zoom in action'''

    __ZOOM_OUT_COEFFICIENT = 0.8
    '''View scale coefficient for zoom out action'''

    __MAX_ZOOM_LEVEL = 8
    '''Maximum value for zoom level'''

    __MIN_ZOOM_LEVEL = -8
    '''Minimum value for zoom level'''

    def _zoom(self, scaleFactor):
        '''
        Change the scale factor of the mind map.
        @parame scaleFactor: the scale of the map to apply (__ZOOM_IN_COEFFICIENT or
        _ZOOM_OUT_COEFFICIENT)
        @type scaleFactor: double
        '''
        self.scale(scaleFactor, scaleFactor)
        self.__zoomLevel += 1 if scaleFactor > 1 else - 1
        self._updateZoomActionsState()

    def _updateZoomActionsState(self):
        '''Update the state of the actions according to
        the zoom level'''
        if self.__zoomLevel == MainView.__MAX_ZOOM_LEVEL:
            self.__actionBag.zoomInAction.setEnabled(False)
        elif self.__zoomLevel == MainView.__MIN_ZOOM_LEVEL:
            self.__actionBag.zoomOutAction.setEnabled(False)
        else:
            self.__actionBag.zoomInAction.setEnabled(True)
            self.__actionBag.zoomOutAction.setEnabled(True)

    def _zoomIn(self):
        '''Zoom in scene view'''
        self._zoom(MainView.__ZOOM_IN_COEFFICIENT)

    def _zoomOut(self):
        '''Zoom out scene view'''
        self._zoom(MainView.__ZOOM_OUT_COEFFICIENT)

    def _resetView(self):
        '''
        Reset zoom transformations. Scroll the contents of
        the viewport to ensure that the root item is centered in the view.
        '''
        self.resetTransform()
        self.scale(1.44, 1.44)
        self.__zoomLevel = 0
        self._updateZoomActionsState()
        self.centerOn(self.scene().rootDelegate)


class MaemoNodeMenu(QObject):
    '''
    Abstract node modification menu for Maemo Platform

    @author: Andrew Vasilev, Alexander Kulikov
    '''

    def __init__(self, parent):
        '''
        @param parent: the main scene view to place menu on top of
        @type parent: MainView
        '''
        QObject.__init__(self, parent)
        self._mapView = parent
        self._menuButtons = {}
        self.connect(parent, SIGNAL('mapClicked'), self.hideMenu)
        self.__hideMenuTimer = QTimer(self)
        self.__hideMenuTimer.setInterval(settings.get('autoHideNodeMenuTimeout'))
        self.__hideMenuTimer.setSingleShot(True)
        self.__hideMenuTimer.connect(SIGNAL('timeout()'), self.hideMenu)

    __INITIAL_ANGLE = math.pi * 1.2
    '''Start angle for button positioning'''

    def addAction(self, action, iconName):
        '''
        Add action to maemo menu
        @type action: QAction
        @type iconName: str
        '''
        action.setIcon(resource.getIcon(iconName))
        button = ActionImageButton(action, self._mapView)
        self._connectButtonToAction(button, action)

    def _connectButtonToAction(self, button, action):
        '''
        Link button with action
        @type button: QAbstractButton
        @type action: QAction
        '''
        self._menuButtons[action] = button
        button.hide()
        button.connect(SIGNAL('clicked()'), self.hideMenu)

    def _restartHideMenuTimer(self):
        '''Restart hide menu timer'''
        self.__hideMenuTimer.stop()
        self.__hideMenuTimer.start()

    def _calculatePosition(self, angle):
        '''
        @param angle: angle of the point in radians
        @type angle: float
        
        @return: the point on ellipse, identified by angle
        @rtype: float tuple 
        '''
        x = self.__basePoint.x() + self.__radiusOne * \
                math.cos(angle + MaemoNodeMenu.__INITIAL_ANGLE)
        y = self.__basePoint.y() + self.__radiusTwo * \
                math.sin(angle + MaemoNodeMenu.__INITIAL_ANGLE)
        return x, y

    def _positionMenu(self):
        '''
        Position buttons on the view and start animation
        '''
        self._generateAnimations()
        step = 0.2
        angle = 0
        while angle < 2 * math.pi:
            if self._isButtonInside(*self._calculatePosition(angle)):
                angle = self._positionMenuButtons(angle)
            angle += step
        self.__showMenuAnimation.start()

    def _isButtonInside(self, x, y):
        '''
        Check whether button fits the view if placed on (x, y) coordinates
          
        @return: True if button fits on the view
        @rtype: bool
        '''
        buttonRect = QRect(x - 32, y - 32, 64, 64)
        viewRectangle = self._mapView.rect()
        return viewRectangle.contains(buttonRect)

    def _distance(self, pointOne, pointTwo):
        '''
        @param pointOne: first point coordinates
        @type pointOne: tuple
        
        @param pointTwo: second point coordinates
        @type pointTwo: tuple
        
        @return: distance between passed points
        @rtype: float 
        '''
        return math.sqrt((pointOne[0] - pointTwo[0]) ** 2
                         + (pointOne[1] - pointTwo[1]) ** 2)

    __MIN_DISTANCE = 68
    '''Minimal distance between buttons'''

    def _positionMenuButtons(self, startAngle):
        '''
        Position menu buttons starting from the passed angle
        
        @param startAngle: base angle for positioning in radian measurement 
        @type startAngle: float
        
        @return: 
        @rtype: float
        '''
        angle = startAngle
        angleStep = math.pi / 60
        basePoint = self.__basePoint
        centerRect = QRect(basePoint.x(), basePoint.y(), 0, 0)
        for action, button in self._menuButtons.iteritems():
            if action.isEnabled():
                curPos = self._calculatePosition(angle)
                endRect = QRect(curPos[0] - 32, curPos[1] - 32, 64, 64)
                showAnimation = self.__showAnimationTransformation[button]
                if button.isVisible():
                    showAnimation.setStartValue(button.geometry())
                else:
                    showAnimation.setStartValue(centerRect)
                    showAnimation.setKeyValueAt(0.65,
                            self._calculateTransitionalPosition(centerRect, endRect, 0.4))
                showAnimation.setEndValue(endRect)
                hideAnimation = self.__hideAnimationTransformation[button]
                hideAnimation.setStartValue(endRect)
                hideAnimation.setEndValue(centerRect)
                # calculate position for next button
                while True:
                    angle += angleStep
                    nextPos = self._calculatePosition(angle)
                    if self._distance(curPos, nextPos) >= MaemoNodeMenu.__MIN_DISTANCE:
                        if not self._isButtonInside(*nextPos):
                            return angle
                        break
        return math.pi * 2

    def _calculateTransitionalPosition(self, startPoint, endPoint, coef):
        '''
        Calculate transitional position for animation between
        startPoint and endPoint.
        
        @param startPoint: start point of animation
        @type startPoint: QRect
        
        @param endPoint: end point of animation
        @type endPoint: QRect
        
        @return: transition point to emulate twisting animation
        @rtype: QRect
        '''
        vectorX = endPoint.x() - startPoint.x()
        vectorY = endPoint.y() - startPoint.y()
        midPointX = vectorX * 0.65 + vectorY * coef + startPoint.x()
        midPointY = vectorY * 0.65 - vectorX * coef + startPoint.y()
        midPointWidth = math.fabs((endPoint.width() - startPoint.width()) / 2.0)
        midPointHeight = math.fabs((endPoint.height() - startPoint.height()) / 2.0)
        return QRect(midPointX, midPointY, midPointWidth, midPointHeight)

    def _generateAnimations(self):
        '''
        Create and set up show and hide animations for menu buttons 
        '''
        self.__showAnimationTransformation = dict()
        self.__hideAnimationTransformation = dict()
        self.__showMenuAnimation = QParallelAnimationGroup()
        self.__hideMenuAnimation = QParallelAnimationGroup()

        for action, button in self._menuButtons.iteritems():
            if button.isVisible() and not action.isEnabled():
                centrRect = QRect(self.__basePoint.x(), self.__basePoint.y(), 0, 0)
                hideAnimation = QSequentialAnimationGroup()

                hideTransformationAnimation = QPropertyAnimation(button, 'geometry')
                hideTransformationAnimation.setDuration(
                        settings.get('hideNodeMenuAnimationDuration'))
                hideTransformationAnimation.setEasingCurve(QEasingCurve.OutExpo)
                hideTransformationAnimation.setStartValue(button.geometry())
                hideTransformationAnimation.setEndValue(centrRect)

                hideVisibilityAnimation = WidgetVisibilityAnimation(button, False)

                hideAnimation.addAnimation(hideTransformationAnimation)
                hideAnimation.addAnimation(hideVisibilityAnimation)
                self.__showMenuAnimation.addAnimation(hideAnimation)

        for action, button in self._menuButtons.iteritems():
            if action.isEnabled():
                showAnimation = QSequentialAnimationGroup()

                showTransformationAnimation = QPropertyAnimation(button, 'geometry')
                showTransformationAnimation.setDuration(
                        settings.get('showNodeMenuAnimationDuration'))
                showTransformationAnimation.setEasingCurve(QEasingCurve.InSine)
                self.__showAnimationTransformation[button] = showTransformationAnimation

                showVisibilityAnimation = WidgetVisibilityAnimation(button, True)

                showAnimation.addAnimation(showVisibilityAnimation)
                showAnimation.addAnimation(showTransformationAnimation)
                self.__showMenuAnimation.addAnimation(showAnimation)

                hideAnimation = QSequentialAnimationGroup()

                hideTransformationAnimation = QPropertyAnimation(button, 'geometry')
                hideTransformationAnimation.setDuration(
                        settings.get('hideNodeMenuAnimationDuration'))
                hideTransformationAnimation.setEasingCurve(QEasingCurve.OutExpo)
                self.__hideAnimationTransformation[button] = hideTransformationAnimation

                hideVisibilityAnimation = WidgetVisibilityAnimation(button, False)

                hideAnimation.addAnimation(hideTransformationAnimation)
                hideAnimation.addAnimation(hideVisibilityAnimation)

                self.__hideMenuAnimation.addAnimation(hideAnimation)

    def _evaluateMenuBoundaries(self, item):
        '''
        @param item: selected item on the scene
        @type item: NodeDelegate
        '''
        rect = item.textRectOnScene()
        self._mapView.ensureVisible(rect, 90, 90)
        self.__basePoint = self._mapView.mapFromScene(rect.center())
        if rect.width() > 100:
            self.__radiusOne = rect.width() / 2.0 + 32
        else:
            self.__radiusOne = 82
        if rect.height() > 60:
            self.__radiusTwo = rect.height() / 2.0 + 32
        else:
            self.__radiusTwo = 62

    def show(self, item):
        '''
        Show main menu on top of selected item
        @param item: selected item on the scene
        @type item: NodeDelegate
        '''
        self.emit(SIGNAL('show'))
        self._evaluateMenuBoundaries(item)
        self._positionMenu()
        self._restartHideMenuTimer()

    def hideMenu(self):
        '''Hide menu'''
        # if main and additional menu are not active then return
        for button in self._menuButtons.values():
            if not button.isHidden():
                self.__hideMenuAnimation.start()


class AbstractMainWindow(SearchingTrMixin, QMainWindow):
    '''
    Abstract main window class

    Subclasses of this class must implement self._mindMapView property

    @author: Alexander Kulikov
    '''

    def _getMindMapView(self):
        return self._mindMapView

    mindMapView = property(_getMindMapView, None, None, 'mindmap view')

    def __init__(self, actionBag, mindMapScene):
        '''Create new window'''
        QMainWindow.__init__(self)
        self._actionBag = actionBag
        self._initNavigateActions(mindMapScene)
        self._initActions()

    def _initNavigateActions(self, mindMapScene):
        '''Appends the navigate actions to mainView's list of actions'''
        self.addAction(mindMapScene.selectLeftNodeAction)
        self.addAction(mindMapScene.selectRightNodeAction)
        self.addAction(mindMapScene.selectUpNodeAction)
        self.addAction(mindMapScene.selectDownNodeAction)
        self.addAction(mindMapScene.selectRootNodeAction)

    def _initActions(self):
        '''Appends the actions to mainView's list of actions'''
        self.addAction(self._actionBag.editEdgeAction)
        self.addAction(self._actionBag.editLabelAction)
        self.addAction(self._actionBag.editNodeAction)
        self.addAction(self._actionBag.editNodeIconsAction)
        self.addAction(self._actionBag.foldNodeAction)
        self.addAction(self._actionBag.addNodeAction)
        self.addAction(self._actionBag.removeNodeAction)
        self.addAction(self._actionBag.enterTransferModeAction)
        self.addAction(self._actionBag.cancelTransferModeAction)
        self.addAction(self._actionBag.putNodeAction)
        self.addAction(self._actionBag.putNodeBelowAction)
        self.addAction(self._actionBag.putNodeAboveAction)

        self.addAction(self._actionBag.addSiblingNodeBelowAction)
        self.addAction(self._actionBag.addSiblingNodeAboveAction)
        self.addAction(self._actionBag.zoomInAction)
        self.addAction(self._actionBag.zoomOutAction)

    def closeEvent(self, event):
        '''Close event handler'''
        self.emit(SIGNAL('closeEvent'))
        event.ignore()
