# -*- coding: utf-8 -*-
# HiveMind - Distributed mind map editor for Maemo 5 platform
# Copyright (C) 2010-2011 HiveMind developers
#
# HiveMind is the legal property of its developers, whose names are
# noticed in @author or @authors annotations at the beginning of each
# module or class.
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02111-1301  USA

'''
Main window class for Mobile platform
'''

from hivemind.main_window import *

class MaemoMainView(MainView):
    '''
    Mouse input filter for the Maemo platform
    @author: Andrew Vasilev
    '''

    __MIN_DRAG_PATH = 50
    '''Minimum drag path length for the Maemo platform'''

    def __init__(self, scene, actionBag):
        '''
        @type scene: MindMapDelegate
        @type actionBag: ActionBag
        '''
        MainView.__init__(self, scene, actionBag)
        self.__singleClickTimer = QTimer()
        self.__singleClickTimer.setInterval(300)
        self.__singleClickTimer.setSingleShot(True)
        self.__singleClickTimer.connect(SIGNAL('timeout()'), self._madeSingleClick)

    def _madeSingleClick(self):
        '''Process delayed single click action'''
        if self.__lastParent is not None:
            self.emit(SIGNAL('itemClicked'), False, self.__lastItem, self.__lastParent,
                    self.__lastChildLocation)
        if self.__lastItem is None:
            self.emit(SIGNAL('mapClicked'))

    def mouseReleaseEvent(self, event):
        '''
        Mouse release event handler.
        Emits itemClicked, mapClicked signals.

        @type event: QMouseEvent
        '''
        MainView.mouseReleaseEvent(self, event)
        if self.draggedPath < self.__MIN_DRAG_PATH:
            item = self.scene().itemAt(self.mapToScene(event.pos()))
            parent, childLocation = self.scene().parentAt(self.mapToScene(event.pos()))
            if self.__singleClickTimer.isActive():
                if item == self.__lastItem:
                    self.__singleClickTimer.stop()
                    self.emit(SIGNAL('itemClicked'), True, item, parent, childLocation)
                    return
            self.__lastItem = item
            self.__lastParent = parent
            self.__lastChildLocation = childLocation
            self.__singleClickTimer.start()


class MaemoMainWindow(AbstractMainWindow):
    '''
    Main window of HiveMind program for Maemo platform

    @author: Alexander Kulikov
    '''
    def __init__(self, actionBag, mindMapScene):
        '''Create new window'''
        AbstractMainWindow.__init__(self, actionBag, mindMapScene)
        self._mindMapView = MaemoMainView(mindMapScene, actionBag)
        self.setCentralWidget(self._mindMapView)
        FullScreenExitButton(self)
        self.__fullScreenAction = QAction(QIcon.fromTheme('view-fullscreen'),
                self.tr('&Full screen'), self)
        self.__fullScreenAction.connect(SIGNAL('triggered()'), self.showFullScreen)
        self._createContextMenu()
        self._createMainMenu()
        self._initMindMapView(actionBag)
        self.setAttribute(Qt.WA_Maemo5StackedWindow)

    def _initMindMapView(self, actionBag):
        '''
        Set widgets on mindMapView and init scroll bar policy
        @type actionBag: ActionBag
        '''
        self.__zoomWidget = MaemoZoomWidget(self._mindMapView, actionBag.zoomInAction,
                actionBag.zoomOutAction)
        self.__connectionStatusWidget = MaemoConnectionStatusWidget(self._mindMapView,
                actionBag.connectionStatusAction, actionBag.subscribeStatusAction)
        mainLayout = QGridLayout()
        mainLayout.addWidget(self.__zoomWidget, 0, 0, Qt.AlignLeft | Qt.AlignTop)
        mainLayout.addWidget(self.__connectionStatusWidget, 1, 0, Qt.AlignLeft | Qt.AlignBottom)
        self._mindMapView.setLayout(mainLayout)
        self._mindMapView.setVerticalScrollBarPolicy(Qt.ScrollBarAlwaysOff)
        self._mindMapView.setHorizontalScrollBarPolicy(Qt.ScrollBarAlwaysOff)

    def setWindowTitle(self, fileName):
        if fileName:
            QMainWindow.setWindowTitle(self, fileName)
        else:
            QMainWindow.setWindowTitle(self, 'HiveMind')

    def _createContextMenu(self):
        self.__menu = MaemoNodeMenu(self._mindMapView)
        self.__additionalMenu = MaemoNodeMenu(self._mindMapView)
        self.__menu.addAction(self._actionBag.addNodeAction, 'add_node')
        self.__menu.addAction(self._actionBag.removeNodeAction, 'remove_node')
        self.__menu.addAction(self._actionBag.editNodeAction, 'edit_node')
        self._actionBag.foldNodeAction.setCheckable(True)
        self.__menu.addAction(self._actionBag.foldNodeAction, 'hide_subtree')
        self.__additionalMenu.addAction(self._actionBag.editNodeIconsAction, 'edit_icons')
        self.__additionalMenu.addAction(self._actionBag.editEdgeAction, 'edit_edge')
        self.__additionalMenu.addAction(self._actionBag.enterTransferModeAction, 'move_node')
        self.__additionalMenu.addAction(self._actionBag.editLabelAction, 'label_edge')
        self.connect(self.__menu, SIGNAL('show'),
                     adaptCallable(self.__additionalMenu.hideMenu))
        self.connect(self.__additionalMenu, SIGNAL('show'),
                     adaptCallable(self.__menu.hideMenu))

    def _createMainMenu(self):
        '''Create main menu'''
        menu = self.menuBar().addMenu('')
        menu.addAction(self._actionBag.fileNewAction)
        menu.addAction(self._actionBag.fileOpenAction)
        menu.addAction(self._actionBag.fileSaveAction)
        menu.addAction(self._actionBag.fileSaveAsAction)
        menu.addAction(self.__fullScreenAction)
        menu.addAction(self._actionBag.aboutAction)
        menu.addAction(self._actionBag.settingsAction)
        menu.addAction(self._actionBag.undoAction)
        menu.addAction(self._actionBag.redoAction)
        menu.addAction(self._actionBag.startNetworkServerAction)
        menu.addAction(self._actionBag.startNetworkClientAction)
        menu.addAction(self._actionBag.editAffiliationAction)
        menu.addAction(self._actionBag.stopNetworkAction)

    def showMenu(self, rightButton, item):
        '''Show one of the two menus according to the button clicked'''
        menu = self.__additionalMenu if rightButton else self.__menu
        menu.show(item)
