# -*- coding: utf-8 -*-
# HiveMind - Distributed mind map editor for Maemo 5 platform
# Copyright (C) 2010-2011 HiveMind developers
#
# HiveMind is the legal property of its developers, whose names are
# noticed in @author or @authors annotations at the beginning of each
# module or class.
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02111-1301  USA

'''
GUI factory

@author: Alexander Kulikov
'''

from hivemind.dialogs import SideSelectionDialog
try:
    import PySide.QtMaemo5 #pylint: disable=F0401,E0611,W0611
    __MAEMO = True
    from hivemind.maemo.main_window import MaemoMainWindow
    from hivemind.maemo.dialogs import MaemoEditNodeDialog, MaemoEditLabelDialog
    from hivemind.maemo.dialogs import MaemoIconDialog, MaemoAboutDialog, MaemoEditEdgeDialog
    from hivemind.maemo.dialogs import MaemoSettingsDialog, MaemoNetworkConnectionDialog
    from hivemind.maemo.dialogs import MaemoAffiliationDialog, MaemoExtendedMenuDialog
except ImportError:
    __MAEMO = None
    from hivemind.desktop.main_window import DesktopMainWindow
    from hivemind.desktop.dialogs import DesktopEditNodeDialog, DesktopEditLabelDialog
    from hivemind.desktop.dialogs import DesktopIconDialog, DesktopAboutDialog
    from hivemind.desktop.dialogs import DesktopEditEdgeDialog
    from hivemind.desktop.dialogs import DesktopSettingsDialog, DesktopNetworkConnectionDialog
    from hivemind.desktop.dialogs import DesktopAffiliationDialog

'''Default parent for all application dialogs (actually, main window)'''
__defaultParent = None

def defaultParent():
    '''
    @return: default parent for all dialogs
    @rtype: QWidget
    '''
    return __defaultParent

def createMainWindow(actionBag, mindMapScene):
    '''
    Factory for creating families of related main windows
    @type actionBag: ActionBag
    @type mindMapScene: MindMapDelegate
    '''
    global __defaultParent #pylint: disable=W0603
    if __MAEMO:
        __defaultParent = MaemoMainWindow(actionBag, mindMapScene)
    else:
        __defaultParent = DesktopMainWindow(actionBag, mindMapScene)
    return __defaultParent

def createEditNodeDialog(node):
    '''
    Factory for creating families of related node editing dialogs
    @type node: Node
    '''
    if __MAEMO:
        return MaemoEditNodeDialog(node, __defaultParent)
    return DesktopEditNodeDialog(node, __defaultParent)

def createEditEdgeDialog(node):
    '''
    Factory for creating families of related edge editing dialogs
    @type node: Node
    '''
    if __MAEMO:
        return MaemoEditEdgeDialog(node, __defaultParent)
    return DesktopEditEdgeDialog(node, __defaultParent)

def createIconDialog(node):
    '''
    Factory for creating families of related icon dialogs
    @type node: Node
    '''
    if __MAEMO:
        return MaemoIconDialog(node, __defaultParent)
    return DesktopIconDialog(node, __defaultParent)

def createAboutDialog():
    '''
    Factory for creating families of related about dialogs
    '''
    if __MAEMO:
        return MaemoAboutDialog(__defaultParent)
    return DesktopAboutDialog(__defaultParent)

def createSettingsDialog():
    '''
    Factory for creating families of related settings dialogs
    @type parent: QWidget
    '''
    if __MAEMO:
        return MaemoSettingsDialog(__defaultParent)
    return DesktopSettingsDialog(__defaultParent)

def createEditLabelDialog(node):
    '''
    Factory for creating families of related edge label editing dialogs
    @type node: Node
    '''
    if __MAEMO:
        return MaemoEditLabelDialog(node, __defaultParent)
    return DesktopEditLabelDialog(node, __defaultParent)

def createNetworkConnectionDialog(credentials, client):
    '''
    Factory for creating families of related network connection dialogs
    '''
    if __MAEMO:
        return MaemoNetworkConnectionDialog(credentials, client, __defaultParent)
    return DesktopNetworkConnectionDialog(credentials, client, __defaultParent)

def createSideSelectionDialog():
    '''
    Factory for creating families of related side selection dialogs
    '''
    return SideSelectionDialog(__defaultParent)

def createAffiliationDialog():
    '''
    Factory for creating families of related side affiliation dialogs
    '''
    if __MAEMO:
        return MaemoAffiliationDialog(__defaultParent)
    return DesktopAffiliationDialog(__defaultParent)

def createExtendedMenuDialog(actionBag, freezeTime = 0):
    '''
    Factory for creating families of related size extended menu dialogs
    @type actionBag: ActionBag
    @type freezeTime: int
    '''
    if __MAEMO:
        return MaemoExtendedMenuDialog(actionBag, freezeTime, __defaultParent)
    assert None
