# -*- coding: utf-8 -*-
# HiveMind - Distributed mind map editor for Maemo 5 platform
# Copyright (C) 2010-2011 HiveMind developers
#
# HiveMind is the legal property of its developers, whose names are
# noticed in @author or @authors annotations at the beginning of each
# module or class.
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02111-1301  USA

'''
GUI factories

@author: Alexander Kulikov
'''

from hivemind.desktop.main_window import *
from hivemind.maemo.main_window import *
from hivemind.desktop.dialogs import *
from hivemind.maemo.dialogs import *

__MAEMO = True
'''True if the application is run on Maemo platform'''

try:
    from PySide.QtMaemo5 import *
except ImportError:
    __MAEMO = None

''' Default parent for all created dialogs '''
__defaultParent = None

def defaultParent(parent = None):
    '''
    @return: default parent for all the dialog if no parent is passed, or link to the
    parent
    @param parent: proposable parent instead of the default one
    @type parent: QWidget
    @rtype: QWidget
    '''
    global __defaultParent
    if parent is None:
        return __defaultParent
    return parent

def createMainWindow(actionBag, mindMapScene):
    '''
    Factory for creating families of related main windows
    @type actionBag: ActionBag
    @type mindMapScene: MindMapDelegate
    '''
    global __defaultParent
    if __MAEMO:
        __defaultParent = MaemoMainWindow(actionBag, mindMapScene)
    else:
        __defaultParent = DesktopMainWindow(actionBag, mindMapScene)
    return __defaultParent

def createEditNodeDialog(node, parent = None):
    '''
    Factory for creating families of related node editing dialogs
    @type node: Node
    @type parent: QWidget
    '''
    parent = defaultParent(parent)
    if __MAEMO:
        return MaemoEditNodeDialog(node, parent)
    return DesktopEditNodeDialog(node, parent)

def createEditEdgeDialog(node, parent = None):
    '''
    Factory for creating families of related edge editing dialogs
    @type node: Node
    @type parent: QWidget
    '''
    parent = defaultParent(parent)
    if __MAEMO:
        return MaemoEditEdgeDialog(node, parent)
    return DesktopEditEdgeDialog(node, parent)

def createIconDialog(node, parent = None):
    '''
    Factory for creating families of related icon dialogs
    @type node: Node
    @type parent: QWidget
    '''
    parent = defaultParent(parent)
    if __MAEMO:
        return MaemoIconDialog(node, parent)
    return DesktopIconDialog(node, parent)

def createAboutDialog(parent = None):
    '''
    Factory for creating families of related about dialogs
    @type parent: QWidget
    '''
    parent = defaultParent(parent)
    if __MAEMO:
        return MaemoAboutDialog(parent)
    return DesktopAboutDialog(parent)

def createSettingsDialog(parent = None):
    '''
    Factory for creating families of related settings dialogs
    @type parent: QWidget
    '''
    parent = defaultParent(parent)
    if __MAEMO:
        return MaemoSettingsDialog(parent)
    return DesktopSettingsDialog(parent)

def createEditLabelDialog(node, parent = None):
    '''
    Factory for creating families of related edge label editing dialogs
    @type node: Node
    @type parent: QWidget
    '''
    parent = defaultParent(parent)
    if __MAEMO:
        return MaemoEditLabelDialog(node, parent)
    return DesktopEditLabelDialog(node, parent)

def createXMPPCredentialsDialog(client, parent = None):
    '''
    Factory for creating families of related xmpp credentials dialogs
    @type parent: QWidget
    '''
    parent = defaultParent(parent)
    if __MAEMO:
        return MaemoXMPPCredentialsDialog(client, parent)
    return DesktopXMPPCredentialsDialog(client, parent)

def createSideSelectionDialog(parent = None):
    '''
    Factory for creating families of related side selection dialogs
    @type parent: QWidget
    '''
    parent = defaultParent(parent)
    return SideSelectionDialog(parent)

def createAffiliationDialog(parent = None):
    '''
    Factory for creating families of related side affiliation dialogs
    @type parent: QWidget
    '''
    if __MAEMO:
        return MaemoAffiliationDialog(parent)
    return DesktopAffiliationDialog(parent)

def createExtendedMenuDialog(actionBag, freezeTime = 0, parent = None):
    '''
    Factory for creating families of related size extended menu dialogs
    @type actionBag: ActionBag
    @type parent: QWidget
    @type freezeTime: int
    '''
    if __MAEMO:
        return MaemoExtendedMenuDialog(actionBag, freezeTime, parent)
    return None
