# -*- coding: utf-8 -*-
# HiveMind - Distributed mind map editor for Maemo 5 platform
# Copyright (C) 2010-2011 HiveMind developers
#
# HiveMind is the legal property of its developers, whose names are
# noticed in @author or @authors annotations at the beginning of each
# module or class.
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02111-1301  USA


'''
Abstract dialog classes
'''

import hivemind.settings as settings
import resource
from PySide.QtGui import *
from PySide.QtCore import *
from hivemind.attribute import *
from hivemind.gui_widgets import *

class CommonDialog(SearchingTrMixin, QDialog):
    '''
    Abstract dialog containing OK and Cancel buttons

    @author: Alexander Kulikov
    '''
    def __init__(self, parent = None):
        '''
        Create OK and Cancel buttons and connect them to appropriate handlers
        '''
        QDialog.__init__(self, parent)
        self._buttonBox = QDialogButtonBox()
        self._okButton = self._buttonBox.addButton(QDialogButtonBox.Ok)
        self._cancelButton = self._buttonBox.addButton(QDialogButtonBox.Cancel)
        self._buttonBox.connect(SIGNAL('accepted()'), self._accept)
        self._buttonBox.connect(SIGNAL('rejected()'), self.close)

    def _accept(self):
        '''
        Some dialogs doesn't need to have _accept method
        '''

class EditNodeDialog(CommonDialog):
    '''
    Main dialog for node editing

    @author: Alexander Kulikov
    '''

    def __init__(self, node = None, parent = None):
        '''
        Initialize dialog with specific node
        @type node: Node
        '''
        CommonDialog.__init__(self, parent)
        self.setWindowTitle(self.tr('Node editor'))
        self.__node = node
        self._isHtml = self.__node.isHtml()
        self.__newAttributes = None
        self._isHtmlSourceMode = False
        self._createComponents()
        self._initEditor()

    def _createComponents(self):
        '''Create components and actions'''
        # init edit widget
        self._editor = SpecialTextEdit()
        self.connect(self._editor, SIGNAL('accept'), self._accept)
        # init html widget
        self._htmlEditor = SpecialTextEdit()
        # create font toolbar
        self._fontToolbar = QToolBar()
        self.__comboFont = QFontComboBox()
        self.__comboSize = QComboBox()
        self.__comboSize.setEditable(True)
        self.__comboFont.connect(SIGNAL('currentFontChanged(QFont)'), self._buildFont)
        self.__comboSize.connect(SIGNAL('currentIndexChanged(int)'), self._buildFont)
        self._fontToolbar.addWidget(self.__comboFont)
        self._fontToolbar.addWidget(self.__comboSize)
        # text format actions
        self._boldAction = QAction(self)
        self._boldAction.setCheckable(True)
        self._italicAction = QAction(self)
        self._italicAction.setCheckable(True)
        self._underlineAction = QAction(self)
        self._underlineAction.setCheckable(True)
        # font action
        self._textFontAction = QAction(self)
        # HTML action
        self._textHtmlAction = QAction(self)
        self._textHtmlAction.setCheckable(True)
        # color actions
        self._colorAction = QAction(self)
        self._backgroundAction = QAction(self)
        # clear format action
        self._clearFormatAction = QAction(self)
        # actions signals connection
        self._boldAction.connect(SIGNAL('triggered(bool)'), self._setBold)
        self._italicAction.connect(SIGNAL('triggered(bool)'), self._setItalic)
        self._underlineAction.connect(SIGNAL('triggered(bool)'), self._setUnderline)
        self._colorAction.connect(SIGNAL('triggered()'), self._textColor)
        self._backgroundAction.connect(SIGNAL('triggered()'), self._textBackground)
        self._textHtmlAction.connect(SIGNAL('triggered()'), self._textHtml)
        self._clearFormatAction.connect(SIGNAL('triggered()'), self._clearFormat)
        self._textFontAction.connect(SIGNAL('triggered()'), self._textFont)
        self._editor.connect(SIGNAL('currentCharFormatChanged(QTextCharFormat)'),
                             self._currentCharFormatChanged)

    def _currentCharFormatChanged(self, format):
        '''
        Update tool bar if the current character format has changed
        @type format: QTextCharFormat
        '''
        self._fontChanged(format.font())

    def _fontChanged(self, font):
        '''
        Update buttons if the current character format has changed
        @type font: QFont
        '''
        self._boldAction.setChecked(font.bold())
        self._italicAction.setChecked(font.italic())
        self._underlineAction.setChecked(font.underline())
        self.__comboFont.blockSignals(True)
        self.__comboSize.blockSignals(True)
        self.__comboFont.setCurrentFont(font)
        self.__comboSize.clear()
        self.__comboSize.addItems([str(value) for value in QFontDatabase().\
                                   pointSizes(self.__comboFont.currentFont().family())])
        index = self.__comboSize.findText(str(font.pointSize()), Qt.MatchExactly)
        if index == -1:
            self.__comboSize.setEditText(str(font.pointSize()))
        else: self.__comboSize.setCurrentIndex(index)
        self.__comboFont.blockSignals(False)
        self.__comboSize.blockSignals(False)

    def _accept(self):
        '''
        Hide the dialog, update attributes of the Node
        and set result code to Accepted
        '''
        storeAttrs = {}
        if self._isHtml:
            if self._isHtmlSourceMode:
                storeAttrs['html'] = self._htmlEditor.toPlainText()
            else:
                storeAttrs['html'] = self._editor.toHtml()
        else:
            storeAttrs['font'] = self.__font
            storeAttrs['text'] = self._editor.toPlainText()
        storeAttrs['color'] = self.__color
        storeAttrs['background_color'] = self.__background
        self.__newAttributes = storeAttrs
        QDialog.accept(self)

    def attributes(self):
        '''
        Get dict of modified attributes with new values
        @return: new attribute values
        @rtype: dict
        '''
        return self.__newAttributes

    def _initEditor(self):
        '''Initialize text editor'''
        self._textHtmlAction.setChecked(self._isHtml)
        if self.__node.isHtml():
            self._editor.setHtml(self.__node.text)
            self.__color = self.__font = None
        else:
            self._editor.setPlainText(self.__node.text)
            self._setFont(self.__node.font)
            self._setColor(self.__node.color)
        self._editor.document().setDefaultFont(settings.get('defaultFont'))
        self._fontChanged(self._editor.textCursor().charFormat().font())
        self._setBackgroundColor(self.__node.background_color)
        self._editor.selectAll()

    def _setHtmlSourceMode(self, isHtmlSourceMode):
        '''
        Set html source mode
        @type isHtmlSourceMode: bool
        '''
        self._isHtmlSourceMode = isHtmlSourceMode

    def _textColor(self):
        '''Choose font color'''
        color = QColorDialog.getColor(settings.get('defaultNodeTextColor') \
                                      if self.__color is None else self.__color,
                                      self)
        if color.isValid():
            self._setColor(color)

    def _textBackground(self):
        '''Choose background text color'''
        color = QColorDialog.getColor(settings.get('defaultNodeBackgroundColor') \
                                      if self.__background is None else self.__background,
                                      self)
        if color.isValid():
            self._setBackgroundColor(color)

    def _textFont(self):
        '''Edit current font'''
        fontDialog = FontDialog(settings.get('defaultFont') \
                                if self.__font is None else self.__font, self)
        if fontDialog.exec_() == QDialog.Accepted:
            self._setFont(fontDialog.font)

    def _textHtml(self):
        '''Switch between html editor and simple editor'''
        self._isHtml = not self._isHtml
        if not self._isHtml:
            self._editor.setPlainText(self._editor.toPlainText())
            if self.__font is not None:
                self.__font = self._editor.currentFont()

    def _setBold(self, checked):
        '''
        Update text editor if Bold button has triggered
        '''
        if self.__font is None:
            self.__font = self._editor.currentFont()
        self.__font.setBold(checked)
        fmt = QTextCharFormat()
        fmt.setFontWeight(QFont.Bold if checked else QFont.Normal)
        self._mergeFormat(fmt)

    def _setItalic(self, checked):
        '''
        Update text editor if Italic button has triggered
        '''
        if self.__font is None:
            self.__font = self._editor.currentFont()
        self.__font.setItalic(checked)
        fmt = QTextCharFormat()
        fmt.setFontItalic(checked)
        self._mergeFormat(fmt)

    def _setUnderline(self, checked):
        '''
        Update text editor if Underline button has triggered
        '''
        if self.__font is None:
            self.__font = self._editor.currentFont()
        self.__font.setUnderline(checked)
        fmt = QTextCharFormat()
        fmt.setFontUnderline(checked)
        self._mergeFormat(fmt)

    def _setBackgroundColor(self, color):
        '''
        Set node's background color
        @type color: QColor
        '''
        palette = QPalette(self._editor.palette())
        self.__background = color
        palette.setColor(QPalette.Base, settings.get('defaultNodeBackgroundColor') \
                         if color is None else color)
        self._editor.setPalette(palette)

    def _setColor(self, color):
        '''
        Set node text color
        @type color: QColor
        '''
        self.__color = color
        fmt = QTextCharFormat()
        fmt.setForeground(QBrush(settings.get('defaultNodeTextColor') \
                                 if color is None else color))
        self._mergeFormat(fmt)

    def _setFont(self, font):
        '''
        Set node text font
        @type font: QFont
        '''
        self.__font = font
        fmt = QTextCharFormat()
        fmt.setFont(settings.get('defaultFont') if font is None else font)
        self._mergeFormat(fmt)

    def _buildFont(self, value):
        '''Build font'''
        font = self.__comboFont.currentFont()
        font.setPointSize(int(self.__comboSize.currentText()))
        font.setBold(self._boldAction.isChecked())
        font.setItalic(self._italicAction.isChecked())
        font.setUnderline(self._underlineAction.isChecked())
        self._setFont(font)

    def _clearFormat(self):
        '''Clear font, color and background of the node'''
        self._setFont(None)
        self._setColor(None)
        self._setBackgroundColor(None)

    def _mergeFormat(self, format):
        '''
        Merge current format in text editor and new format
        @type format: QTextCharFormat
        '''
        cursor = self._editor.textCursor()
        if not self._isHtml:
            cursor.select(QTextCursor.Document)
        elif not cursor.hasSelection():
            cursor.select(QTextCursor.WordUnderCursor)
        cursor.mergeCharFormat(format)
        self._editor.mergeCurrentCharFormat(format)


class FontDialog(CommonDialog):
    '''
    Font selection dialog

    @author: Alexander Kulikov
    '''
    readable('font')
    def __init__(self, font, parent = None):
        '''
        Initialize widget with specified font
        @type font: QFont
        @type parent: QWidget
        '''
        CommonDialog.__init__(self, parent)
        self.setWindowFlags(Qt.Window)
        self.setAttribute(Qt.WA_Maemo5StackedWindow)
        self._initComponents(font)

    def _initComponents(self, font):
        self.__comboFont = QFontComboBox()
        self.__comboFont.setCurrentFont(font)
        self.__comboSize = QComboBox()
        self.__comboSize.addItems([str(value) for value in QFontDatabase().pointSizes(self.__comboFont.currentFont().family())])
        self.__comboSize.setEditable(True)
        index = self.__comboSize.findText(str(font.pointSize()), Qt.MatchExactly)
        if index == -1:
            self.__comboSize.setEditText(str(font.pointSize()))
        else: self.__comboSize.setCurrentIndex(index)
        self.__sampleText = QLineEdit(self.tr('ABCDE abcde'))
        self.__sampleText.setFrame(False)
        self.__sampleText.setFont(QFont(self.__comboFont.currentFont().family(), int(self.__comboSize.currentText())))
        self.__sampleText.setAlignment(Qt.AlignHCenter | Qt.AlignVCenter)
        layout = QGridLayout()
        layout.addWidget(self.__comboFont, 0, 0, 1, 3)
        layout.addWidget(self.__comboSize, 0, 3)
        scalingWidget = ScalingWidget(self.__sampleText, 2.0)
        layout.addWidget(scalingWidget, 1, 0, 1, 4)
        layout.addWidget(self._buttonBox, 2, 0, 1, 4)
        self.setLayout(layout)
        self.__comboFont.connect(SIGNAL('currentFontChanged(QFont)'), self._setLabelFont)
        self.__comboSize.connect(SIGNAL('currentIndexChanged(int)'), self._setLabelSize)
        self._setLabelFormat()

    def _accept(self):
        '''
        Hide the dialog and set result code to Accepted
        '''
        QDialog.accept(self)

    def _setLabelFormat(self):
        '''Update sample text font'''
        self.__font = self.__comboFont.currentFont()
        self.__font.setPointSize(int(self.__comboSize.currentText()))
        self.__sampleText.setFont(self.__font)

    def _setLabelFont(self, font):
        '''
        Update sample text font
        @type font: QFont
        '''
        self._setLabelFormat()

    def _setLabelSize(self, index):
        '''
        Update sample text font
        @type index : int
        '''
        self._setLabelFormat()

class EditLabelDialog(CommonDialog):
    '''
    Dialog for edge label editing

    @author: Oleg Kandaurov
    '''
    def __init__(self, node = None, parent = None):
        '''
        Initialize dialog with specific node

        @type node: Node
        '''
        CommonDialog.__init__(self, parent)
        self.setWindowTitle(self.tr('Edit Label'))
        self.__node = node
        self._createComponents()
        self._initEditor()

    def _createComponents(self):
        '''Create components and actions'''
        self._editor = SpecialTextEdit()
        self._boldAction = QAction(self)
        self._boldAction.setCheckable(True)
        self._italicAction = QAction(self)
        self._italicAction.setCheckable(True)
        self._underlineAction = QAction(self)
        self._underlineAction.setCheckable(True)
        self._comboFont = QFontComboBox()
        self._comboSize = QComboBox()
        self.connect(self._editor, SIGNAL('accept'), self._accept)
        self._comboFont.connect(SIGNAL('currentFontChanged(QFont)'), self._setFont)
        self._comboSize.connect(SIGNAL('currentIndexChanged(int)'), self._setSize)
        self._colorParent = QCheckBox(self.tr("Parent's color"))
        self._colorButton = QPushButton(self.tr('Choose color'))
        self._fontParent = QCheckBox(self.tr("Parent's font"))
        self._boldAction.connect(SIGNAL('triggered(bool)'), self._setBold)
        self._italicAction.connect(SIGNAL('triggered(bool)'), self._setItalic)
        self._underlineAction.connect(SIGNAL('triggered(bool)'), self._setUnderline)
        self._colorButton.connect(SIGNAL('clicked()'), self._textColor)
        self._colorParent.connect(SIGNAL('toggled(bool)'), self._colorButton,
                                  SLOT('setDisabled(bool)'))
        self._colorParent.connect(SIGNAL('toggled(bool)'), self._updateEditor)
        self._fontParent.connect(SIGNAL('toggled(bool)'), self._comboFont,
                                 SLOT('setDisabled(bool)'))
        self._fontParent.connect(SIGNAL('toggled(bool)'), self._comboSize,
                                 SLOT('setDisabled(bool)'))
        self._fontParent.connect(SIGNAL('toggled(bool)'), self._boldAction,
                                 SLOT('setDisabled(bool)'))
        self._fontParent.connect(SIGNAL('toggled(bool)'), self._italicAction,
                                 SLOT('setDisabled(bool)'))
        self._fontParent.connect(SIGNAL('toggled(bool)'), self._underlineAction,
                                 SLOT('setDisabled(bool)'))
        self._fontParent.connect(SIGNAL('toggled(bool)'), self._updateEditor)

    def _initEditor(self):
        '''Initialize editor'''
        self._editor.setPlainText(self.__node.labelText)
        self._editor.document().setDefaultFont(settings.get('defaultLabelFont'))
        self.__font = self.__node.labelFont
        self._setEditorFont(self.__node.labelFont)
        self.__color = self.__node.labelColor
        self._setEditorColor(self.__node.labelColor)
        if self.__node.labelColor is None:
            self._colorParent.setChecked(True)
        if self.__node.labelFont is None:
            self._fontParent.setChecked(True)
        self._comboSize.setEditable(True)
        font = self.__node.effectiveLabelFont
        self._comboFont.setCurrentFont(font)
        self._editor.selectAll()

    def _fontSizes(self, font):
        '''
        Add to comboSize sizes supported by current font

        @type font: QFont
        '''
        self._comboSize.clear()
        self._comboSize.addItems([str(value) for value in QFontDatabase().\
                                   pointSizes(font.family())])

    def _setSizeIndex(self, font):
        '''
        Set font size in comboSize

        @type font: QFont
        '''
        index = self._comboSize.findText(str(font.pointSize()), Qt.MatchExactly)
        if index == -1:
            self._comboSize.setEditText(str(font.pointSize()))
        else:
            self._comboSize.setCurrentIndex(index)

    def _updateButtonsState(self, font):
        '''
        Update state of buttons according to font

        @type font: QFont
        '''
        self._boldAction.setChecked(font.bold())
        self._italicAction.setChecked(font.italic())
        self._underlineAction.setChecked(font.underline())

    def _setFont(self, font):
        '''
        Set and update font

        @type font: QFont
        '''
        if font is None:
            font = self.__node.effectiveLabelFont
        if self._comboSize.currentText() != '':
            font.setPointSize(int(self._comboSize.currentText()))
        self._fontSizes(font)
        self._setSizeIndex(font)
        self._setEditorFont(font)
        self._updateButtonsState(font)
        # don't save current formatting if parent button checked
        if not self._fontParent.isChecked():
            self.__font = font

    def _setSize(self, index):
        '''
        Set and update size of font

        @type index: int
        '''
        font = self._comboFont.currentFont()
        if index != -1:
            font.setPointSize(int(self._comboSize.itemText(index)))
        self._setEditorFont(font)
        self._updateButtonsState(font)
        # don't save current formatting if parent button checked
        if not self._fontParent.isChecked():
            self.__font = font

    def _setBold(self, checked):
        '''
        Update text editor if Bold button has triggered
        '''
        fmt = QTextCharFormat()
        fmt.setFontWeight(QFont.Bold if checked else QFont.Normal)
        self._mergeFormat(fmt)
        # don't save current formatting if parent button checked
        if not self._fontParent.isChecked():
            self.__font.setBold(checked)

    def _setItalic(self, checked):
        '''
        Update text editor if Italic button has triggered
        '''
        fmt = QTextCharFormat()
        fmt.setFontItalic(checked)
        self._mergeFormat(fmt)
        # don't save current formatting if parent button checked
        if not self._fontParent.isChecked():
            self.__font.setItalic(checked)

    def _setUnderline(self, checked):
        '''
        Update text editor if Underline button has triggered
        '''
        fmt = QTextCharFormat()
        fmt.setFontUnderline(checked)
        self._mergeFormat(fmt)
        # don't save current formatting if parent button checked
        if not self._fontParent.isChecked():
            self.__font.setUnderline(checked)

    def _textColor(self):
        '''Choose font color'''
        color = QColorDialog.getColor(self.__node.effectiveLabelColor, self)
        if color.isValid():
            self.__color = color
            self._setEditorColor(color)

    def _updateEditor(self):
        '''
        Update text editor according to "parent" buttons
        '''
        if self._colorParent.isChecked():
            self._setEditorColor(None)
        else:
            self._setEditorColor(self.__color)
        if self._fontParent.isChecked():
            font = self.__node.effectiveLabelFont
        else:
            font = self.__font
            if font is None:
                font = self.__node.effectiveLabelFont
        self._fontSizes(font)
        self._setSizeIndex(font)
        self._comboFont.setCurrentFont(font)

    def _setEditorColor(self, color):
        '''
        Update text color of text editor

        @type color: QColor
        '''
        fmt = QTextCharFormat()
        newColor = color if color else self.__node.effectiveLabelColor
        fmt.setForeground(QBrush(newColor))
        self._mergeFormat(fmt)

    def _setEditorFont(self, font):
        '''
        Update text font of text editor

        @type font: QFont
        '''
        fmt = QTextCharFormat()
        newFont = font if font else self.__node.effectiveLabelFont
        fmt.setFont(newFont)
        self._mergeFormat(fmt)

    def _mergeFormat(self, format):
        '''
        Merge current format in text editor and new format

        @type format: QTextCharFormat
        '''
        cursor = self._editor.textCursor()
        cursor.select(QTextCursor.Document)
        cursor.mergeCharFormat(format)
        self._editor.mergeCurrentCharFormat(format)

    def _accept(self):
        '''
        Hide the dialog, update attributes of the label
        and set result code to Accepted
        '''
        modified = {}
        if self._colorParent.isChecked():
            self.__color = None
        else:
            if self.__color is None:
                self.__color = self.__node.effectiveLabelColor
        if self._fontParent.isChecked():
            self.__font = None
        else:
            if self.__font is None:
                self.__font = self.__node.effectiveLabelFont
        modified['labelColor'] = self.__color
        modified['labelFont'] = self.__font
        modified['labelText'] = self._editor.toPlainText().replace('\n', ' ')
        self.__newAttributes = modified
        QDialog.accept(self)

    def attributes(self):
        '''
        Get dict of modified attributes with new values

        @return: new attribute values
        @rtype: dict
        '''
        return self.__newAttributes


class IconDialog(CommonDialog):
    '''
    Abstract class for icon selection dialog

    @author: Alexander Kulikov
    '''

    def __init__(self, node, parent = None):
        '''
        Initialize with specific node
        @type node: Node
        @type parent: QWidget
        '''
        CommonDialog.__init__(self, parent)
        self.setWindowTitle(self.tr('Icon selector'))
        self.__node = node
        self._createComponents()

    def _createComponents(self):
        '''Create components'''
        if len(self.__node.icons) > IconList.MAX_ICON_COUNT:
            self._listScene = IconList(self.__node.icons[:10], self)
        else: self._listScene = IconList(self.__node.icons, self)
        self._iconTable = IconTable(self)
        self._iconTable.setIconSize(QSize(64, 64))
        self._iconList = QGraphicsView(self._listScene, self)
        self._iconList.setHorizontalScrollBarPolicy(Qt.ScrollBarAlwaysOff)
        self._iconList.setVerticalScrollBarPolicy(Qt.ScrollBarAlwaysOff)
        self._iconTable.connect(SIGNAL('iconClicked'), \
                                 adaptCallable(self._listScene.addIcon))

    def _accept(self):
        '''
        Hide the dialog, update attributes of the Node
        and set result code to Accepted
        '''
        icons = self._iconList.scene().getIcons()
        if len(self.__node.icons) > IconList.MAX_ICON_COUNT:
            icons.extend(self.__node.icons[10:])
        self.__newAttributes = {'icons' : icons}
        QDialog.accept(self)

    def attributes(self):
        '''
        Get dict of modified attributes with new values
        @return: new attribute values
        @rtype: dict
        '''
        return self.__newAttributes


class AboutDialog(CommonDialog):
    '''
    Abstract class for about dialog

    @author: Oleg Kandaurov, Alexander Kulikov
    '''
    def __init__(self, parent = None):
        '''
        Constructor
        @type parent: QWidget
        '''
        CommonDialog.__init__(self, parent)
        self.setWindowTitle(self.tr('About HiveMind'))
        self._createComponents()

    def _createComponents(self):
        self._logo = QLabel()
        self._logoImage = resource.getImage('logo')
        self._info = QLabel(self.tr("Collaborative mind map editor"))


class EditEdgeDialog(CommonDialog):

    '''
    Abstract class for edit edge dialog
    @author: Oleg Kandaurov
    '''
    def __init__(self, node = None, parent = None):
        '''
        Initialize dialog with specific node
        @type node: Node
        '''
        CommonDialog.__init__(self, parent)
        self.setWindowTitle(self.tr('Edge editor'))
        self.__node = node
        self._createComponents()
        self._initEditor()

    def _createComponents(self):
        '''Create components'''
        # Setting up width view
        self._widthLabel = QLabel(self.tr('Width'))
        self._widthParent = QRadioButton(self.tr('Parent'))
        self._widthThin = QRadioButton(self.tr('Thin'))
        self._widthOne = QRadioButton('1')
        self._widthTwo = QRadioButton('2')
        self._widthFour = QRadioButton('4')
        self._widthEight = QRadioButton('8')
        widthGroup = QButtonGroup(self)
        widthGroup.addButton(self._widthParent)
        widthGroup.addButton(self._widthThin)
        widthGroup.addButton(self._widthOne)
        widthGroup.addButton(self._widthTwo)
        widthGroup.addButton(self._widthFour)
        widthGroup.addButton(self._widthEight)
        # Setting up style view
        self._styleLabel = QLabel(self.tr('Style'))
        self._styleParent = QRadioButton(self.tr('Parent'))
        self._styleLinear = QRadioButton(self.tr('Linear'))
        self._styleBezier = QRadioButton(self.tr('Bezier'))
        self._styleSharpLinear = QRadioButton(self.tr('Sharp linear'))
        self._styleSharpBezier = QRadioButton(self.tr('Sharp bezier'))
        styleGroup = QButtonGroup(self)
        styleGroup.addButton(self._styleParent)
        styleGroup.addButton(self._styleLinear)
        styleGroup.addButton(self._styleBezier)
        styleGroup.addButton(self._styleSharpLinear)
        styleGroup.addButton(self._styleSharpBezier)
        # Setting up color view
        self._colorLabel = QLabel(self.tr('Color'))
        self._colorParent = QCheckBox(self.tr('Parent'))
        self._colorButton = QPushButton(self.tr('Choose color'))
        self._colorButton.connect(SIGNAL('clicked()'), self._edgeColor)
        self._colorParent.connect(SIGNAL('toggled(bool)'), self._colorButton,
                                    SLOT('setDisabled(bool)'))

    def _initEditor(self):
        '''Initialize editor'''
        # Initialize color of the Edge
        self.__color = self.__node.edgeColor
        if self.__color is None:
            self._colorParent.setChecked(True)
        # Initialize width of the Edge
        self.__widthToButton = {1 : self._widthOne, 2 : self._widthTwo,
                         4 : self._widthFour, 8: self._widthEight,
                         11 : self._widthThin,
                         None : self._widthParent}
        self.__widthToButton[self.__node.edgeWidth].setChecked(True)
        # Initialize style of the Edge
        self.__styleToButton = {'linear' : self._styleLinear,
                          'bezier' : self._styleBezier,
                          'sharp_linear' : self._styleSharpLinear,
                          'sharp_bezier' : self._styleSharpBezier,
                          None : self._styleParent}
        self.__styleToButton[self.__node.edgeStyle].setChecked(True)

    def _edgeColor(self):
        '''Choose color of the Edge'''
        color = QColorDialog.getColor(self.__node.effectiveEdgeColor, self)
        if color.isValid():
            self.__color = color

    def _accept(self):
        '''
        Hide the dialog, update attributes of the Edge
        and set result code to Accepted
        '''
        modified = {}
        if self._colorParent.isChecked():
            self.__color = None
        elif self.__color is None:
            self.__color = self.__node.effectiveEdgeColor
        modified['edgeColor'] = self.__color
        for width, button in self.__widthToButton.iteritems():
            if button.isChecked():
                modified['edgeWidth'] = width
        for style, button in self.__styleToButton.iteritems():
            if button.isChecked():
                modified['edgeStyle'] = style
        self.__newAttributes = modified
        QDialog.accept(self)

    def attributes(self):
        '''
        Get dict of modified attributes with new values
        @return: new attribute values
        @rtype: dict
        '''
        return self.__newAttributes

class SettingsDialog(CommonDialog):
    '''
    Abstract class for settings dialog
    @author: Oleg Kandaurov
    '''
    def __init__(self, parent = None):
        '''
        @type parent: QWidget
        '''
        CommonDialog.__init__(self, parent)
        self.setWindowTitle(self.tr('Settings'))
        self._newSettings = list()

    def _accept(self):
        '''
        Apply new settings
        '''
        oldLocale = settings.get('locale')
        newSettings = {}
        for group in self._newSettings:
            for setting in group:
                if setting: newSettings.update(setting)
        settings.set(**newSettings)
        if oldLocale != settings.get('locale'):
            QMessageBox.information(self, self.tr('Settings changed'),
                    self.tr('The language settings will be applied on restart of the application!'))
        QDialog.accept(self)

class XMPPCredentialsDialog(CommonDialog):
    '''
    Abstract class for xmpp credentials dialog
    @author: Oleg Kandaurov
    '''
    def __init__(self, client = False, parent = None):
        '''
        @type parent: QWidget
        '''
        CommonDialog.__init__(self, parent)
        self.__client = client
        self.setWindowTitle(self.tr('XMPP Credentials'))
        self._createComponents()
        self._initComponents()

    def _createComponents(self):
        '''
        Create Components
        '''
        self.__newAttributes = dict()
        mailRegexp = QRegExp('[A-Za-z0-9._%+-]+@(?:[A-Za-z0-9-]+\\.)+[A-Za-z]{2,6}')
        self._userBinder = LineEditBinder('userJid', settings.get('userJid'), mailRegexp)
        self._userLabel = QLabel(self.tr('User JID:'))
        self._passwordBinder = LineEditBinder('password', '')
        self._passwordBinder.setEchoMode(QLineEdit.Password)
        self._passwordLabel = QLabel(self.tr('Password:'))
        self._serviceBinder = LineEditBinder('serviceJid', settings.get('serviceJid'), mailRegexp)
        self._serviceLabel = QLabel(self.tr('Service JID:'))
        self._readOnlyBinder = CheckBoxBinder('readOnly', settings.get('readOnly'))
        self._readOnlyBinder.setText(self.tr('Read only'))

    def _initComponents(self):
        mainLayout = QVBoxLayout()
        formLayout = QFormLayout()
        formLayout.addRow(self._userLabel, self._userBinder)
        formLayout.addRow(self._passwordLabel, self._passwordBinder)
        if self.__client:
            formLayout.addRow(self._serviceLabel, self._serviceBinder)
        else:
            formLayout.addRow(self._readOnlyBinder)
        mainLayout.addLayout(formLayout)
        mainLayout.addWidget(self._buttonBox)
        self.setLayout(mainLayout)

    def _accept(self):
        '''
        Hide the dialog, update attributes
        set result code to Accepted and save new jids
        '''
        if not self._serviceBinder.isValid() and self.__client or not self._userBinder.isValid():
            QMessageBox.critical(self, self.tr('Incorrect JID!'),
                    self.tr('Please, enter correct jabber id.'))
            return
        self.__newAttributes['userJid'] = self._userBinder.retrieve().values().pop()
        if self.__client:
            self.__newAttributes['serviceJid'] = self._serviceBinder.retrieve().values().pop()
        else:
            self.__newAttributes['readOnly'] = self._readOnlyBinder.retrieve().values().pop()
        settings.set(**self.__newAttributes)
        self.__newAttributes['password'] = self._passwordBinder.retrieve().values().pop()
        QDialog.accept(self)

    def attributes(self):
        '''
        Get dict of modified attributes with new values
        @return: new attribute values
        @rtype: dict
        '''
        return self.__newAttributes


class SideSelectionDialog(CommonDialog):
    '''
    Side selection dialog
    @author: Oleg Kandaurov
    '''

    readable('leftSide')

    def __init__(self, parent = None):
        '''
        @type parent: QWidget
        '''
        CommonDialog.__init__(self, parent)
        self.setWindowTitle(self.tr('Node Position'))
        self.__leftSide = None
        self._createComponents()
        self._initComponents()

    def _createComponents(self):
        '''
        Create components
        '''
        self.__label = QLabel(self.tr('Select node position'))
        self.__left = QPushButton(self.tr('Left'))
        self.__right = QPushButton(self.tr('Right'))
        self.connect(self.__left, SIGNAL('clicked()'), self._accept)
        self.connect(self.__right, SIGNAL('clicked()'), self._accept)

    def _initComponents(self):
        '''
        Init components
        '''
        buttonLayout = QHBoxLayout()
        buttonLayout.addWidget(self.__left)
        buttonLayout.addWidget(self.__right)
        mainLayout = QVBoxLayout()
        self.__label.setAlignment(Qt.AlignCenter)
        mainLayout.addWidget(self.__label)
        mainLayout.addLayout(buttonLayout)
        self.setLayout(mainLayout)

    def _accept(self):
        '''
        Hide the dialog and sets side
        '''
        self.__leftSide = True if self.sender() == self.__left else False
        QDialog.accept(self)


class AffiliationDialog(CommonDialog):
    '''
    Abstract class for affiliation dialog
    @author: Oleg Kandaurov
    '''

    def __init__(self, parent = None):
        CommonDialog.__init__(self, parent)
        self.setWindowTitle(self.tr('Edit affiliations'))
        self.__supportedStatuses = {'subscribed': self.tr('Subscribed'),
                'unsubscribed': self.tr('Unsubscribed')}
        self._createComponents()
        self._initComponents()

    def _createComponents(self):
        '''
        Create components
        '''
        self.__model = QStandardItemModel(0, 3)
        self.__delegate = AffiliationDelegate()
        self.__view = QTableView()
        self.__model.setHorizontalHeaderLabels([self.tr('JID'),
                self.tr('Affiliation'), self.tr('Status')])
        self.__view.setModel(self.__model)
        self.__view.setItemDelegateForColumn(1, self.__delegate)
        self.__jidLabel = QLabel(self.tr('JID'))
        self.__jidEdit = QLineEdit()
        self.__affiliationLabel = QLabel(self.tr('Affiliation'))
        self.__affiliationCombo = QComboBox()
        for data, name in self.__delegate.supportedAffiliations.iteritems():
            self.__affiliationCombo.addItem(name, data)
        self.__additionButton = QPushButton(self.tr('Add participant'))
        self.__additionButton.clicked.connect(self._addParticipant)

    def _initComponents(self):
        '''
        Initialize components
        '''
        jidLayout = QVBoxLayout()
        self.__jidLabel.setAlignment(Qt.AlignCenter)
        jidLayout.addWidget(self.__jidLabel)
        jidLayout.addWidget(self.__jidEdit)
        affilationLayout = QVBoxLayout()
        self.__affiliationLabel.setAlignment(Qt.AlignCenter)
        affilationLayout.addWidget(self.__affiliationLabel)
        affilationLayout.addWidget(self.__affiliationCombo)
        buttonLayout = QVBoxLayout()
        buttonLayout.addWidget(QLabel(''))
        buttonLayout.addWidget(self.__additionButton)
        additionLayout = QHBoxLayout()
        additionLayout.setAlignment(Qt.AlignBottom)
        additionLayout.addLayout(jidLayout)
        additionLayout.addLayout(affilationLayout)
        additionLayout.addLayout(buttonLayout)
        mainLayout = QVBoxLayout()
        mainLayout.addWidget(self.__view)
        mainLayout.addLayout(additionLayout)
        mainLayout.addWidget(self._buttonBox)
        self.setLayout(mainLayout)

    def _addParticipant(self):
        '''
        Retrieve info from components and create participant with given access
        '''
        jidRegexp = QRegExp('[A-Za-z0-9._%+-]+@(?:[A-Za-z0-9-]+\\.)+[A-Za-z]{2,6}')
        if not jidRegexp.exactMatch(self.__jidEdit.text()):
            QMessageBox.critical(self, self.tr('Incorrect JID!'),
                    self.tr('Please, enter correct jabber id.'))
            return
        self.setAffiliation(self.__jidEdit.text(),
                self.__affiliationCombo.itemData(self.__affiliationCombo.currentIndex()))
        self.__jidEdit.clear()

    def updateStatus(self, event, subscriber, node, optional):
        '''
        Update status of widgets according to received event
        @param event: Type of event
        @type event: str
        @param subscriber: Subcriber that generates event
        @type subscriber: str
        @param node: Node from which event received
        @type node: str
        @param optional: Subscribers count
        @type optional: int
        '''
        for row in range(0, self.__model.rowCount()):
            if self.__model.index(row, 0).data() == subscriber:
                self.__model.setData(self.__model.index(row, 2),
                        self.__supportedStatuses[event])

    def setAffiliation(self, jid, newAffiliation):
        '''
        Set new affiliation
        @param jid: jid to set affiliation
        @type jid: str
        @param affiliation: Affiliation for jid
        @type affiliation: str
        '''
        for row in range(0, self.__model.rowCount()):
            if self.__model.index(row, 0).data() == jid:
                self.__model.setData(self.__model.index(row, 1),
                        newAffiliation)
                return
        self.__model.setRowCount(self.__model.rowCount() + 1)
        self.__model.setData(self.__model.index(self.__model.rowCount() - 1, 2),
                self.__supportedStatuses['subscribed'])
        self.__model.setData(self.__model.index(self.__model.rowCount() - 1, 1),
                newAffiliation)
        self.__model.setData(self.__model.index(self.__model.rowCount() - 1, 0),
                jid)

    def getAffiliations(self):
        '''
        Extract affiliations from model
        @return: Dictionary of jid and affiliation
        @rtype: dict
        '''
        affiliations = {}
        for row in range(0, self.__model.rowCount()):
            affiliations[self.__model.index(row, 0).data()] = \
                    self.__model.index(row, 1).data()
        return affiliations

    def _accept(self):
        '''
        Hide the dialog
        '''
        QDialog.accept(self)
