# -*- coding: utf-8 -*-
# HiveMind - Distributed mind map editor for Maemo 5 platform
# Copyright (C) 2010-2011 HiveMind developers
#
# HiveMind is the legal property of its developers, whose names are
# noticed in @author or @authors annotations at the beginning of each
# module or class.
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02111-1301  USA

'''
Core classes
'''

import hivemind.settings as settings
import time
from PySide.QtGui import *
from PySide.QtCore import *
from hivemind.attribute import *
from sets import Set


class Element(object):
    '''
    Base class for mind map objects

    @authors: Andrew Golovchenko
    '''
    _ATTRIBUTES = ()
    '''List of element attributes'''

    def __init__(self, **attributes):
        '''
        Initialize attributes with specified values and the others to defaults

        @param attributes: dictionary of initialization attributes
        @type attributes: dict
        '''
        # class specified prefix of private field
        mangle = '_%s__' % self.__class__.__name__
        # init private fields and set it to default values
        for attr in self.__class__._ATTRIBUTES:
            self.__dict__[mangle + attr] = None
        self.initAttributes(attributes)

    def initAttributes(self, attributes):
        '''
        Set class attributes to specified values names and values

        @type attributes: dict
        '''
        # class specified prefix of private field
        mangle = '_%s__' % self.__class__.__name__
        # replace some of default values with new
        for attrName, attrValue in attributes.items():
            self.__dict__[mangle + attrName.lower()] = attrValue

    @property
    def attributes(self):
        '''
        Return dict of inner element attributes
        @rtype: dict
        '''
        elementAttrs = {}
        mangle = '_%s__' % self.__class__.__name__
        for attr in self.__class__._ATTRIBUTES:
            if self.__dict__[mangle + attr] is not None:
                elementAttrs[attr] = self.__dict__[mangle + attr]
        return elementAttrs


class MindMap(Element):
    '''
    Mind map

    @authors: Andrew Golovchenko
    '''
    writable('attributeRegistry')

    def __init__(self):
        '''Initialize mind map'''
        self.__attributeRegistry = None
        root = Node()
        root.setAttributes(text = QApplication.translate('MindMap', 'Root'))
        self.setRoot(root)

    def processNotification(self, sender, *values, **args):
        '''
        Update the id-to-node dictionary after the node addition/deletion

        @param sender: source of notification
        @type sender: Observable
        '''
        notifyType = args.get('type')
        if notifyType == 'childNodeAdded':
            self._registerSubtree(*values)
        elif notifyType == 'childNodeRemoved':
            self._unregisterSubtree(*values)

    def _registerSubtree(self, root):
        '''Register the node and its children in the id-to-node dictionary'''
        self.__idToNode[root.id] = root
        root.addObserver(self)
        for child in root.children:
            self._registerSubtree(child)

    def _unregisterSubtree(self, root):
        '''Unregister the node and its children in the id-to-node dictionary'''
        self.__idToNode.pop(root.id)
        for child in root.children:
            self._unregisterSubtree(child)

    def getNodeById(self, id):
        '''
        @return: node with the passed id attribute or None if no node found
        @rtype: Node
        '''
        return self.__idToNode.get(id)

    def getRoot(self):
        '''
        @return: root node of the mind map
        @rtype: Node
        '''
        return self.__root

    def setRoot(self, newRoot):
        '''
        Set new root for the mind map
        @param newRoot: new root
        @type newRoot: Node
        '''
        self.__root = newRoot
        self.__idToNode = {}
        self._registerSubtree(newRoot)

    root = property(getRoot, setRoot, None, "Root node of the mind map")


class Node(Element, Observable):
    '''
    Mind map node

    @authors: Ilya Paramonov, Andrew Golovchenko, Andrew Vasilev
    '''
    _ATTRIBUTES = ('created', 'id', 'modified', 'style', 'left', 'folded', 'link', 'hgap',
                'vgap', 'vshift', 'encrypted_content', 'background_color', 'color')
    '''List of node attributes'''
    readable('created', 'id', 'modified', 'encrypted_content', 'nodeAttribute', 'note',
             'parent', 'icons', 'children', 'font', 'color', 'edgeColor',
             'edgeWidth', 'edgeStyle', 'hgap', 'vgap', 'vshift', 'cloud', 'folded',
             'background_color', 'link', 'style', 'left',
             'labelText', 'labelFont', 'labelColor')

    def _setStyle(self, style):
        '''Set style value of Node'''
        self.__style = style
        self._propagateEffectiveStyle(style)

    @property
    def effectiveStyle(self):
        return self.__effectiveStyle if self.__effectiveStyle else settings.get('defaultStyle')

    def _propagateEffectiveStyle(self, style):
        '''Set style value of Node according inheritance policy'''
        if self.__effectiveStyle == style: return
        if self.__style:
            self.__effectiveStyle = self.__style
        else:
            self.__effectiveStyle = self.__parent.__effectiveStyle if self.__parent else style
        for child in self.__children:
            child._propagateEffectiveStyle(self.__effectiveStyle)

    def _setLeft(self, left):
        '''Set left position flag of Node according inheritance policy'''
        if self.__left == left: return
        self.__left = left
        for child in self.__children:
            child._setLeft(left)

    def _setLabelColor(self, color):
        '''Set label color attribute to Node'''
        self.__labelColor = color
        self._propagateEffectiveLabelColor(color)

    @property
    def effectiveLabelColor(self):
        return self.__effectiveLabelColor if self.__effectiveLabelColor else settings.get('defaultLabelColor')

    def _propagateEffectiveLabelColor(self, color):
        '''Set label color attribute to Node according inheritance policy'''
        if equal(self.__effectiveLabelColor, color): return
        if self.__labelColor:
            self.__effectiveLabelColor = self.__labelColor
        else:
            self.__effectiveLabelColor = self.__parent.__effectiveLabelColor if self.__parent else color
        if self.__effectiveLabelColor is not None:
            self.__effectiveLabelColor = QColor(self.__effectiveLabelColor)
        for child in self.__children:
            child._propagateEffectiveLabelColor(self.__effectiveLabelColor)

    def _setLabelFont(self, font):
        '''Set label font attribute to Node'''
        self.__labelFont = font
        self._propagateEffectiveLabelFont(font)

    @property
    def effectiveLabelFont(self):
        return self.__effectiveLabelFont if self.__effectiveLabelFont else settings.get('defaultLabelFont')

    def _propagateEffectiveLabelFont(self, font):
        '''Set label font attribute to Node according inheritance policy'''
        if equal(self.__effectiveLabelFont, font): return
        if self.__labelFont:
            self.__effectiveLabelFont = self.__labelFont
        else:
            self.__effectiveLabelFont = self.__parent.__effectiveLabelFont if self.__parent else font
        if self.__effectiveLabelFont is not None:
            self.__effectiveLabelFont = QFont(self.__effectiveLabelFont)
        self.__labelFontMetrics = QFontMetricsF(self.effectiveLabelFont)
        for child in self.__children:
            child._propagateEffectiveLabelFont(self.__effectiveLabelFont)

    def _setEdgeColor(self, color):
        '''Set edge color attribute to Node'''
        self.__edgeColor = color
        self._propagateEffectiveEdgeColor(color)

    @property
    def effectiveEdgeColor(self):
        return self.__effectiveEdgeColor if self.__effectiveEdgeColor else settings.get('defaultEdgeColor')

    def _propagateEffectiveEdgeColor(self, color):
        '''Set edge color attribute to Node according inheritance policy'''
        if equal(self.__effectiveEdgeColor, color): return
        if self.__edgeColor:
            self.__effectiveEdgeColor = self.__edgeColor
        else:
            self.__effectiveEdgeColor = self.__parent.__effectiveEdgeColor if self.__parent else color
        if self.__effectiveEdgeColor is not None:
            self.__effectiveEdgeColor = QColor(self.__effectiveEdgeColor)
        for child in self.__children:
            child._propagateEffectiveEdgeColor(self.__effectiveEdgeColor)

    def _setEdgeWidth(self, width):
        '''Set edge width attribute to Node'''
        self.__edgeWidth = width if width else None
        self._propagateEffectiveEdgeWidth(width)

    @property
    def effectiveEdgeWidth(self):
        edgeWidth = self.__effectiveEdgeWidth if self.__effectiveEdgeWidth else settings.get('defaultEdgeWidth')
        return float(edgeWidth % 10)

    def _propagateEffectiveEdgeWidth(self, width):
        '''Set edge width attribute to Node according inheritance policy'''
        if self.__effectiveEdgeWidth == width: return
        if self.__edgeWidth:
            self.__effectiveEdgeWidth = self.__edgeWidth
        else:
            self.__effectiveEdgeWidth = self.__parent.__effectiveEdgeWidth if self.__parent else width
        for child in self.__children:
            child._propagateEffectiveEdgeWidth(self.__effectiveEdgeWidth)

    def _setEdgeStyle(self, style):
        '''Set edge style attribute to Node'''
        self.__edgeStyle = style if style else None
        self._propagateEffectiveEdgeStyle(style)

    @property
    def effectiveEdgeStyle(self):
        return self.__effectiveEdgeStyle if self.__effectiveEdgeStyle else settings.get('defaultEdgeStyle')

    def _propagateEffectiveEdgeStyle(self, style):
        '''Set edge style attribute to Node according inheritance policy'''
        if self.__effectiveEdgeStyle == style: return
        if self.__edgeStyle:
            self.__effectiveEdgeStyle = self.__edgeStyle
        else:
            self.__effectiveEdgeStyle = self.__parent.__effectiveEdgeStyle if self.__parent else style
        for child in self.__children:
            child._propagateEffectiveEdgeStyle(self.__effectiveEdgeStyle)

    def _setFont(self, font):
        '''Set font value of Node text'''
        self.__font = font
        self._propagateEffectiveFont(font)

    @property
    def effectiveFont(self):
        return self.__effectiveFont if self.__effectiveFont else settings.get('defaultFont')

    def _propagateEffectiveFont(self, font):
        '''Set font value of Node text according inheritance policy'''
        if equal(self.__effectiveFont, font): return
        if self.__font:
            self.__effectiveFont = self.__font
        else:
            self.__effectiveFont = self.__parent.__effectiveFont if self.__parent else font
        if self.__effectiveFont is not None:
            self.__effectiveFont = QFont(self.__effectiveFont)
        self.__richcontent.setDefaultFont(self.effectiveFont)
        for child in self.__children:
            child._propagateEffectiveFont(self.__effectiveFont)

    def _setColor(self, color):
        '''Set color value of Node text content'''
        self.__color = color
        self._propagateEffectiveColor(color)

    @property
    def effectiveColor(self):
        return self.__effectiveColor if self.__effectiveColor else settings.get('defaultNodeTextColor')

    def _propagateEffectiveColor(self, color):
        '''Set color value of Node text content according inheritance policy'''
        if equal(self.__effectiveColor, color): return
        if self.__color:
            self.__effectiveColor = self.__color
        else:
            self.__effectiveColor = self.__parent.__effectiveColor if self.__parent else color
        if self.__effectiveColor is not None:
            self.__effectiveColor = QColor(self.__effectiveColor)
        self.__colorPalette.setColor(QPalette.Text, self.effectiveColor)
        for child in self.__children:
            child._propagateEffectiveColor(self.__effectiveColor)

    def _setBgColor(self, color):
        '''Set background color value of Node'''
        self.__background_color = color
        self._propagateEffectiveBgColor(color)

    @property
    def effectiveBgColor(self):
        return self.__effectiveBgColor if self.__effectiveBgColor else settings.get('defaultNodeBackgroundColor')

    def _propagateEffectiveBgColor(self, color):
        '''Set background color value of Node according inheritance policy'''
        if equal(self.__effectiveBgColor, color): return
        if self.__background_color:
            self.__effectiveBgColor = self.__background_color
        else:
            self.__effectiveBgColor = self.__parent.__effectiveBgColor if self.__parent else color
        if self.__effectiveBgColor is not None:
            self.__effectiveBgColor = QColor(self.__effectiveBgColor)
        for child in self.__children:
            child._propagateEffectiveBgColor(self.__effectiveBgColor)

    def updateProperties(self):
        '''Update cached properties of the node'''
        self.__richcontent.setDefaultFont(self.effectiveFont)
        self.__colorPalette.setColor(QPalette.Text, self.effectiveColor)
        self.__labelFontMetrics = QFontMetricsF(self.effectiveLabelFont)
        for child in self.__children:
            child.updateProperties()

    def __init__(self, **attributes):
        '''
        Initialize node attributes
        
        @param attributes: dictionary of initialization attributes
        @type attributes: dict
        '''
        Element.__init__(self, **attributes)
        Observable.__init__(self)
        # has similar name properties
        self.__parent = None
        self.__labelText = None
        self.__labelFont = None
        self.__labelFontMetrics = QFontMetricsF(settings.get('defaultLabelFont'))
        self.__effectiveLabelFont = None
        self.__labelColor = None
        self.__effectiveLabelColor = None
        self.__note = None
        self.__effectiveStyle = None
        self.__left = None
        self.__effectiveEdgeStyle = None
        self.__edgeStyle = None
        self.__effectiveEdgeColor = None
        self.__edgeColor = None
        self.__effectiveEdgeWidth = None
        self.__edgeWidth = None
        self.__cloud = None
        self.__font = None
        self.__effectiveFont = None
        self.__effectiveColor = None
        self.__effectiveBgColor = None
        self.__icons = []
        self.__nodeAttribute = None
        self.__children = []
        if self.__id is None:
            self.__created = int(time.time() * 1000.0)
            self.__modified = self.__created
#           FIXME: use integer value instead of string in new file format
            self.__id = 'ID_0%d' % id(self)
        # local fields
        self.__isHtmlContent = False
        self.__colorPalette = QPalette()
        self.__colorPalette.setColor(QPalette.Text, settings.get('defaultNodeTextColor'))
        self.__richcontent = QTextDocument()
        self.__richcontent.setDefaultFont(settings.get('defaultFont'))

    def _setField(self, field, val):
        '''Create/set private field value'''
        self.__dict__['_Node__' + field] = val

    def _getField(self, fieldName):
        '''Return private field value'''
        return self.__dict__['_Node__' + fieldName]

#    TODO: implement class
#    class _Notifier(object):
#        def __init__(self):
#            self.__maxNotifyLevel = -1 # -1 <=> no notification will be send
#        def reset(self):
#            self.__maxNotifyLevel = -1
#        def updateNotifyLevel(self, level):
#            self.__maxNotifyLevel = max(self.__maxNotifyLevel, level)
#        def notify(self):
#            # emit only one signal with max level of change
#            self.notifyObservers(self, type = _Notifier.__SIGNALS[self.__maxNotifyLevel])

    __LOCAL_ATTRIBUTES = Set(('cloud', 'folded', 'icons', 'labelText', 'link'))
    '''Properties which emit "localeAttributeChanged" notification and not has a setter'''

    __NONEVENT_ATTRIBUTES = Set(('created', 'id', 'modified', 'encrypted_content', 'nodeAttribute', 'note',
                                 'hgap', 'vgap', 'vshift'))
    '''Properties which not emit any notifications and not has a setter'''

    __LOCAL_CHANGE_LEVEL, __INHERITED_CHANGE_LEVEL = range(2)
    '''Numerical constants of signal types'''

    __SIGNALS = ('localAttributeChanged', 'inheritedAttributeChanged')
    '''Main signals to notify about mind map changes'''

    def setAttributes(self, **attributes):
        '''
        Set one or many node attributes by name

        @param attributes: name = value pairs of attributes
        @type attributes: dict
        '''
        maxNotifyLevel = -1 # -1 <=> no notification will be send
        # TODO optimize: use more efficient switch
        for attrName, attrValue in attributes.items():
            if attrName == 'font':
                self._setFont(attrValue)
                maxNotifyLevel = max(maxNotifyLevel, Node.__INHERITED_CHANGE_LEVEL)
            elif attrName == 'color':
                self._setColor(attrValue)
                maxNotifyLevel = max(maxNotifyLevel, Node.__INHERITED_CHANGE_LEVEL)
            elif attrName == 'background_color':
                self._setBgColor(attrValue)
                maxNotifyLevel = max(maxNotifyLevel, Node.__INHERITED_CHANGE_LEVEL)
            elif attrName == 'text':
                self._setPlainText(attrValue)
                maxNotifyLevel = max(maxNotifyLevel, Node.__LOCAL_CHANGE_LEVEL)
            elif attrName == 'html':
                self._setHtml(attrValue)
                maxNotifyLevel = max(maxNotifyLevel, Node.__LOCAL_CHANGE_LEVEL)
#           FIXME: differ levels by type
            elif attrName == 'edgeWidth':
                self._setEdgeWidth(attrValue)
                maxNotifyLevel = max(maxNotifyLevel, Node.__INHERITED_CHANGE_LEVEL)
            elif attrName == 'edgeStyle':
                self._setEdgeStyle(attrValue)
                maxNotifyLevel = max(maxNotifyLevel, Node.__INHERITED_CHANGE_LEVEL)
            elif attrName == 'edgeColor':
                self._setEdgeColor(attrValue)
                maxNotifyLevel = max(maxNotifyLevel, Node.__INHERITED_CHANGE_LEVEL)
            elif attrName == 'labelFont':
                self._setLabelFont(attrValue)
                maxNotifyLevel = max(maxNotifyLevel, Node.__INHERITED_CHANGE_LEVEL)
            elif attrName == 'labelColor':
                self._setLabelColor(attrValue)
                maxNotifyLevel = max(maxNotifyLevel, Node.__INHERITED_CHANGE_LEVEL)
            elif attrName == 'left':
                self._setLeft(attrValue)
                maxNotifyLevel = max(maxNotifyLevel, Node.__INHERITED_CHANGE_LEVEL)
            elif attrName == 'style':
                self._setStyle(attrValue)
                maxNotifyLevel = max(maxNotifyLevel, Node.__INHERITED_CHANGE_LEVEL)
            elif attrName in Node.__LOCAL_ATTRIBUTES:
                self._setField(attrName, attrValue)
                maxNotifyLevel = max(maxNotifyLevel, Node.__LOCAL_CHANGE_LEVEL)
            elif attrName in Node.__NONEVENT_ATTRIBUTES:
                self._setField(attrName, attrValue)
            else: print 'Unknown attribute name: ' + attrName
        # emit only one signal with max level of change
        if maxNotifyLevel != -1:
            self.notifyObservers(self, type = Node.__SIGNALS[maxNotifyLevel])

    WRITABLE_PROPERTIES = ('created', 'id', 'modified', 'encrypted_content', 'nodeAttribute',
                             'note', 'icons', 'font', 'color', 'left', 'edgeColor',
                             'edgeWidth', 'edgeStyle', 'link', 'hgap', 'vgap', 'vshift',
                             'cloud', 'folded', 'background_color', 'style', 'text', 'labelText',
                             'labelFont', 'labelColor')
    '''Node writable properties that can be set with setAttributes() method'''

    def getAttributes(self, *propNames, **options):
        '''
        Return dict of specified property values. If used method call with
        argument 'getAll = True' returns overall writable values

        @param propNames: list of attributes to be returned
        @type propNames: tuple

        @param options: named control options (one yet): getAll = True
        @type options: dict
        '''
        resultAttributes = {}
        if options.get('getAll'):
            propNames = Node.WRITABLE_PROPERTIES
        # automatically choose content type to be returned)
        if 'text' in propNames:
            propNames = list(propNames)
            propNames.remove('text')
            if self.__isHtmlContent:
                resultAttributes['html'] = self.text
            else:
                resultAttributes['text'] = self.text
        for attrName in propNames:
            resultAttributes[attrName] = self._getField(attrName)
        return resultAttributes

    @property
    def text(self):
        '''
        Return text of the node in either plain text or HTML
        @rtype: unicode
        '''
        # unicode type casting must be used for QString
        if self.__isHtmlContent:
            return unicode(self.__richcontent.toHtml())
        return unicode(self.__richcontent.toPlainText())

    def isHtml(self):
        '''
        Returns True if node has rich content; otherwise return False
        '''
        return self.__isHtmlContent

    def _setPlainText(self, text):
        '''
        Set node content as simple text
        '''
        self.__isHtmlContent = False
        self.__richcontent.setPlainText(text)

    def _setHtml(self, html):
        '''
        Set node content as formated html
        '''
        self.__isHtmlContent = True
        self.__richcontent.setHtml(html)

    def addChild(self, child, position = None):
        '''
        Add a child node to the current node

        @type child: Node

        @param position: position in children list
        @type position: int
        '''
        # prevent duplicates
        if child in self.__children: return
        child.__parent = self
        if position is not None:
            self.__children.insert(position, child)
        else:
            self.__children.append(child)
        self.notifyObservers(child, type = 'childNodeAdded')

    def removeChild(self, child):
        '''
        Remove a child node from the current node
        
        @type child: Node
        '''
        self.__children.remove(child)
        child.__parent = None
        self.notifyObservers(child, type = 'childNodeRemoved')

    def getChildPosition(self, child):
        '''
        Return position of a child in the current node
        
        @type child: Node
        '''
        if not self.isAncestorOf(child):
            return None
        return self.__children.index(child)

    def moveNode(self, dest, position = None):
        '''
        Relocate current node to a destination node

        @type dest: Node

        @param position: position in children list
        @type position: int
        '''
        if self.__parent is None or self.isAncestorOf(dest):
            return
        if self.__parent == dest:
            if position - dest.children.index(self) > 1:
                position -= 1
        self.__parent.__children.remove(self)
        self.__parent = dest
        if position is None:
            dest.__children.append(self)
        else:
            dest.__children.insert(position, self)
        self._setLeft(self.left if dest.parent is None else dest.left)
        self._propagateEffectiveStyle(dest.effectiveStyle)
        self._propagateEffectiveEdgeColor(dest.effectiveEdgeColor)
        self._propagateEffectiveEdgeStyle(dest.effectiveEdgeStyle)
        self._propagateEffectiveEdgeWidth(dest.effectiveEdgeWidth)
        self._propagateEffectiveLabelColor(dest.effectiveLabelColor)
        self._propagateEffectiveLabelFont(dest.effectiveLabelFont)
        self._propagateEffectiveColor(dest.effectiveColor)
        self._propagateEffectiveBgColor(dest.effectiveBgColor)
        self._propagateEffectiveFont(dest.effectiveFont)
        self.notifyObservers(self, dest, type = 'nodeMoved')

    def isAncestorOf(self, child):
        '''
        Check if the current node is an ancestor of the child

        @type child: Node
        @rtype: bool
        '''
        node = child
        while node is not None:
            if self == node:
                return True
            node = node.parent
        return False

    def depth(self):
        '''
        Return the depth of the node
        @rtype: int
        '''
        node = self
        _depth = 0
        while node.parent is not None:
            _depth += 1
            node = node.parent
        return _depth

    def createChildPrototype(self, **attributes):
        '''
        Create a new node without adding it into the children list.
        Attributes of the created node are set according to the inheritance policy
        
        @param attributes: initialization attributes
        @type attributes: dict

        @return: created child node
        @rtype: Node
        '''
        child = Node(**attributes)
        # FIXME prototype is not a child yet!
        child.__parent = self
        # set inheritable attributes
        child.__left = self.__left
        child.__effectiveStyle = self.__effectiveStyle
        child.__effectiveEdgeColor = self.__effectiveEdgeColor
        child.__effectiveEdgeWidth = self.__effectiveEdgeWidth
        child.__effectiveEdgeStyle = self.__effectiveEdgeStyle
        child.__effectiveLabelColor = self.__effectiveLabelColor
        child.__effectiveBgColor = self.__effectiveBgColor
        child._propagateEffectiveLabelFont(self.__effectiveLabelFont)
        child._propagateEffectiveColor(self.__effectiveColor)
        child._propagateEffectiveFont(self.__effectiveFont)
        return child

    @property
    def allNodeAttributes(self):
        '''
        Return list of pairs (attributeName, attributeValue)
        @rtype: list
        '''
        if self.__nodeAttribute:
            # Attribute.allAttributes always has not None value
            return self.__nodeAttribute.allAttributes.items()
        return []

    def addNodeAttribute(self, attrName, attrValue):
        '''
        Add attribute to the current node
        
        @type attrName: str
        @type attrValue: str    
        '''
        if self.__nodeAttribute is None:
            self.__nodeAttribute = Attribute()
        self.__nodeAttribute.addAttribute(attrName, attrValue)

    def paint(self, painter):
        '''
        Paint node text content
        
        @type painter: QPainter
        '''
        painter.save()
        paintContext = QAbstractTextDocumentLayout.PaintContext()
#        painter.setFont(self.effectiveFont)
        paintContext.palette = self.__colorPalette
        self.__richcontent.documentLayout().draw(painter, paintContext)
        painter.restore()

    def textSize(self):
        '''
        Return area size of node text content
        
        @rtype: QSizeF
        '''
        return self.__richcontent.size()

    @property
    def labelSize(self):
        '''
        Return minimal area size of label text

        @rtype: QSizeF
        '''
        if self.__labelText is None: return QSizeF(0.0, 0.0)
#       FIXME: 1) QFontMetrics.xxx methods are very slow! Use cached size value
#              2) QFontMetrics.width() works incorrectly with italic fonts => use QFontMetrics.size() instead
        return QSizeF(self.__labelFontMetrics.width(self.__labelText),
                      self.__labelFontMetrics.overlinePos() -
                      self.__labelFontMetrics.underlinePos())


class Cloud(Element):
    '''
    Cloud inner node

    @authors: Andrew Golovchenko
    '''
    _ATTRIBUTES = ('style', 'color', 'width')
    '''List of cloud attributes'''
    writable(*_ATTRIBUTES)

    def __init__(self, **attributes):
        '''
        Initialize cloud attributes
        
        @param attributes: dictionary of initialization attributes
        @type attributes: dict
        '''
        Element.__init__(self, **attributes)


class AttributeRegistry(Element):
    '''
    Mind map attribute registry

    @authors: Andrew Golovchenko
    '''
    _ATTRIBUTES = ('restricted', 'font_size', 'show_attributes')
    '''List of "attribute registry" element attributes'''
    writable('viewOptions', 'attrValues')
    writable(*_ATTRIBUTES)

    def __init__(self, **attributes):
        '''
        Initialize empty attribute registry
        
        @param attributes: dictionary of initialization attributes
        @type attributes: dict
        '''
        Element.__init__(self, **attributes)
        # dict of attributes options in following format: 
        # {'name_A' : {'option_x' : 'value_x', 'option_y' : 'value_y'}
        #  'name_B' : {'option_z' : 'value_z'} ...}
        self.__viewOptions = {}
        # dict of attributes values in following format: 
        # {'name_A' : ['value_1', 'value_2'], 'name_B' : ['value_1'] ...}
        self.__attrValues = {}


class Attribute(Element):
    '''
    Composite container class for attribute and attribute_layout inner nodes

    @authors: Andrew Golovchenko
    '''
    _ATTRIBUTES = ('name_width', 'value_width')
    '''List of "attribute" element attributes'''
    readable('allAttributes')
    writable(*_ATTRIBUTES)

    def __init__(self, **attributes):
        '''
        Initialize class attributes
        
        @param attributes: dictionary of initialization attributes
        @type attributes: dict
        '''
        Element.__init__(self, **attributes)
        # dict of attributes in following format: 
        # {'name_A' : 'value_A', 'name_B' : 'value_B' ...}
        self.__allAttributes = {}

    def addAttribute(self, attrName, attrValue):
        '''
        Add attribute to the container
        
        @type attrName: str
        @type attrValue: str    
        '''
        self.__allAttributes[attrName] = attrValue

def equal(objectOne, objectTwo):
    '''
    Compare two objects for equality. Both objects can be None.
    @return: True if objects are equal, False otherwise
    @rtype: bool
    '''
    if objectOne is None and objectTwo is None: return True
    if objectOne is None or objectTwo is None: return False
    return objectOne == objectTwo
