# -*- coding: utf-8 -*-
# HiveMind - Distributed mind map editor for Maemo 5 platform
# Copyright (C) 2010-2011 HiveMind developers
#
# HiveMind is the legal property of its developers, whose names are
# noticed in @author or @authors annotations at the beginning of each
# module or class.
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02111-1301  USA

'''
Easy-to-use property creation functions.

This module allows creation of properties of various types inside a class.
Read only, read/write and bound properties are supported. Names of properties
to be generated are passed to the necessary function in the form of positional
or named parameters. In the latter case values of the parameters must contain
docstrings for the corresponding properties.

Example::

  readable('prop1', 'prop2')
  writable('prop1', prop2 = 'Very important property')
  bound('prop', signalName = 'customSignal')

For the bound properties the special parameter C{signalName} is used to indicate
the name of the signal to be emitted on property change. If it is not set,
C{'propertyChanged'} value used by default. 

If the signal name ends with an @-sign, the reference to sender is passed as
a parameter of the signal.

@authors: Andrew Vasilev, Ilya Paramonov
'''

import sys
from PySide.QtCore import SIGNAL

def _mangle(className, attrName):
    '''
    Mangle name according to python name-mangling conventions for private
    variables
    '''
    return "_%s__%s" % (className, attrName)

def _classSpace(classLevel = 3):
    '''
    Get the calling class' name and dictionary
    
    @param classLevel: number of the class frame in the stack
    '''
    frame = sys._getframe(classLevel)
    className = frame.f_code.co_name
    classDict = frame.f_locals
    return className, classDict


def readable(*propNames, **propNamesWithDoc):
    '''Create read-only properties'''
    _attribute('r', propNames, propNamesWithDoc)

def writable(*propNames, **propNamesWithDoc):
    '''Create read-write properties'''
    _attribute('rw', propNames, propNamesWithDoc)

def bound(*propNames, **propNamesWithDoc):
    '''Create bound properties (for QObject descendants only)'''
    signalName = propNamesWithDoc.pop('signalName', 'propertyChanged')
    _attribute('rb', propNames, propNamesWithDoc, signalName)

def _attribute(permission, propNames, propNamesWithDoc, signalName = None):
    '''
    Create properties
    
    @param permission: property access level (permission)
    @type permission: str    
    @type propNames: list
    @type propNamesWithDoc: dict
    '''
    className, classDict = _classSpace()
    for attrName in propNames:
        doc = "%s property" % attrName
        classDict[attrName] = _property(className, attrName, permission, doc, signalName)
    for attrName, doc in propNamesWithDoc.iteritems():
        classDict[attrName] = _property(className, attrName, permission, doc, signalName)

def _property(className, attrName, permission, doc, signalName):
    '''Generate class property with specified parameters'''
    mangledName = _mangle(className, attrName)
    fget, fset, fdel = None, None, None
    if 'r' in permission:
        def fget(self):
            return getattr(self, mangledName)
    if 'w' in permission:
        def fset(self, value):
            setattr(self, mangledName, value)
    if 'b' in permission:
        if signalName[-1] == '@':
            signalName = signalName[:-1]
            def fset(self, value):
                setattr(self, mangledName, value)
                self.emit(SIGNAL(signalName), self)
        else:
            def fset(self, value):
                setattr(self, mangledName, value)
                self.emit(SIGNAL(signalName))
        doc += ('.\nEmits B{%s} signal on change' % signalName)
    return property(fget = fget, fset = fset, fdel = fdel, doc = doc)

def adaptCallable(callable):
    '''
    Create an adapter function forwarding its call to the certain
    callable object
    
    @param callable: callable object to forward calls
    @rtype: function
    '''
    def adapter(*args):
        return callable(*args)
    return adapter

class Observable():
    '''
    The observable object. Notifies all connected observers by invoking
    processNotification method.
    
    Data class objects will notify observers on value change. Sample code::
    
        def Data(Observable):
            def __init__(self, *args):
                Observable.__init__(self, *args)
                self.__uniqueData = 0
            def updateValue(newValue):
                self.__uniqueData = newValue
                self.notifyObservers()
    
        def DataView():
            def setData(self, dataObject):
                self.__dataObject = dataObject
                dataObject.addObserver(self)
            def processNotification(self, observable):
                self.repaint()
    
    @author: Andrew Vasilev
    '''
    def __init__(self, *args):
        '''
        Constructor
        '''
        self.__observers = []

    def addObserver(self, observer):
        '''
        Add observer object to the list of observers. Observer object must
        have update method.
        
        @param observer: observer object to add
        '''
        if observer not in self.__observers:
            self.__observers.append(observer)

    def deleteObserver(self, observer):
        '''
        Remove observer from the list of observer if it is present. Never raises
        an exception.
        
        @param observer: observer object to remove
        '''
        if observer in self.__observers:
            self.__observers.remove(observer)

    def deleteObservers(self):
        '''
        Remove all observers from the list.
        '''
        self.__observers = []

    def notifyObservers(self, *values, **args):
        '''
        Notify all observers from the list. All parameters are passed to observers.
        '''
        for observer in self.__observers:
            observer.processNotification(self, *values, **args)
