# -*- coding: utf-8 -*-
# HiveMind - Distributed mind map editor for Maemo 5 platform
# Copyright (C) 2010-2011 HiveMind developers
#
# HiveMind is the legal property of its developers, whose names are
# noticed in  or  annotations at the beginning of each
# module or class.
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02111-1301  USA

'''
Action bag class for actions storing

@author: Alexander Kulikov
'''

from hivemind.attribute import *
from PySide.QtCore import *
from PySide.QtGui import *
from hivemind.gui_widgets import *

class ActionBag(QObject):
    '''
    Action bag for actions storing

    @author: Alexander Kulikov
    '''

    readable('fileNewAction', 'fileOpenAction', 'fileSaveAction',
             'fileSaveAsAction', 'editEdgeAction', 'editLabelAction', 'editNodeAction',
             'editNodeIconsAction', 'foldNodeAction', 'addNodeAction',
             'addSiblingNodeBelowAction', 'removeNodeAction', 'enterTransferModeAction',
             'undoAction', 'redoAction', 'startNetworkClientAction',
             'startNetworkServerAction', 'stopNetworkAction', 'zoomInAction',
             'zoomOutAction', 'exitAction', 'aboutAction', 'settingsAction',
             'parentWidthAction', 'thinWidthAction', 'oneWidthAction', 'twoWidthAction',
             'fourWidthAction', 'eightWidthAction', 'parentStyleAction',
             'linearStyleAction', 'bezierStyleAction', 'sharpLinearStyleAction',
             'sharpBezierStyleAction', 'connectionStatusAction', 'subscribeStatusAction',
             'addSiblingNodeAboveAction', 'editAffiliationAction',
             'cancelTransferModeAction', 'putNodeAction', 'putNodeBelowAction',
             'putNodeAboveAction')

    def __init__(self, parent = None):
        QObject.__init__(self, parent)
        self._createActions()
        self._initializeActions()

    def _createActions(self):
        '''Create actions'''
        self.__fileNewAction = self.createAction(self.tr('&New'), 'Ctrl+N', 'document-new')
        self.__fileOpenAction = self.createAction(self.tr('&Open'), 'Ctrl+O', 'document-open')
        self.__fileSaveAction = self.createAction(self.tr('&Save'), 'Ctrl+S', 'document-save')
        self.__fileSaveAsAction = self.createAction(self.tr('Save as'), None, 'document-save-as')
        self.__editEdgeAction = self.createAction(self.tr('Edit edge'), 'Ctrl+E')
        self.__editLabelAction = self.createAction(self.tr('Edit label'), 'Ctrl+L')
        self.__editNodeAction = self.createAction(self.tr('Edit node'), 'F2', 'document-edit')
        self.__editNodeIconsAction = self.createAction(self.tr('Select icons'), 'Alt+I')
        self.__foldNodeAction = self.createAction(self.tr('Fold node'), 'Space')
        self.__addNodeAction = self.createAction(self.tr('New child node'), 'Insert', 'list-add')
        self.__addSiblingNodeBelowAction = self.createAction(self.tr('New sibling node below'), Qt.Key_Return, 'list-add')
        self.__addSiblingNodeAboveAction = self.createAction(self.tr('New sibling node above'),
            'Shift+Return', 'list-add')
        self.__removeNodeAction = self.createAction(self.tr('Remove node'), 'Delete', 'list-remove')
        self.__enterTransferModeAction = self.createAction(self.tr('Move node'), 'Ctrl+M')
        self.__cancelTransferModeAction = self.createAction(self.tr('Cancel transfer mode'),
                'Escape')
        self.__putNodeAction = self.createAction(self.tr('Paste moved node'), 'Insert')
        self.__putNodeBelowAction = self.createAction(self.tr('Put node below'), 'Return')
        self.__putNodeAboveAction = self.createAction(self.tr('Put node above'), 'Shift+Return')
        self.__undoAction = self.createAction(self.tr('&Undo'), 'Ctrl+Z', 'edit-undo')
        self.__redoAction = self.createAction(self.tr('&Redo'), 'Ctrl+Y', 'edit-redo')
        self.__startNetworkServerAction = self.createAction(self.tr('&Share mind map'))
        self.__startNetworkClientAction = self.createAction(self.tr('&Edit shared mind map'))
        self.__stopNetworkAction = self.createAction(self.tr('&Finish collaboration'))
        self.__zoomInAction = self.createAction(self.tr('Zoom in'), '+', 'zoom-in')
        self.__zoomOutAction = self.createAction(self.tr('Zoom out'), '-', 'zoom-out')
        self.__aboutAction = self.createAction(self.tr('&About'), None, 'help-about')
        self.__exitAction = self.createAction(self.tr('E&xit'), 'Ctrl+Q', 'application-exit')
        self.__settingsAction = self.createAction(self.tr('Settings'), None, 'preferences-other')
        self.__parentWidthAction = self.createAction(self.tr('Parent'))
        self.__thinWidthAction = self.createAction(self.tr('Thin'))
        self.__oneWidthAction = self.createAction('1')
        self.__twoWidthAction = self.createAction('2')
        self.__fourWidthAction = self.createAction('4')
        self.__eightWidthAction = self.createAction('8')
        self.__parentStyleAction = self.createAction(self.tr('Parent'))
        self.__linearStyleAction = self.createAction(self.tr('Linear'))
        self.__bezierStyleAction = self.createAction(self.tr('Bezier'))
        self.__sharpLinearStyleAction = self.createAction(self.tr('Sharp linear'))
        self.__sharpBezierStyleAction = self.createAction(self.tr('Sharp bezier'))
        self.__connectionStatusAction = StatusAction('Connection status', 'connection', 'no')
        self.__subscribeStatusAction = StatusAction('Subscribe status', 'subscribe', 'no')
        self.__editAffiliationAction = self.createAction(self.tr('&Edit participants access'))

    def _initializeActions(self):
        '''Make specific action initializations'''
        self.__fileSaveAction.setEnabled(False)

    def createAction(self, text, shortcut = None, icon = None):
        '''
        Create action
        @param text: descriptive text for tool buttons
        @type text: string
        @param handler: slot
        @type shortcut: string
        @rtype: QAction
        '''
        action = QAction(text, self)
        if shortcut: action.setShortcut(shortcut)
        if icon: action.setIcon(QIcon.fromTheme(icon))
        return action
