# -*- coding: utf-8 -*-
# HiveMind - Distributed mind map editor for Maemo 5 platform
# Copyright (C) 2010 HiveMind developers
#
# HiveMind is the legal property of its developers, whose names are
# noticed in  or  annotations at the beginning of each
# module or class.
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02111-1301  USA

from wokkel.pubsub import *
from wokkel.client import XMPPClient
from twisted.words.protocols.jabber import jid
from twisted.words.protocols.jabber.sasl import *
from twisted.words.xish.domish import Element
from twisted.python.failure import Failure
from twisted.words.protocols.jabber.error import *
from hivemind import parser
from twisted.internet.defer import Deferred
from hivemind.attribute import Observable, writable
import random

'''
Network subsystem of the application

@author: Andrew Vasilev
'''

class NetworkController(Observable):
    '''
    Asynchronous command processor. Sends commands into shared space, listens to incoming messages
    and sends correct commands to main application controller.
    '''

    def __init__(self):
        '''Constructor'''
        Observable.__init__(self)
        self.__xmppClient = None
        self.__protocol = None
        self.__mindmap = None

    def publishCommand(self, command):
        '''
        Publish command in common space
        @param command: command to publish in shared space
        @type command: QUndoCommand
        '''
        data = parser.serializeCommand(command)
        if self.__protocol:
            self.__protocol.proposeChangeset(Changeset(data, 'command'))
        else:
            self.notifyObservers(command = command)

    def _startConnection(self, userJid, password, protocol):
        '''
        Setup network connection
        @param userJid: JID for XMPP connection
        @type userJid: str
        @param password: password for XMPP connection
        @type password: str
        @param protocol: protocol handler on top of xmpp
        @type protocol: XMPPHandler
        '''
        self.__xmppClient = self._configureXMPPClient(userJid, password)
        protocol.setHandlerParent(self.__xmppClient)
        self.__protocol = protocol
        self.__protocol.addObserver(self)
        self.__xmppClient.startService()
        self.notifyObservers(state = 'connecting')

    def startHiveMindService(self, userJid, password, mindmap, undoStack):
        '''
        Start HiveMind service
        @param userJid: JID for XMPP connection
        @type userJid: str
        @param password: password for XMPP connection
        @type password: str
        @param mindmap: current mind map to publish first
        @type mindmap: MindMap
        @param undoStack: Undo stack associated with mindmap
        @type undoStack: QUndoStack
        '''
        self.__mindmap = mindmap
        protocol = HivemindService(parser.mindMapToString(mindmap, False),
                parser.serializeCommandStack(undoStack))
        self._startConnection(userJid, password, protocol)

    def _configureXMPPClient(self, userJid, password):
        '''
        Configure XMPP client
        @param userJid: user JID to use for connection
        @type userJid: str
        @param password: password for passed JID
        @type password: str
        @return: configured XMPP client
        @rtype: XMPPClient
        '''
        xmppClient = HivemindXMPPClient(jid.internJID(userJid), password)
        xmppClient.addObserver(self)
        xmppClient.logTraffic = True
        return xmppClient

    def startHiveMindClient(self, userJid, password, serviceJid):
        '''
        Start HiveMind client, connect to specified service
        @param userJid: user JID for XMPP connection
        @type userJid: str
        @param password: password for passed JID
        @type password: str
        @param serviceJid: HiveMind service JID
        @type serviceJid: str
        '''
        protocol = HivemindClient(jid.internJID(serviceJid))
        self._startConnection(userJid, password, protocol)

    def stopCommunication(self):
        '''Stop communication with the service'''
        # TODO: gracefull shutdown
        if self.__protocol:
            self.__protocol.disownHandlerParent(self.__xmppClient)
            self.__protocol.deleteObserver(self)
            self.__protocol = None
        if self.__xmppClient:
            self.__xmppClient.stopService()
            self.__xmppClient = None
        self.__mindmap = None
        self.notifyObservers(state = 'offline')

    def _loadMindMap(self, mindmap):
        '''
        Load mind map recieved from the service
        @param mindmap: serialized into string mind map
        @type mindmap: str
        '''
        self.__mindmap = parser.mindMapFromString(mindmap)
        self.notifyObservers(mindmap = self.__mindmap)

    def processNotification(self, *values, **kvargs):
        '''Process notification from protocols'''
        if 'connection' in kvargs:
            self._processProtocolConnectionState(kvargs['connection'])
        elif 'error' in kvargs:
            self._processError(kvargs['error'])
        elif 'changeset' in kvargs:
            self._addChangeset(kvargs['changeset'])

    def _processError(self, failure):
        ''' Process various errors '''
        if isinstance(failure, Failure):
            error = failure.trap(SASLError, StanzaError)
            self.stopCommunication()
            if error == SASLError:
                self.notifyObservers(error = 'connection')
            elif error == StanzaError:
                self.notifyObservers(error = 'subscription')
        else:
            self.notifyObservers(error = 'unknown')

    def _processProtocolConnectionState(self, state):
        '''
        Process connection state change notification
        @param state: current state of connection
        @type state: str
        '''
        if state == 'initialized':
            self.notifyObservers(state = 'connected')

    def _addChangeset(self, changeset):
        '''
        Add changeset recieved from hivemind network
        @type changeset: Changeset
        '''
        if changeset.type == 'mindmap':
            self._loadMindMap(changeset.data)
        elif changeset.type == 'command':
            self.notifyObservers(command = parser.deserializeCommand(changeset.data, self.__mindmap))
        elif changeset.type == 'undostack':
            self.notifyObservers(undostack = parser.deserializeCommandStack(changeset.data, self.__mindmap))


class HivemindXMPPClient(XMPPClient, Observable):
    '''
    Special XMPP client for hivemind
    '''
    def __init__(self, jid, password, host = None, port = 5222):
        XMPPClient.__init__(self, jid, password, host, port)
        Observable.__init__(self)

    def initializationFailed(self, reason):
        '''
        Called when stream initialization has failed
        @param reason: A failure instance indicating why stream initialization failed
        @type reason: Failure
        '''
        self.notifyObservers(error = reason)


XMPP_NODE_NAME = unicode('HiveMind_1')
'''XMPP PubSub node name'''


class HivemindService(PubSubService, Observable):
    '''Pubsub service for hivemind implementation'''

    def __init__(self, mindMap, undoStack):
        '''
        @param mindMap: initial record in the service with whole mind map
        @type mindMap: str
        @param undoStack: undostack associated with mindMap
        @type undoStack: str
        '''
        PubSubService.__init__(self)
        Observable.__init__(self)
        self.discoIdentity = {'category' : 'pubsub', 'type' : 'generic', 'name' : 'Hivemind Map'}
        self.__items = []
        self.__changesets = []
        self.__subscriptions = {}
        self._updateLists(Changeset(mindMap, 'mindmap'))
        self._updateLists(Changeset(undoStack, 'undostack'))

    def _errorHandler(self, error):
        ''' Handles various errors '''
        self.notifyObservers(error = error)

    def connectionInitalized(self):
        '''Connection to XMPP network has been initialized'''
        PubSubService.connectionInitialized(self)
        self.notifyObservers(connection = 'initialized')

    def publish(self, requestor, service, nodeIdentifier, items):
        '''Handler of XMPP subscribe event'''
        self._checkSubscription(requestor, nodeIdentifier)
        for item in items:
            changeset = Changeset.fromElement(item)
            self._updateLists(changeset)
            self.notifyObservers(changeset = changeset)
        self._publishChangeset(len(items))
        d = Deferred()
        d.callback([])
        d.addErrback(self._errorHandler)
        return d

    def subscribe(self, requestor, service, nodeIdentifier, subscriber):
        '''Handler of XMPP subscribe event'''
        if nodeIdentifier != XMPP_NODE_NAME:
            raise BadRequest('not-hivemind-node')
        if subscriber in self.__subscriptions:
            info = self.__subscriptions[subscriber]
        else:
            info = Subscription(XMPP_NODE_NAME, subscriber, 'subscribed')
            self.__subscriptions[subscriber] = info
        d = Deferred()
        d.callback(info)
        d.addErrback(self._errorHandler)
        return d

    def items(self, requestor, service, nodeIdentifier, maxItems, itemIdentifiers):
        '''Handler of XMPP request for items'''
        self._checkSubscription(requestor, nodeIdentifier)
        if maxItems is None or maxItems > len(self.__items):
            maxItems = len(self.__items)
        d = Deferred()
        d.callback(self.__items[-maxItems:])
        d.addErrback(self._errorHandler)
        return d

    def _checkSubscription(self, requestor, nodeIdentifier):
        '''
        Check subscription of the specified requestor
        @param requestor: jid of the requestor
        @type requestor: jid
        @param nodeIdentifier: the identifier of the node
        @type nodeIdentifier: unicode
        '''
        if nodeIdentifier != XMPP_NODE_NAME:
            raise PubSubError('service-unavailable', 'Service only serves ' + XMPP_NODE_NAME + ' node.')
        if requestor not in self.__subscriptions:
            raise PubSubError('subscription-required', 'You are not subscribe to current node')

    def _generateUniqueId(self):
        '''
        Generate unique identifier for the changeset
        @return: new unique identifier for the changeset
        @rtype: str
        '''
        while True:
            newId = self._generateId()
            for changeset in self.__changesets:
                if changeset.id == newId:
                    newId = None
            if newId is None: continue
            return newId

    def _generateId(self, len = 10):
        '''
        Generate identifier string
        @param len: length of the identifier
        @type len: int
        '''
        return ("%%0%dX" % len) % random.getrandbits(len * 4)

    def proposeChangeset(self, changeset):
        '''
        Propose changeset to the HiveMind network
        @param changeset: changeset to publish inside network
        @type changeset: Changeset
        '''
        self.notifyObservers(changeset = changeset)
        self._updateLists(changeset)
        self._publishChangeset(1)

    def _publishChangeset(self, number):
        '''
        Publish several last changesets into network
        @param number: number of changesets to publish
        @type number: int
        '''
        sendList = []
        count = min(number, len(self.__items))
        items = self.__items[-count:]
        for subscription in self.__subscriptions.values():
            sendList.append([subscription.subscriber, None, items])
        self.notifyPublish(self.parent.jid, XMPP_NODE_NAME, sendList)

    def _updateLists(self, changeset):
        '''
        Fill identification fields and add changeset and corresponding element to the lists
        @param changeset: changeset to add to the lists
        @type changeset: Changeset
        '''
        changeset.id = self._generateUniqueId()
        changeset.number = len(self.__changesets)
        self.__changesets.append(changeset)
        self.__items.append(changeset.toElement())


class HivemindClient(PubSubClient, Observable):
    '''Pubsub client for HiveMind service'''

    def __init__(self, hivemindService):
        '''
        @param hivemindService: jid of hivemind service
        @type hivemindService: jid
        '''
        PubSubClient.__init__(self)
        Observable.__init__(self)
        self.__service = hivemindService

    def _errorHandler(self, error):
        ''' Handles various errors '''
        self.notifyObservers(error = error)

    def connectionInitialized(self):
        '''Connection initialized event handler'''
        def cb(result):
            self.notifyObservers(connection = 'subscribed')
            return result

        PubSubClient.connectionInitialized(self)
        self.notifyObservers(connection = 'initialized')
        d = self.subscribe(self.__service, XMPP_NODE_NAME, self.parent.jid)
        d.addCallback(cb)
        d.addCallback(self.retrieveItems)
        d.addErrback(self._errorHandler)

    def retrieveItems(self, result = None):
        '''retrieve items from the server, all of them'''
        d = self.items(self.__service, XMPP_NODE_NAME)
        d.addCallback(self._parseInitItems)
        d.addErrback(self._errorHandler)

    def _parseInitItems(self, items):
        '''
        Parse items on initial request
        @param items: a set of items retrieved from the service
        @type items: list
        '''
        for item in items:
            changeset = Changeset.fromElement(item)
            self.notifyObservers(changeset = changeset)

    def itemsReceived(self, event):
        '''Process recieved items on publish command'''
        if event.sender != self.__service or event.nodeIdentifier != XMPP_NODE_NAME:
            return
        self._parseInitItems(event.items)

    def proposeChangeset(self, changeset):
        '''
        Propose changeset to the network controller
        @param changeset: changeset to send to the controller
        @type changeset: str
        '''
        d = self.publish(self.__service, XMPP_NODE_NAME, [changeset.toElement()])
        d.addErrback(self._errorHandler)


class Changeset():
    '''
    Changeset representation
    @author: Andrew Vasilev
    '''

    writable('id', 'number', 'data', 'type')
    def __init__(self, data = None, type = None, id = None, number = None):
        '''
        Initialize object from the several components
        @param data: xml-serialized data of the changeset
        @type data: str
        @param type: type of the changeset
        @type type: str
        @param id: unique id of the changeset
        @type id: str
        @param number: the number of the changeset
        @type number: int
        '''
        self.__id = id
        self.__number = number
        self.__data = data
        self.__type = type

    def toElement(self):
        '''
        Convert this changeset into domish xml Element
        @return: Element representation of current object
        @rtype: Element
        '''
        element = Element((None, 'item'))
        if self.__id: element.attributes['id'] = self.__id
        if self.__number: element.attributes['number'] = str(self.__number)
        if self.__type: element.attributes['type'] = self.__type
        element.addRawXml(self.__data)
        return element

    @staticmethod
    def fromElement(element):
        '''
        Create HiveElement object from xml Element
        @param element: xml representation of the object
        @type element: Element
        '''
        xmlChangeset = element.firstChildElement()
        _removePoisonedUri(xmlChangeset)
        changeset = xmlChangeset.toXml()
        return Changeset(changeset, element.getAttribute('type'), element.getAttribute('id'),
            element.getAttribute('number'))


def _removePoisonedUri(element):
    '''
    Remove poisoning uri and defaultUri properties from xml tree
    @param element: root element of xml tree
    @type element: Element
    '''
    element.uri, element.defaultUri = None, None
    for child in element.elements():
        _removePoisonedUri(child)
