# -*- coding: utf-8 -*-
# HiveMind - Distributed mind map editor for Maemo 5 platform
# Copyright (C) 2010 HiveMind developers
#
# HiveMind is the legal property of its developers, whose names are
# noticed in @author or @authors annotations at the beginning of each
# module or class.
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02111-1301  USA

'''
Main window classes
'''

import getopt
import hivemind.parser as parser
import hivemind.__dirs__ as __dirs__
import os
import re
from hivemind.core import *
from hivemind.attribute import *
from hivemind.gui_delegates import *
from hivemind.gui_widgets import *
from hivemind.commands import *
from PySide.QtGui import *
from PySide import QtGui
from PySide.QtCore import *
from twisted.python import log
import hivemind.gui_factory as gui_factory


# Install qt4reactor as main twisted framework reactor
from twisted.internet import main
from qt4reactor.qt4reactor import QTReactor
app = QApplication(sys.argv)
reactor = QTReactor()
main.installReactor(reactor)

from hivemind.network.network_core import NetworkController

class MainWindowController(QObject):
    '''
    Base controller class for main window widgets
    @author: Andrew Vasilev, Alexander Kulikov, Ilya Paramonov
    '''

    readable('fileNewAction', 'fileOpenAction', 'fileSaveAction',
             'fileSaveAsAction', 'editEdgeAction', 'editNodeAction',
             'editLabelAction', 'foldNodeAction',
             'addNodeAction', 'exitAction', 'editNodeIconsAction',
             'removeNodeAction', 'aboutAction', 'undoAction', 'redoAction',
             'startNetworkClientAction', 'startNetworkServerAction',
             'stopNetworkAction', 'settingsAction', 'parentWidthAction',
             'thinWidthAction', 'oneWidthAction', 'twoWidthAction',
             'fourWidthAction', 'eightWidthAction', 'parentStyleAction',
             'linearStyleAction', 'bezierStyleAction', 'sharpLinearStyleAction',
             'sharpBezierStyleAction', 'moveNodeAction', 'selectLeftNodeAction',
             'selectRightNodeAction', 'selectUpNodeAction', 'selectDownNodeAction',
             'selectRootNodeAction', 'addAdjacentNodeAction', 'zoomInAction',
             'zoomOutAction')
    writable('mode')

    MODE_NORMAL = 'normal'
    '''All actions are parsed as normal'''

    MODE_TRANSFER = 'transfer'
    '''Map is in transfer mode'''

    def __init__(self):
        QObject.__init__(self)
        '''Init widgets and configure application'''
        self.__mode = self.MODE_NORMAL
        self.__undoStack = QUndoStack(self)
        self.__undoStack.connect(SIGNAL('cleanChanged(bool)'), self._undoStackStateChanged)
        self.__networkController = NetworkController()
        self.__networkController.addObserver(self)
        self._createActions()
        self._createStateModificationActions()
        self.__mindMapScene = MindMapDelegate()
        self.connect(self.__mindMapScene, SIGNAL('currentNodeChanged'),
                     adaptCallable(self._setActionsState))
        self.__mainWindow = gui_factory.createMainWindow(self, self.__mindMapScene)
        _configureApplication(self)
        self.__mainWindow.show()

    def createAction(self, text, handler, shortcut = None, icon = None):
        '''
        Create action
        @param text: descriptive text for tool buttons
        @type text: string
        @param handler: slot
        @type shortcut: string
        @rtype: QAction
        '''
        action = QAction(text, self)
        if shortcut: action.setShortcut(shortcut)
        if icon: action.setIcon(QIcon.fromTheme(icon))
        if handler: action.connect(SIGNAL('triggered()'), handler)
        return action

    def _setActionsState(self, currentNode):
        '''
        Set properties of the actions according to the current node selection
        @type currentNode: Node
        '''
        self.__widthActions[currentNode.edgeWidth].setChecked(True)
        self.__styleActions[currentNode.edgeStyle].setChecked(True)
        self.__foldNodeAction.setEnabled(len(currentNode.children) > 0)
        self.__foldNodeAction.setText(
               self.tr('Unfold node') if currentNode.folded else self.tr('Fold node'))
        self.__foldNodeAction.setChecked(currentNode.folded is not None)
        self.__removeNodeAction.setEnabled(currentNode.parent is not None)
        self.__editLabelAction.setEnabled(currentNode.parent is not None)
        self.__moveNodeAction.setEnabled(currentNode.parent is not None)

    def _createStateModificationAction(self, text, alteredAttributes, shortcut = None, icon = None, checkable = True):
        '''
        Create an action and connect it to a handler which alters given attributes
        @type text: string
        @param alteredAttributes: attributes to be altered by the handler
        @type alteredAttributes: dict
        @type checkable: bool
        @type shortcut: string
        @rtype: QAction
        '''
        def handler():
            self.createStateModificationCommand(self.__mindMapScene.currentNode, alteredAttributes)
        action = self.createAction(text, handler, shortcut, icon)
        action.setCheckable(checkable)
        return action

    def _createStateModificationActions(self):
        '''Create state modification actions'''
        self.__parentWidthAction = self._createStateModificationAction(self.tr('Parent'),
                {'edgeWidth': None})
        self.__thinWidthAction = self._createStateModificationAction(self.tr('Thin'),
                {'edgeWidth': 11})
        self.__oneWidthAction = self._createStateModificationAction('1', {'edgeWidth': 1})
        self.__twoWidthAction = self._createStateModificationAction('2', {'edgeWidth': 2})
        self.__fourWidthAction = self._createStateModificationAction('4', {'edgeWidth': 4})
        self.__eightWidthAction = self._createStateModificationAction('8', {'edgeWidth': 8})
        self.__parentStyleAction = self._createStateModificationAction(self.tr('Parent'),
                {'edgeStyle': None})
        self.__linearStyleAction = self._createStateModificationAction(self.tr('Linear'),
                {'edgeStyle': 'linear'})
        self.__bezierStyleAction = self._createStateModificationAction(self.tr('Bezier'),
                {'edgeStyle': 'bezier'})
        self.__sharpLinearStyleAction = self._createStateModificationAction(self.tr('Sharp linear'),
                {'edgeStyle': 'sharp_linear'})
        self.__sharpBezierStyleAction = self._createStateModificationAction(self.tr('Sharp bezier'),
                {'edgeStyle': 'sharp_bezier'})
        self.__widthActions = {None : self.__parentWidthAction,
                               11 : self.__thinWidthAction,
                               1 : self.__oneWidthAction,
                               2 : self.__twoWidthAction,
                               4 : self.__fourWidthAction,
                               8 : self.__eightWidthAction}
        self.__styleActions = {'linear' : self.__linearStyleAction,
                               'bezier' : self.__bezierStyleAction,
                               'sharp_linear' : self.__sharpLinearStyleAction,
                               'sharp_bezier' : self.__sharpBezierStyleAction,
                               None : self.__parentStyleAction}
        group = QActionGroup(self)
        for action in self.__widthActions.values():
            group.addAction(action)
        group = QActionGroup(self)
        for action in self.__styleActions.values():
            group.addAction(action)

    def _createActions(self):
        '''Create actions'''
        self.__fileNewAction = self.createAction(self.tr('&New'), self.onFileNew,
                'Ctrl+N', 'document-new')
        self.__fileOpenAction = self.createAction(self.tr('&Open'), self.onFileOpen,
                'Ctrl+O', 'document-open')
        self.__fileSaveAction = self.createAction(self.tr('&Save'), self.onFileSave,
                'Ctrl+S', 'document-save')
        self.__fileSaveAsAction = self.createAction(self.tr('Save as'), self.onFileSaveAs,
                None, 'document-save-as')
        self.__editEdgeAction = self.createAction(self.tr('Edit edge'), self.editEdge,
                'Ctrl+E')
        self.__editLabelAction = self.createAction(self.tr('Edit label'), self.editLabel,
                'Ctrl+L')
        self.__editNodeAction = self.createAction(self.tr('Edit node'), self.editNode,
                'F2', 'document-edit')
        self.__editNodeIconsAction = self.createAction(self.tr('Select icons'),
                self.editNodeIcons, 'Alt+I')
        self.__foldNodeAction = self.createAction(self.tr('Fold node'), self.foldNode,
                'Space')
        self.__addNodeAction = self.createAction(self.tr('Add child node'), self.addNode,
                'Insert', 'list-add')
        self.__removeNodeAction = self.createAction(self.tr('Remove node'), self.removeNode,
                'Delete', 'list-remove')
        self.__moveNodeAction = self.createAction(self.tr('Move node'), self._startNodeMoving)
        self.__aboutAction = self.createAction(self.tr('&About'), self.onAbout,
                None, 'help-about')
        self.__undoAction = self.__undoStack.createUndoAction(self, self.tr('&Undo'))
        self.__undoAction.setIcon(QIcon.fromTheme('edit-undo'))
        self.__undoAction.setShortcut('Ctrl+Z')
        self.__redoAction = self.__undoStack.createRedoAction(self, self.tr('&Redo'))
        self.__redoAction.setIcon(QIcon.fromTheme('edit-redo'))
        self.__redoAction.setShortcut('Ctrl+Y')
        self.__exitAction = self.createAction(self.tr('E&xit'), self._closeMainWindow,
                'Ctrl+Q', 'application-exit')
        self.__startNetworkServerAction = self.createAction(self.tr('&Share mind map'),
            self._startHiveMindService, None, None)
        self.__startNetworkClientAction = self.createAction(self.tr('&Edit shared mind map'),
            self._startHiveMindClient, None, None)
        self.__stopNetworkAction = self.createAction(self.tr('&Finish collaboration'),
            self.__networkController.stopCommunication, None, None)
        self.__stopNetworkAction.setEnabled(False)
        self.__settingsAction = self.createAction(self.tr('Settings'), self.onSettings,
                None, 'preferences-other')
        self.__selectLeftNodeAction = self.createAction(None, self._selectLeftNode, 'Left')
        self.__selectRightNodeAction = self.createAction(None, self._selectRightNode, 'Right')
        self.__selectUpNodeAction = self.createAction(None, self._selectUpNode, 'Up')
        self.__selectDownNodeAction = self.createAction(None, self._selectDownNode, 'Down')
        self.__selectRootNodeAction = self.createAction(None, self._selectRootNode, 'Escape')
        self.__addAdjacentNodeAction = self.createAction(None, self._addAdjacentNode,
                Qt.Key_Return)
        self.__zoomInAction = self.createAction(self.tr('Zoom in'), None, '+', 'zoom-in')
        self.__zoomOutAction = self.createAction(self.tr('Zoom out'), None, '-', 'zoom-out')

    def _addCommand(self, command):
        '''
        Record command, created outside main controller
        @param command: command to execute on top of undo controller
        @type command: QUndoCommand
        '''
        self.__undoStack.push(command)

    def createStateModificationCommand(self, selectedNode, alteredAttributes):
        '''
        Create modification command using modified attributes on selected node

        @param selectedNode: the object of modification
        @type selectedNode: Node

        @param alteredAttributes: modified attributes
        @type alteredAttributes: Hash
        '''
        initialAttributeValues = selectedNode.getAttributes(getAll = True)
        unmodifiedAttributes = {}
        for key in alteredAttributes.keys():
            unmodifiedAttributes[key] = initialAttributeValues[key]
        self._recordCommand(StateModificationCommand(selectedNode, unmodifiedAttributes,
                                                     alteredAttributes))

    def _recordStateModificationCommand(self, node, dialog):
        '''
        Store node modifications made by dialog in the undo stack

        @param node: the object of modification
        @type node: Node

        @param dialog: dialog that performs change, must emit accepted() signal
                       after node modification
        @type dialog: QDialog
        '''
        initialAttributeValues = node.getAttributes(getAll = True)
        def _accept():
            alteredAttributes = dialog.attributes()
            # need to fix it, must use initialAttributeValues
            unmodifiedAttributes = {}
            for key in alteredAttributes.keys():
                if initialAttributeValues.has_key(key):
                    unmodifiedAttributes[key] = initialAttributeValues[key]
            self._recordCommand(StateModificationCommand(node, unmodifiedAttributes,
                    alteredAttributes))
        dialog.connect(SIGNAL('accepted()'), _accept)
        dialog.exec_()

    def _recordCommand(self, command):
        '''
        Process created command
        
        @param command: undo stack command to 
        '''
        self.__networkController.publishCommand(command)

    def _setFileName(self, fileName):
        '''Set __filename field and window header'''
        self.__fileName = fileName
        if fileName is not None:
            self.__mainWindow.setWindowTitle(os.path.split(fileName)[1])
        else:
            self.__mainWindow.setWindowTitle('HiveMind')

    def onItemClicked(self, rightButton, item, parent, childLocation):
        if self.__mode == MainWindowController.MODE_NORMAL:
            if item is not None:
                self.__mindMapScene.currentNode = item.node
                self.__currentLayerDepth = item.node.depth()
                self.__mainWindow.showMenu(rightButton, item)
        else:
            if parent.node.folded is None:
                self._finishNodeMoving(parent, childLocation)
            else:
                # Unfold node in transfer mode (tricky)
                roamingNode = self.__mindMapScene.currentNode
                self.__mindMapScene.currentNode = item.node
                self.foldNode()
                self.__mindMapScene.currentNode = roamingNode

    def _selectNextLeftRightNode(self, leftSubtree):
        '''
        Switch currentNode to left or to right node
        @type leftSubtree: boolean
        @param leftSubtree: select left node if true.
        '''
        nextNode = None
        currentNode = self.__mindMapScene.currentNode
        if currentNode.folded and (currentNode.left == leftSubtree or\
            not currentNode.parent):
            return
        if not currentNode.parent:
            for node in currentNode.children:
                if node.left == leftSubtree:
                    nextNode = node
                    break
        elif currentNode.left == leftSubtree:
            if len(currentNode.children) > 0:
                nextNode = currentNode.children[0]
        else:
            nextNode = currentNode.parent

        if nextNode:
            self.__mindMapScene.currentNode = nextNode
            self.__currentLayerDepth = nextNode.depth()

    def _createNodesLayer(self, root, maxDepth, leftSubtree, selectedNodes):
        '''
        Find nodes with the depth equal to maxDepth in the both left and right subtrees
        and put them to the list.
        @type root: Node
        @type maxDepth: int
        @type leftSubtree: bool
        @type selectNodes: list
        '''
        if not root.folded and maxDepth > 0:
            for node in root.children:
                if node.left == leftSubtree:
                    self._createNodesLayer(node, maxDepth - 1, leftSubtree, selectedNodes)
        if root.folded or maxDepth == 0 or len(root.children) == 0:
            selectedNodes.append(root)

    def _addAdjacentNode(self):
        '''
        Create and edit a adjacent node, add it to parent of selected node
        '''
        if self.__mindMapScene.currentNode.parent is None:
            return
        def _accept():
            '''Process accept action'''
            newNode.setAttributes(left = self.__mindMapScene.currentNode.left)
            newNode.setAttributes(**editor.attributes())
            parent = self.__mindMapScene.currentNode.parent
            index = parent.children.index(self.__mindMapScene.currentNode)
            self._recordCommand(AddNodeCommand(parent, newNode, index + 1))
            self.__currentLayerDepth = newNode.depth()
        newNode = self.__mindMapScene.currentNode.parent.createChildPrototype()
        editor = gui_factory.createEditNodeDialog(newNode, self.__mainWindow)
        editor.connect(SIGNAL('accepted()'), _accept)
        editor.show()

    def _getNodesLayer(self, leftSubtree):
        '''
        Get current nodes layer
        @type leftSubtree: bool
        @rtype: list of Node
        '''
        nodes = []
        self._createNodesLayer(self.__mindMapScene.rootDelegate.node,
                self.__currentLayerDepth, leftSubtree, nodes)
        for node in nodes:
            if self.__mindMapScene.currentNode.isAncestorOf(node):
                self.__mindMapScene.currentNode = node
                break
        return nodes

    def _selectRootNode(self):
        '''Select root node'''
        self.__mindMapScene.currentNode = self.__mindMapScene.rootDelegate.node

    def _selectLeftNode(self):
        '''Select left node'''
        self._selectNextLeftRightNode(True)

    def _selectRightNode(self):
        '''Select right node'''
        self._selectNextLeftRightNode(False)

    def _selectUpNode(self):
        '''Select up node'''
        nodes = self._getNodesLayer(self.__mindMapScene.currentNode.left)
        index = nodes.index(self.__mindMapScene.currentNode)
        if index > 0:
            self.__mindMapScene.currentNode = nodes[index - 1]

    def _selectDownNode(self):
        '''Select down node'''
        nodes = self._getNodesLayer(self.__mindMapScene.currentNode.left)
        index = nodes.index(self.__mindMapScene.currentNode)
        if index + 1 < len(nodes):
            self.__mindMapScene.currentNode = nodes[index + 1]

    def _startNodeMoving(self):
        '''
        Switch controller and view to transfer mode
        '''
        self.__mode = self.MODE_TRANSFER
        self.__inactiveEffect = QGraphicsBlurEffect()
        self.__inactiveEffect.setBlurRadius(3)
        self.__mindMapScene.getDelegate(self.__mindMapScene.currentNode).setGraphicsEffect(self.__inactiveEffect)

    def _finishNodeMoving(self, parent, location):
        '''
        Switch controller to normal mode
        @type parent: NodeDelegate
        @type location: NodeLocation
        '''
        self.__mode = self.MODE_NORMAL
        self.__mindMapScene.getDelegate(self.__mindMapScene.currentNode).setGraphicsEffect(None)
        if parent is None: return
        if self.__mindMapScene.currentNode == parent: return
        self.moveNode(self.__mindMapScene.currentNode, parent.node, location)

    def onFileNew(self):
        '''Process file new action'''
        if not self.prohibitChangeMapAction():
            return
        self.loadFile(None)

    def onFileOpen(self):
        '''Process file open action'''
        if not self.prohibitChangeMapAction():
            return
        fileName = QFileDialog.getOpenFileName(self.__mainWindow, self.tr('Open'),
                self._fileDirectory(), self.tr('Mind maps (*.mm);;All Files (*)'))[0]
        if not fileName: return
        self.loadFile(fileName)

    def onFileSave(self):
        '''
        Process file save action
        @return: True if mind map was saved, False otherwize
        @rtype: bool
        '''
        if self.__fileName is None:
            return self.onFileSaveAs()
        else:
            parser.storeMindMap(self.__fileName, self.__mindMapScene.mindMap)
            self.__undoStack.setClean()
            return True

    def onFileSaveAs(self):
        '''
        Process file save as action
        @return: True if mind map was saved False otherwise
        @rtype: bool
        '''
        fileName = QFileDialog.getSaveFileName(self.__mainWindow, self.tr('Save as'),
                self._fileDirectory(), self.tr('Mind maps (*.mm);;All Files (*)'))[0];
        if not fileName: return False
        if not re.match('.*\.mm', fileName): fileName += '.mm'
        parser.storeMindMap(fileName, self.__mindMapScene.mindMap)
        self._setFileName(fileName)
        self.__undoStack.setClean()
        return True

    def _fileDirectory(self):
        '''
        @return: Directory of the oppenned file or home
        @rtype: str
        '''
        if not self.__fileName: return os.path.expanduser('~')
        return os.path.dirname(self.__fileName)

    def loadFile(self, path):
        '''
        Load mind map file and show chosen map.
        If path is empty brand new mind map is created
        
        @param path: path to mind map file
        @type path: unicode
        '''
        self.__undoStack.setClean()
        self._undoStackStateChanged(True)
        if path is not None:
            mindMap = parser.loadMindMap(path)
        else:
            mindMap = MindMap()
        self._loadMindMap(mindMap, path)

    def _loadMindMap(self, mindMap, path = None):
        '''
        Load externally created mind map

        @param mindMap: mind map object
        @type mindMap: MindMap

        @param path: path to the file to store
        @type path: sting
        '''
        self._setFileName(path)
        self.__mindMapScene.setMap(mindMap)
        self.__mainWindow.resetView()
        self.__undoStack.clear()
        self.__currentLayerDepth = 0

    def onAbout(self):
        '''
        Process about action
        '''
        gui_factory.createAboutDialog(self.__mainWindow).exec_()

    def onSettings(self):
        '''
        Process settings action
        '''
        gui_factory.createSettingsDialog(self.__mainWindow).exec_()

    def editEdge(self):
        '''Edit node's edge decorations and style'''
        dialog = gui_factory.createEditEdgeDialog(self.__mindMapScene.currentNode, self.__mainWindow)
        self._recordStateModificationCommand(self.__mindMapScene.currentNode, dialog)

    def editNode(self):
        '''Edit selected node'''
        dialog = gui_factory.createEditNodeDialog(self.__mindMapScene.currentNode, self.__mainWindow)
        self._recordStateModificationCommand(self.__mindMapScene.currentNode, dialog)

    def editNodeIcons(self):
        '''Edit set of icons for selected node'''
        dialog = gui_factory.createIconDialog(self.__mindMapScene.currentNode, self.__mainWindow)
        self._recordStateModificationCommand(self.__mindMapScene.currentNode, dialog)

    def foldNode(self):
        '''Fold node subtree'''
        unmodifiedAttributes = {}
        alteredAttributes = {}
        unmodifiedAttributes['folded'] = self.__mindMapScene.currentNode.folded
        alteredAttributes['folded'] = True if self.__mindMapScene.currentNode.folded is None else None
        self._recordCommand(StateModificationCommand(
                self.__mindMapScene.currentNode, unmodifiedAttributes, alteredAttributes))

    def addNode(self):
        '''
        Create and edit a child node, add it to selected node
        '''
        def _accept():
            '''Process accept action'''
            if self.__mindMapScene.currentNode.parent is None:
                dialog = gui_factory.createSideSelectionDialog(self.__mainWindow)
                dialog.exec_()
                if dialog.leftSide is None:
                    # User doesn't accept dialog
                    return
                childNode.setAttributes(left = dialog.leftSide)
            if self.__mindMapScene.currentNode.folded:
                self.foldNode()
            childNode.setAttributes(**editor.attributes())
            self._recordCommand(AddNodeCommand(self.__mindMapScene.currentNode, childNode))
            self.__currentLayerDepth = childNode.depth()
        childNode = self.__mindMapScene.currentNode.createChildPrototype()
        editor = gui_factory.createEditNodeDialog(childNode, self.__mainWindow)
        editor.connect(SIGNAL('accepted()'), _accept)
        editor.show()

    def removeNode(self):
        '''
        Remove selected node
        '''
        def _recordCommand():
            self._recordCommand(RemoveNodeCommand(self.__mindMapScene.currentNode))
        if settings.get('confirmDelete'):
            ret = QMessageBox.question(self.__mainWindow, self.tr('Confirm delete'),
                  self.tr('Do you really want to delete the node?'),
                  QMessageBox.Yes, QMessageBox.No)
            if ret == QMessageBox.Yes:
                _recordCommand()
        else:
            _recordCommand()

    def moveNode(self, roamingNode, parentNode, location):
        '''
        Move node from current position to the new ancestor
        @param roamingNode: node to move
        @type roamingNode: Node
        @param parentNode: new parent for selected node
        @type parentNode: Node
        @param location: location relative to parent
        @type location: NodeLocation
        '''
        newPosition = location.position
        if newPosition is None:
            if location.top:
                newPosition = 0
            else:
                newPosition = len(parentNode.children)
        newSide = None
        if parentNode.parent is None:
            newSide = not location.right
        self._recordCommand(MoveNodeCommand(roamingNode, parentNode, newPosition, newSide))

    def editLabel(self):
        '''Edit edge label'''
        dialog = gui_factory.createEditLabelDialog(self.__mindMapScene.currentNode, self.__mainWindow)
        self._recordStateModificationCommand(self.__mindMapScene.currentNode, dialog)

    def _showSaveChangesBox(self):
        '''
        Ask user to make a decision about unsaved changes

        @return: result of message box invocation
        @rtype: int
        '''
        msgBox = QMessageBox(self.__mainWindow)
        msgBox.setIcon(QMessageBox.Question)
        msgBox.setText(self.tr('MindMap has been modified'))
        msgBox.setInformativeText(self.tr('Do you want to save changes?'))
        msgBox.setStandardButtons(QMessageBox.Save | QMessageBox.Discard | QMessageBox.Cancel)
        msgBox.setDefaultButton(QMessageBox.Save)
        return msgBox.exec_()

    def prohibitChangeMapAction(self):
        '''
        Ask user permition to forget mind map changes
        @return: False if user canced action, True otherwise
        @rtype: bool
        '''
        if self.__undoStack.isClean():
            return True
        decition = self._showSaveChangesBox()
        if decition == QMessageBox.Save:
            return self.onFileSave()
        elif decition == QMessageBox.Cancel:
            return False
        return True

    def _mindMapSaved(self):
        '''Setup stack and actions to present saved form of document'''
        self.__undoStack.setClean()
        self.__fileSaveAction.setEnabled(False)

    def _undoStackStateChanged(self, clean):
        '''
        Update internal's according to undo stack state
        @param clean: undo stack clean state indication
        @type clean: bool
        '''
        self.__fileSaveAction.setDisabled(clean)

    def _closeMainWindow(self):
        '''Close main window of the application'''
        self.__mainWindow.close()

    def processNotification(self, *values, **kvargs):
        '''Process notifications from network core'''
        if 'state' in kvargs:
            self._networkControllerStateChanged(kvargs['state'])
        elif 'command' in kvargs:
            self._addCommand(kvargs['command'])
        elif 'error' in kvargs:
            self._networkControllerFailed(kvargs['error'])
        elif 'mindmap' in kvargs:
            self._loadMindMap(kvargs['mindmap'])
        elif 'undostack' in kvargs:
            self.__undoStack = kvargs['undostack']

    def _networkControllerFailed(self, reason):
        '''
        Process network errors
        @type reason: str
        '''
        if reason == 'connection':
            QMessageBox.information(self.__mainWindow, self.tr('Connection problem'),
                    self.tr('Connection with XMPP server can not be established!'))
        elif reason == 'subscription':
            QMessageBox.information(self.__mainWindow, self.tr('Subscription problem'),
                    self.tr('Subscription error. You are disconnected.'))
        else:
            QMessageBox.information(self.__mainWindow, self.tr('Unknown problem'),
                    self.tr('Unknown problem has occured. You are disconnected.'))

    def _networkControllerStateChanged(self, state):
        '''
        Process network controller state change
        @param state: current state of the controller
        @type state: str
        '''
        canStart = True if state == 'offline' else False
        self.__startNetworkClientAction.setEnabled(canStart)
        self.__startNetworkServerAction.setEnabled(canStart)
        self.__stopNetworkAction.setEnabled(not canStart)

    def _startHiveMindService(self):
        '''Enable HiveMind network service'''
        def _accept():
            settings = dialog.attributes()
            self.__networkController.startHiveMindService(settings['userJid'] + '/hivemind',
                    settings['password'], self.__mindMapScene.mindMap, self.__undoStack)
        dialog = gui_factory.createXMPPCredentialsDialog(False, self.__mainWindow)
        dialog.connect(SIGNAL('accepted()'), _accept)
        dialog.exec_()

    def _startHiveMindClient(self):
        '''Enable HiveMind network client'''
        if not self.prohibitChangeMapAction():
            return
        def _accept():
            settings = dialog.attributes()
            self.__networkController.startHiveMindClient(settings['userJid'] + '/hivemind',
                    settings['password'], settings['serviceJid'] + '/hivemind')
        dialog = gui_factory.createXMPPCredentialsDialog(True, self.__mainWindow)
        dialog.connect(SIGNAL('accepted()'), _accept)
        dialog.exec_()

    def stopNetworkCommunication(self):
        '''Shutdown network communication'''
        self.__networkController.stopCommunication()
        reactor.fireSystemEvent('shutdown')


def _printHelp():
    '''Print general information about the application'''
    print '''HiveMind - Distributed mind map editor for Maemo 5 platform
Copyright (C) 2010 HiveMind developers

Usage: hivemind [filename]

Options:
    -h  Display this help message and exit
'''

def _configureApplication(controller):
    '''
    Parse command line arguments and configure application
    
    @param controller: the application controller
    @type controller: MainWindowController
    '''
    options, arguments = getopt.getopt(sys.argv[1:], 'hl')
    # parsing options
    for option in options:
        if option[0] == '-h':
            _printHelp()
            sys.exit()
        elif option[0] == '-l':
            log.startLogging(sys.stdout)
    # parsing arguments
    controller.loadFile(arguments[0] if arguments else None)

def _getLocale():
    '''Get locale for the application'''
    locale = settings.get('locale')
    if locale == '':
        locale = QLocale.system().name()
        settings.set(locale = locale)
    return locale

def run():
    '''Run HiveMind application'''
    app = QApplication.instance()
    # translator installation
    locale = _getLocale()[0:2]
    translatorStd = QTranslator()
    if translatorStd.load("%s/qt_%s.qm" %
            (QLibraryInfo.location(QLibraryInfo.TranslationsPath), locale)):
        app.installTranslator(translatorStd)
    translator = QTranslator()
    if translator.load(__dirs__.TRANSLATION_PATH_TEMPLATE % locale):
        app.installTranslator(translator)
    # launch of reactor and controller
    global reactor
    reactor.runReturn()
    MainWindowController()
    sys.exit(app.exec_())
