#include "actionlauncher.h"
#include <QDBusConnection>
#include <QDBusInterface>

ActionLauncher::ActionLauncher(QObject *parent) :
    QObject(parent)
{
}

QStringList ActionLauncher::parseAction(const QString &action) {
    QStringList parseList = action.split(' ', QString::SkipEmptyParts);
    QStringList actionList;

    while (!parseList.isEmpty()) {
        QString s = parseList.takeFirst();

        while ((s.endsWith('\\')) && (!parseList.isEmpty())) {
            s.append(QString(" %1").arg(parseList.takeFirst()));
        }

        actionList.append(s.remove('\\'));
    }

    return actionList;
}

void ActionLauncher::launchAction(const QString &action) {
    QStringList actionList = parseAction(action);

    if (!actionList.isEmpty()) {
        QString actionType = actionList.takeFirst();

        if ((actionType.startsWith("dbus")) && (actionList.size() > 3)) {
            QString service = actionList.takeFirst();
            QString path = actionList.takeFirst();
            QString interface = actionList.takeFirst();
            QString method = actionList.takeFirst();

            QDBusConnection bus = actionType.endsWith("system") ? QDBusConnection::systemBus() : QDBusConnection::sessionBus();
            QDBusInterface dbusInterface(service, path, interface, bus);

            if (actionList.isEmpty()) {
                dbusInterface.call(QDBus::NoBlock, method);
            }
            else {
                QVariantList args;

                while (!actionList.isEmpty()) {
                    args.append(actionList.takeFirst());
                }

                dbusInterface.callWithArgumentList(QDBus::NoBlock, method, args);
            }
        }
        else if ((actionType == "exec") && (!actionList.isEmpty())) {
            QProcess *process = new QProcess(this);
            process->start(actionList.takeFirst(), actionList);
            connect(process, SIGNAL(finished(int)), process, SLOT(deleteLater()));
            connect(process, SIGNAL(error(QProcess::ProcessError)), this, SLOT(onProcessError(QProcess::ProcessError)));
        }
        else {
            emit error(tr("Failed to launch action"));
        }
    }
    else {
        emit error(tr("Failed to launch action"));
    }
}

void ActionLauncher::launchActions(QStringList actions) {
    while (!actions.isEmpty()) {
        launchAction(actions.takeFirst());
    }
}

void ActionLauncher::onProcessError(QProcess::ProcessError e) {
    switch (e) {
    case QProcess::FailedToStart:
        emit error(tr("Failed to launch action"));
        break;
    case QProcess::Crashed:
        emit error(tr("Action has crashed"));
        break;
    case QProcess::Timedout:
        emit error(tr("Action has timed out"));
        break;
    default:
        emit error(tr("Failed to launch action"));
    }
}
