#include "eventlistdelegate.h"
#include "eventlistmodel.h"
#include "utils.h"
#include <qmath.h>
#include <QStaticText>
#include <QRegExp>

EventListDelegate::EventListDelegate(QObject *parent) :
    QStyledItemDelegate(parent)
{
}

void EventListDelegate::paint(QPainter *painter, const QStyleOptionViewItem &option, const QModelIndex &index) const {
    painter->save();

    if ((option.state) & (QStyle::State_Selected)) {
        painter->drawImage(option.rect, QImage("/etc/hildon/theme/images/TouchListBackgroundPressed.png"));
    }

    QRect appIconRect = option.rect;
    appIconRect.moveTopLeft(QPoint(appIconRect.left() + 5, appIconRect.top() + 5));
    appIconRect.setSize(QSize(48, 48));

    QImage appIcon(index.data(EventListModel::IconRole).toString());

    if (!appIcon.isNull()) {
        if ((appIcon.width() > 48) || (appIcon.height() > 48)) {
            appIcon = appIcon.scaled(QSize(48, 48), Qt::IgnoreAspectRatio, Qt::SmoothTransformation);
        }

        painter->drawImage(appIconRect, appIcon);
    }

    QRect titleRect = option.rect;
    titleRect.setTopLeft(QPoint(appIconRect.right() + 10, appIconRect.top()));
    titleRect.setRight(titleRect.right() - 5);
    titleRect.setHeight(30);

    QStaticText text(index.data(EventListModel::TitleRole).toString());
    text.setTextWidth(titleRect.width());

    QTextOption textOption(Qt::AlignTop);
    textOption.setWrapMode(QTextOption::NoWrap);
    text.setTextOption(textOption);

    painter->setClipRect(titleRect);
    painter->drawStaticText(titleRect.topLeft(), text);

    QRect bodyRect = option.rect;
    bodyRect.setTopLeft(QPoint(titleRect.left(), titleRect.bottom() + 10));
    bodyRect.setRight(titleRect.right());

    QFont font;
    font.setPixelSize(20);

    painter->setFont(font);

    QString body = index.data(EventListModel::BodyRole).toString();
    QStaticText bodyText(body);
    bodyText.setTextWidth(bodyRect.width());
    textOption.setWrapMode(QTextOption::WordWrap);
    bodyText.setTextOption(textOption);

    bodyRect.setHeight(qMin(option.fontMetrics.boundingRect(option.rect, Qt::TextWordWrap, body.replace(QRegExp("(<br />|<br/>)"), "\n")).height(), 50)); // HTML linebreaks result in wrong value for text height using QFontMetrics::boundingRect().

    painter->setClipRect(bodyRect);
    painter->drawStaticText(bodyRect.topLeft(), bodyText);
    painter->setClipping(false);

    QStringList imagePaths = index.data(EventListModel::ImagesRole).toStringList();

    QRect imageRect;

    if (!imagePaths.isEmpty()) {

        imageRect = option.rect;
        imageRect.setTopLeft(QPoint(bodyRect.left(), bodyRect.bottom() + 10));
        imageRect.setSize(QSize(100, 100));

        for (int i = 0; i < imagePaths.size(); i++) {

            QImage image(imagePaths.at(i));

            if (!image.isNull()) {
                if ((image.width() > 100) || (image.height() > 100)) {
                    image = image.scaled(QSize(100, 100), Qt::KeepAspectRatioByExpanding, Qt::SmoothTransformation);
                }

                int hOffset = image.width() > 100 ? qFloor((image.width() - 100) / 2) : 0;
                int vOffset = image.height() > 100 ? qFloor((image.height() - 100) / 2) : 0;
                QRect sourceRect(hOffset, vOffset, 100, 100);
                painter->drawImage(imageRect, image, sourceRect);
            }

            imageRect.moveLeft(imageRect.right() + 10);
        }
    }

    QRect footerRect = option.rect;
    footerRect.setRight(footerRect.right() - 5);

    QString footer = index.data(EventListModel::FooterRole).toString();
    QString date = Utils::dateFromMSecs(index.data(EventListModel::TimestampRole).toLongLong());
    QStaticText footerText(footer.isEmpty() ? date : QString ("%1 - %2").arg(footer).arg(date));
    footerText.setTextWidth(footerRect.width());
    textOption.setWrapMode(QTextOption::NoWrap);
    footerText.setTextOption(textOption);

    if (!imagePaths.isEmpty()) {
        footerRect.setTopLeft(QPoint(bodyRect.left(), imageRect.bottom() + 10));
    }
    else {
        footerRect.setTopLeft(QPoint(bodyRect.left(), bodyRect.bottom() + 10));
    }

    font.setPixelSize(18);

    painter->setClipping(false);
    painter->setFont(font);
    painter->drawStaticText(footerRect.topLeft(), footerText);
    painter->restore();
}

QSize EventListDelegate::sizeHint(const QStyleOptionViewItem &option, const QModelIndex &index) const {
    int bodyHeight = qMin(option.fontMetrics.boundingRect(option.rect, Qt::TextWordWrap, index.data(EventListModel::BodyRole).toString().replace(QRegExp("(<br />|<br/>)"), "\n")).height(), 50);
    QStringList imagePaths = index.data(EventListModel::ImagesRole).toStringList();

    return imagePaths.isEmpty() ? QSize(option.rect.width(), bodyHeight + 80) : QSize(option.rect.width(), bodyHeight + 200);
}
