#include "eventlistwindow.h"
#include "eventlistdelegate.h"
#include "settingsdialog.h"
#include "aboutdialog.h"
#include <QMenuBar>
#include <QDBusConnection>
#include <QMaemo5InformationBox>

EventListWindow::EventListWindow(QWidget *parent) :
    StackedWindow(parent),
    m_actionLauncher(new ActionLauncher(this)),
    m_database(new Database(this)),
    m_model(new EventListModel(m_database, this)),
    m_listView(new ListView(this)),
    m_contextMenu(new QMenu(this)),
    m_removeAction(m_contextMenu->addAction(tr("Remove"), this, SLOT(removeEvent()))),
    m_removeAllAction(m_contextMenu->addAction("", this, SLOT(removeEventsBySourceName()))),
    m_refreshAllAction(menuBar()->addAction(tr("Refresh"), this, SLOT(refreshAll()))),
    m_settingsAction(menuBar()->addAction(tr("Settings"), this, SLOT(showSettingsDialog()))),
    m_aboutAction(menuBar()->addAction(tr("About"), this, SLOT(showAboutDialog()))),
    m_noResultsLabel(new QLabel(QString("<p style='color:#4d4d4d;font-size:40px;'><center><b>%1</b></center></p>").arg(tr("No events")), this))
{
    setWindowTitle(tr("Event Feed"));
    setCentralWidget(m_listView);
    setAttribute(Qt::WA_DeleteOnClose, false);

    m_noResultsLabel->setVisible(m_model->rowCount() == 0);
    m_listView->setModel(m_model);
    m_listView->setItemDelegate(new EventListDelegate(m_listView));
    m_refreshAllAction->setEnabled(!m_database->getRefreshActions().isEmpty());

    connect(m_listView, SIGNAL(clicked(QModelIndex)), this, SLOT(onItemClicked(QModelIndex)));
    connect(m_listView, SIGNAL(customContextMenuRequested(QPoint)), this, SLOT(showContextMenu(QPoint)));
    connect(m_model, SIGNAL(countChanged(int)), this, SLOT(onCountChanged(int)));
    connect(m_actionLauncher, SIGNAL(error(QString)), this, SLOT(onActionError(QString)));

    QDBusConnection bus = QDBusConnection::sessionBus();
    bus.connect("com.maemo.eventFeed", "/", "com.maemo.eventFeed", "eventsUpdated", this, SLOT(onEventsUpdated()));
    bus.connect("com.maemo.eventFeed", "/", "com.maemo.eventFeed", "refreshActionsUpdated", this, SLOT(onRefreshActionsUpdated()));

    onOrientationChanged();
}

void EventListWindow::setLandscapeLayout() {
    m_noResultsLabel->setFixedSize(800, 400);
}

void EventListWindow::setPortraitLayout() {
    m_noResultsLabel->setFixedSize(480, 700);
}

void EventListWindow::onActionError(const QString &errorString) {
    QMaemo5InformationBox::information(this, errorString, QMaemo5InformationBox::NoTimeout);
}

void EventListWindow::onEventsUpdated() {
    m_model->reloadEvents();
    m_listView->positionAtBeginning();
}

void EventListWindow::onRefreshActionsUpdated() {
    m_refreshAllAction->setEnabled(!m_database->getRefreshActions().isEmpty());
}

void EventListWindow::onCountChanged(int count) {
    m_noResultsLabel->setVisible(count == 0);
}

void EventListWindow::onItemClicked(const QModelIndex &index) {
    m_actionLauncher->launchAction(index.data(EventListModel::ActionRole).toString());
}

void EventListWindow::showContextMenu(const QPoint &pos) {
    if (m_model->rowCount() > 0) {
        QString appName = m_listView->currentIndex().data(EventListModel::SourceDisplayNameRole).toString();
        m_removeAllAction->setText(tr("Remove %1").arg(appName));
        m_contextMenu->popup(pos, m_removeAction);
    }
}

void EventListWindow::removeEvent() {
    if (m_model->removeEvent(m_listView->currentIndex().row())) {
        emit eventsUpdated();
    }
}

void EventListWindow::removeEventsBySourceName() {
    if (m_model->removeEventsBySourceName(m_listView->currentIndex().data(EventListModel::SourceNameRole).toString())) {
        emit eventsUpdated();
    }
}

void EventListWindow::refreshAll() {
    m_actionLauncher->launchActions(m_database->getRefreshActions());
}

void EventListWindow::showSettingsDialog() {
    SettingsDialog *dialog = new SettingsDialog(this);
    dialog->open();
}

void EventListWindow::showAboutDialog() {
    AboutDialog *dialog = new AboutDialog(this);
    dialog->open();
}
