#include "eventlistwidget.h"
#include "eventlistdelegate.h"
#include "settingsdialog.h"
#include <QDBusConnection>
#include <QPainter>

EventListWidget::EventListWidget(QWidget *parent) :
    QWidget(parent),
    m_actionLauncher(new ActionLauncher(this)),
    m_database(new Database(this)),
    m_model(new EventListModel(m_database, this)),
    m_listView(new ListView(this)),
    m_upButton(new WidgetButton(QIcon::fromTheme("rss_reader_move_up"), "RSSAppletLeftButton", this)),
    m_refreshButton(new WidgetButton(QIcon::fromTheme("general_refresh"), "RSSAppletCenterButton", this)),
    m_downButton(new WidgetButton(QIcon::fromTheme("rss_reader_move_down"), "RSSAppletRightButton", this)),
    m_noResultsLabel(new QLabel(QString("<p style='color:#ffffff;font-size:24px;'><center><b>%1</b></center></p>").arg(tr("No events")), this)),
    m_grid(new QGridLayout(this))
{
    setAttribute(Qt::WA_TranslucentBackground, true);
    setFixedSize(390, 330);

    m_grid->setContentsMargins(0, 15, 0, 0);
    m_grid->setSpacing(0);
    m_grid->addWidget(m_listView, 0, 0, 1, 3, Qt::AlignHCenter);
    m_grid->addWidget(m_upButton, 1, 0);
    m_grid->addWidget(m_refreshButton, 1, 1);
    m_grid->addWidget(m_downButton, 1, 2);

    m_listView->setFixedWidth(370);
    m_listView->setModel(m_model);
    m_listView->setItemDelegate(new EventListDelegate(m_listView));
    m_refreshButton->setEnabled(!m_database->getRefreshActions().isEmpty());
    m_noResultsLabel->setFixedSize(390, 330);
    m_noResultsLabel->setVisible(m_model->rowCount() == 0);
    m_upButton->raise();
    m_refreshButton->raise();
    m_downButton->raise();

    connect(m_upButton, SIGNAL(clicked()), m_listView, SLOT(moveUp()));
    connect(m_refreshButton, SIGNAL(clicked()), this, SLOT(refresh()));
    connect(m_downButton, SIGNAL(clicked()), m_listView, SLOT(moveDown()));
    connect(m_listView, SIGNAL(clicked(QModelIndex)), this, SLOT(onItemClicked(QModelIndex)));
    connect(m_model, SIGNAL(countChanged(int)), this, SLOT(onCountChanged(int)));

    QDBusConnection bus = QDBusConnection::sessionBus();
    bus.connect("com.maemo.eventFeedUI", "/", "com.maemo.eventFeedUI", "eventsUpdated", this, SLOT(onEventsUpdated()));
    bus.connect("com.maemo.eventFeed", "/", "com.maemo.eventFeed", "eventsUpdated", this, SLOT(onEventsUpdated()));
    bus.connect("com.maemo.eventFeed", "/", "com.maemo.eventFeed", "refreshActionsUpdated", this, SLOT(onRefreshActionsUpdated()));
}

void EventListWidget::onCountChanged(int count) {
    m_noResultsLabel->setVisible(count == 0);
}

void EventListWidget::onItemClicked(const QModelIndex &index) {
    m_actionLauncher->launchAction(index.data(EventListModel::ActionRole).toString());
}

void EventListWidget::onEventsUpdated() {
    m_model->reloadEvents();
    m_listView->positionAtBeginning();
}

void EventListWidget::onRefreshActionsUpdated() {
    m_refreshButton->setEnabled(!m_database->getRefreshActions().isEmpty());
}

void EventListWidget::refresh() {
    m_actionLauncher->launchActions(m_database->getRefreshActions());
}

void EventListWidget::showSettingsDialog() {
    SettingsDialog *dialog = new SettingsDialog(this);
    dialog->open();
}

void EventListWidget::paintEvent(QPaintEvent *event) {
    QPainter painter(this);
    QImage image("/etc/hildon/theme/images/HildonEventFeedAppletHead.png");

    if (!image.isNull()) {
        painter.drawImage(0, 0, image);
    }
    else {
        painter.drawImage(0, 0, QImage(":/HildonEventFeedAppletHead.png"));
    }

    QWidget::paintEvent(event);
}
