#include "eventlistmodel.h"

EventListModel::EventListModel(Database *database, QObject *parent) :
    QAbstractListModel(parent),
    m_database(database)
{
    loadEvents();
}

EventListModel::~EventListModel() {
}

void EventListModel::loadEvents() {
    m_list = m_database->getEvents();
    emit countChanged(rowCount());
}

void EventListModel::reloadEvents() {
    beginResetModel();
    m_list.clear();
    m_list = m_database->getEvents();
    endResetModel();
    emit countChanged(rowCount());
}

int EventListModel::rowCount(const QModelIndex &parent) const {
    Q_UNUSED(parent);

    return m_list.size();
}

QVariant EventListModel::data(const QModelIndex &index, int role) const {
    switch (role) {
    case EventIdRole:
        return m_list.at(index.row()).eventId();
    case SourceNameRole:
        return m_list.at(index.row()).sourceName();
    case SourceDisplayNameRole:
        return m_list.at(index.row()).sourceDisplayName();
    case IconRole:
        return m_list.at(index.row()).iconPath();
    case TitleRole:
        return m_list.at(index.row()).title();
    case BodyRole:
        return m_list.at(index.row()).body();
    case ImagesRole:
        return m_list.at(index.row()).imagePaths();
    case FooterRole:
        return m_list.at(index.row()).footer();
    case TimestampRole:
        return m_list.at(index.row()).timestamp();
    case ActionRole:
        return m_list.at(index.row()).action();
    default:
        return QVariant();
    }
}

Event EventListModel::get(int row) const {
    return (row >= 0) && (row < m_list.size()) ? m_list.at(row) : Event();
}

void EventListModel::removeRow(int row, const QModelIndex &parent) {
    if ((row >= 0) && (row <= m_list.size())) {
        beginRemoveRows(parent, row, row);
        m_list.removeAt(row);
        endRemoveRows();
        emit countChanged(rowCount());
    }
}

bool EventListModel::removeEvent(int row) {
    if (m_database->removeEvent(data(index(row, 0), EventIdRole).toLongLong())) {
        removeRow(row);
        return true;
    }

    return false;
}

bool EventListModel::removeEventsBySourceName(const QString &sourceName) {
    if (m_database->removeEventsBySourceName(sourceName)) {
        reloadEvents();
        return true;
    }

    return false;
}
