#include <gdk/gdkx.h>
#include "systray-process.h"

#define SYSTEM_TRAY_REQUEST_DOCK 0

typedef struct {
	GtkWidget *systray_widget;
	GtkWidget *vbox;
	Atom tray_opcode;
	GtkWidget *invisible;
	int icon_size;
	WindowIDCallback cb;
	gpointer cb_user_data;
} SystrayProcessParams;

static SystrayProcessParams *
systray_process_params_new()
{
	SystrayProcessParams *spp = g_new0(SystrayProcessParams, 1);

	spp->vbox = NULL;
	spp->tray_opcode = 0;
	spp->invisible = NULL;
	spp->icon_size = 16;
	spp->cb = NULL;
	spp->cb_user_data = NULL;

	return spp;
}

static SystrayProcessParams *
systray_process_params_free(SystrayProcessParams *spp)
{
	if (spp->invisible)
		gtk_widget_destroy(spp->invisible);
	g_free(spp);

	return NULL;
}

typedef struct {
	int cx, cy;
	GtkWidget *hbox;
	GtkWidget *systray_widget;
} FindHBoxParams;

static void
find_hbox(GtkWidget *hbox, FindHBoxParams *fhbp)
{
	if (!(fhbp->hbox))
		if (g_type_is_a(G_TYPE_FROM_INSTANCE(hbox), GTK_TYPE_HBOX)) {
			GtkRequisition rq = { .width = 0, .height = 0 };

			gtk_widget_size_request(hbox, &rq);
			if ((rq.width + fhbp->cx) <= fhbp->systray_widget->allocation.width)
				fhbp->hbox = hbox;
		}
}

static void
embed_tray_item(SystrayProcessParams *spp, Display *dpy, GdkNativeWindow wid)
{
	FindHBoxParams fhbp = {
		.cx = 0,
		.cy = 0,
		.hbox = NULL,
		.systray_widget = spp->systray_widget
	};
	GtkWidget *socket;
	GtkWidget *frame;

	fhbp.cx = spp->icon_size;
	fhbp.cy = 0;

	gtk_container_foreach(GTK_CONTAINER(spp->vbox), (GtkCallback)find_hbox, &fhbp);

	if (!(fhbp.hbox)) {
		fhbp.hbox = g_object_new(GTK_TYPE_HBOX, "visible", TRUE, "spacing", 8, NULL);
		gtk_box_pack_start(GTK_BOX(spp->vbox), fhbp.hbox, FALSE, TRUE, 0);
	}

	socket = g_object_new(GTK_TYPE_SOCKET, "visible", TRUE, NULL);
	gtk_widget_set_size_request(socket, -1, spp->icon_size);
	gtk_box_pack_start(GTK_BOX(fhbp.hbox), 
		frame = g_object_new(GTK_TYPE_FRAME, "visible", TRUE, "child", socket, NULL),
		FALSE, TRUE, 0);
	g_signal_connect_swapped(G_OBJECT(socket), "plug-removed", (GCallback)gtk_widget_destroy, frame);
	gtk_socket_add_id(GTK_SOCKET(socket), wid);

	if (spp->cb)
		spp->cb(wid, spp->cb_user_data);
}

static GdkFilterReturn
invisible_window_x_event_filter(GdkXEvent *x_event, GdkEvent *event, SystrayProcessParams *spp)
{
	XEvent *xev = (XEvent *)x_event;

	g_print("invisible_window_x_event_filter: Entering\n");

	if (ClientMessage == xev->type) {
		if (xev->xclient.message_type == spp->tray_opcode &&
		    xev->xclient.data.l[1] == SYSTEM_TRAY_REQUEST_DOCK) {
			embed_tray_item(spp, xev->xany.display, xev->xclient.data.l[2]);
		}
	}

	return GDK_FILTER_CONTINUE;
}

void
systray_setup(GtkWidget *widget, int icon_spacing, int icon_size, WindowIDCallback cb, gpointer cb_user_data)
{
	SystrayProcessParams *spp = NULL;

	g_return_if_fail(GTK_IS_CONTAINER(widget) && GTK_WIDGET_REALIZED(widget));

	if (g_object_get_data(G_OBJECT(widget), "spp") != NULL) return;

	spp = systray_process_params_new();

	if (spp) {
		Display *dpy;
		Window root;
		Atom tray_selection;
		GdkScreen *default_screen;
		char *tray_string = NULL;

		g_object_set_data_full(G_OBJECT(widget), "spp", spp, (GDestroyNotify)systray_process_params_free);
		spp->icon_size = icon_size;
		spp->cb = cb;
		spp->cb_user_data = cb_user_data;
		spp->systray_widget = widget;

		spp->vbox = g_object_new(GTK_TYPE_VBOX, "visible", TRUE, "spacing", icon_spacing, "border-width", icon_spacing, NULL);
		gtk_container_add(GTK_CONTAINER(widget), 
			g_object_new(GTK_TYPE_ALIGNMENT,
				"visible", TRUE,
				"xalign", 0.5, "yalign", 0.5,
				"xscale", 0.0, "yscale", 0.0,
				"child", spp->vbox, 
				NULL));
#if (0)
		gtk_box_pack_start(GTK_BOX(spp->vbox),
			g_object_new(GTK_TYPE_ALIGNMENT,
				"visible", TRUE,
				"xalign", 0.5, "yalign", 0.5, "xscale", 0.0, "yscale", 0.0,
				"child", g_object_new(GTK_TYPE_LABEL,
					"visible", TRUE,
					"label", "Systray:",
					NULL), NULL), FALSE, TRUE, 0);
#endif /* (0) */

		default_screen = gdk_screen_get_default();
		dpy = GDK_DISPLAY();
		tray_string = g_strdup_printf("_NET_SYSTEM_TRAY_S%d", GDK_SCREEN_XNUMBER(default_screen));
		tray_selection = XInternAtom(dpy, tray_string, False);
		g_free(tray_string); tray_string = NULL;
		spp->tray_opcode = XInternAtom(dpy, "_NET_SYSTEM_TRAY_OPCODE", False);
		spp->invisible = gtk_invisible_new_for_screen(default_screen);
		gtk_widget_realize(spp->invisible);
		gtk_widget_add_events(spp->invisible, GDK_PROPERTY_CHANGE_MASK | GDK_STRUCTURE_MASK);
		XSetSelectionOwner(dpy, tray_selection, GDK_WINDOW_XID(spp->invisible->window), CurrentTime);

		root = GDK_WINDOW_XID(gdk_get_default_root_window());

		if (XGetSelectionOwner(dpy, tray_selection) == GDK_WINDOW_XID(spp->invisible->window))
		{
			XClientMessageEvent xev;

			xev.type = ClientMessage;
			xev.window = root;
			xev.message_type = XInternAtom(dpy, "MANAGER", False);
			xev.format = 32;
			xev.data.l[0] = CurrentTime;
			xev.data.l[1] = tray_selection;
			xev.data.l[2] = GDK_WINDOW_XID(spp->invisible->window);
			xev.data.l[3] = 0;
			xev.data.l[4] = 0;

			gdk_error_trap_push();

			XSendEvent(dpy, root, False, StructureNotifyMask, (XEvent *)&xev);

			gdk_error_trap_pop();

			g_print("systray_setup: Adding filter\n");
			gdk_window_add_filter(spp->invisible->window, (GdkFilterFunc)invisible_window_x_event_filter, spp);
		}
		else
			g_print("systray_setup: XGetSelectionOwner failed\n");
	}
}
