/*
 * This file is part of hildon-input-method-plugins-example 
 *
 * Copyright (C) 2006-2007 Nokia Corporation. All rights reserved.
 *
 * Author: Joaquim Rocha <jrocha@igalia.com>
 *
 * Redistribution and use in source and binary forms, with or without modification, are permitted provided that the following conditions are met:
 *
    * Redistributions of source code must retain the above copyright notice, this list of conditions and the following disclaimer.
    * Redistributions in binary form must reproduce the above copyright notice, this list of conditions and the following disclaimer in the documentation and/or other materials provided with the distribution.
    * Neither the name of Nokia Corporation nor the names of its contributors may be used to endorse or promote products derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF 
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, 
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED
 * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 * 
*/

#include "hildon-im-plugin.h"
#include "hildon-im-ui.h"
#include "chewing.h"

#include <stdio.h>
#include <string.h>
#include <glib.h>
#include <gdk/gdk.h>
#include <gdk/gdkx.h>
#include <libosso.h>
#include <mce/dbus-names.h>
#include <gtk/gtk.h>
#include <hildon/hildon.h>
#include "../config.h"

#define SLIDING_TIMEOUT 1000
#define FAVORITE_HAND RIGHT
#define KEYS_VALUES_DATA "key_values"
#define PERSISTENT_VALUE_DATA "persistent_value"

#define HILDON_IM_CHEWING_VKB_TYPE hildon_im_chewing_vkb_get_type ()
#define HILDON_IM_CHEWING_VKB(obj) GTK_CHECK_CAST(obj, hildon_im_chewing_vkb_get_type (), HildonIMChewingVKB)
#define HILDON_IM_CHEWING_VKB_CLASS(klass) \
        GTK_CHECK_CLASS_CAST(klass, hildon_im_chewing_vkb_get_type, \
                             HildonIMChewingVKBClass)
#define HILDON_IS_IM_CHEWING_VKB(obj) \
        GTK_CHECK_TYPE(obj, hildon_im_chewing_vkb_get_type ())
#define HILDON_IM_CHEWING_VKB_GET_PRIVATE(obj) \
        (G_TYPE_INSTANCE_GET_PRIVATE ((obj), HILDON_IM_CHEWING_VKB_TYPE,\
                                      HildonIMChewingVKBPrivate))

static gchar* qwerty_keys[] = {"1","2","3","4","5","6","7","8","9","0",
				"q","w","e","r","t","y","u","i","o","p","-",
				"a","s","d","f","g","h","j","k","l",";",
				"z","x","c","v","b","n","m",",",".","/"};
static gchar* chewing_keys[] = {"ㄅ","ㄉ","ˇ","ˋ","ㄓ","ˊ","˙","ㄚ","ㄞ","ㄢ",
				"ㄆ","ㄊ","ㄍ","ㄐ","ㄔ","ㄗ","ㄧ","ㄛ","ㄟ","ㄣ","ㄦ",
				"ㄇ","ㄋ","ㄎ","ㄑ","ㄕ","ㄘ","ㄨ","ㄜ","ㄠ","ㄤ",
				"ㄈ","ㄌ","ㄏ","ㄒ","ㄖ","ㄙ","ㄩ","ㄝ","ㄡ","ㄥ"};
static gchar* qwerty_symbol_keys[] = {"1","2","3","4","5","6","7","8","9","0",
				"<",">","[","]","{","}","|","%","~","^","-",
				"*","+","#","-","_","(",")","&","=","!",
				";","$",":","/","\\","\"","\'","@","...","?"};
static gchar* full_symbol_keys[] ={"1","2","3","4","5","6","7","8","9","0",
				"〈","〉","『","』","「","」","％","＊","～","“","”",
				"×","＋","＃","－","√","（","）","＆","＝","！",
				"；","＄","︰","÷","、","。","，","＠","…","？"};

static int selKey_define[ 9 ] = {'1','2','3','4','5','6','7','8',0}; /* Default */

ChewingContext *ctx;

typedef enum
{
  PORTRAIT,
  LANDSCAPE
} ScreenMode;

typedef enum
{
  RIGHT,
  LEFT
} HandMode;

typedef struct
{
  GtkContainerClass parent;
}
HildonIMChewingVKBClass;

typedef struct
{
  GtkContainer parent;
  
}
HildonIMChewingVKB;

typedef struct
{
  HildonIMUI *ui;

  GtkWidget *window;
  GtkWidget *text_view;
  GtkWidget *chewing_buffer;
	GtkWidget *container;
  GtkWidget *keyboard;
  GtkWidget *keyboard_symbols;

  GtkWidget *keyboard_keys[G_N_ELEMENTS (qwerty_keys)];
  GtkWidget *text_view_row;
  GtkWidget *keyboard_rows[7];

  GtkWidget *caps_button;
  GtkWidget *enter_button;
  GtkWidget *backspace_button;
  GtkWidget *passage_button;
  GtkWidget *forwardpage_button;
  GtkWidget *symbol_page_button;
  GtkWidget *modechange_button;
  GtkWidget *space_button;
  GtkWidget *close_button;
  GtkWidget *send_button;
  GtkWidget *select_button;
  GtkWidget *dialog;

  gint saved_offset;

  gint keys_width;

  ScreenMode screen_mode;
  HandMode hand_mode;
  gint fkb_width;
  gint fkb_height;

  gint slide_index;
  gint replace_previous_char;
  gint timeout_id;

  GtkWidget *repeating_key;
  GtkWidget *pressed_key;
  GdkScreen *screen;

  gint keyboard_page;
  
}
HildonIMChewingVKBPrivate;

static GType hildon_im_chewing_vkb_type = 0;
static GtkWidgetClass *parent_class = NULL;

GType hildon_im_chewing_vkb_get_type (void);
GtkWidget *hildon_im_chewing_vkb_new (HildonIMUI *kbd);

/* 
 * HildonIMPlugin interface
 */
static void hildon_im_chewing_vkb_iface_init (HildonIMPluginIface *iface);

static void hildon_im_chewing_vkb_enable (HildonIMPlugin *plugin, gboolean init);
static void hildon_im_chewing_vkb_disable (HildonIMPlugin *plugin);
static void hildon_im_chewing_vkb_surrounding_received (HildonIMPlugin *plugin,
                                                       const gchar *surrounding,
                                                       gint offset);

/*
 * GObject functions
 */
static void hildon_im_chewing_vkb_finalize     (GObject *obj);
static void hildon_im_chewing_vkb_get_property (GObject *object,
                                                guint prop_id,
                                                GValue *value,
                                                GParamSpec *pspec);
static void hildon_im_chewing_vkb_set_property (GObject *object,
                                                guint prop_id,
                                                const GValue *value,
                                                GParamSpec *pspec);
 
static void hildon_im_chewing_vkb_class_init (HildonIMChewingVKBClass *klass);
static void hildon_im_chewing_vkb_init       (HildonIMChewingVKB *self);

/*
 * Internal functions
 */
static void create_window (HildonIMChewingVKB *self);

static GtkWidget *
create_keyboard(HildonIMChewingVKB *self, gchar **keys, gint page);

static void set_fkb_size (HildonIMChewingVKB *self);
static void set_kb_mode (HildonIMChewingVKB *self, ScreenMode mode);
static void backspace (HildonIMChewingVKB *self);
static void write (HildonIMChewingVKB *self, const gchar *text);
static void chewing_write (HildonIMChewingVKB *self, const gchar *text);
static gboolean get_cursor_offset (HildonIMChewingVKB *self);
static void clear_timeout_id (HildonIMChewingVKB *self);
static void chewing_candkey_refresh(HildonIMChewingVKB *self);

static void swap_keyboard_page (HildonIMChewingVKB *self,gint mode_number);
/*
 * Module functions
 */

HildonIMPlugin* 
module_create (HildonIMUI *keyboard)
{
  return HILDON_IM_PLUGIN (hildon_im_chewing_vkb_new (keyboard));
}

void
module_exit(void)
{
  /* empty */
}

void
module_init(GTypeModule *module)
{
  static const GTypeInfo type_info = {
    sizeof(HildonIMChewingVKBClass),
    NULL, /* base_init */
    NULL, /* base_finalize */
    (GClassInitFunc) hildon_im_chewing_vkb_class_init,
    NULL, /* class_finalize */
    NULL, /* class_data */
    sizeof(HildonIMChewingVKB),
    0,    /* n_preallocs */
    (GInstanceInitFunc) hildon_im_chewing_vkb_init,
  };

  static const GInterfaceInfo plugin_info = {
    (GInterfaceInitFunc) hildon_im_chewing_vkb_iface_init,
    NULL, /* interface_finalize */
    NULL, /* interface_data */
  };

  hildon_im_chewing_vkb_type =
          g_type_module_register_type(module,
                                      GTK_TYPE_CONTAINER, "HildonIMChewingVKB",
                                      &type_info,
                                      0);
  
  g_type_module_add_interface(module,
                              HILDON_IM_CHEWING_VKB_TYPE,
                              HILDON_IM_TYPE_PLUGIN,
                              &plugin_info);
}

/*
 * This is used to know the plugin's information when loading the module
 */
const HildonIMPluginInfo *
hildon_im_plugin_get_info(void)
{
  static const HildonIMPluginInfo info =
  {
    "HIM CHEWING VKB",                  /* description */
    "hildon_im_chewing_vkb",            /* name */
    NULL,                               /* menu title */
    NULL,                               /* gettext domain */
    TRUE,                               /* visible in menu */
    FALSE,                              /* cached */
    HILDON_IM_TYPE_FULLSCREEN,  	/* UI type */
    HILDON_IM_GROUP_LATIN,              /* group */
    HILDON_IM_DEFAULT_PLUGIN_PRIORITY,  /* priority */
    NULL,                               /* special character plugin */
    NULL,                               /* help page */
    TRUE,                               /* disable common UI buttons */
    0,                                  /* plugin height */
    HILDON_IM_TRIGGER_FINGER            /* trigger */
  };

  return &info;
}

/*
 * This function returns the list of available languages supported
 * by the plugin.
 */
gchar ** 
hildon_im_plugin_get_available_languages (gboolean *free)
{
  static gchar *langs[] = {"en_GB","zh_TW", NULL};
  *free = FALSE;

  return langs;
}

GType
hildon_im_chewing_vkb_get_type (void)
{
  return hildon_im_chewing_vkb_type;
}

/*
 * Implement the interface.
 */
static void
hildon_im_chewing_vkb_iface_init (HildonIMPluginIface *iface)
{
  iface->enable = hildon_im_chewing_vkb_enable;
  iface->disable = hildon_im_chewing_vkb_disable;
  iface->surrounding_received = hildon_im_chewing_vkb_surrounding_received;
}

static void
hildon_im_chewing_vkb_class_init (HildonIMChewingVKBClass *klass)
{
  GObjectClass *object_class;
  GtkObjectClass *gtk_object_class;
  GtkWidgetClass *widget_class;
  GtkContainerClass *container_class;

  parent_class = g_type_class_peek_parent (klass);
  g_type_class_add_private (klass, sizeof (HildonIMChewingVKBPrivate));

  object_class = G_OBJECT_CLASS(klass);
  gtk_object_class = GTK_OBJECT_CLASS(klass);
  widget_class = GTK_WIDGET_CLASS(klass);
  container_class = GTK_CONTAINER_CLASS(klass);

  object_class->set_property  = hildon_im_chewing_vkb_set_property;
  object_class->get_property  = hildon_im_chewing_vkb_get_property;
  object_class->finalize      = hildon_im_chewing_vkb_finalize;
  
  g_object_class_install_property (object_class, HILDON_IM_PROP_UI,
                                   g_param_spec_object (HILDON_IM_PROP_UI_DESCRIPTION, 
                                                        HILDON_IM_PROP_UI_DESCRIPTION,
                                                        "UI that uses plugin",
                                                        HILDON_IM_TYPE_UI,
                                                        G_PARAM_READWRITE
                                                        | G_PARAM_CONSTRUCT_ONLY));
}

static void
hildon_im_chewing_vkb_init (HildonIMChewingVKB *self)
{
  HildonIMChewingVKBPrivate *priv;

  g_return_if_fail (HILDON_IS_IM_CHEWING_VKB (self));

  priv = HILDON_IM_CHEWING_VKB_GET_PRIVATE (self);

  priv->enter_button = NULL;
  priv->backspace_button = NULL;
  priv->passage_button = NULL;
  priv->forwardpage_button = NULL;
  priv->symbol_page_button = NULL;
  priv->text_view = NULL;
  priv->chewing_buffer = NULL;
  priv->keyboard = NULL;
  priv->caps_button = NULL;
  priv->modechange_button = NULL;
  priv->close_button = NULL;
  priv->space_button = NULL;
  priv->send_button = NULL;
  priv->select_button = NULL;
	
  priv->repeating_key = NULL;
  priv->pressed_key = NULL;
  
  priv->dialog = NULL;

  priv->screen_mode = LANDSCAPE;
  priv->hand_mode = FAVORITE_HAND;
  priv->replace_previous_char = FALSE;
  priv->slide_index = -1;
  priv->timeout_id = 0;

  priv->keyboard_page = 0;
  
}

static void 
hildon_im_chewing_vkb_finalize(GObject *obj)
{
  if (G_OBJECT_CLASS (parent_class)->finalize)
  {
    G_OBJECT_CLASS (parent_class)->finalize (obj);
  }
}

GtkWidget *
hildon_im_chewing_vkb_new (HildonIMUI *kbd)
{
  return g_object_new (HILDON_IM_CHEWING_VKB_TYPE,
                       HILDON_IM_PROP_UI_DESCRIPTION, kbd, NULL);
}

static void
hildon_im_chewing_vkb_get_property (GObject *object,
                                    guint prop_id,
                                    GValue *value,
                                    GParamSpec *pspec)
{
  HildonIMChewingVKBPrivate *priv;

  g_return_if_fail (HILDON_IS_IM_CHEWING_VKB(object));
  priv = HILDON_IM_CHEWING_VKB_GET_PRIVATE(object);

  switch (prop_id)
  {
    case HILDON_IM_PROP_UI:
      g_value_set_object(value, priv->ui);
      break;

    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
  }
}

static void
hildon_im_chewing_vkb_set_property (GObject *object,
                                    guint prop_id,
                                    const GValue *value,
                                    GParamSpec *pspec)
{
  HildonIMChewingVKBPrivate *priv;

  g_return_if_fail (HILDON_IS_IM_CHEWING_VKB (object));
  priv = HILDON_IM_CHEWING_VKB_GET_PRIVATE(object);

  switch (prop_id)
  {
    case HILDON_IM_PROP_UI:
      priv->ui = g_value_get_object(value);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
      break;
  }
}



static ScreenMode
get_screen_mode (GdkScreen *screen){
    
    int width = gdk_screen_get_width(screen);
    int height = gdk_screen_get_height(screen);
    if (width > height) {
        return LANDSCAPE;
    } else {
        return PORTRAIT;
    }

}
static void
unparent_contents (HildonIMChewingVKB *self)
{
  HildonIMChewingVKBPrivate *priv;
  priv = HILDON_IM_CHEWING_VKB_GET_PRIVATE (self);

  if (!GTK_IS_WIDGET (priv->container))
  {
    return;
  }


  gtk_container_remove (GTK_CONTAINER (priv->text_view_row), priv->text_view);
  gtk_container_remove (GTK_CONTAINER (priv->text_view_row), priv->chewing_buffer);
  gtk_container_remove (GTK_CONTAINER (priv->container), priv->keyboard);
  gtk_container_remove (GTK_CONTAINER (priv->container), priv->text_view_row);
  gtk_container_remove (GTK_CONTAINER (priv->window), priv->container);
}

static void
hildon_im_chewing_vkb_enable (HildonIMPlugin *plugin, gboolean init)
{

  HildonIMChewingVKB *self;
  HildonIMChewingVKBPrivate *priv;
  
  g_return_if_fail (HILDON_IS_IM_CHEWING_VKB (plugin));
  self = HILDON_IM_CHEWING_VKB(plugin);
  priv = HILDON_IM_CHEWING_VKB_GET_PRIVATE(self);
  
  priv->screen = gdk_screen_get_default();
  priv->screen_mode = get_screen_mode (priv->screen);

  set_fkb_size (self);

    if (priv->window == NULL){
	chewing_Init( "/usr/share/chewing", "/tmp" );
	ctx = chewing_new();
	    
	chewing_set_KBType( ctx, chewing_KBStr2Num( "KB_DEFAULT" ) );

	chewing_set_candPerPage( ctx, 8 );
	chewing_set_maxChiSymbolLen( ctx, 16 );
	chewing_set_selKey( ctx, selKey_define, 10 );
	chewing_set_addPhraseDirection( ctx, 1 );
	chewing_set_spaceAsSelection( ctx, 1 );
	    
	create_window (self);

	/*dbus_bus_add_match (connection, MCE_RULE, NULL);
	dbus_connection_add_filter (connection, screen_mode_dbus_handler, self, NULL);*/
    }

  hildon_im_ui_send_communication_message (priv->ui,
					   HILDON_IM_CONTEXT_REQUEST_SURROUNDING_FULL);
  gtk_window_fullscreen (GTK_WINDOW (priv->window));
  gtk_widget_show_all (priv->window);

  gtk_widget_hide( priv->keyboard_symbols );
  gtk_widget_show( priv->keyboard );
  gtk_widget_hide( priv->chewing_buffer );
  gtk_widget_hide( priv->passage_button );
  gtk_widget_hide( priv->forwardpage_button );

	swap_keyboard_page(self,priv->keyboard_page);
  gdk_window_set_transient_for (GTK_WIDGET (priv->window)->window,
                                gtk_widget_get_root_window(GTK_WIDGET(priv->window)));

  priv->saved_offset = get_cursor_offset (self);	
}

static void
hildon_im_chewing_vkb_disable (HildonIMPlugin *plugin)
{

  HildonIMChewingVKB *self;
  HildonIMChewingVKBPrivate *priv;

  g_return_if_fail (HILDON_IS_IM_CHEWING_VKB (plugin));
  self = HILDON_IM_CHEWING_VKB (plugin);
  priv = HILDON_IM_CHEWING_VKB_GET_PRIVATE (self);

	//gtk_widget_hide (GTK_WIDGET (priv->window));
    
    unparent_contents(self);
    chewing_Reset(ctx);
    chewing_Terminate();
	gtk_widget_destroy(GTK_WIDGET(priv->window));
	priv->window=NULL;
    //hildon_im_ui_restore_previous_mode (priv->ui);
}

static void
hildon_im_chewing_vkb_surrounding_received(HildonIMPlugin *plugin,
                                           const gchar *surrounding,
                                           gint offset)
{
  HildonIMChewingVKB *self;
  HildonIMChewingVKBPrivate *priv;
  GtkTextBuffer *buffer;
  GtkTextIter cursor_iter;

  self = HILDON_IM_CHEWING_VKB(plugin);
  priv = HILDON_IM_CHEWING_VKB_GET_PRIVATE(self);

  buffer = gtk_text_view_get_buffer(GTK_TEXT_VIEW(priv->text_view));

  /* I don't know why, but the view doesn't reflect the changes immediately */
  gtk_text_buffer_set_text(buffer, surrounding, -1);
  gtk_text_buffer_get_iter_at_offset(buffer, &cursor_iter, offset);
  gtk_text_buffer_place_cursor(buffer, &cursor_iter);
  
  priv->saved_offset = offset;
}

static gint
get_cursor_offset (HildonIMChewingVKB *self)
{
  HildonIMChewingVKBPrivate *priv;
  GtkTextBuffer *buffer;
  GtkTextIter iter;

  priv = HILDON_IM_CHEWING_VKB_GET_PRIVATE (self);

  buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW (priv->text_view));
  gtk_text_buffer_get_iter_at_mark (buffer,
                                    &iter,
                                    gtk_text_buffer_get_selection_bound (buffer));

  return gtk_text_iter_get_offset (&iter);
}

static gboolean
textview_button_press_cb (GtkWidget *textview, GdkEventButton *event, gpointer data)
{
  HildonIMChewingVKB *self;
  HildonIMChewingVKBPrivate *priv;
  gint offset;

  g_return_val_if_fail (HILDON_IS_IM_CHEWING_VKB (data), FALSE);
  self = HILDON_IM_CHEWING_VKB (data);
  priv = HILDON_IM_CHEWING_VKB_GET_PRIVATE (self);

  priv->saved_offset = get_cursor_offset (self);

  GTK_WIDGET_GET_CLASS (priv->text_view)->button_press_event (priv->text_view, event);

  if (hildon_im_ui_get_commit_mode (priv->ui) == HILDON_IM_COMMIT_REDIRECT)
  {
    offset = get_cursor_offset (self);

    hildon_im_ui_send_surrounding_offset (priv->ui,
                                          TRUE,
                                          offset - priv->saved_offset);
    priv->saved_offset = offset;
  }

  return TRUE;
}

static gboolean
delete_selection (HildonIMChewingVKB *self)
{
  HildonIMChewingVKBPrivate *priv;
  GtkTextIter start, end;
  GtkTextBuffer *buffer;

  priv = HILDON_IM_CHEWING_VKB_GET_PRIVATE (self);

  buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW (priv->text_view));

  if (gtk_text_buffer_get_selection_bounds (buffer, &start, &end))
  {
    if (hildon_im_ui_get_commit_mode (priv->ui) == HILDON_IM_COMMIT_REDIRECT)
    {
      gchar *selected_text = gtk_text_buffer_get_text (buffer, &start, &end, FALSE);

      hildon_im_ui_send_surrounding_offset (priv->ui,
                                            TRUE,
                                            gtk_text_iter_get_offset (&end) - get_cursor_offset (self));

      gint i;
      for (i = 0; i < g_utf8_strlen (selected_text, -1); i++)
      {
        hildon_im_ui_send_communication_message (priv->ui,
                                                 HILDON_IM_CONTEXT_HANDLE_BACKSPACE);
      }
    }
    gtk_text_buffer_delete (buffer, &start, &end);

    return TRUE;
  }

  return FALSE;
}

static void
close_fkb_cb (GtkWidget *widget, gpointer data)
{
    HildonIMChewingVKBPrivate *priv;
    HildonIMChewingVKB *self = HILDON_IM_CHEWING_VKB (data);
    priv = HILDON_IM_CHEWING_VKB_GET_PRIVATE (self);
   
    //gtk_widget_hide (GTK_WIDGET (priv->window));
    
    unparent_contents(self);
    chewing_Reset(ctx);
    chewing_Terminate();
	gtk_widget_destroy(GTK_WIDGET(priv->window));
    //hildon_im_ui_restore_previous_mode (priv->ui);
//  hildon_im_chewing_vkb_disable (HILDON_IM_PLUGIN (data));
}
static void 
change_keys (HildonIMChewingVKB *self,gchar **keys) {
  HildonIMChewingVKBPrivate *priv;	
  priv = HILDON_IM_CHEWING_VKB_GET_PRIVATE (self);
  PangoFontDescription *desc_info = pango_font_description_from_string("Sans 24");
  int i = 0;
	if(priv->keyboard_page == 1 && chewing_cand_TotalChoice(ctx)){
            desc_info = pango_font_description_from_string("Sans 24");
		while (i < 8){
			gchar* label = keys[i];
			/*g_object_set_data (G_OBJECT (priv->keyboard_keys[i]), KEYS_VALUES_DATA, label);
			g_object_set_data (G_OBJECT (priv->keyboard_keys[i]), PERSISTENT_VALUE_DATA, label);*/
			//hildon_button_set_title (HILDON_BUTTON (priv->keyboard_keys[i]), label);
			gtk_button_set_label ( GTK_BUTTON (priv->keyboard_keys[i]), label );
                        gtk_widget_modify_font (GTK_BIN(priv->keyboard_keys[i])->child, desc_info);
			i++;
		}
	}
	else{
            if(priv->keyboard_page == 1)
                desc_info = pango_font_description_from_string("Sans 24");
            
            while (i < G_N_ELEMENTS (qwerty_keys)) {
                gchar* label = keys[i];
                /*if(priv->keyboard != 1){
                        g_object_set_data (G_OBJECT (priv->keyboard_keys[i]), KEYS_VALUES_DATA, label);
                        g_object_set_data (G_OBJECT (priv->keyboard_keys[i]), PERSISTENT_VALUE_DATA, label);
                }*/
                //hildon_button_set_title (HILDON_BUTTON (priv->keyboard_keys[i]), label);
                gtk_button_set_label ( GTK_BUTTON (priv->keyboard_keys[i]),label );
                gtk_widget_modify_font (GTK_BIN(priv->keyboard_keys[i])->child, desc_info);
                i++;
            }
	}
	
	pango_font_description_free(desc_info);
	
}

static void
backspace_pressed_cb (GtkWidget *widget, gpointer data){

	HildonIMChewingVKB *self = HILDON_IM_CHEWING_VKB (data);
	HildonIMChewingVKBPrivate *priv;
        gint i;
	priv = HILDON_IM_CHEWING_VKB_GET_PRIVATE (self);
  	if (! delete_selection (self)){
		if(priv->keyboard_page == 1){ 
			if(chewing_buffer_Check(ctx) || !chewing_zuin_Check(ctx)){
				if(chewing_cand_TotalChoice(ctx)){
					chewing_handle_Esc(ctx);
                                        for(i=0;i<chewing_buffer_Len(ctx);i++)
                                            chewing_handle_Right(ctx);
					change_keys(self,chewing_keys);
				}
				else{
                                    chewing_handle_Backspace(ctx);
                                    chewing_write (self, "null");
                                }
			}
			else
				backspace (self);
		}
		else
    		backspace (self);
  	}
}
static void
enter_key_pressed_cb (GtkWidget *widget, gpointer data)
{
  HildonIMChewingVKB *self = HILDON_IM_CHEWING_VKB (data);
  HildonIMChewingVKBPrivate *priv;

  priv = HILDON_IM_CHEWING_VKB_GET_PRIVATE (self);

  clear_timeout_id (self);
	if(priv->keyboard_page == 1 && chewing_buffer_Check(ctx)){
		if(chewing_cand_TotalChoice(ctx)){
			chewing_handle_Esc(ctx);
			change_keys(self,chewing_keys);
		}
		chewing_handle_Enter(ctx);
		chewing_write (self,"enter");
	}
	else
  		write (self, "\n");
}

static void
backspace (HildonIMChewingVKB *self)
{
  HildonIMChewingVKBPrivate *priv;
  GtkTextIter iter;
  GtkTextBuffer *buffer;

  priv = HILDON_IM_CHEWING_VKB_GET_PRIVATE (self);

  clear_timeout_id (self);
	if(priv->keyboard_page == 1)
		buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW (priv->chewing_buffer));
	else
		buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW (priv->text_view));
		
  gtk_text_buffer_get_iter_at_mark(buffer,
                                   &iter,
                                   gtk_text_buffer_get_insert (buffer));

	if(priv->keyboard_page != 1 && (gtk_text_buffer_get_char_count (buffer) > 0))
  		if (hildon_im_ui_get_commit_mode(priv->ui) == HILDON_IM_COMMIT_REDIRECT){
    		hildon_im_ui_send_communication_message(priv->ui,
                                            HILDON_IM_CONTEXT_HANDLE_BACKSPACE);
    		gtk_text_buffer_get_iter_at_mark(buffer,
                                     &iter,
                                     gtk_text_buffer_get_selection_bound (buffer));
    		priv->saved_offset = gtk_text_iter_get_offset(&iter);
		}
	gtk_text_buffer_backspace (buffer, &iter, TRUE, TRUE);

}
static void 
chewing_write (HildonIMChewingVKB *self, const gchar *text)
{
	HildonIMChewingVKBPrivate *priv;
	GtkTextBuffer *buffer;
	int index = 0;
	gchar *text_to_write;
	GtkTextIter start;
	GtkTextIter end;
	static int zuin_count_before = 0;
	static int chewingBufferCount = 0;
	priv = HILDON_IM_CHEWING_VKB_GET_PRIVATE(self);

	buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW (priv->chewing_buffer));
	
	if(!strcmp(text,"clear")){
		zuin_count_before = 0;
		chewingBufferCount = 0;
	}
	else if(!strcmp(text,"enter")){
		gtk_text_buffer_get_start_iter(buffer,&start);
		gtk_text_buffer_get_end_iter(buffer,&end);
		/*text_to_write = gtk_text_buffer_get_text(buffer,&start,&end,FALSE);
		write(self,text_to_write);*/
		gtk_text_buffer_delete(buffer,&start,&end);
		zuin_count_before = 0;
		chewingBufferCount = 0;
                chewing_Reset(ctx);
		text="\0";
	}
	else if(!(!strcmp(text," "))){
		if( chewing_cand_TotalChoice(ctx)){
			if( !(!strcmp(text,"\0"))){
				/*chewing_cand_Enumerate(ctx);
				while(!(!(strcmp(text,chewing_cand_String(ctx)))) && index < chewing_cand_TotalChoice(ctx))
					index++;
				
				while(index>9){
					chewing_handle_Space(ctx);
					index = index - 9;
				}*/
				chewing_handle_Default(ctx,text[0]);
				chewing_handle_Down(ctx);
				chewing_handle_Down(ctx);
				chewing_cand_Enumerate(ctx);
			
				chewing_candkey_refresh(self);
			}
		}
		else if(!(!strcmp(text,"null"))){	
			if( !strcmp(text,"\0"))
				chewingBufferCount = 0;

			/*index = 0;
			while( !(!(strcmp(chewing_keys[index],text))) && (index < G_N_ELEMENTS (qwerty_keys)))
				index++;

			text = qwerty_keys[index];*/
			chewing_handle_Default( ctx, text[0] );
		}
		while( zuin_count_before > 0){
			backspace ( self );
			zuin_count_before--;
		};
		text = chewing_zuin_String( ctx,&index);
		zuin_count_before = strlen(text)/3;
		if ( zuin_count_before == 0){
			text = chewing_buffer_String( ctx );
			while( chewingBufferCount > 0){
				backspace ( self );
				chewingBufferCount--;
			};
			chewingBufferCount = chewing_buffer_Len( ctx );
		}			
	}
	if ( chewing_commit_Check (ctx) ){
            text_to_write = chewing_commit_String(ctx);
            write(self,text_to_write);
        }
	
	gtk_text_buffer_insert_at_cursor (buffer, text, -1);
	
	gtk_text_view_scroll_mark_onscreen (GTK_TEXT_VIEW (priv->chewing_buffer),
                                      gtk_text_buffer_get_insert (buffer));
}
static void
write (HildonIMChewingVKB *self, const gchar *text)
{
	HildonIMChewingVKBPrivate *priv;
	GtkTextBuffer *buffer;
  
	priv = HILDON_IM_CHEWING_VKB_GET_PRIVATE(self);

	buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW (priv->text_view));

	if (hildon_im_ui_get_commit_mode (priv->ui) == HILDON_IM_COMMIT_REDIRECT){
	    delete_selection (self);
	    hildon_im_ui_send_utf8(priv->ui, text);
	    
		
	}
		
	if (hildon_im_ui_get_commit_mode (priv->ui) == HILDON_IM_COMMIT_SURROUNDING){
	    hildon_im_ui_send_utf8(priv->ui, text);
	}
	gtk_text_buffer_insert_at_cursor (buffer, text, -1);
	
	gtk_text_view_scroll_mark_onscreen (GTK_TEXT_VIEW (priv->text_view),
                                      gtk_text_buffer_get_insert (buffer));
}

static GtkWidget *
create_special_key (HildonIMChewingVKB *self,
                    HildonSizeType size,
                    gchar *icon_name,
                    GCallback callback)
{
  GtkWidget *image;
  GtkWidget *key;

  key = hildon_gtk_button_new (size);

  if (icon_name != NULL)
  {
    image = gtk_image_new_from_icon_name (icon_name, -1);
    gtk_container_add (GTK_CONTAINER (key), image);
  }

  if (callback != NULL)
  {
    g_signal_connect (key,
                      "clicked",
                      G_CALLBACK (callback),
                      self);
  }

  return key;
}


static void
set_fkb_size (HildonIMChewingVKB *self)
{
  HildonIMChewingVKBPrivate *priv;
  GdkScreen *screen;

  priv = HILDON_IM_CHEWING_VKB_GET_PRIVATE(self);
  screen = gdk_screen_get_default ();

  priv->fkb_width = gdk_screen_get_width (screen);
  priv->fkb_height = gdk_screen_get_height (screen);
}



static gchar *
get_first_char (gchar *full_string)
{
  gunichar chr = g_utf8_get_char_validated (full_string, -1);
  gchar character[7];
  int len = g_unichar_to_utf8 (chr, character);
  character[len] = '\0';

  return g_strndup (character, len);
}

static void
clear_timeout_id (HildonIMChewingVKB *self)
{
  HildonIMChewingVKBPrivate *priv;
  priv = HILDON_IM_CHEWING_VKB_GET_PRIVATE (self);

  if (priv->timeout_id > 0)
  {
    g_source_remove (priv->timeout_id);
  }
}

static gboolean
press_expired (gpointer data)
{
  HildonIMChewingVKB *self = HILDON_IM_CHEWING_VKB (data);
  HildonIMChewingVKBPrivate *priv;
  priv = HILDON_IM_CHEWING_VKB_GET_PRIVATE (self);

  priv->replace_previous_char = FALSE;
  priv->slide_index = -1;
  priv->repeating_key = NULL;

  gchar *text;

    gchar *key_values = g_object_get_data (G_OBJECT (priv->pressed_key),
                                           KEYS_VALUES_DATA);
    text = g_strdup (get_first_char (key_values));


  if (text != NULL)
  {
    write (self, text);
  }

  priv->pressed_key = NULL;

  return FALSE;
}

static gboolean
slide_expired (gpointer data)
{
	HildonIMChewingVKB *self = HILDON_IM_CHEWING_VKB ( data );
	HildonIMChewingVKBPrivate *priv;
	priv = HILDON_IM_CHEWING_VKB_GET_PRIVATE (self);

	priv->replace_previous_char = FALSE;
	priv->slide_index = -1;
	priv->repeating_key = NULL;

	return FALSE;
}	

static void
sliding_key_pressed_cb (GtkWidget *key, gpointer data)
{
  HildonIMChewingVKB *self = HILDON_IM_CHEWING_VKB (data);
  HildonIMChewingVKBPrivate *priv;
  priv = HILDON_IM_CHEWING_VKB_GET_PRIVATE (self);

  clear_timeout_id (self);
  priv->pressed_key = key;
  priv->timeout_id = g_timeout_add (SLIDING_TIMEOUT, press_expired, self);
}

static void
sliding_key_released_cb (GtkWidget *key, gpointer data)
{
  const gchar *key_value = NULL;
  gchar *sub_str = NULL;
  gint nr_values = 1;
  gboolean is_slide_key = TRUE;
  gboolean caps_on = FALSE;
	gint index = 0;
	gchar ** keys;
  HildonIMChewingVKB *self = HILDON_IM_CHEWING_VKB (data);
  HildonIMChewingVKBPrivate *priv;
  priv = HILDON_IM_CHEWING_VKB_GET_PRIVATE (self);

  if (priv->pressed_key == NULL)
  {
    return;
  }

  caps_on = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (priv->caps_button));

  key_value = g_object_get_data (G_OBJECT (key), KEYS_VALUES_DATA);
  //key_value = gtk_button_get_label (GTK_BUTTON(key));
  nr_values = g_utf8_strlen (key_value, -1);

  if (nr_values <= 1)
		is_slide_key = FALSE;

  clear_timeout_id (self);

  if (is_slide_key)
  {
    if (priv->repeating_key == key)
		priv->slide_index++;
    else
		priv->slide_index = 0;

    if (priv->slide_index >= nr_values)
		priv->slide_index = 0;

    sub_str = g_utf8_offset_to_pointer (key_value, priv->slide_index);
    key_value = g_strdup (get_first_char (sub_str));

    if (key_value == NULL)
		return;

    if (priv->replace_previous_char && priv->repeating_key == key)
		backspace (self);

    priv->timeout_id = g_timeout_add (SLIDING_TIMEOUT, slide_expired, self);
    priv->replace_previous_char = TRUE;
  }
  priv->repeating_key = key;
	switch( priv->keyboard_page ){
		case 2:
			keys = qwerty_symbol_keys;
			break;
		case 3:
			keys = full_symbol_keys;
			break;
		default:
			keys = qwerty_symbol_keys;
			break;
	}
	if( priv->keyboard_page != 0 && priv->keyboard_page != 1){
		while( !(!(strcmp(qwerty_keys[index],key_value))) && (index < G_N_ELEMENTS (qwerty_keys)))
				index++;	
		key_value = keys[index];
	}
	if(priv->keyboard_page == 1)
		chewing_write(self,key_value);
	else
		write (self, caps_on ? g_utf8_strup (key_value, -1) : key_value);
}


static void 
swap_keyboard_page (HildonIMChewingVKB *self,gint mode_number)
{
  HildonIMChewingVKBPrivate *priv;

  priv = HILDON_IM_CHEWING_VKB_GET_PRIVATE (self);

	
	/*if (priv->keyboard_page == 0) {
		gtk_widget_show (priv->caps_button); 
	 else 
		gtk_widget_hide (priv->caps_button);*/

	gtk_widget_hide(priv->chewing_buffer);
        gtk_widget_hide(priv->forwardpage_button);
        gtk_widget_hide(priv->passage_button);
	gtk_widget_hide(priv->caps_button);
	switch (mode_number) {
		case 0:
		    gtk_widget_show(priv->caps_button);
		    change_keys(self,qwerty_keys);	    
		    break;
		case 1:
                    gtk_widget_show(priv->chewing_buffer);
                    gtk_widget_show(priv->forwardpage_button);
                    gtk_widget_show(priv->passage_button);
		    change_keys(self,chewing_keys);
		    break;
		case 2:
		    gtk_widget_show(priv->caps_button);
		    change_keys(self,qwerty_symbol_keys);
			
		    break;
		case 3:
		    gtk_widget_show(priv->chewing_buffer);
		    change_keys(self,full_symbol_keys);
		    break;
		default :
		    break;
	}
}

static void
sliding_keyboard_change_page_cb(GtkWidget *key, gpointer data)
{
	HildonIMChewingVKBPrivate *priv;
	HildonIMChewingVKB *self = HILDON_IM_CHEWING_VKB (data);
	priv = HILDON_IM_CHEWING_VKB_GET_PRIVATE (self);
	
	switch(priv->keyboard_page){
		case 0:
			priv->keyboard_page=2;
			swap_keyboard_page(self,2);
			break;
		case 1:
			priv->keyboard_page=3;
			swap_keyboard_page(self,3);
			break;
		case 2:
			priv->keyboard_page=0;
			swap_keyboard_page(self,0);
			break;
		case 3:
			priv->keyboard_page=1;
			swap_keyboard_page(self,1);
			break;
		default:
			break;
	}
}
static void
spacekey_press_cb(GtkWidget *key, gpointer data){
    
	HildonIMChewingVKBPrivate *priv;
	HildonIMChewingVKB *self = HILDON_IM_CHEWING_VKB (data);
	priv = HILDON_IM_CHEWING_VKB_GET_PRIVATE (self);
	
	if( priv->keyboard_page == 1 ){
		chewing_handle_Space(ctx);
		if( chewing_cand_TotalChoice(ctx))
		    chewing_candkey_refresh(self);
		else
			chewing_write(self,"null");
	}
	else
	    write(self," ");
			
	

    
}

static void
keyboard_mode_change_cb(GtkWidget *key, gpointer data){

	HildonIMChewingVKBPrivate *priv;
	HildonIMChewingVKB *self = HILDON_IM_CHEWING_VKB (data);
	priv = HILDON_IM_CHEWING_VKB_GET_PRIVATE (self);
	
	if( priv->keyboard_page == 1 || priv->keyboard_page == 3){
		priv->keyboard_page=0;
		swap_keyboard_page (self,0);

	}
	else{
		priv->keyboard_page=1;
		swap_keyboard_page (self,1);
	}
}
static void
passbutton_press_cb(GtkWidget *key, gpointer data){

	HildonIMChewingVKBPrivate *priv;
	HildonIMChewingVKB *self = HILDON_IM_CHEWING_VKB (data);
	priv = HILDON_IM_CHEWING_VKB_GET_PRIVATE (self);

	if( priv->keyboard_page == 1){
	
		if(chewing_cand_TotalChoice(ctx))
			chewing_handle_Esc(ctx);
			
		chewing_handle_Left(ctx);		
		chewing_handle_Down(ctx);
		chewing_handle_Down(ctx);
		chewing_cand_Enumerate(ctx);

		if(chewing_cand_TotalChoice(ctx))
			chewing_candkey_refresh(self);
	}
	
}
static void
forwardbutton_press_cb(GtkWidget *key, gpointer data){

	HildonIMChewingVKBPrivate *priv;
	HildonIMChewingVKB *self = HILDON_IM_CHEWING_VKB (data);
	priv = HILDON_IM_CHEWING_VKB_GET_PRIVATE (self);
		
	if(priv->keyboard_page == 1){
	
		if(chewing_cand_TotalChoice(ctx))
			chewing_handle_Esc(ctx);

		chewing_handle_Right(ctx);
		chewing_handle_Down(ctx);
		chewing_handle_Down(ctx);
		chewing_cand_Enumerate(ctx);

		if(chewing_cand_TotalChoice(ctx))		
			chewing_candkey_refresh(self);
	}

	
}
static void
sendbutton_press_cb(GtkWidget *key, gpointer data){

	HildonIMChewingVKBPrivate *priv;
	HildonIMChewingVKB *self = HILDON_IM_CHEWING_VKB (data);
	priv = HILDON_IM_CHEWING_VKB_GET_PRIVATE (self);
		
	hildon_im_ui_send_communication_message(priv->ui,HILDON_IM_CONTEXT_ENTER_ON_FOCUS);
	
	gtk_widget_hide (GTK_WIDGET (priv->window));
        
        unparent_contents(self);
        chewing_Reset(ctx);
        chewing_Terminate();
	hildon_im_ui_restore_previous_mode (priv->ui);    
	
}

static void
select_function_cb(GtkWidget *key, gpointer data){

	HildonIMChewingVKBPrivate *priv;
	HildonIMChewingVKB *self = HILDON_IM_CHEWING_VKB (data);
        gchar *key_value;
        GtkTextBuffer *buffer;
	        
        GtkClipboard *clipboard;
        priv = HILDON_IM_CHEWING_VKB_GET_PRIVATE (self);
        key_value = g_object_get_data (G_OBJECT (key), KEYS_VALUES_DATA);
        
        clipboard = gtk_clipboard_get (GDK_SELECTION_CLIPBOARD);
        buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW (priv->text_view));
        
        if(!strcmp(key_value,"copy"))
            gtk_text_buffer_copy_clipboard (buffer, clipboard);
        else if(!strcmp(key_value,"paste")){
            gtk_text_buffer_paste_clipboard (buffer, clipboard, NULL, TRUE);
            hildon_im_ui_send_communication_message (priv->ui,HILDON_IM_CONTEXT_CLIPBOARD_PASTE);
        }
        else if(!strcmp(key_value,"cut")){
            gtk_text_buffer_copy_clipboard (buffer, clipboard);
            delete_selection (self);          
        }
        //hildon_im_ui_send_communication_message(priv->ui,HILDON_IM_CONTEXT_CLIPBOARD_PASTE);
    
	gtk_widget_hide(priv->dialog);
}


static void
selectbutton_press_cb(GtkWidget *key, gpointer data)
{
    HildonIMChewingVKBPrivate *priv;
    HildonIMChewingVKB *self = HILDON_IM_CHEWING_VKB (data);
    priv = HILDON_IM_CHEWING_VKB_GET_PRIVATE (self);
    
    GtkWidget *keys;
    
    /* Create dialog */
        priv->dialog = gtk_dialog_new ();
        gtk_window_set_title(GTK_WINDOW(priv->dialog), "注音虛擬鍵盤");
        if(priv->screen_mode == PORTRAIT)
            hildon_gtk_window_set_portrait_flags (GTK_WINDOW (priv->dialog), HILDON_PORTRAIT_MODE_REQUEST);
    //cut button
        keys = gtk_dialog_add_button(GTK_DIALOG(priv->dialog),"剪下",GTK_RESPONSE_OK);
        g_object_set_data (G_OBJECT (keys), KEYS_VALUES_DATA, "cut");
        g_signal_connect (keys,"clicked",G_CALLBACK (select_function_cb),self);
        gtk_widget_set_size_request (keys, priv->fkb_width/2 , 70);
        //gtk_dialog_add_action_widget(GTK_DIALOG(priv->dialog),keys,GTK_RESPONSE_CLOSE);
    //copy button    
        keys = gtk_dialog_add_button(GTK_DIALOG(priv->dialog),"複製",GTK_RESPONSE_OK);
        g_object_set_data (G_OBJECT (keys), KEYS_VALUES_DATA, "copy");
        g_signal_connect (keys,"clicked",G_CALLBACK (select_function_cb),self);
        gtk_widget_set_size_request (keys, priv->fkb_width/2 , 70);
        //gtk_dialog_add_action_widget(GTK_DIALOG(priv->dialog),keys,GTK_RESPONSE_OK);
    //paste button
        keys = gtk_dialog_add_button(GTK_DIALOG(priv->dialog),"貼上",GTK_RESPONSE_OK);
        g_object_set_data (G_OBJECT (keys), KEYS_VALUES_DATA, "paste");
        g_signal_connect (keys,"clicked",G_CALLBACK (select_function_cb),self);
        gtk_widget_set_size_request (keys, priv->fkb_width/2 , 70);
        //gtk_dialog_add_action_widget(GTK_DIALOG(priv->dialog),keys,GTK_RESPONSE_CLOSE);

//end of create dialog
    
    gtk_widget_show_all (priv->dialog);
    
}
static void
chewing_candkey_refresh(HildonIMChewingVKB *self){

	HildonIMChewingVKBPrivate *priv;
	priv = HILDON_IM_CHEWING_VKB_GET_PRIVATE (self);
	gchar* cand_keys[8];

	int i=0;
		if(!chewing_cand_hasNext(ctx))
			chewing_cand_Enumerate(ctx);
		while(i<8){
			if(chewing_cand_hasNext(ctx))
				cand_keys[i]=chewing_cand_String(ctx);
			else
				cand_keys[i]="\0";
			i++;
		}
		
		change_keys(self,cand_keys);
	
}



static GtkWidget *
create_sliding_key (HildonIMChewingVKB *self, gchar *title, gchar *key_values, gint visible_offset)
{
  gchar *label = key_values;
  GtkWidget *key = gtk_button_new();
  PangoFontDescription *desc_info = pango_font_description_from_string("Sans 24");
  //GtkWidget *key = hildon_button_new (HILDON_SIZE_AUTO_HEIGHT,HILDON_BUTTON_ARRANGEMENT_VERTICAL);

  g_object_set_data (G_OBJECT (key), KEYS_VALUES_DATA, key_values);
  //g_object_set_data (G_OBJECT (key), PERSISTENT_VALUE_DATA, title);
  if (visible_offset > 0)
  {
    gint byte_count = g_utf8_offset_to_pointer (key_values, visible_offset) - key_values;
    label = g_strndup (key_values, byte_count);
  }
  
  //hildon_button_set_value ( HILDON_BUTTON (key), key_values);
  //sprintf ( markup, "<span size='x-large'>%s</span>",label);
  gtk_button_set_label(GTK_BUTTON(key),title);
  gtk_widget_modify_font (GTK_BIN(key)->child, desc_info);
  
  gtk_button_set_alignment (GTK_BUTTON (key), 0.5, 0.5);
  
  pango_font_description_free(desc_info);
  
  //gtk_label_set_markup(GTK_LABEL(GTK_BIN(key)->child), "<span size='x-large'>G</span>");


  /*hildon_button_set_value (HILDON_BUTTON (key), label);
  hildon_button_set_title (HILDON_BUTTON (key), label);

  hildon_button_set_title_alignment (HILDON_BUTTON (key), 0.5, 0.5);*/

  g_signal_connect (key,
                    "pressed",
                    G_CALLBACK (sliding_key_pressed_cb),
                    self);

  g_signal_connect (key,
                    "released",
                    G_CALLBACK (sliding_key_released_cb),
                    self);

  return key;
}

//set keyboard layout
static void
set_kb_mode (HildonIMChewingVKB *self, ScreenMode mode)
{
  //CURRENTLY BROKEN FOR LAYOUT CHANGE REINIT
  HildonIMChewingVKBPrivate *priv;
  priv = HILDON_IM_CHEWING_VKB_GET_PRIVATE (self);
  gint i,j,k;
  i = 0; j = 0;k = 0;


  if (mode == LANDSCAPE)
  {
	gtk_widget_set_size_request (priv->chewing_buffer, 50, -1);
	k=1;
	for(i = 0; i < G_N_ELEMENTS (qwerty_keys)-1; ){
		for(j = 0;j < 10; j++,i++){
			if( i != 20){
				//gtk_widget_reparent(priv->keyboard_keys[i], priv->keyboard_rows[k]);
				gtk_box_pack_start (GTK_BOX (priv->keyboard_rows[k]), priv->keyboard_keys[i], TRUE, TRUE, 0);
                                //gtk_box_reorder_child(GTK_BOX(priv->keyboard_rows[k]),priv->keyboard_keys[i],j);
			}
			else
				j--;		
		}
		k++;
	}
	
        gtk_box_pack_start (GTK_BOX (priv->keyboard_rows[k]), priv->keyboard_keys[i], TRUE, TRUE, 0);
        gtk_box_pack_start (GTK_BOX (priv->keyboard_rows[1]), priv->backspace_button, TRUE, TRUE, 0);
        gtk_box_pack_start (GTK_BOX (priv->keyboard_rows[2]), priv->keyboard_keys[20], TRUE, TRUE, 0);
        gtk_box_pack_start (GTK_BOX (priv->keyboard_rows[3]), priv->enter_button, TRUE, TRUE, 0);
        gtk_box_pack_start (GTK_BOX (priv->keyboard_rows[4]), priv->send_button, TRUE, TRUE, 0);
        gtk_box_pack_start (GTK_BOX (priv->keyboard_rows[5]), priv->modechange_button, TRUE, TRUE, 0);
        gtk_box_pack_start (GTK_BOX (priv->keyboard_rows[5]), priv->symbol_page_button, TRUE, TRUE, 0);
        gtk_box_pack_start (GTK_BOX (priv->keyboard_rows[5]), priv->select_button, TRUE, TRUE, 0);
        gtk_box_pack_start (GTK_BOX (priv->keyboard_rows[5]), priv->space_button, TRUE, TRUE, 0);
        gtk_box_pack_start (GTK_BOX (priv->keyboard_rows[5]), priv->caps_button, TRUE, TRUE, 0);
        gtk_box_pack_start (GTK_BOX (priv->keyboard_rows[5]), priv->passage_button, TRUE, TRUE, 0);
        gtk_box_pack_start (GTK_BOX (priv->keyboard_rows[5]), priv->forwardpage_button, TRUE, TRUE, 0);
        gtk_box_pack_start (GTK_BOX (priv->keyboard_rows[5]), priv->close_button, TRUE, TRUE, 0);
	
	//gtk_widget_hide(priv->keyboard_rows[0]);
	//gtk_widget_hide(priv->keyboard_rows[6]);

  }
  else
  {
	gtk_widget_set_size_request (priv->chewing_buffer, 50, -1);
	
	gtk_widget_show(priv->keyboard_rows[0]);
	gtk_widget_show(priv->keyboard_rows[6]);

        gtk_box_pack_start (GTK_BOX (priv->keyboard_rows[0]), priv->select_button, TRUE, TRUE, 0);
        gtk_box_pack_start (GTK_BOX (priv->keyboard_rows[0]), priv->passage_button, TRUE, TRUE, 0);
	gtk_box_pack_start (GTK_BOX (priv->keyboard_rows[0]), priv->forwardpage_button, TRUE, TRUE, 0);
        gtk_box_pack_start (GTK_BOX (priv->keyboard_rows[0]), priv->enter_button, TRUE, TRUE, 0);
        gtk_box_pack_start (GTK_BOX (priv->keyboard_rows[0]), priv->send_button, TRUE, TRUE, 0);
        gtk_box_pack_start (GTK_BOX (priv->keyboard_rows[0]), priv->caps_button, TRUE, TRUE, 0);
        gtk_box_pack_start (GTK_BOX (priv->keyboard_rows[0]), priv->backspace_button, TRUE, TRUE, 0);
	
	k = 1;	
        for(i = 0; i < G_N_ELEMENTS (qwerty_keys)-1; ){
                for(j = 0;j < 8; j++){
                        gtk_box_pack_start (GTK_BOX (priv->keyboard_rows[k]), priv->keyboard_keys[i], TRUE, TRUE, 0);
                        //gtk_widget_reparent(priv->keyboard_keys[i], priv->keyboard_rows[k]);
                        //gtk_box_reorder_child(GTK_BOX(priv->keyboard_rows[k]),priv->keyboard_keys[i],j);
                            i++;
                }
                i+=2;
                if(i == 20)
                    i++;
                k++;
        }
        gtk_box_pack_start (GTK_BOX (priv->keyboard_rows[5]), priv->keyboard_keys[8], TRUE, TRUE, 0);
        gtk_box_pack_start (GTK_BOX (priv->keyboard_rows[5]), priv->keyboard_keys[9], TRUE, TRUE, 0);
        gtk_box_pack_start (GTK_BOX (priv->keyboard_rows[5]), priv->keyboard_keys[18], TRUE, TRUE, 0);
        gtk_box_pack_start (GTK_BOX (priv->keyboard_rows[5]), priv->keyboard_keys[19], TRUE, TRUE, 0);
        gtk_box_pack_start (GTK_BOX (priv->keyboard_rows[5]), priv->keyboard_keys[20], TRUE, TRUE, 0);
        gtk_box_pack_start (GTK_BOX (priv->keyboard_rows[5]), priv->keyboard_keys[29], TRUE, TRUE, 0);
        gtk_box_pack_start (GTK_BOX (priv->keyboard_rows[5]), priv->keyboard_keys[30], TRUE, TRUE, 0);
        gtk_box_pack_start (GTK_BOX (priv->keyboard_rows[5]), priv->keyboard_keys[39], TRUE, TRUE, 0);
        gtk_box_pack_start (GTK_BOX (priv->keyboard_rows[6]), priv->keyboard_keys[40], TRUE, TRUE, 0);

        gtk_box_pack_start (GTK_BOX (priv->keyboard_rows[6]), priv->modechange_button, TRUE, TRUE, 0);
        gtk_box_pack_start (GTK_BOX (priv->keyboard_rows[6]), priv->symbol_page_button, TRUE, TRUE, 0);
        gtk_box_pack_start (GTK_BOX (priv->keyboard_rows[6]), priv->space_button, TRUE, TRUE, 0);
        gtk_box_pack_start (GTK_BOX (priv->keyboard_rows[6]), priv->close_button, TRUE, TRUE, 0);
	//gtk_widget_reparent(priv->keyboard_keys[i], priv->keyboard_rows[k]);
  }	

}
//add all key to keyboard
static GtkWidget *
create_keyboard(HildonIMChewingVKB *self, gchar **keys, gint page)
{
  HildonIMChewingVKBPrivate *priv;
  GtkWidget *image;
  gint j,i;
  priv = HILDON_IM_CHEWING_VKB_GET_PRIVATE(self);
  GtkSizeGroup *group = gtk_size_group_new (GTK_SIZE_GROUP_BOTH);
  GtkWidget *contents = gtk_vbox_new (FALSE, 0);

  for(j=0;j<7;j++){
    priv->keyboard_rows[j] = gtk_hbox_new (FALSE, 0);
    gtk_box_pack_start (GTK_BOX (contents), priv->keyboard_rows[j], TRUE, TRUE, 0);
  }
  
  gint num_keys = G_N_ELEMENTS (qwerty_keys);
  gint visible_offset;
  for (i = 0; i < num_keys; i++)
  {
    visible_offset = 7; 
    
    priv->keyboard_keys[i] = create_sliding_key (self, keys[i],qwerty_keys[i], visible_offset);
    gtk_size_group_add_widget (group, priv->keyboard_keys[i]);
    //gtk_container_add(GTK_CONTAINER (contents), priv->keyboard_keys[i]);
  }
  
    priv->backspace_button = create_special_key (self,
                            HILDON_SIZE_AUTO_HEIGHT,
                            "keyboard_backspace",
                            G_CALLBACK (backspace_pressed_cb));
    gtk_size_group_add_widget (group, priv->backspace_button);
    //gtk_container_add(GTK_CONTAINER (contents), priv->backspace_button);
      
    priv->enter_button = create_special_key (self,
                HILDON_SIZE_AUTO_HEIGHT,
                "keyboard_enter",
                G_CALLBACK (enter_key_pressed_cb));
    gtk_size_group_add_widget (group, priv->enter_button);
    //gtk_container_add(GTK_CONTAINER (contents), priv->enter_button);

//send_key
priv->send_button = create_special_key (self,
                    HILDON_SIZE_AUTO_HEIGHT,
                    "general_sent",
                    G_CALLBACK (sendbutton_press_cb));  
gtk_size_group_add_widget (group, priv->send_button);
//gtk_container_add(GTK_CONTAINER (contents), priv->send_button);


//symbol page toggle
  priv->symbol_page_button = create_special_key (self,
                    HILDON_SIZE_AUTO_HEIGHT,
                    "control_keyboard",
                    G_CALLBACK (keyboard_mode_change_cb));  
  gtk_size_group_add_widget (group, priv->symbol_page_button);
  //gtk_container_add(GTK_CONTAINER (contents), priv->symbol_page_button);


//mode_change_key
  priv->modechange_button = create_special_key (self,
                    HILDON_SIZE_AUTO_HEIGHT,
                    "general_refresh",
                    G_CALLBACK (sliding_keyboard_change_page_cb));  
  gtk_size_group_add_widget (group, priv->modechange_button);
  //gtk_container_add(GTK_CONTAINER (contents), priv->modechange_button);
  
//select_key
  priv->select_button = create_special_key (self,
                    HILDON_SIZE_AUTO_HEIGHT,
                    "browser_mover",
                    G_CALLBACK (selectbutton_press_cb));  
  gtk_size_group_add_widget (group, priv->select_button);
  //gtk_container_add(GTK_CONTAINER (contents), priv->select_button);
  
//space_key
  priv->space_button = create_special_key (self,
                    HILDON_SIZE_HALFSCREEN_WIDTH,
                    NULL,
                    G_CALLBACK (spacekey_press_cb));
  gtk_widget_set_size_request (priv->space_button, 250 , -1);
  //gtk_container_add(GTK_CONTAINER (contents), priv->space_button);

//passpage_select_key  
        priv->passage_button = create_special_key (self,
                    HILDON_SIZE_AUTO_HEIGHT,
                    "general_back",
                    G_CALLBACK (passbutton_press_cb));  
        gtk_size_group_add_widget (group, priv->passage_button);
        //gtk_container_add(GTK_CONTAINER (contents), priv->passage_button);

//forwardpage_entermode_key  
        priv->forwardpage_button = create_special_key (self,
                    HILDON_SIZE_AUTO_HEIGHT,
                    "general_forward",
                    G_CALLBACK (forwardbutton_press_cb));  
        gtk_size_group_add_widget (group, priv->forwardpage_button);
        //gtk_container_add(GTK_CONTAINER (contents), priv->forwardpage_button);
        
//caps_button
priv->caps_button = hildon_gtk_toggle_button_new (HILDON_SIZE_AUTO_HEIGHT);
image = gtk_image_new_from_icon_name ("keyboard_move_up", -1);
gtk_container_add (GTK_CONTAINER (priv->caps_button), image);
                  
gtk_size_group_add_widget (group, priv->caps_button);
//gtk_container_add(GTK_CONTAINER (contents), priv->caps_button);
                 

//close_key
        priv->close_button = create_special_key (self,
                            HILDON_SIZE_AUTO_HEIGHT,
                            "keyboard_close",
                            G_CALLBACK (close_fkb_cb));
  
        gtk_size_group_add_widget (group, priv->close_button);
        //gtk_container_add(GTK_CONTAINER (contents), priv->close_button);
  
        
  return contents;
}
//add all widget to window
static void
create_window (HildonIMChewingVKB *self)
{
	char **keys;
	HildonIMChewingVKBPrivate *priv;
	priv = HILDON_IM_CHEWING_VKB_GET_PRIVATE(self);
	GtkWidget *text_area;
	
	priv->window = gtk_window_new (GTK_WINDOW_TOPLEVEL);
       

	switch(priv->keyboard_page){
		case 0:
			keys = qwerty_keys;
			break;
		case 1:
			keys = chewing_keys;
			break;
		case 2:
			keys = qwerty_symbol_keys;
			break;
		case 3:
			keys = full_symbol_keys;
			break;
		default:
			keys = qwerty_keys;
			break;
	}
 
        priv->keyboard = create_keyboard(self, keys, 0);
    
	priv->container = gtk_vbox_new (FALSE, 0);
	gtk_window_set_type_hint (GTK_WINDOW (priv->window), GDK_WINDOW_TYPE_HINT_DIALOG);
	gtk_window_set_decorated (GTK_WINDOW (priv->window), FALSE);
	//gtk_window_fullscreen (GTK_WINDOW (priv->window));
        if(priv->screen_mode == PORTRAIT)
            hildon_gtk_window_set_portrait_flags (GTK_WINDOW (priv->window), HILDON_PORTRAIT_MODE_REQUEST);
										
	priv->text_view_row = gtk_hbox_new (FALSE, 0);

	priv->text_view = gtk_text_view_new ();
	gtk_text_view_set_wrap_mode (GTK_TEXT_VIEW (priv->text_view), GTK_WRAP_WORD_CHAR);
	gtk_text_view_set_cursor_visible (GTK_TEXT_VIEW (priv->text_view), TRUE);

	gtk_widget_set_name (priv->text_view, "him-textview");
	g_signal_connect (priv->text_view,
                    "button-press-event",
                    G_CALLBACK (textview_button_press_cb),
                    self);
	
	priv->chewing_buffer = gtk_text_view_new ();
	gtk_text_view_set_wrap_mode (GTK_TEXT_VIEW (priv->chewing_buffer), GTK_WRAP_WORD_CHAR);
	gtk_text_view_set_editable (GTK_TEXT_VIEW (priv->chewing_buffer),FALSE);
	
	text_area = gtk_scrolled_window_new (NULL, NULL);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (text_area),
                                  GTK_POLICY_AUTOMATIC,
                                  GTK_POLICY_AUTOMATIC);
	gtk_container_add (GTK_CONTAINER (text_area), priv->text_view);
	gtk_box_pack_start (GTK_BOX (priv->text_view_row), text_area, TRUE, TRUE, 0);
	
	gtk_box_pack_start (GTK_BOX (priv->text_view_row), priv->chewing_buffer, TRUE, TRUE, 0);
	
	gtk_container_add (GTK_CONTAINER (priv->container), priv->text_view_row);
	gtk_container_add (GTK_CONTAINER (priv->container), priv->keyboard);
	gtk_container_add (GTK_CONTAINER (priv->window), priv->container);

	
        set_kb_mode (self, priv->screen_mode);	
}
