from org.maemo.hermes.engine.hermes import Hermes
from org.maemo.hermes.engine.friend import Friend
from org.maemo.hermes.engine.names import canonical

import unittest

class FakeService():
    def __init__(self):
        self.friends_to_create = []
        self.unmatched_friends = []
        self.contacts_pre_processed = []
        self.contacts_processed = []
        self.process_friends_called = False
        self._response_for_contact = {}
        
    def get_id(self): 
        return "fakebook"
    def pre_process_contact(self, contact): 
        self.contacts_pre_processed.append(contact)
    def process_friends(self): 
        self.process_friends_called = True
    def process_contact(self, contact): 
        self.contacts_processed.append(contact)
        econtact = contact.get_econtact()
        if self._response_for_contact.has_key(econtact):
            return self._response_for_contact[econtact]
        return None
    def get_friends_to_create_contacts_for(self): 
        return self.friends_to_create
    def get_unmatched_friends(self): 
        return self.unmatched_friends
    def finalise(self, updated, overwrite=False): 
        pass
    def _will_return_friend_for_contact(self, friend, contact):
        self._response_for_contact[contact] = friend
    
class FakeAddressBook():
    def __init__(self):
        self.contacts = []
        self.contacts_committed = []
    def get_all_contacts(self): 
        return self.contacts
    def add_contact(self, contact):
        self.contacts.append(contact)
    def commit_contact(self, contact): 
        self.contacts_committed.append(contact)
        
class FakeContact(Friend):
    def __init__(self, book=None, econtact=None, props=None):
        Friend.__init__(self, props)
        self.econtact = econtact
    def get_property(self, key):
        return self._safe_get(key)
    def get_identifiers(self):
        return [self.get_name(), self.get_nickname()]
    def get_econtact(self):
        return self.econtact
    def add_mapping(self, id):
        self._add('mapping', id)
    def get_mappings(self):
        return self._safe_get('mapping')
    def get_photo(self):
        return None
    def get_emails(self):
        return []
    def set_photo(self, value):
        self.set_photo_url(value)
    def get_birthday(self):
        return self.get_birthday_date()

class FakeEContact():
    def __init__(self, name):
        self.name = name
    def get_name(self):
        return self.name
    def get_property(self, key):
        return None

class TestHermes(unittest.TestCase):

    def setUp(self):
        self.service = FakeService()
        self.abook = FakeAddressBook()
        self.hermes = Hermes([self.service], None)
        self.hermes._get_address_book = self._get_address_book
        self.hermes._create_empty_contact = self._create_empty_contact
        self.hermes._create_contact_wrapper = self._create_contact_wrapper
    
    
    def test_empty_run(self):
        self.hermes.run()

        assert self.service.process_friends_called
        assert len(self.abook.contacts) == 0
        assert len(self.service.contacts_processed) == 0
        assert len(self.service.contacts_pre_processed) == 0
        assert len(self.hermes.unmatched) == 0
        assert len(self.hermes.matched) == 0
        assert len(self.hermes.updated) == 0

        
    def test_empty_abook_and_service_returns_one_friend_to_create(self):
        self.service.friends_to_create = [Friend("Fredrik")]
        
        self.hermes.run()
        
        assert len(self.abook.contacts) == 1
        assert len(self.service.contacts_processed) == 0
        assert len(self.service.contacts_pre_processed) == 0
        
    
    def test_one_contact_and_no_match(self):
        self.abook.contacts.append(FakeEContact("Fredrik"))
        
        self.hermes.run()
        
        assert len(self.abook.contacts) == 1
        assert len(self.service.contacts_processed) == 1
        assert len(self.service.contacts_pre_processed) == 1
        assert len(self.hermes.unmatched) == 1
        assert len(self.hermes.matched) == 0
        assert len(self.hermes.updated) == 0
        
        
    def test_one_contact_and_one_match_and_no_field_updated(self):
        contact = FakeEContact("Fredrik")
        friend = Friend("Fredrik", props={'birthday_date':'1980-10-15'})
        self.abook.contacts.append(contact)
        self.service._will_return_friend_for_contact(friend, contact)
        
        self.hermes.run()
        
        assert len(self.abook.contacts) == 1
        assert len(self.service.contacts_processed) == 1
        assert len(self.service.contacts_pre_processed) == 1
        assert len(self.hermes.unmatched) == 0
        assert len(self.hermes.matched) == 1
        assert len(self.hermes.updated) == 0
        
        
    # faked to avoid evolution/ebook dependency
    def _get_address_book(self):
        return self.abook
    def _create_empty_contact(self, friend):
        return FakeEContact(friend.get_name())
    def _create_contact_wrapper(self, econtact):
        return FakeContact(econtact=econtact)

if __name__ == '__main__':
    unittest.main()
