from facebook import Facebook, FacebookError
import gnome.gconf
import gtk, hildon
import org.maemo.hermes.engine.provider
import org.maemo.hermes.engine.facebook.service

class Provider(org.maemo.hermes.engine.provider.Provider):
    """Facebook provider for Hermes. 

       This requires two gconf paths to contain Facebook application keys:
           /apps/maemo/hermes/facebook_app
           /apps/maemo/hermes/facebook_secret
       
       Copyright (c) Andrew Flegg <andrew@bleb.org> 2010.
       Released under the Artistic Licence."""

    # -----------------------------------------------------------------------
    def __init__(self):
        """Initialise the provider, and ensure the environment is going to work."""

        self._gc = gnome.gconf.client_get_default()

        key_app    = self._gc.get_string('/apps/maemo/hermes/facebook_app')
        key_secret = self._gc.get_string('/apps/maemo/hermes/facebook_secret')
        if key_app is None or key_secret is None:
            raise Exception('No Facebook application keys found. Installation error.')
        
        self.fb = Facebook(key_app, key_secret)
        self.fb.desktop = True

        if self.fb.session_key is None:
            self.fb.session_key = self._gc.get_string('/apps/maemo/hermes/facebook_session_key')
            self.fb.secret = self._gc.get_string('/apps/maemo/hermes/facebook_secret_key')
            self.fb.uid = self._gc.get_string('/apps/maemo/hermes/facebook_uid')


    # -----------------------------------------------------------------------
    def get_name(self):
        """Return the display name of this service. An icon, of with the lower-case,
           all-alphabetic version of this name is expected to be provided."""
           
        return 'Facebook'

    
    # -----------------------------------------------------------------------
    def get_account_detail(self):
        """Return the email address associated with the user, if available."""
        
        name = self._gc.get_string('/apps/maemo/hermes/facebook_user')
        return name and name or _('Pending authorisation')
    
    
    # -----------------------------------------------------------------------
    def has_preferences(self):
        """Whether or not this provider has any preferences. If it does not,
           open_preferences must NOT be called; as the behaviour is undetermined."""
           
        return True
    
    
    # -----------------------------------------------------------------------
    def open_preferences(self, parent):
        """Open the preferences for this provider as a child of the 'parent' widget."""

        dialog = gtk.Dialog(self.get_name(), parent)
        dialog.add_button(_('Disable'), gtk.RESPONSE_NO)
        dialog.add_button(_('Enable'), gtk.RESPONSE_YES)
        
        checkbox = hildon.CheckButton(gtk.HILDON_SIZE_FINGER_HEIGHT)
        checkbox.set_label(_('Create birthday-only contacts'))
        checkbox.set_active(self._gc.get_bool('/apps/maemo/hermes/facebook_birthday_only'))
        dialog.vbox.add(checkbox)
        dialog.vbox.add(gtk.Label("\n" + _('Note: authentication via web page') + "\n"))
        
        clear = hildon.Button(gtk.HILDON_SIZE_FINGER_HEIGHT,
                              hildon.BUTTON_ARRANGEMENT_VERTICAL,
                              title = _("Clear authorisation"))
        clear.connect('clicked', self._clear_auth)
        dialog.vbox.add(clear)
        
        dialog.show_all()
        result = dialog.run()
        dialog.hide()
        if result == gtk.RESPONSE_CANCEL or result == gtk.RESPONSE_DELETE_EVENT:
            return None
    
        self._gc.set_bool('/apps/maemo/hermes/facebook_birthday_only', checkbox.get_active())
        return result == gtk.RESPONSE_YES


    # -----------------------------------------------------------------------
    def _clear_auth(self, event):
        """Clear Facebook authorisation information. Triggered by pressing
           the 'clear' button in the preferences dialogue."""
        
        self._gc.unset('/apps/maemo/hermes/facebook_session_key')
        self._gc.unset('/apps/maemo/hermes/facebook_secret_key')
        self._gc.unset('/apps/maemo/hermes/facebook_uid')
        self._gc.unset('/apps/maemo/hermes/facebook_user')

    
    # -----------------------------------------------------------------------
    def service(self, gui_callback):
        """Return a service instance."""
        
        self._gui = gui_callback
        
        # Check the available session is still valid...
        while True:
            try:
                if self.fb.users.getLoggedInUser() and self.fb.session_key:
                    break
            except FacebookError:
                pass
            self._do_fb_login()

        return org.maemo.hermes.engine.facebook.service.Service(self.get_id(), self.fb, self._gc.get_bool('/apps/maemo/hermes/facebook_birthday_only'))


    # -----------------------------------------------------------------------
    def _do_fb_login(self):
        """Perform authentication against Facebook and store the result in gconf
             for later use. Uses the 'need_auth' and 'block_for_auth' methods on
             the callback class. The former allows a message to warn the user
             about what is about to happen to be shown; the second is to wait
             for the user to confirm they have logged in."""
        self.fb.session_key = None
        self.fb.secret = None
        self.fb.uid = None
        
        if self._gui:
            self._gui.need_auth()
            
        self.fb.auth.createToken()
        self.fb.login()
        
        if self._gui:
            self._gui.block_for_auth()
          
        session = self.fb.auth.getSession()
        self._gc.set_string('/apps/maemo/hermes/facebook_session_key', session['session_key'])
        self._gc.set_string('/apps/maemo/hermes/facebook_secret_key', session['secret'])
        self._gc.set_string('/apps/maemo/hermes/facebook_uid', str(session['uid']))

        info = self.fb.users.getInfo([self.fb.uid], ['name'])
        self._gc.set_string('/apps/maemo/hermes/facebook_user', info[0]['name'])

