import urllib
import Image
import ImageOps
import StringIO
import datetime
import re
from org.maemo.hermes.engine.names import canonical, variants
from pygobject import *
from ctypes import *

# Constants from http://library.gnome.org/devel/libebook/stable/EContact.html#EContactField
ebook = CDLL('libebook-1.2.so.5')
E_CONTACT_HOMEPAGE_URL = 42
E_CONTACT_PHOTO = 94
E_CONTACT_EMAIL = 97
E_CONTACT_BIRTHDAY_DATE = 107


class Contact:
    """Provide an abstraction of contact, working around limitations
       in the evolution-python bindings. Properties are defined at:
       
          http://library.gnome.org/devel/libebook/stable/EContact.html#id3066469

       Copyright (c) Andrew Flegg <andrew@bleb.org> 2009.
       Released under the Artistic Licence."""

       
    # -----------------------------------------------------------------------
    def __init__(self, book, econtact):
        """Create a new contact store for modifying contacts in the given
           EBook."""
        
        self._book = book
        self._contact = econtact
        self._identifiers = self._find_ids()
        self._mapped_to = set([])
        
        
    # -----------------------------------------------------------------------
    def _find_ids(self):
        """Return a set of the identifiers which should be used to match this
           contact. Includes variants of first name, last name, nickname and
           email address. These are all Unicode-normalised into the traditional
           US-ASCII namespace"""
           
        result = set()
        for name in variants(self._contact.get_name()):
            result.add(canonical(name))
            
        for name in variants(self._contact.get_property('nickname')):
            result.add(canonical(name))
            
        for email in self.get_emails():
            user = canonical(email.split('@')[0])
            if len(user) > 4:
                result.add(user)

        return result
    
    
    # -----------------------------------------------------------------------
    def get_name(self):
        """Return this contact's name."""
        
        return self._contact.get_name() or self._contact.get_property('nickname')
    
    
    # -----------------------------------------------------------------------
    def get_identifiers(self):
        """Return the lowercase, Unicode-normalised, all-alphabetic
           versions of identifiers for this contact."""
           
        return self._identifiers

    
    # -----------------------------------------------------------------------
    def get_econtact(self):
        """Return the EContact which backs this contact."""
        
        return self._contact
    
    
    # -----------------------------------------------------------------------
    def add_mapping(self, id):
        """Record the fact that this contact is mapped against a provider.
           'id' MUST match the string returned by Provider.get_id()."""
        
        self._mapped_to.add(id)
        
        
    # ----------------------------------------------------------------------
    def get_mappings(self):
        """Return the set of IDs of providers which are mapped to this contact.
           The data can only be relied upon after services have run."""
           
        return self._mapped_to


    # -----------------------------------------------------------------------
    def get_photo(self):
        """Return the photo property, or None. The result is of type
        EContactPhoto."""
        
        photo = self._contact.get_property('photo')
        #print "Photo[" + re.sub('[^A-Za-z0-9_-]', '.', string_at(cast(c_void_p(hash(photo)), c_char_p))) + "]"
        pi = cast(c_void_p(hash(photo)), POINTER(EContactPhoto))
        if photo is None or pi.contents.data.uri == '':
            return None
        else:
            return pi
    
    
    # -----------------------------------------------------------------------
    def set_photo(self, url):
        """Set the given contact's photo to the picture found at the URL. If the
           photo is wider than it is tall, it will be cropped with a bias towards
           the top of the photo."""
        
        try:
            f = urllib.urlopen(url)
            data = ''
            while True:
                read_data = f.read()
                data += read_data
                if not read_data:
                    break
            
            im = Image.open(StringIO.StringIO(data))
            if im.mode != 'RGB':
                im.convert('RGB')
                
            (w, h) = im.size
            if (h > w):
                ##print u"Shrinking photo for %s as it's %d x %d" % (self._contact.get_name(), w, h)
                im = ImageOps.fit(im, (w, w), Image.NEAREST, 0, (0, 0.1))
              
            ## print u"Updating photo for %s" % (self._contact.get_name())
            f = StringIO.StringIO()
            im.save(f, "JPEG")
            image_data = f.getvalue()
            photo = EContactPhoto()
            photo.type = 0
            photo.data = EContactPhoto_data()
            photo.data.inlined = EContactPhoto_inlined()
            photo.data.inlined.mime_type = cast(create_string_buffer("image/jpeg"), c_char_p)
            photo.data.inlined.length = len(image_data)
            photo.data.inlined.data = cast(create_string_buffer(image_data), c_void_p)
            ebook.e_contact_set(hash(self._contact), E_CONTACT_PHOTO, addressof(photo))
            return True
        except:
            print "FAILED to get photo from URL %s" % url
            return False
      
      
    # -----------------------------------------------------------------------
    def set_birthday(self, day, month, year = 0):
        """Set the birthday for this contact to the given day, month and year."""
        
        if year == 0:
            year = datetime.date.today().year
          
        birthday = EContactDate()
        birthday.year = year
        birthday.month = month
        birthday.day = day
        print u"Setting birthday for [%s] to %d-%d-%d" % (self._contact.get_name(), year, month, day)
        ebook.e_contact_set(hash(self._contact), E_CONTACT_BIRTHDAY_DATE, addressof(birthday))
        return True
    
    
    # -----------------------------------------------------------------------
    def get_birthday(self):
        date = self._contact.get_property('birth-date')
        return date is not None and cast(c_void_p(hash(date)), POINTER(EContactDate)) or None
        
    
    # -----------------------------------------------------------------------
    def set_nickname(self, nickname):
        """Set the nickname for this contact to the given nickname."""
        
        # FIXME does this work?
        self._contact.set_property('nickname', nickname)
        #ebook.e_contact_set(hash(self._contact), E_NICKNAME, addressof(nickname))
    
    
    # -----------------------------------------------------------------------
    def get_nickname(self):
        """Get the nickname for this contact."""
        
        return self._contact.get_property('nickname')
    
    
    # -----------------------------------------------------------------------
    def get_emails(self):
        """Return the email addresses associated with this contact."""
        
        emails = []
        ai = GList.new(ebook.e_contact_get_attributes(hash(self._contact), E_CONTACT_EMAIL))
        while ai.has_next():
            attr = ai.next(as_a = EVCardAttribute)
            if not attr:
                raise Exception(u"Unexpected null attribute for [" + self._contact.get_name() + "] with emails " + emails)
            emails.append(string_at(attr.value().next()))
          
        return emails
        
      
      
    # -----------------------------------------------------------------------
    def get_urls(self):
        """Return a list of URLs which are associated with this contact."""
        
        urls = []
        ai = GList.new(ebook.e_contact_get_attributes(hash(self._contact), E_CONTACT_HOMEPAGE_URL))
        while ai.has_next():
            attr = ai.next(as_a = EVCardAttribute)
            if not attr:
                raise Exception(u"Unexpected null attribute for [" + self._contact.get_name() + "] with URLs " + urls)
            urls.append(string_at(attr.value().next()))
          
        return urls
    
      
    # -----------------------------------------------------------------------
    def add_url(self, str, unique = ''):
        """Add a new URL to the set of URLs for the given contact."""
        
        urls = re.findall('(?:(?:ftp|https?):\/\/|\\bwww\.|\\bftp\.)[,\w\.\-\/@:%?&=%+#~_$\*]+[\w=\/&=+#]', str, re.I | re.S)
        updated = False
        for url in urls:
            updated = self._add_url(url, unique or re.sub('(?:.*://)?(\w+(?:[\w\.])*).*', '\\1', url)) or updated
        
        return updated
    
    
    # -----------------------------------------------------------------------
    def _add_url(self, url, unique):
        """Do the work of adding a unique URL to a contact."""
        
        url_attr = None
        ai = GList.new(ebook.e_contact_get_attributes(hash(self._contact), E_CONTACT_HOMEPAGE_URL))
        while ai.has_next():
            attr = ai.next(as_a = EVCardAttribute)
            existing = string_at(attr.value().next())
            #print "Existing URL [%s] when adding [%s] to [%s] with constraint [%s]" % (existing, url, contact.get_name(), unique)
            if existing == unique or existing == url:
                return False
            elif existing.find(unique) > -1:
                url_attr = attr
          
        if not url_attr:
            ai.add()
            url_attr = EVCardAttribute()
            url_attr.group = ''
            url_attr.name = 'URL'
        
        val = GList()
        ##print u"Setting URL for [%s] to [%s]" % (self._contact.get_name(), url)
        val.set(create_string_buffer(url))
        ai.set(addressof(url_attr))
        url_attr.values = cast(addressof(val), POINTER(GList))
        ebook.e_contact_set_attributes(hash(self._contact), E_CONTACT_HOMEPAGE_URL, addressof(ai))
        return True

