from org.maemo.hermes.engine.twitter.service import Service
from org.maemo.hermes.engine.names import canonical
from org.maemo.hermes.engine.friend import Friend
import unittest

class FakeContact():
    def __init__(self, name, addr):
        self.name = name
        self.urls = addr
    def get_urls(self):
        return self.urls
    def get_name(self):
        return self.name
    def get_identifiers(self):
        return [canonical(self.name)]

class FakeTweeter():
    def __init__(self, name, screen_name, props=None):
        self._props = {}
        self["name"] = name
        self["screen_name"] = screen_name
        if props:
            for key in props:
                self[key] = props[key]
    def __setitem__(self, key, value):
        self._props[key] = value
    def __getattr__(self, key):
        try:
            return self._props[key]
        except:
            return ""
        
    
class TestTwitterService(unittest.TestCase):
    
    def setUp(self):
        self.testee = Service('twitter', None)
    
    
    def test_that_process_contact_returns_None_for_unknown_contact(self):
        contact = FakeContact('Person Person', [])
        self._fake_server([])
        
        self.testee.pre_process_contact(contact);
        self.testee.process_friends()
        friend = self.testee.process_contact(contact)
        assert friend is None
        

    def test_that_process_contact_returns_friend_object_if_contact_is_known(self):
        twitterName = "Same Name"
        known_tweeter = FakeTweeter(twitterName, "samename")
        contact = FakeContact(twitterName, [])
        self._fake_server([known_tweeter])
        
        self.testee.pre_process_contact(contact);
        self.testee.process_friends()
        friend = self.testee.process_contact(contact)
        assert isinstance(friend, Friend)

        
    def test_main_flow(self):
        twitterName = "Twitter Name"
        self._fake_server([FakeTweeter(twitterName, "wendtse", {"url":"http://wendt.se"})])
        contact = FakeContact("Fredrik Wendt", ["http://twitter.com/wendtse"])
        
        self._exercise_service([contact])
        
        assert len(self.testee.get_unmatched_friends()) == 0
        matchers = self.testee.get_contacts_with_match()
        assert contact in matchers
        assert twitterName == matchers[contact].get_name()
        assert contact == self.testee._get_friend_by_contact(contact)['contact']
    
    
    def test_name_collision_avoided_by_previous_matching(self):
        twitterName = "Same Name"
        known_tweeter = FakeTweeter(twitterName, "samename")
        other_tweeter = FakeTweeter(twitterName, "otherscreenname")
        self._fake_server([other_tweeter, known_tweeter])
        contact = FakeContact(twitterName, [])
        
        self._exercise_service([contact])
        
        assert len(self.testee.get_unmatched_friends()) == 1
        matchers = self.testee.get_contacts_with_match()
        assert contact in matchers
        assert contact == self.testee._get_friend_by_contact(contact)['contact']


    def test_name_collision_avoided_only_one_person_matched(self):
        twitter_name = "Same Name"
        screen_name = "samename"
        known_tweeter = FakeTweeter(twitter_name, screen_name)
        other_tweeter = FakeTweeter(twitter_name, "otherscreenname")
        self._fake_server([other_tweeter, known_tweeter])
        contact = FakeContact(twitter_name, ["http://twitter.com/" + screen_name])
        
        self._exercise_service([contact])
        
        assert len(self.testee.get_unmatched_friends()) == 1
        matchers = self.testee.get_contacts_with_match()
        assert contact in matchers
        assert screen_name == matchers[contact].get_nickname()
        assert contact == matchers[contact]['contact']
        assert contact == self.testee._get_friend_by_contact(contact)['contact']
        

    def _exercise_service(self, contacts):
        for contact in contacts:
            self.testee.pre_process_contact(contact)
        self.testee.process_friends()
        for contact in contacts:
            self.testee.process_contact(contact)

    def _fake_server(self, data):
        self._fake_data = data
        self.testee._get_tweeters = self._get_tweeters
        
    def _get_tweeters(self):
        return self._fake_data
    
if __name__ == '__main__':
    unittest.main()
