#!/usr/bin/env python

import gtk, gobject
import traceback
import time
import thread
import contactview
import urllib2
import hildon
import wimpworks
import mapcontact
from wimpworks import WimpWorks
from hermes import Hermes

class HermesGUI(WimpWorks):
  """Provides the GUI for Hermes, allowing the syncing of Facebook and
     Twitter friends' information with the Evolution contacts' database.
       
     Copyright (c) Andrew Flegg <andrew@bleb.org> 2009.
     Released under the Artistic Licence."""


  # -----------------------------------------------------------------------
  def __init__(self):
    WimpWorks.__init__(self, 'Hermes', version = '0.2.0', dbus_name = 'org.maemo.hermes')
    self.set_background('background.png')
    
    layout = wimpworks.HildonMainScreenLayout(offset = 0.8, container = self)
    layout.add_button('Retrieve', "Get contacts' missing info")
    layout.add_button('Refresh', "Update contacts' info")

    self.add_menu_action("Accounts")
    self.menu.show_all()

  
  # -----------------------------------------------------------------------
  def do_retrieve(self, widget):
    self.sync(widget, False)
    
    
  # -----------------------------------------------------------------------
  def do_refresh(self, widget):
    self.sync(widget, True)


  # -----------------------------------------------------------------------
  def do_accounts(self, widget = None):
    dialog = gtk.Dialog('Accounts', self.main_window)
    dialog.add_button('Save', gtk.RESPONSE_OK)

    #pa = hildon.PannableArea()
    #dialog.vbox.add(pa)
    content = dialog.vbox 
    #content = gtk.VBox()
    #pa.add(content)
    #pa.set_size_request(600, 380)

    use_facebook = self.new_checkbox('Use Facebook', content)
    use_facebook.set_active(self.get_use_facebook())

    indent = self.new_indent(content)
    self.link_control(use_facebook, gtk.Label('Note: authentication via web page'), indent)
    
    fb_empty = self.link_control(use_facebook, self.new_checkbox('Create birthday-only contacts'), indent)
    fb_empty.set_active(self.get_create_empty())

    use_twitter = self.new_checkbox('Use Twitter', content)
    use_twitter.set_active(self.get_use_twitter())

    indent = self.new_indent(content)
    tw_user = self.link_control(use_twitter, self.new_input('Twitter username'), indent)
    tw_user.set_text(self.get_twitter_credentials()[0])

    tw_pass = self.link_control(use_twitter, self.new_input('Twitter password', password = True), indent)
    tw_pass.set_text(self.get_twitter_credentials()[1])

    dialog.show_all()
    result = dialog.run()
    dialog.hide()
    if result == gtk.RESPONSE_OK:
      self.set_use_facebook(use_facebook.get_active())
      self.set_create_empty(fb_empty.get_active())
      self.set_use_twitter(use_twitter.get_active(), tw_user.get_text(), tw_pass.get_text())

    return result
   

  # -----------------------------------------------------------------------
  def sync(self, widget, force, main = True):
    if main and not self.get_use_facebook() and not self.get_use_twitter():
      saved = self.do_accounts()
      if saved == gtk.RESPONSE_DELETE_EVENT:
        return

    if main:
      self.main_window.set_property('sensitive', False)
      thread.start_new_thread(self.sync, (widget, force, False))
    else:
      try:
        fb2c = Hermes(self,
                      twitter = (self.get_use_twitter() and self.get_twitter_credentials()) or None,
                      facebook = self.get_use_facebook(),
                      empty = self.get_create_empty())
        fb2c.load_friends()
        fb2c.sync_contacts(resync = force)
        gobject.idle_add(self.open_summary, fb2c)

      except urllib2.HTTPError, e:
        traceback.print_exc()
        if e.code == 401:
          gobject.idle_add(self.report_error, 'Authentication problem. Check credentials.', True)
        else:
          gobject.idle_add(self.report_error, 'Network connection error. Check connectivity.')

      except urllib2.URLError, e:
        traceback.print_exc()
        gobject.idle_add(self.report_error, 'Network connection error. Check connectivity.')
      
      except Exception, e:
        traceback.print_exc()
        gobject.idle_add(self.report_error, 'Something went wrong: ' + e.message)


  # -----------------------------------------------------------------------
  def open_summary(self, fb2c):
    gobject.idle_add(self.main_window.set_property, 'sensitive', True)

    dialog = gtk.Dialog('Summary', self.main_window)
    dialog.add_button('Done', gtk.RESPONSE_OK)
    
    button = hildon.Button(gtk.HILDON_SIZE_FINGER_HEIGHT, hildon.BUTTON_ARRANGEMENT_VERTICAL,
                           title = 'Updated %d contacts' % (len(fb2c.updated)))
    button.connect('clicked', self.show_contacts, fb2c, fb2c.updated)
    button.set_property('sensitive', len(fb2c.updated) > 0)
    dialog.vbox.add(button)
    
    button = hildon.Button(gtk.HILDON_SIZE_FINGER_HEIGHT, hildon.BUTTON_ARRANGEMENT_VERTICAL,
                           title = 'Matched %d contacts' % (len(fb2c.matched)))
    button.connect('clicked', self.show_contacts, fb2c, fb2c.matched)
    button.set_property('sensitive', len(fb2c.matched) > 0)
    dialog.vbox.add(button)
    
    button = hildon.Button(gtk.HILDON_SIZE_FINGER_HEIGHT, hildon.BUTTON_ARRANGEMENT_VERTICAL,
                           title = '%d contacts unmatched' % (len(fb2c.unmatched)))
    button.connect('clicked', self.show_contacts, fb2c, fb2c.unmatched)
    button.set_property('sensitive', len(fb2c.unmatched) > 0)
    dialog.vbox.add(button)
    
    dialog.show_all()
    dialog.run()
    dialog.hide()
  

  # -----------------------------------------------------------------------
  def show_contacts(self, widget, fb2c, contacts):
    view = contactview.ContactView(contacts)

    dialog = gtk.Dialog('Contacts', self.main_window)
    view.connect('contact-activated', self.map_contact, fb2c)
    dialog.vbox.add(view)
    dialog.show_all()
    
    dialog.run()
    dialog.hide()
    
    
  # -----------------------------------------------------------------------
  def map_contact(self, widget, contact, fb2c):
    view = mapcontact.MapContact(fb2c.friends, contact)

    dialog = gtk.Dialog(contact.get_name(), self.main_window)
    dialog.add_button('Update', gtk.RESPONSE_OK)
    dialog.vbox.add(view)
    dialog.show_all()
    
    result = dialog.run()
    dialog.hide()
    if result == gtk.RESPONSE_OK:
      friend = view.get_selected_friend()
      if friend:
        if 'contact' in friend and friend['contact'] == contact:
          hildon.hildon_banner_show_information(self.main_window, '', "Removing existing mappings is not yet supported")
        elif view.contact_mapped:
          hildon.hildon_banner_show_information(self.main_window, '', "Changing existing mappings is not yet supported")
        else:
          if fb2c.update_contact(contact, friend, False):
            fb2c.addresses.commit_contact(contact)

    
  # -----------------------------------------------------------------------
  def need_auth(self, main = False):
    if main:
      hildon.hildon_banner_show_information(self.main_window, '', "Need to authenticate with Facebook")
    else:
      gobject.idle_add(self.need_auth, True)
    

  # -----------------------------------------------------------------------
  def block_for_auth(self, main = False, lock = None):
    if main:
      note = gtk.Dialog('Facebook authorisation', self.main_window)
      note.add_button("Validate", gtk.RESPONSE_OK)
      note.vbox.add(gtk.Label("\nPress 'Validate' once Facebook has\nbeen authenticated in web browser.\n"))

      note.show_all()
      result = note.run()
      note.hide()
      lock.release()
      
    else:
      time.sleep(2)
      lock = thread.allocate_lock()
      lock.acquire()
      gobject.idle_add(self.block_for_auth, True, lock)
      lock.acquire()
      lock.release()


  # -----------------------------------------------------------------------
  def progress(self, i, j, main = False):
    if main:
      if i == 0:
        self.progressbar = gtk.ProgressBar()
        self.progressnote = gtk.Dialog("Fetching friends' info", self.main_window)
        self.progressnote.vbox.add(self.progressbar)
        hildon.hildon_gtk_window_set_progress_indicator(self.progressnote, 1)
        
        self.progressnote.show_all()
        
      elif i < j:
        if i == 1:
          self.progressnote.set_title("Updating contacts")
          hildon.hildon_gtk_window_set_progress_indicator(self.progressnote, 0)
          
        self.progressbar.set_fraction(float(i) / float(j))
        
      else:
        self.progressnote.destroy()
        
      print i,j
    else:
      gobject.idle_add(self.progress, i, j, True)


  # -----------------------------------------------------------------------
  def report_error(self, e, prefs = False):
    if self.progressnote:
      self.main_window.set_property('sensitive', True)
      self.progressnote.destroy()

    hildon.hildon_banner_show_information(self.main_window, '', e)
    if prefs:
      self.do_accounts()

      
  def get_use_facebook(self):
    return self.gconf.get_bool("/apps/maemo/hermes/use_facebook")


  def set_use_facebook(self, value):
    self.gconf.set_bool("/apps/maemo/hermes/use_facebook", value)

  def get_create_empty(self):
    return self.gconf.get_bool("/apps/maemo/hermes/create_empty")


  def set_create_empty(self, value):
    self.gconf.set_bool("/apps/maemo/hermes/create_empty", value)


  def get_use_twitter(self):
    return self.gconf.get_bool("/apps/maemo/hermes/use_twitter")


  def set_use_twitter(self, value, user, password):
    self.gconf.set_bool("/apps/maemo/hermes/use_twitter", value)
    self.gconf.set_string("/apps/maemo/hermes/twitter_user", user)
    self.gconf.set_string("/apps/maemo/hermes/twitter_pwd", password)
    
  
  def get_twitter_credentials(self):
    return (self.gconf.get_string("/apps/maemo/hermes/twitter_user") or '',
            self.gconf.get_string("/apps/maemo/hermes/twitter_pwd") or '')


# -------------------------------------------------------------------------
if __name__ == '__main__':
  gui = HermesGUI()
  gui.run()

