#include "filelistview.h"

FileListView::FileListView(QWidget *parent) :
    QListView(parent)
{
    this->setUniformItemSizes(true);
    this->setEditTriggers(QAbstractItemView::NoEditTriggers);
    this->setMovement(QListView::Static);

    enableGridMode(false);

    // Initialize the model
    filesystemModel = new FileSystemModel(this);
    this->setModel(filesystemModel);

    connect(this, SIGNAL(activated(QModelIndex)), this, SLOT(onActivated(QModelIndex)));
}

void FileListView::keyPressEvent(QKeyEvent *e)
{
    switch (e->key()) {
        case Qt::Key_Up:
        case Qt::Key_Down:
        case Qt::Key_Enter:
            QListView::keyPressEvent(e); break;

        case Qt::Key_Left:
            if (this->viewMode() == QListView::IconMode)
                QListView::keyPressEvent(e);
            else
                openPath("..");
            break;

        case Qt::Key_Right:
            if (this->viewMode() == QListView::IconMode)
                QListView::keyPressEvent(e);
            else if (QFileInfo(this->currentIndex().data(FileSystemModel::PathRole).toString()).isDir())
                onActivated(this->currentIndex());
            break;

        default:
            e->ignore();
    }
}

void FileListView::enableGridMode(bool enable)
{
    if (enable) {
        this->itemDelegate()->deleteLater();
        this->setItemDelegate(new GridDelegate(this));

        this->setViewMode(QListView::IconMode);
        this->setGridSize(QSize(142,80));
        this->setFlow(QListView::LeftToRight);
    } else {
        this->itemDelegate()->deleteLater();
        this->setItemDelegate(new DetailedDelegate(this));

        this->setViewMode(QListView::ListMode);
        this->setGridSize(QSize());
        this->setFlow(QListView::TopToBottom);
    }
}

void FileListView::resizeEvent(QResizeEvent *e)
{
    QListView::resizeEvent(e);
    this->setFlow(this->flow());
}

void FileListView::onActivated(QModelIndex index)
{
    openPath(index.data(FileSystemModel::PathRole).toString());
}

QString FileListView::currentPath()
{
    return filesystemModel->location();
}

void FileListView::setDetails(FileSystemModel::Detail firstDetail, FileSystemModel::Detail secondDetail)
{
    filesystemModel->setDetails(firstDetail, secondDetail);
}

void FileListView::openPath(QString path)
{
    if (!path.startsWith("/"))
        path = filesystemModel->location() + '/' + path;

    QFileInfo file(path);
    path = file.absoluteFilePath();

    if (file.isDir()) {
        qDebug() << "Changing directory" << path;
        filesystemModel->setLocation(path);
        emit locationChanged(path);

        this->clearSelection();
        this->scrollToTop();
    } else {
        qDebug() << "Opening file" << path;
        hildon_mime_open_file(dbus_g_connection_get_connection(dbus_g_bus_get(DBUS_BUS_SESSION, NULL)), path.toUtf8());
    }
}

void FileListView::contextMenuEvent(QContextMenuEvent *e)
{
    this->selectionModel()->select(this->currentIndex(), QItemSelectionModel::SelectCurrent);

    QMenu *contextMenu = new QMenu(this);
    contextMenu->setAttribute(Qt::WA_DeleteOnClose);

    contextMenu->addAction(tr("Details"), this, SLOT(examineCurrentItem()));
    contextMenu->addAction(tr("Clip"), this, SLOT(clipSelected()));
    contextMenu->addAction(tr("Open with..."), this, SLOT(openSelected()));
    /*contextMenu->addAction(tr("Copy to..."), this, SLOT(copySelected()));
    contextMenu->addAction(tr("Move to..."), this, SLOT(moveSelected()));
    contextMenu->addAction(tr("Link to..."), this, SLOT(linkSelected()));*/
    contextMenu->addAction(tr("Share"), this, SLOT(showShareDialog()));
    contextMenu->addAction(tr("Delete"), this, SLOT(deleteSelected()));

    contextMenu->exec(e->globalPos());
}

void FileListView::enableHidden(bool enable)
{
    filesystemModel->enableHidden(enable);
}

void FileListView::enableThumbnails(bool enable)
{
    filesystemModel->enableThumbnails(enable);
}

void FileListView::enableLocationItem(bool enable)
{
    filesystemModel->enableDotDot(enable);
}

void FileListView::enableMultiSelection(bool enable)
{
    if (enable) {
        disconnect(this, SIGNAL(activated(QModelIndex)), this, SLOT(onActivated(QModelIndex)));
        this->setSelectionMode(QAbstractItemView::MultiSelection);
        this->setSelectionRectVisible(true);
    } else {
        this->setSelectionMode(QAbstractItemView::SingleSelection);
        this->setSelectionRectVisible(false);
        this->clearSelection();
        connect(this, SIGNAL(activated(QModelIndex)), this, SLOT(onActivated(QModelIndex)), Qt::UniqueConnection);
    }
}

void FileListView::sortByName()
{
    QSettings().setValue("Browser/Sorting", QDir::Name);
    filesystemModel->sortByName();
}

void FileListView::sortByTime()
{
    QSettings().setValue("Browser/Sorting", QDir::Time);
    filesystemModel->sortByTime();
}

void FileListView::sortBySize()
{
    QSettings().setValue("Browser/Sorting", QDir::Size);
    filesystemModel->sortBySize();
}

void FileListView::sortByType()
{
    QSettings().setValue("Browser/Sorting", QDir::Type);
    filesystemModel->sortByType();
}

void FileListView::examineCurrentItem()
{
    FileDetailsDialog(this, this->currentIndex().data(FileSystemModel::PathRole).toString()).exec();
}

void FileListView::clipSelected()
{
    QItemSelectionModel *selection = this->selectionModel();

    if (selection->hasSelection()) {
        QFileInfoList files;
        QModelIndexList indices = selection->selectedIndexes();
        foreach (QModelIndex index, indices)
            files.append(index.data(FileSystemModel::PathRole).toString());

        Clipboard::get()->add(files);
        QMaemo5InformationBox::information(this, tr("Added %n item(s) to clipboard", "", files.size()));
    } else {
        QMaemo5InformationBox::information(this, tr("Selection is empty"));
    }
}

void FileListView::openSelected()
{
    QStringList files;
    QModelIndexList indices = this->selectionModel()->selectedIndexes();
    foreach (QModelIndex index, indices)
            files.append(index.data(FileSystemModel::PathRole).toString());

    ApplicationsWindow *applications = new ApplicationsWindow(this, files);
    applications->show();
}

void FileListView::deleteSelected()
{
    if (ConfirmDialog(this, ConfirmDialog::DeleteSelected).exec() == QMessageBox::Yes) {
        QFileInfoList files;
        QModelIndexList indices = this->selectionModel()->selectedIndexes();
        foreach (QModelIndex index, indices)
            files.append(index.data(FileSystemModel::PathRole).toString());

        OperationManager::get()->add(new FileOperation(FileOperation::Remove, files));
    }
}

void FileListView::showShareDialog()
{
    // Build a list of files to share, exclude directories
    QStringList files;
    QModelIndexList indices = this->selectionModel()->selectedIndexes();
    foreach (QModelIndex index, indices) {
        QFileInfo info(index.data(FileSystemModel::PathRole).toString());
        if (info.isFile())
            files.append(info.absoluteFilePath());
    }

    if (files.isEmpty())
        QMaemo5InformationBox::information(this, tr("There are no files to share"));
    else
        ShareDialog(files, this).exec();
}
