#include "filesystemmodel.h"

FileSystemModel::FileSystemModel(QObject *parent) :
    QSortFilterProxyModel(parent)
{
    qRegisterMetaType<QFileInfo>("QFileInfo");

    this->setSourceModel(model = new QStandardItemModel(this));
    this->setFilterCaseSensitivity(Qt::CaseInsensitive);

    watcher = new QFileSystemWatcher(this);
    connect(watcher, SIGNAL(directoryChanged(QString)), this, SLOT(reload()));

    firstDetail = Size;
    secondDetail = Date;

    enableDotDot(false);
    enableHidden(false);
    sortByName();
}

bool FileSystemModel::filterAcceptsRow(int sourceRow, const QModelIndex &sourceParent) const
{
    return model->index(sourceRow, 0, sourceParent).data(Qt::DisplayRole).toString() == ".." ||
           QSortFilterProxyModel::filterAcceptsRow(sourceRow, sourceParent);
}

QString FileSystemModel::location()
{
    return currentDir.absolutePath();
}

void FileSystemModel::setLocation(QString path)
{
    QStringList watchDirs = watcher->directories();
    if (!watchDirs.isEmpty())
        watcher->removePaths(watchDirs);

    currentDir.cd(path);
    watcher->addPath(currentDir.absolutePath());

    reload();
}

void FileSystemModel::setDetails(Detail firstDetail, Detail secondDetail)
{
    this->firstDetail = firstDetail;
    this->secondDetail = secondDetail;

    reload();
}

void FileSystemModel::reload()
{
    model->clear();

    if (showDotDot && currentDir.path() != "/") {
        QStandardItem *item = new QStandardItem();
        item->setText("..");
        item->setIcon(QIcon::fromTheme("filemanager_folder_up"));
        item->setData(QVariant::fromValue(QFileInfo("..")), FileInfoRole);
        model->appendRow(item);
    }

    if (currentDir.exists()) {
        currentDir.refresh();
        QFileInfoList entries = currentDir.entryInfoList();
        foreach (QFileInfo entry, entries) {
            QStandardItem *item = new QStandardItem();

            item->setText(entry.fileName());
            item->setIcon(showThumbnails ? thumbnail(entry) : icon(entry));

            item->setData(QVariant::fromValue(entry), FileInfoRole); // Deprecate this thing

            item->setData(detailString(entry, firstDetail), FirstDetailRole);
            item->setData(detailString(entry, secondDetail), SecondDetailRole);
            item->setData(entry.isSymLink(), SymLinkRole);
            item->setData(!entry.exists(), BrokenRole);

            model->appendRow(item);
        }
    }
}

void FileSystemModel::enableDotDot(bool enable)
{
    showDotDot = enable;
    reload();
}

void FileSystemModel::enableHidden(bool enable)
{
    showHidden = enable;
    rebuildFilter();
}

void FileSystemModel::enableThumbnails(bool enable)
{
    showThumbnails = enable;
    reload();
}

void FileSystemModel::rebuildFilter()
{
    QDir::Filters oldFilters = currentDir.filter();
    QDir::Filters newFilters = QDir::AllEntries | QDir::AllDirs | QDir::System | QDir::NoDotAndDotDot;

    if (showHidden) newFilters |= QDir::Hidden;

    if (newFilters != oldFilters) {
        currentDir.setFilter(newFilters);
        reload();
    }
}

void FileSystemModel::sortByName()
{
    currentDir.setSorting(QDir::DirsFirst | QDir::LocaleAware | QDir::IgnoreCase | QDir::Name);
}

void FileSystemModel::sortByTime()
{
   currentDir.setSorting(QDir::DirsFirst | QDir::LocaleAware | QDir::IgnoreCase | QDir::Time);
}

void FileSystemModel::sortBySize()
{
   currentDir.setSorting(QDir::DirsFirst | QDir::LocaleAware | QDir::IgnoreCase | QDir::Size);
}

void FileSystemModel::sortByType()
{
    currentDir.setSorting(QDir::DirsFirst | QDir::LocaleAware | QDir::IgnoreCase | QDir::Type);
}


QIcon FileSystemModel::icon(QFileInfo info)
{
        if (info.isDir()) {
            if (info.fileName() == "..")
                return QIcon::fromTheme("filemanager_folder_up");
            else
                return QIcon::fromTheme("general_folder");
        } else {
            gchar *g_mime = g_content_type_guess(info.fileName().toUtf8(), NULL, 0, NULL);
            QString mime = QString(g_mime);
            g_free(g_mime);

            return QIcon::fromTheme("gnome-mime-" + mime.replace('/', '-'),
                   QIcon::fromTheme(mime.replace('/', '-'),
                   QIcon::fromTheme("filemanager_unknown_file")));
        }
}

QIcon FileSystemModel::thumbnail(QFileInfo info)
{
    gchar* uri = g_filename_to_uri(info.absoluteFilePath().toUtf8(), NULL, NULL);
    if (uri) {
        gchar* file = hildon_thumbnail_get_uri(uri, 48, 48, true);
        g_free(uri);

        if (file) {
            QString thumbFile = QString::fromUtf8(file).mid(7); // "file://"
            g_free(file);

            if (QFileInfo(thumbFile).exists())
                return QIcon(thumbFile);
        }
    }
    return icon(info);
}

QString FileSystemModel::sizeString(qint64 size)
{
    const qint64 K = 1024;
    const qint64 M = 1024 * K;
    const qint64 G = 1024 * M;

    if (size >= G*10)
        return QString("%1 GiB").arg(size/G);
    else if (size >= M*10)
        return QString("%1 MiB").arg(size/M);
    else if (size >= K)
        return QString("%1 KiB").arg(size/K);
    else
        return QString("%1 B").arg(size);
}

QString FileSystemModel::detailString(QFileInfo info, Detail detail)
{
    switch (detail) {
        case Size:
            return info.isDir() ? tr("Directory") : sizeString(info.size());

        case Date:
            return info.lastModified().toString();
    }

    return QString();
}
