/* GStreamer Tag Reader interface
 * Copyright (C) 2009 Tim-Philipp Müller <tim centricular net>
 * Copyright (C) 2008 Nokia Corporation.
 *
 * Contact: Stefan Kost <stefan.kost@nokia.com>
 *
 * tagreader.h: Tag Reader interface design
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#ifndef __GST_TAG_READER_H__
#define __GST_TAG_READER_H__

#include <gst/gst.h>

G_BEGIN_DECLS

/**
 * GstTagReadingMode:
 * @GST_TAG_READING_MODE_DEFAULT: parse both metadata and audio/video stream
 *     data
 * @GST_TAG_READING_MODE_TAGS_ONLY: parse metadata and process only as much
 *     audio/video data as required to extract basic information about the
 *     stream such as tags, available streams and duration
 * @GST_TAG_READING_MODE_SKIP_TAGS: don't extract any metadata, just process
 *     the audio/video stream data (behaviour should generally be the same as
 *     with #GST_TAG_READING_MODE_DEFAULT, only that tag extraction is skipped)
 *
 * Available tag reading modes.
 *
 * Since: 0.10.24
 */
/* keep gst_tag_reading_mode_get_type() in sync when adding enums here */
typedef enum {
  GST_TAG_READING_MODE_DEFAULT = 0,
  GST_TAG_READING_MODE_TAGS_ONLY,
  GST_TAG_READING_MODE_SKIP_TAGS
} GstTagReadingMode;

#define GST_TYPE_TAG_READING_MODE  (gst_tag_reading_mode_get_type())
GType  gst_tag_reading_mode_get_type (void);


#define GST_TYPE_TAG_READER \
  (gst_tag_reader_get_type ())
#define GST_TAG_READER(obj) \
  (G_TYPE_CHECK_INSTANCE_CAST ((obj), GST_TYPE_TAG_READER, GstTagReader))
#define GST_IS_TAG_READER(obj) \
  (G_TYPE_CHECK_INSTANCE_TYPE ((obj), GST_TYPE_TAG_READER))
#define GST_TAG_READER_GET_IFACE(inst) \
  (G_TYPE_INSTANCE_GET_INTERFACE ((inst), GST_TYPE_TAG_READER, \
      GstTagReaderInterface))

/**
 * GstTagReader:
 *
 * Opaque #GstTagReader structure (dummy typedef to denote object instances
 * which implement the #GstTagReader interface).
 *
 * Since: 0.10.24
 */
typedef struct _GstTagReader GstTagReader;
typedef struct _GstTagReaderInterface GstTagReaderInterface;

/**
 * GstTagReaderInterface:
 *
 * Dummy interface structure. Elements do not need to implement any interface
 * methods, all they need to do is query the desired tag reading mode via
 * gst_tag_reader_get_tag_reading_mode() where and when appropriate/required.
 *
 * Since: 0.10.24
 */
struct _GstTagReaderInterface {
  GTypeInterface iface;

  /* methods: no methods to implement at this time */

  /* interfaces don't need padding, so no padding here */
};

GType              gst_tag_reader_get_type (void);

void               gst_tag_reader_add_interface_to_type (GType type);

/* public methods */

void               gst_tag_reader_set_tag_reading_mode (GstTagReader * reader,
                                                        GstTagReadingMode mode);

GstTagReadingMode  gst_tag_reader_get_tag_reading_mode (GstTagReader * reader);

G_END_DECLS

#endif /* __GST_TAG_READER_H__ */

